// SSLeayServerSocket.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeayServerSocket.java,v 1.4 1999/02/20 19:49:53 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.net.*;
import javax.net.ssl.*;
import java.lang.reflect.*;

/**
 * Class SSLServerSocket is a implementation of the <code>SSLSocket</code>
 * abstract class. A <code>SSLeayServerSocket</code> is used to create
 * <code>SSLSocket</code> objects through its <code>accept</code> method.
 * These sockets are the <em>server side SSLSockets</em>. The server side
 * SSL Sockets are usual <code>SSLSockets</code>, except they have different
 * SSL protocol policies (i.e. they may not require client authentication).
 * <p>
 * Several <code>SSLServerSocket</code> methods may be employed to control
 * the type and behavior of the server side sockets created by <code>accept</code>.
 * These are:
 * <ol>
 * <li>setSocketPrototype -- specify a socket prototype; sockets of the same class
 * will be created by accept.
 * <li>setNeedClientAuth
 * <li>setUseClientMode
 * <li>setEnabledCipherSuites
 * <li>setEnableSessionCreation
 * </ol>
 * These settings will be <em>forwarded</em> to the created server side
 * sockets, which have similar methods.
 * <p>
 * The method <code>setEnableAcceptHandshake</code> controls whether the
 * created server side sockets delivered by <code>accept</code> are
 * already in the handshaking sequence or not.
 * <p>
 * The server side sockets are SSLSockets and thus need a
 * <code>SSLeaySessionContext</code> for their sessions. Every
 * <code>SSLeayServerSocket</code>
 * uses a <code>SSLeaySessionContext</code> for the server side sockets it creates.
 * The <code>SSLeaySessionContext</code> can be specified only when constructing a
 * <code>SSLeayServerSocket</code>.
 *
 * @revison	$Revision: 1.4 $
 * @author	Andrei Popovici
 */
public
class SSLeayServerSocket extends SSLServerSocket{

  /// the context used for connections
  protected SSLeaySessionContext context;

  /// The sockets produced by <code>accept</code>
  protected Class   socketPrototypeClass;

  private boolean  useClientMode         = false;
  private boolean  needClientAuth        = false;
  private boolean  enableSessionCreation = true;
  private String[] enabledCipherSuites   = null;
  private boolean  enableAcceptHandshake = true;

 /** Create a TCP server socket on a port, using the default
   * authentication context. The connection backlog defaults to
   * fifty connections queued up before the system starts to reject new connection requests.
   *
   * @param port  the port number, or <code>0</code> to use any free port.
   * @param ctx   the <code>SSLeaySessionContext</code> for the server side sockets created
   * by this server socket
   * @exception  IOException  if an I/O error occurs when opening the socket.
   * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>.
   *
   */
  public SSLeayServerSocket(int port, SSLeaySessionContext ctx) throws IOException
    {
      this(port, 50, null,ctx);
    }

  /** Create a TCP server socket on a port, using the default
    * authentication context and a specified backlog of
    * connections.
    *
    * @param      port     the specified port, or <code>0</code> to use any free port.
    * @param      backlog  the maximum length of the queue.
    * @param ctx   the <code>SSLeaySessionContext</code> for the server side sockets created
    * by this server socket
    * @exception  IOException  if an I/O error occurs when opening the socket.
    * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>.
    */
  public SSLeayServerSocket(int port, int backlog, SSLeaySessionContext ctx)
    throws IOException {
        this(port, backlog, null,ctx);
    }

  /** Create a TCP server socket on a port, using the default
    * authentication context and a specified backlog of
    * connections as well as a particular specified network interface.
    * This constructor is used on multihomed hosts, such as those used
    * for firewalls or as routers, to control through which interface
    * a network service is provided.
    * <p>
    * The handshake of the accepted connections will be performed by default
    * in server mode. Use the <code>setUseClientMode</code> method
    * to change this behavior.
    *
    * @param port the local TCP port
    * @param backlog the listen backlog
    * @param bindAddr the local InetAddress the server will bind to
    * @param ctx   the <code>SSLeaySessionContext</code> for the server side sockets created
    * by this server socket
    * @exception IOException  if an I/O error occurs when opening the socket.
    * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>.
    */
  public SSLeayServerSocket(int port, int backlog, InetAddress bindAddr, SSLeaySessionContext ctx)
    throws IOException
    {
      super(port,backlog,bindAddr);
      if (ctx == null)
	{ throw new IllegalArgumentException("SSLeayServerSocket.init:ctx is null");}
      context = ctx;
      enabledCipherSuites   = ctx.getDefaultCipherSuites();
      enableSessionCreation = true;
      useClientMode         = false;
      needClientAuth        = false;
      setSocketPrototype(new SSLeaySocket(ctx));
    }

  /** Listens for a connection to be made to this socket and accepts
    * it. The method blocks until a connection is made. If accept handshake is
    * enabled (method <code>enableAcceptHandshake</code>) the created
    * socket will negotiate a <code>SSL</code> connection. Depending
    * the value of <code>useClientMode</code> the connection will
    * be negotiated in server mode (<code>useClientMode=false</code> or
    * in client mode <code>useClientMode=true</code>.
    * <p>
    * The type of returned socket depends on the prototype specified
    * in the <code>setSocketPrototype</code> method.
    *
    * @return a ssl-connected socket using a SSL connection
    * @exception IOException if server side socket could not be created,
    * or if the handshake (if enabled) failed
    */
  public Socket accept() throws IOException
    {
      Class[] paramTypes = { context.getClass() };
      Object[] params    = {context};
      SSLeaySocket s     = null;
      String errorMsg    = "SSLeayServerSocket.accept():";

      try
      	{
	  s = (SSLeaySocket)socketPrototypeClass.
	    getConstructor(paramTypes).newInstance(params);
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
       catch (InvocationTargetException e3)
 	{
 	  handleInvocationException(e3);
 	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}


      super.implAccept(s);
      s.setUseClientMode(useClientMode);
      s.setNeedClientAuth(needClientAuth);
      s.setEnableSessionCreation(enableSessionCreation);
      s.setEnabledCipherSuites(enabledCipherSuites);
      if(enableAcceptHandshake)
	s.startHandshake();
      return s;
    }

  private void handleInvocationException(InvocationTargetException e)
    throws IOException
    {
      Throwable e1 = e.getTargetException();
      if((new IOException()).getClass().isInstance(e1))
	throw (IOException)e1;
      else  throw new Error(e1.getMessage());
    }


  /** Controls whether accepted connections are in the (default) SSL
    * server mode, or the SSL client mode.
    *
    * @param mode - <code>true</code> if newly accepted connections
    *    should use SSL client mode.
    */
  public void setUseClientMode(boolean mode)
    {
      useClientMode = mode;
    }

  /** Returns <code>true</code> if accepted connections will be in SSL
    * client mode.
    *
    * @return <code>true</code> if the connection should use SSL client mode.
    */
  public boolean getUseClientMode()
    {
      return useClientMode;
    }

  /** Controls whether the connections which are accepted must include
    * client authentication.By default, clients do not need to provide
    * authentication information .
    * This control requires the connection to be in the (default) SSL
    * server mode.
    *
    * @param flag - true if the clients must authenticate themselves.
    */
  public void setNeedClientAuth(boolean flag)
    {
      needClientAuth = flag;
    }
  /** Returns true if client authentication is required on newly
    * accepted connection
    */
  public boolean getNeedClientAuth()
    {
      return needClientAuth;
    }

  /* Returns the names of the cipher suites which could be enabled for use
   * on an SSL connection. Normally, only a subset of these will actually
   * be enabled by default, since this list may include cipher suites which
   * do not meet quality of service requirements for those defaults. Such
   * cipher suites are useful in specialized applications.
   *
   * @return an array of cipher suite names
   */
    public String[] getSupportedCipherSuites()
    {
      return context.getSupportedCipherSuites();
    }

  /* Controls which particular SSL cipher suites are enabled for use by
   * accepted connections. The cipher suites must have been listed by
   * getSupportedCipherSuites() as being supported. Suites which require
   * authentication information which is not available in this
   * ServerSocket's authentication context will not be used in any case,
   * even if they are enabled.
   *
   * @param suites Names of all the cipher suites to enable;
   * <code>null</code> means to accept system defaults.
   */

  public void setEnabledCipherSuites(String[] ciphers)
    {
      enabledCipherSuites = ciphers;
    }

  /*
   * Returns the list of cipher suites which are currently enabled for use
   * by newly accepted connections. If this has not been explicitly
   * modified, a system-provided default guarantees a minimum quality of
   * service in all enabled cipher suites. That default is accessible
   * through the SSLSocket API.
   * <p>
   * There are several reasons why an enabled cipher suite might not
   * actually be used. For example: the server socket might not have
   * appropriate private keys available to it or the cipher suite might be
   * anonymous, precluding the use of client authentication, while the
   * server socket has been told to require that sort of authentication.
   *
   * @return an array of cipher suite names
   */
  public String[] getEnabledCipherSuites()
    {
      return enabledCipherSuites;
    }

  /**
    * Controls whether new SSL sessions may be established by  the
    * server side sockets created by this socket.
    *
    * @param flag <code>true</code> the server side sockets are to
    * establish new connections.
    */
  public void setEnableSessionCreation(boolean flag)
    {
      enableSessionCreation = flag;
    }

  /**
   * Returns true if new SSL sessions may be established by the
   * server side sockets created by this socket.
   *
   * @return <code>true</code> if the server side sockets are to
    * establish new connections.
   */
  public boolean getEnableSessionCreation()
    {
      return enableSessionCreation;
    }

  /** Controls whether the last action of the <code>accept</code>
    * method is the handshake initiation of the newly created server side sockets.
    *
    * @param flag if <code>true</code>, the last action in accept is
    * <code>startHandshake</code> for the new server side socket
    */
  public void setEnableAcceptHandshake(boolean flag)
    {
      enableAcceptHandshake = flag;
    }

  /* Returns <code>true</code> id last action of the <code>accept</code>
    * method is the handshake initiation of the newly created server side sockets.
    *
    * @return <code>true</code> if the last action in accept is
    * <code>startHandshake</code> for the new server side socket.
    */
  public boolean getEnabelAcceptHandshake()
    {
      return enableAcceptHandshake;
    }

  /** Set the prototype of sockets the <code>accept</code> methods produces.
    * This is actually a template method, which enables further extensions
    * of <code>SSLeayServerSocket</code> to accept connections using other
    * type of sockets than the default.
    * <p>
    * This method should be used in constructors.
    *
    * @param proto an unconnected <code>SSLeaySocket</code>
    * @exception IllegalArgumentException <code>proto</code> is <code>null</code>
    */
  protected  void setSocketPrototype(SSLeaySocket proto)
    {
      if (proto == null)
	{
	  throw new IllegalArgumentException("setSocketPrototype(null)");
	}
      socketPrototypeClass=proto.getClass();
    }

  /** Return the class of the prototype this server socket uses to
    * accept connections. Coresponds to the class of the prototype
    * specified through the <code>setSocketPrototype</code> method.
    *
    * @return the class of the sockets produced by <code>accept</code>.
    */
  public Class getSocketPrototypeClass()
    {
      return socketPrototypeClass;
    }
 }


//======================================================================
//
// $Log: SSLeayServerSocket.java,v $
// Revision 1.4  1999/02/20 19:49:53  apopovic
//  - methods 'set/getSocketPrototype(Class)' added
//  - 'accept' changed to enable using of socket prototypes
//
// Revision 1.3  1999/02/13 15:31:19  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.2  1999/01/27 18:47:53  apopovic
// Documentation spell-check
//
// Revision 1.1  1999/01/08 10:42:37  apopovic
// Initial Revision
//
//
