// SSLeayServerSocketFactory.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeayServerSocketFactory.java,v 1.6 1999/02/21 14:21:19 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.lang.reflect.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.net.*;
import javax.net.ssl.*;
import java.security.*;

/**
 * Class SSLeayServerSocketFactory creates <code>SSLeayServerSocket</code>s.
 * Each <code>SSLeayServerSocketFactory</code> uses exaclty one
 * <code>SSLeaySessionContext</code>. Every socket created using this factory
 * will use this <code>SSLeaySessionContext</code> for the sessions
 * corresponding to its own connection. The default ciphers which are
 * going to be used by the sockets created by this factory are actually
 * default ciphers of the <code>SSLeaySessionContext</code> of this
 * factory.
 *
 * <h4> Session Context Instantition </h4>
 * <p>
 * The <code>SSLeaySessionContext</code> is instantiated "lazy", namely in
 * the  first call of one of the following methods:
 * <ul>
 * <li> the <code>createSocket</code> methods
 * <li> the <code>get/setEnabledCipherSuites</code> methods
 * <li> the <code>getSupportedCipherSuites</code>
 * </ul>
 * This property is very useful when serializing and deserializing
 * <code>SSLeaySocketFactory</code> objects. If the context would have
 * been initialized upon creation, the deserialization of a factory would
 * work only in the presence of cryptographic information, like certificate
 * files. Using "lazy" instantiation for the used context means the context
 * (and thus the capability of produces sockets) will be created only
 * when somebody realy wants to use a factory object for socket creation.
 * <P>
 * The methods specified above will fail by throwing
 * a <code>SSLException</code> if the necessary information for setting
 * up the context is not there. Possible fail reasons:
 * <ol>
 * <li> The CA information is not present or incomplete, so
 * the created socket is could not verify its peer.
 *
 * <li> If the system property <code>iti.ssl.serverIdentityRequired</code>
 * is <code>true</code>, but neither of the environment variables or
 * system properties controlling the user identity are defined (See
 * <code>SSLeaySessionContext</code>). In this case, a
 * <code>SSLeayIdentity</code> will be thrown. For any other
 * value of the property, no exception will be raised. By default,
 * the <code>iti.ssl.serverIdentityRequired</code> property has the
 * value <code>true</code>. This means server socket factories require
 * the user's identity by default.
 * <P>
 *  Secure connections
 * <em>can</em> be created even if this information is not available, but
 * the peer will not be able to check our identity. This exception is
 * catchable; the new socket factory will create sockets.
 * <code>false</code>.
 * </ol>
 * <p>
 * The <code>createServerSocket</code> methods <em>do not</em> create connections
 * between sockets. You have to use the <code>accept</code> method of the
 * created server sockets to create SSL connections.
 * <p>
 * <h4>Extending and customizing <code>SSLeayServerFactory</code><h4>
 * <p>
 * Using  the <code>setSSLPolicy</code> method you can specify the modes
 * of the connections obtained through the  <code>accept</code> method.
 * This method is a <em>template method</em>, which you can redefine
 * in subclasses of this class in order to specify different policies
 * for the server sockets of this factory. <code>setSSLPolicy</code> is called
 * <em>after</em> creating and server socket in all <code>createSocket</code>
 * methods.
 * <p>
 * The <code>setSocketPrototypeClass</code> specifies the type of server sockets
 * created by the <code>createServerSocket</code> methods.
 *
 * @version	$Revision: 1.6 $
 * @author	Andrei Popovici
 */
public
class SSLeayServerSocketFactory extends SSLServerSocketFactory implements Serializable {

  /* The context used to store the sockets of this factory */
  transient protected SSLeaySessionContext context;

  protected Class         socketType;
  private boolean userIdentitySet = false;

  private Class STRING_CLASS;
  private Class CONTEXT_CLASS;
  private Class INETADDRESS_CLASS;

  /** Create a new <code>SSLeayServerSocketFactory</code> using a new
    *  <code>SSLeaySessionContext</code>.The instantiation of the
    * factory will be performed lazy, upon the first call of a communication
    * method.

    *
    * @exception SSLeayRuntimeException no identity information
    * @exception Error no CA information
    *
    */
  public SSLeayServerSocketFactory()
    {
      // the contructor 'constants'
      try
	{
	  STRING_CLASS      = Class.forName("java.lang.String");
	  CONTEXT_CLASS     = Class.forName("de.tu_darmstadt.sp.ssl.SSLeaySessionContext");
	  INETADDRESS_CLASS = Class.forName("java.net.InetAddress");
	  socketType        = Class.forName("de.tu_darmstadt.sp.ssl.SSLeayServerSocket");
	}
      catch(Exception e)
	{
	  throw new Error("SSLeayServerSocketFactory.init():" +
			  e.getMessage());
	}
    }

  /** create the context -- runtime exception if failed -- first call the real one, rest = nops*/
  private synchronized  void initContext()
    {
      if (context!=null)
	return;
      try
	{
	  context = new SSLeaySessionContext();
	}
      catch (SSLException e)
	{
	  throw new SSLeayRuntimeException("Could not create context");
	}
    }

  /** set the user identtity */
  private synchronized void setUserIdentity() throws SSLeayIdentityException
    {
      if (userIdentitySet)
	return;
      try
	{
	  context.setUserIdentity();
	}
      catch (SSLeayIdentityException e2)
	{
	  String idReq =
	    System.getProperties().getProperty("iti.ssl.serverIdentityRequired",
					   "true");
	  if (!idReq.equals("false"))
	    throw e2;
	}
      userIdentitySet=true;
    }

  /** Create a new <code>SSLeayServerSocketFactory</code>  using
    *  <code>context</code> as the <code>SSLeaySessionContext</code>
    * for all connection initiated by the server sockets created by this factory.
    *
    * @param context the SessionConxtext of all connections of this factory
    */
  public SSLeayServerSocketFactory(SSLeaySessionContext context)
    {
      // the contructor 'constants'
      try
	{
	  STRING_CLASS      = Class.forName("java.lang.String");
	  CONTEXT_CLASS     = Class.forName("de.tu_darmstadt.sp.ssl.SSLeaySessionContext");
	  INETADDRESS_CLASS = Class.forName("java.net.InetAddress");
	  socketType        = Class.forName("de.tu_darmstadt.sp.ssl.SSLeayServerSocket");
	}
      catch(Exception e)
	{
	  throw new Error("SSLeayServerSocketFactory.init(context):" +
			  e.getMessage());
	}

      // initialize opaque pointer
      if (context == null)
	throw new IllegalArgumentException("context cannot be null in SSLeayFactory constructor");
      this.context = context;
    }

  private void handleInvocationException(InvocationTargetException e)
    throws IOException
    {
      Throwable e1 = e.getTargetException();
      if((new IOException()).getClass().isInstance(e1))
	throw (IOException)e1;
      else  throw new Error(e1.getMessage());
    }

  /* Returns a server socket which uses all network interfaces on the
   * host, and is bound to the specified port.
   *
   * @param      port  the port number, or <code>0</code> to use any
   *                   free port.
   * @exception IOException  if an I/O error occurs when opening the socket.
   * @exception SSLException could not create the session context for this factory
   * @exception SSLeayIdentityException could not set the identity of the user
   */
  public ServerSocket createServerSocket(int port)
    throws IOException
    {
      // initialize opaque pointer
      initContext();
      setUserIdentity();

      Class[] paramTypes = {Integer.TYPE,CONTEXT_CLASS};
      Object[] params    = {new Integer(port), context};
      SSLeayServerSocket s = null;
      String errorMsg = "SSLeayServerSocketFactory.createSeverSocket(int):";

      try
      	{
	  s = (SSLeayServerSocket)socketType.getConstructor(paramTypes).newInstance(params);
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
       catch (InvocationTargetException e3)
 	{
 	  handleInvocationException(e3);
 	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}

      setSSLPolicy(s);
      return s;
    }

  /* Returns a server socket which uses all network interfaces on the
   * host, is bound to a the specified port, and uses the specified
   * connection backlog.
   *
   * @param      port     the specified port, or <code>0</code> to use
   *                      any free port.
   * @param      backlog  the maximum length of the queue.
   * @exception IOException  if an I/O error occurs when opening the socket.
   * @exception SSLException could not create the session context for this factory
   * @exception SSLeayIdentityException could not set the identity of the user
   */
  public ServerSocket createServerSocket(int port, int backlog)
    throws IOException
    {
      // initialize context
      initContext();
      setUserIdentity();

      Class[] paramTypes = {Integer.TYPE,Integer.TYPE,CONTEXT_CLASS};
      Object[] params    = {new Integer(port),new Integer(backlog),context};
      SSLeayServerSocket s = null;
      String errorMsg =
	"SSLeayServerSocketFactory.createSeverSocket(int,int,context):";

      try
      	{
	  s = (SSLeayServerSocket)socketType.getConstructor(paramTypes).newInstance(params);
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
       catch (InvocationTargetException e3)
 	{
	  handleInvocationException(e3);
 	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}

      setSSLPolicy(s);
      return s;
    }

  /* Returns a server socket which uses only the specified network
   * interface on the local host, is bound to a the specified port, and
   * uses the specified connection backlog.
   *
   * @param port the local TCP port
   * @param backlog the listen backlog
   * @param bindAddr the local InetAddress the server will bind to
   * @exception IOException  if an I/O error occurs when opening the socket.
   * @exception SSLException could not create the session context for this factory
   * @exception SSLeayIdentityException could not set the identity of the user
   */
  public ServerSocket createServerSocket(int port,int backlog,
					 InetAddress bindAddr)
     throws IOException
    {
      // initialize context
      initContext();
      setUserIdentity();

      // create socket using Reflection
      Class[] paramTypes =
      {Integer.TYPE,Integer.TYPE,INETADDRESS_CLASS,CONTEXT_CLASS};
      Object[] params    =
      {new Integer(port),new Integer(backlog), bindAddr, context};
      SSLeayServerSocket s=null;
      String errorMsg =
	"SSLeayServerSocketFactory.createSeverSocket(int,int,InetAddr,ctx):";

      try
      	{
	  s = (SSLeayServerSocket)socketType.
	    getConstructor(paramTypes).newInstance(params);
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
       catch (InvocationTargetException e3)
 	{
	  handleInvocationException(e3);
 	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}

      setSSLPolicy(s);
      return s;
    }
  /* Returns the list of cipher suites which are enabled by default.
   * Unless a different list is enabled, handshaking on an SSL connection
   *  will use one of these cipher suites. The minimum quality of service
   * for these defaults requires confidentiality protection and server
   * authentication.
   *
   * @return a <code>String</code> array representing the default ciphers,
   * in order of their preference
   */
  public  String[] getDefaultCipherSuites()
    {
      // initialize context
      initContext();

      return context.getDefaultCipherSuites();
    }

  /* Returns the names of the cipher suites which could be enabled for use
   * on an SSL connection created by this factory. Normally, only a subset
   * of these will actually be enabled by default, since this list may
   * include cipher suites which do not meet quality of service
   * requirements for those defaults. Such cipher suites are useful in
   * specialized applications.
   *
   * @return a <code>String</code> array representing the supported ciphers
   */
  public  String[] getSupportedCipherSuites()
    {
      // initialize context
      initContext();
      return context.getSupportedCipherSuites();
    }
 /* Specify the list of cipher suites which are enabled by default.
   * Handshaking on an SSL connectionwill use one of these cipher
   *suites. The minimum quality of service for these defaults
   * requires confidentiality protection and server  authentication.
   *
   * @param ciphers <code>String</code> array consisting of the new
   * default ciphers, in order of their preference
   */
  public void setDefaultCipherSuites(String[] ciphers)
    {
      // initialize context
      initContext();
      context.setDefaultCipherSuites(ciphers);
    }

  /** This is a template method called in the <code>createSocket</code>
    * methods <em>after</em> creating and binding a
    * <code>SSLeayServerSocket</code>.
    * <p>
    * Subclasses of <code>SSLeayServerSocketFactory</code> can redefine this
    * method to specify alternative SSL configuration policies for the
    * SSLeayServersockets and thus for the server side sockets.
    * can be specified using methods of <code>SSLSocket</code> i.e.
    * <ul>
    * <li> <code>setEnabledCipherSuites</code>
    * <li> <code>setNeedClientAuth</code>
    * <li> <code>setUseClientMode</code>
    * <li> <code>setEnableAcceptHandshake</code>
    * <li> <code>setEnableSessionCreation</code>
    * </ul>
    */
  protected void setSSLPolicy(SSLeayServerSocket s)
    {
    }

  /** Specify the type of sockets (class) returned by the
    * <code>createServerSocket</code> methods. The class has to be a
    * subclass of <code>SSLeayServerSocket</code>. The constructors of
    *  <code>cls</code> have to be a superset of the
    *  <code><code>SSLeaySocket</code>'s constructors.
    *
    * @param cls the class of the sockets to create
    * @exception IllegalArgumentException <code>cls</code> is not a subclass of
    * <code>SSLeaySocket</code> or constructor mismatch between <code>cls</code> and
    * <code>SSLeaySocket</code>.
    */

  protected synchronized void setServerSocketClass(Class cls)
    {
      if (cls == null)
	{
	  throw new IllegalArgumentException("setSocketPrototypeClass(null)");
	}
      socketType = cls;
      // class check: TODO
    }

  /** Return the type of sockets created by this factory. The returned
    * <code>Class</code> object will be a subclass of <code>SSLeayServerSocket</code>.
    *
    * @return the class of the objects returned by <code>createSocket</code>
    */
  public Class getServerSocketClass()
    {
      return socketType;
    }

  private void readObject(java.io.ObjectInputStream in)
     throws IOException, ClassNotFoundException
    {
      in.defaultReadObject();
      context=null;
      userIdentitySet=false;
    }

}


//======================================================================
//
// $Log: SSLeayServerSocketFactory.java,v $
// Revision 1.6  1999/02/21 14:21:19  apopovic
// system properties renamed to initial 'iti.ssl..' insted of the new 'de.tu_darmstadt..'
//
// Revision 1.5  1999/02/20 19:40:53  apopovic
// - serialization support added
// - initContext split into initContext, setUserIdentity and moved into
//   the createServerSocket Procedures
// - 'setSocketType' replaced with 'setSocketPrototypeClass'
// - 'getSocketType' replaced with 'getSocketPrototypeClass'
// - Bug Fix: INT_CLASS replaced with Integer.TYPE -- was not serializable
// - Documentation update
//
// Revision 1.4  1999/02/13 15:31:19  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.3  1999/01/27 18:51:26  apopovic
// - 'createServerSocket' methods changed to use reflection when creating object
// - support for dynamic specification of socket Types added
// - methods 'setSocketType' and 'getSocketType' added
// - documentation spell-check
//
// Revision 1.2  1999/01/22 09:37:09  apopovic
// Minor tag fix; context.setUserIdentity now used in constructor
//
// Revision 1.1  1999/01/08 10:42:37  apopovic
// Initial Revision
//
