/* SSLeaySession.c -- ITISSL Program */
/* ITISSL - a Java 2 implementation for  Sun's reference SSL API  using SSLeay
 * Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* $Id: SSLeaySession.c,v 1.5 1999/02/21 14:26:04 apopovic Exp $
 *----------------------------------------------------------------------
 * Written by Andrei Popovici.
 */

/*
 * Implementation for the 'SSLeaySession' native methods
 *
 * (history at end)
 */


#include "SSLeaySession.h"
#include "itissl.h"
/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    getPeerCertificateChainString
 * Signature: ()[Ljava/lang/String;
 */
JNIEXPORT jobjectArray JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_getPeerCertificateChainString
  (JNIEnv * env, jobject this)
{

  /* variables */
  SSL_SESSION* sess;
  int          cert_chain_len;
  STACK*       cert_chain;
  X509*        cert;
  X509*        peer_cert;
  jarray       result;
  jsize        i;
  jstring      string_cert;
  jclass       string_class;

  TRACE("SSLeaySession_getPeerCertificateChainString: Entering\n");

  /* initialization */
  sess           = ITISSL_get_Session_session(env,this);
  cert_chain     = SSL_SESSION_get_cert_chain(sess); /** FIXME */
  peer_cert      = SSL_SESSION_get_peer_certificate(sess);

  if (cert_chain == NULL)
      cert_chain_len = 0;
  else
      cert_chain_len = sk_num(cert_chain);

  if (peer_cert != NULL)
      cert_chain_len++;

  string_class   = (*env)->FindClass(env,"java/lang/String");
  result         = (*env)->NewObjectArray(env,
					  cert_chain_len,
					  string_class,
					  NULL);
  ITISSL_assert(env,
		(result!=NULL),
		"cannot alloc array");
  TRACE("  Initialization done..\n");

  /* check if we have work to do.. */
  if (cert_chain_len == 0)
      return NULL;

  /* the first array element is the certificate of the peer */
  string_cert = ITISSL_X509_to_jstring(env,peer_cert);
  (*env)->SetObjectArrayElement(env,
				result,
				0,
				string_cert);

  TRACE("  Peer certificate created\n");
  /* create the jarray */
  if (cert_chain != NULL)
    {
      for(i=1; i<cert_chain_len; i++)
	{
	  /* print certs into buffer */
	  cert = (X509*)sk_value(cert_chain,i);
	  string_cert  = ITISSL_X509_to_jstring(env,cert);
	  (*env)->SetObjectArrayElement(env,
					result,
					i,
					string_cert);
	}
    }

  TRACE("SSLeaySession_getPeerCertificateChainString: Leaving\n");
 return result;

}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    getId
 * Signature: ()[B
 */
JNIEXPORT jbyteArray JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_getId
  (JNIEnv * env, jobject this)
{
    SSL_SESSION* sess;
    jbyteArray   result;
    jbyte*       resultElems;
    int i;
    unsigned char* session_id;

    TRACE("SSLeaySession_getId: Entering\n");
    /* get the real SSLeay structure */
    sess = ITISSL_get_Session_session(env,this);

    /* create the java result array */
    result = (*env)->NewByteArray(env,
				 (jsize)SSL_SESSION_get_session_id_len(sess));
    resultElems       = (*env)->GetByteArrayElements( env, result , NULL );

    /* copy the id into our element array */
    session_id = SSL_SESSION_get_session_id(sess);
    ITISSL_error(env,
		 (session_id!=NULL),
		 "Session id missing in session object)");
    for (i=0; i<SSL_SESSION_get_session_id_len(sess); i++)
	resultElems[i] = session_id[i];

    /* free our element array (copy back into java Object) */
    (*env)->ReleaseByteArrayElements( env, result, resultElems, 0 );
    TRACE("SSLeaySession_getId: Leaving\n");
    return result;
}



/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    getCreationTime
 * Signature: ()J
 * Returns the time in s since epoch this session was created at.
 */
JNIEXPORT jlong JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_doGetCreationTime
  (JNIEnv * env, jobject this)
{
   SSL_SESSION* sess;
   TRACE("SSLeaySession_getCreationTime: Entering\n");
   sess = ITISSL_get_Session_session(env,this);
   TRACE1("  secconds since epoch: %d\n",sess->time);
   TRACE("SSLeaySession_getCreationTime: Leaving\n");
   return (jlong) (SSL_get_time(sess));
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    getLastAccessedTime
 * Signature: ()J
 * Dummy implementation, returnes the session creation time, since
 * last acces not present in SSLeay.
 */
JNIEXPORT jlong JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_doGetLastAccessedTime
  (JNIEnv * env, jobject this)
{
   SSL_SESSION* sess;
   TRACE("SSLeaySession_getLastAccessedTime: Entering\n");
   sess = ITISSL_get_Session_session(env,this);
   TRACE("SSLeaySession_getLastAccessedTime: Leaving\n");
   return (jlong) (SSL_get_time(sess));
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    doInvalidate
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_doInvalidate
   (JNIEnv * env, jobject this)
{
  SSL_SESSION* sess;
  SSL_CTX*     context;
  TRACE("SSLeaySession_doInvalidate: Entering\n");

  sess = ITISSL_get_Session_session(env,this);
  context = ITISSL_get_Session_context(env,this);

  SSL_CTX_remove_session(context,sess);
  TRACE("SSLeaySession_doInvalidate: Leaving\n");
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    getCipherSuite
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_getCipherSuite
  (JNIEnv * env, jobject this)
{
  SSL_SESSION* sess;
  char *       utfCipherName;
  jstring      result;
  TRACE("SSLeaySession_getCipherSuite: Entering\n");

  sess = ITISSL_get_Session_session(env,this);
  utfCipherName = SSL_CIPHER_get_name(SSL_SESSION_get_cipher(sess));
  result = (*env)->NewStringUTF(env,utfCipherName);

  TRACE("SSLeaySession_getCipherSuite: Leaving\n");
  return result;
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySession
 * Method:    applicationData
 * Signature: ()Ljava/util/Hashtable;
 */
JNIEXPORT jobject JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySession_applicationData
  (JNIEnv * env, jobject this)
{
  /* variables */
  jobject   result;
  jclass    hashtable_class;
  jmethodID hashtable_constructor_ID;
  SSL_SESSION* sess;
  void*     application_data;
  jobject   new_application_data;
  TRACE("SSLeaySession_applicationData: Entering");

  /* initialization */
  sess = ITISSL_get_Session_session(env,this);
  application_data = SSL_SESSION_get_app_data(sess);

  /* if application data (hashtable) not present, create-it */
  if (application_data == NULL)
    {
      hashtable_class = (*env)->FindClass(env,"/java/util/Hashtable");
      hashtable_constructor_ID = (*env) ->GetMethodID(env,hashtable_class,
						      "<init>","Void(V)");
      result = (*env) -> NewObject(env,hashtable_class,
				   hashtable_constructor_ID);

      new_application_data  = (*env)->NewGlobalRef(env,result);
      SSL_SESSION_set_app_data(sess,((void*)new_application_data));

    }
  else
    result = (jobject)application_data;

  TRACE("SSLeaySession_applicationData: Leaving");
  return result;
}





/*======================================================================
 *
 * $Log: SSLeaySession.c,v $
 * Revision 1.5  1999/02/21 14:26:04  apopovic
 *  'doGetPeerCertificateStrings' fixed: it used to return a
 * void certificate array even if the peer sent the its own cert
 *
 * Revision 1.4  1999/02/20 19:47:15  apopovic
 * Adaption to time bugs in the java file
 *
 * Revision 1.3  1999/01/27 18:54:41  apopovic
 * Documentation spell-check
 *
 * Revision 1.2  1999/01/22 10:16:41  apopovic
 * Minor bug fixes, debug message enhancements.
 *
 * Revision 1.1  1999/01/08 10:41:13  apopovic
 * Initial Revision
 *
 */
