// SSLeaySessionContext.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeaySessionContext.java,v 1.6 1999/02/21 14:21:59 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import javax.net.ssl.*;

/**
 * Class SSLeaySessionContext is a grouping of SSLSessions associated
 * with a single entity. For example, it could be associated with a
 * server or client who participates in many sessions concurrently.
 * This interface provides methods for retrieving a SSLSession based
 *  on its ID, and allows such IDs to be listed.
 *
 * <h4>Implementation</h4>
 * A <code>SSLeaySessionContext</code> is just a proxy for the real session
 * cache of the underlying SSLeay layer. At a certain point , its
 * contents (list of sessions) may not be consistent with the existing sessions
 * in the underlying SSLeay layer. It rather offers a <em>snapshot</em> of the
 * existing sessions and their respective session Id's at a given point time in the past.
 *  The <code>update</code> method is used to re-read the session
 * information of the SSL_CTX structure. After a call to  <code>update</code>,
 * the information stored in this object and the one in the SSLeay layer are
 * consistent. Note that while navigating through the sessions in the
 * Session Context, new connections may be issued in the underlying layer.
 *
 * <p>
 * The <code>getIds</code>  method always updates the session cache view.
 *
 * @version	$Revision: 1.6 $
 * @author	Andrei Popovici
 */
public
class SSLeaySessionContext implements SSLSessionContext{


  /** the opaque pointer of this object */
  protected int context;

  /** The variable holding the view over the underlying session hash */
  protected Hashtable sessionHash;


  /** Create a new SSLeay Session object <em>together</em> with the
    * corresponding <code>SSL_CTX</code> structure in the underlying
    * SSLeay layer.
    * The initialization is performed using the following properties
    * or environment variables.
    * <ul>
    * <li> The environment variable <code>SSL_CIPHERS</code> and the
    * system property <code>de.tu_darmstadt.sp.ssl.ssl_ciphers</code> control the
    * default cipher suites of this context object. If both are <code>null</code>
    * the <em>SSLeay</em> defaults will be used.
    * <li> The environment variable <code>CA_FILE</code> and the
    * system property <code>iti.ssl.ca_file</code> control the location
    * of the Certification Agency file. This information not mandatory.
    * <li> The environment variable <code>CA_PATH</code> and the
    * system property <code>iti.ssl.ca_path</code> control the location
    * of the Certification Agency Pem-Format directory. This information not mandatory.
    * </ul>
    * <p> For all  cases described above, the property (if existent)
    * will override the environment variable. If all 4 values
    * are <code>null</code> or corresponding files are inexistent,
    * this context won't have an initial CA information, this is, sockets
    * created with it won't be able to verify peers. However, the socket
    * factory which created this context can specify the CA loations
    * using the <code>addTrustedCACerts</code> method.
    *
    * @exception SSLException could not create context
    */
  public  SSLeaySessionContext() throws SSLException
    {
      // get the cipher default cipher list and the ca-file and
      // setup system
      String cipher_list =
	System.getProperty("de.tu_darmstadt.sp.ssl.cipher_list");
      setup(cipher_list);

      String ca_file =
	System.getProperty("iti.ssl.ca_file");
      String ca_path =
	System.getProperty("iti.ssl.ca_path");

      try
	{
	  addTrustedCACerts(ca_file,ca_path);
	}
      catch (IOException e)
	{
	  // we allow the context to live, maybe the factory which
	  // created this context wants to specify different
	  // verification locations
	}
      sessionHash = new Hashtable();

    }

  /**
    * Initialize the context  with the identity of the user. The
    *  initialization is performed using the following properties or
    *  environment variables:
    * <ul>
    * <li> The environment variable <code>CERT_FILE</code> and the
    * system property <code>iti.ssl.cert_file</code> control the
    * location of the certificate of the user. If both variables are
    * <code>null</code> or if the specified file does not exist,
    * a <code>SSLeayIdentityException</code> will be thrown.
    * This exception <em>should</em> be caught if the context is used
    * for connection which do not authenticate themselves.
    * <li> The environment variable <code>KEY_FILE </code>and the
    * system property <code>iti.ssl.key_file</code> control the
    * location of the private key of the user. If both variables are
    * <code>null</code> or if the specified file does not exist,
    * a <code>SSLeayIdentityException</code> will be thrown.
    * This exception <em>should</em> be caught if the context is used
    * for connection which do not authenticate themselves.
    * </ul>
    * <p> For all  cases described above, the property (if existent)
    * will override the environment variable.
    *
    * @exception SSLeayIdentityException if the identity information
    * (certificate and private key) does not exist.
    */
  public void setUserIdentity() throws SSLeayIdentityException
    {
         // set identity
      String cert_file =
	System.getProperty("iti.ssl.cert_file");
      String key_file  =
	System.getProperty("iti.ssl.key_file");


      doSetUserIdentity(cert_file,key_file);
    }

  /** Set the identity of the current user in the underlying
    * <code>SSLeay</code> layer. This identity
    * will be used in all connections/sessions managed by this context.
    *
    * @param cert_file the certificate file to be used. If <code>null</code>,
    * the value of the <code>CERT_FILE</code>  environment variable
    * will be used instead.
    * @param key_file  the private key of the user to be used. If <code>null</code>,
    * the value of the <code>KEY_FILE</code> environment variable
    * will be used instead.
    * @exception SSLeayIdentityException if the certificate file or the
    * key file were not specified or inexistent
    */
  protected native void doSetUserIdentity(String cert_file, String key_file)
    throws SSLeayIdentityException;


  /** Create a proxy of the real SSLeay Session context. The SSLeay session
    * context is identified by the <code>context</code> opaque pointer.
    *
    * @param context opaque pointer, hiding a <code>SSL_CTX*</code>
    */
  protected  SSLeaySessionContext(int context)
    {
      this.context = context;
      sessionHash = new Hashtable();
    }

  /** Returns an Enumeration of all session id's.
    * Note that this methods first 'updates' the internal view of
    * existing sessions. This view can be queried later via the
    * <code>getSession</code> Method.
    *
    * @return a Enumeration of the session Id's currently in the cache.
    * Each element of the enumeration is a <code>byte[]</code> object.
    */
  public  Enumeration getIds()
    {
      update();
      return new SessionIdEnumeration(sessionHash.keys());
    }

  /** Returns the SSLSession bound to the specified session id, or
    * <code>null</code> if the specified session id does not refer to a valid SSLSession.
    * <p>
    * This method would reflect the context state corresponding to the last
    * call of either <code>getIds</code> or <code>update</code>
    *
    * @param sessionId a session Id
    * @return a <code>SSLeaySession</code> object, the session corresponding
    * to the specified session id
    */
  public synchronized SSLSession getSession(byte[] sessionId)
    {
         SSLeaySession sess =
	   (SSLeaySession)sessionHash.get(new SessionIdContainer(sessionId));
	 return sess;
    }

  /** Return a array consisting of the opaque pointers for all the existing
    * sessions in the underlying <code>SSL_CTX</code> structure.
    */
  native protected int[] getSessionOpaquePointers();

  /* Update the internal view of the underlying SSLeay Session Context.
   */
  public void  update()
    {
      int[] opaques = getSessionOpaquePointers();
      synchronized(sessionHash)
	{
	  sessionHash.clear();
	  for (int i= 0; i<opaques.length;i++)
	    {
	      SSLeaySession sess = new SSLeaySession(opaques[i],this);
	      sessionHash.put(new SessionIdContainer(sess.getId()),
			      sess);
	    }
	}
    }

  /** Return the cipher suites which will be used by default by all connections
    * created in this context.
    *
    * @return an array of string, each string representing a cipher name
    */
  native public String[] getDefaultCipherSuites();

  /** Return the cipher suites the current implementation supports.
    *
    * @return an array of string, each string representing a cipher name
    */
  native public String[] getSupportedCipherSuites();

  /** Set which cipher suites should be enabled by default for the connections
    * created in this context.
    *
    * @param ciphers the cipher suites to be enabled by default for all
    * connections created via this context. If <code>null</code>,
    * throw a <code>IllegalArgumentException</code>
    *
    * @param ciphers a list of strings, each string being a cipher name
    * @exception IllegalArgumentException if ciphers is <code>null</code>
    */
  public void setDefaultCipherSuites(String[] ciphers)
    {
      if (ciphers == null)
	throw new IllegalArgumentException("null value for default ciphers");

      StringBuffer strbuf = new StringBuffer();
      for (int i = 0; i < ciphers.length; i++)
	{
	  if (i != 0)
	    strbuf.append(":");
	  strbuf.append(ciphers[i]);
	}
      doSetDefaultCiphers(strbuf.toString(),ciphers.length);
    }

  /** Set which ciphers should be used by default for connections created
    * through this context.
    *
    * @param cipher_list a list o ciphers separated by <code>:</code>
    * @param cipher_no the number of ciphers in <code>cipher_list</code>.
    * This is just a convenience parameter, to avoid parsing <code>cipher_list</code>
    */
  private native void doSetDefaultCiphers(String cipher_list, int cipher_no);

  /** Create a <code>SSL_CTX</code> structure, initialize the
    * opaque <code>context</code> pointer and
    * and set default connection parameters. This method
    * must be used in constructors.
    *
    * @exception SSLException  could not create the context.
    */
  private native void setup(String ciphers)
    throws SSLException;


  /** Add the CA certificates in the specified <code>ca_file</code> or
    *  <code>ca_path</code>to the
    * list of trusted ca's. The contents of the
    *  <code>ca_file</code>/<code>will</code> be looked up lazy, upon certificate verification.
    *
    *
    * @param ca_file a file containg a list of trusted certificates
    * @param ca_path a directory in pem-format containg a list of trusted
    * certificates.
    * @exception IOException both <code>ca_file</code>  and <code>ca_path</code>
    * do not exist
    */
  public native void addTrustedCACerts(String ca_file, String ca_path)
   throws IOException;


  /** Free the underlying <code>SSL_CTX</code> structure. This method
   * is used in the <code>finalize</code> method */
  private native void cleanup();

  /** Free the underlying structures */
  protected void finalize()
    {
      cleanup();
    }

 // fetch native implementation
  static
    {
     //  System.loadLibrary("gcc");
      //  System.loadLibrary("crypto");
      //  System.loadLibrary("ssl");
      System.loadLibrary("itissl");

    }


  static class SessionIdEnumeration implements Enumeration {
    Enumeration scn;
    SessionIdEnumeration(Enumeration other)
      {
	scn = other;
      }

    public boolean hasMoreElements()
      {
	return scn.hasMoreElements();
      }

    public Object nextElement()
      {
	return ((SessionIdContainer)scn.nextElement()).getId();
      }
  }

  class SessionIdContainer {
    String stringId;
    byte[]  byteId;
    SessionIdContainer(byte[] sid)
      { stringId = sid.toString();
        byteId   = sid;
      }
    public boolean equals(Object other)
      {
	return stringId.equals( ((SessionIdContainer)other).stringId);
      }
    public int hashCode()
      {
	return stringId.hashCode();
      }
    public byte[] getId()
      {
	return byteId;
      }

  }
}
//======================================================================
//
// $Log: SSLeaySessionContext.java,v $
// Revision 1.6  1999/02/21 14:21:59  apopovic
// system properties renamed to initial 'iti.ssl..' insted of the new 'de.tu_darmstadt..'
//
// Revision 1.5  1999/02/20 19:43:52  apopovic
// - setup split into 'setup' and 'addTrustedCerts'
// - constructor is succcesful even if ca information could not be found.
//
// Revision 1.4  1999/02/13 15:31:20  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.3  1999/01/27 18:55:30  apopovic
// Documentation spell-check
//
// Revision 1.2  1999/01/22 09:47:21  apopovic
// - Documentation enhancements
// - method 'setup' added. Cipher suites and CA location from system properties added.
// - methods 'setUserIdentity' and 'doSetUserIdentity' added.
// - 'set_default_cipher_suites' replaced with 'doSetDefaultCipherSuites'
// - method 'finalize' added (with cleanup actions
//
// Revision 1.1  1999/01/08 10:42:39  apopovic
// Initial Revision
//
