// SSLeaySocketFactory.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeaySocketFactory.java,v 1.7 1999/04/23 09:03:32 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.lang.reflect.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.net.ssl.*;

/**
 * Class SSLeaySocketFactory creates <code>SSLeaySocket</code>s.
 * Each <code>SSLeaySocketFactory</code> uses exactly one
 * <code>SSLeaySessionContext</code>. Every socket created using this factory
 * will use this <code>SSLeaySessionContext</code> for the session
 * corresponding to its own connection. The default ciphers which are
 * going to be used by the sockets created by this factory are actually
 * default ciphers of the <code>SSLeaySessionContext</code> of this
 * factory.
 * <p>
 * <h4>Instantiation of the Session Context</h4>
 * <p>
 * The <code>SSLeaySessionContext</code> is instantiated "lazy", namely in
 * the  first call of one of the following methods:
 * <ul>
 * <li> the <code>createSocket</code> methods
 * <li> the <code>get/setEnabledCipherSuites</code> methods
 * <li> the <code>getSupportedCipherSuites</code>
 * </ul>
 * This property is very useful when serializing and deserializing
 * <code>SSLeaySocketObject</code> factories. If the context would have
 * been initialized upon creation, de deserialization of a factory would
 * work only in the presence of cryptographic information, like certificate
 * files. Using "lazy" instantiation for the used context means the context
 * (and thus the capability of produces sockets) will be created only
 * when somebody realy wants to use a factory object for socket creation.
 * <P>
 * The methods specified above will fail by throwing
 * a <code>SSLException</code> if the necessary information for setting
 * up the context is not there. Socket creation will
 * fail in one of the two cases:
 * <ol>
 * <li> The CA information is not present or incomplete, so
 * the <code>SSLeaySessionContext</code> could not verify a peer.
 *
 * <li> If the system property <code>iti.ssl.clientIdentityRequired</code>
 * is <code>true</code>, but neither of the environment variables or
 * system properties controlling the user identity are defined (See
 * <code>SSLeaySessionContext</code>). In this case, a
 * <code>SSLeayRuntimeException</code> will be thrown. For any other
 * value of the property, no exception will be raised. By default,
 * the <code>iti.ssl.clientIdentityRequired</code> property has the
 * value <code>false</code>.
 * <P>
 *  Secure connections
 * <em>can</em> be created even if this information is not available, but
 * the peer will not be able to check our identity. This exception is
 * catchable; the new socket factory will create sockets.
 * <code>false</code>.
 * </ol>
 *
 * <p>
 * <h4>Extending and customizing <code>SSLeaySocketFactory</code></h4>
 * <p>
 * Using  the <code>setSSLPolicy</code> method you can specify the modes
 * of the connections to be established or decide whether to start or not
 * a SSL handshake in the <code>createSocket</code> methods.
 * This method is a <em>template method</em>, which you can redefine
 * in subclasses of this class in order to specify different policies
 * for the sockets of this factory. <code>setSSLPolicy</code> is called
 * <em>after</em> creating a socket in all <code>createSocket</code>
 * methods.
 * <p>
 * SSLeaySocketFactories together with SSLeaySockets are the means to
 * specify the socket security on a given platform. Some applications need
 * to extend the current SSLSocket implemtation (which is <code>SSLeaySocket</code>).
 * To help extensions of this factory use other socket types than the default,
 * the protected method <code>setSocketPrototype</code> is provided. Used in
 * a constructor, sets the type of sockets the factory will produce.
 *
 *
 * @version	$Revision: 1.7 $
 * @author	Andrei Popovici
 */
public
class SSLeaySocketFactory extends SSLSocketFactory implements Serializable {

  /// the session context
  transient protected SSLeaySessionContext context;

  // the socket type
  protected Class         socketType;

  private boolean userIdentitySet =false;

  private Class STRING_CLASS;
  private Class CONTEXT_CLASS;
  private Class INETADDRESS_CLASS;


  /** Create a new <code>SSLeaySocketFactory</code>  using
    *  <code>context</code> as the <code>SSLeaySessionContext</code>
    * for all connections initiated by this factory.
    *
    * @param context the SessionConxtext of all connections of this factory
    */
  public SSLeaySocketFactory(SSLeaySessionContext context)
    {

      // the constructor 'constants'
      try
	{
	  STRING_CLASS      = Class.forName("java.lang.String");
	  CONTEXT_CLASS     = Class.forName("de.tu_darmstadt.sp.ssl.SSLeaySessionContext");
	  INETADDRESS_CLASS = Class.forName("java.net.InetAddress");
	  socketType        = Class.forName("de.tu_darmstadt.sp.ssl.SSLeaySocket");
	}
      catch(Exception e)
	{
	  throw new Error("SSLeaySpcketFactory.init(context):" + e.getMessage());
	}

      // initialize opaque pointer
      if (context == null)
	throw new IllegalArgumentException("context cannot be null in SSLeayFactory constructor");
      this.context = context;
      userIdentitySet=false;
    }

 /** Create a new <code>SSLeaySocketFactory</code> using a new
    *  <code>SSLeaySessionContext</code>. The instantiation of the
    * factory object is lazy.
    * @exception SSLeayRuntimeException no identity information
    * @exception Error no CA information
    */
  public SSLeaySocketFactory()
  {
    // the constructor 'constants'
    try
      {
	STRING_CLASS      = Class.forName("java.lang.String");
	CONTEXT_CLASS     = Class.forName("de.tu_darmstadt.sp.ssl.SSLeaySessionContext");
	INETADDRESS_CLASS = Class.forName("java.net.InetAddress");
	socketType        = Class.forName("de.tu_darmstadt.sp.ssl.SSLeaySocket");
      }
    catch(Exception e)
      {
	throw new Error("SSLeaySocketFactory.init():" + e.getMessage());
      }

    context = null;
    userIdentitySet=false;
  }
  /** create the context -- runtime exception if failed -- first call the real one, rest = nops*/
  private synchronized  void initContext()
    {
      if (context!=null)
	return;
      try
	{
	  context = new SSLeaySessionContext();
	}
      catch (SSLException e)
	{
	  throw new SSLeayRuntimeException("Could not create context");
	}
    }

  /** set the user identtity */
  private synchronized void setUserIdentity() throws SSLeayIdentityException
    {
      if (userIdentitySet)
	return;
      try
	{
	  context.setUserIdentity();
	  userIdentitySet=true;
	}
      catch (SSLeayIdentityException e2)
	{
	  String idReq =
	    System.getProperties().getProperty("iti.ssl.clientIdentityRequired",
					   "false");
	  if (!idReq.equals("false"))
	    throw e2;
	}

    }

  private void handleInvocationException(InvocationTargetException e)
    throws UnknownHostException, IOException
    {
      Throwable e1 = e.getTargetException();
      if((new IOException()).getClass().isInstance(e1))
	throw (IOException)e1;
      else if( (new UnknownHostException()).getClass().isInstance(e1))
	throw (UnknownHostException)e1;
      else throw new Error(e1.getMessage());
    }
  /** Returns a socket connected to a ServerSocket on the named host,
    * at the given port. This socket is configured using the socket
    * options established for this factory.
    * <p>
    * By default, the class of the returned socket is <code>SSLeaySocket</code>.
    * Subclasses can tune the type of socket they return using the
    * <code>setSocketPrototype</code> method.
    *
    * @param      host   the host name.
    * @param      port   the port number.
    * @exception IOException  if an I/O error occurs when creating the socket.
    * @exception SSLException could not create the session context for this factory
    * @exception SSLeayIdentityException could not set the identity of the user
    */
  public Socket createSocket(String host, int port)
    throws UnknownHostException, IOException
    {
      // first of ALL, enable SSLeay
      initContext();
      setUserIdentity();

      // create the socket using Reflection
      Class[] paramTypes = {STRING_CLASS,Integer.TYPE,CONTEXT_CLASS};
      Object[] params    = {host, new Integer(port), context};
      SSLeaySocket s     = null;
      String errorMsg = "SSLeaySocketFactory.createSocket(String,int):";


      try
      	{
	  s = (SSLeaySocket)socketType.getConstructor(paramTypes).newInstance(params);
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
       catch (InvocationTargetException e3)
 	{
	  handleInvocationException(e3);
 	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}

      setSSLPolicy(s);
      return (Socket)s;
    }

  /**
    * Returns a socket connected to a ServerSocket at the specified
    *  network address and port. This socket is configured using the
    *  socket options established for this factory.
    * <p>
    * By default, the class of the returned socket is <code>SSLeaySocket</code>.
    * Subclasses can tune the type of socket they return using the
    * <code>setSocketPrototype</code> method.
    *
    * @param      address   the IP address.
    * @param      port      the port number.
    * @exception IOException if the connection can't be established
    * @exception SSLException could not create the session context for this factory
    * @exception SSLeayIdentityException could not set the identity of the user
    */
  public
    Socket createSocket(InetAddress address, int port) throws IOException
    {
      // first of ALL, enable SSLeay
      initContext();
      setUserIdentity();

      // create the socket using Reflection
      Class[]      paramTypes = {INETADDRESS_CLASS,Integer.TYPE,CONTEXT_CLASS};
      Object[]     params     = {address, new Integer(port),context};
      SSLeaySocket s          =null;
      String       errorMsg   = "SSLeaySocketFactory.createSocket(InetAddress,Int):";

      try
	{
	  s = (SSLeaySocket)socketType.getConstructor(paramTypes).newInstance(params);
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
      catch (InvocationTargetException e3)
	{
	  handleInvocationException(e3);
	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}

      setSSLPolicy(s);
      return (Socket)s;
    }

  /** Returns a socket connected to a ServerSocket on the named host,
    *  at the given port. The client address address is the specified
    *   host and port. This socket is configured using the socket
    *   options established for this factory.
    * <p>
    * By default, the class of the returned socket is <code>SSLeaySocket</code>.
    * Subclasses can tune the type of socket they return using the
    * <code>setSocketType</code> method.
    *
    * @param host the name of the remote host
    * @param port the remote port
    * @param clientAddr the local address the socket is bound to
    * @param clientPort the local port the socket is bound to
    * @exception IOException  if an I/O error occurs when creating the socket.
    * @exception SSLException could not create the session context for this factory
    * @exception SSLeayIdentityException could not set the identity of the user
    */
  public
    Socket createSocket(String host, int port, InetAddress clientAddr,
		  int clientPort) throws IOException,UnknownHostException
    {
      // first of ALL, enable SSLeay
      initContext();
      setUserIdentity();

      // create the socket using Reflection
      Class[] paramTypes =
        {STRING_CLASS,Integer.TYPE,INETADDRESS_CLASS,Integer.TYPE,CONTEXT_CLASS};
      Object[] params =
        {host,new Integer(port),clientAddr,new Integer(clientPort),context};
      SSLeaySocket s  = null;
      String errorMsg = "SSLeaySocketFactory.createSocket(String,int,InetAddress,int):";

      try
	{
	  s = (SSLeaySocket)(socketType.getConstructor(paramTypes).newInstance(params));
	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
      catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
      catch (InvocationTargetException e3)
	{
	  handleInvocationException(e3);
	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}
      setSSLPolicy(s);
      return (Socket)s;
    }

    /**
      * Returns a socket connected to a ServerSocket at the specified
      * network address and port. The client is bound to the specified
      * network address and port, and the socket is configured using the
      * socket options established for this factory. *
      * <p>
      * By default, the class of the returned socket is <code>SSLeaySocket</code>.
      * Subclasses can tune the type of socket they return using the
      * <code>setSocketPrototype</code> method.
      *
      * @param address the remote address
      * @param port the remote port
      * @param clientAddr the local address the socket is bound to
      * @param clientPort the local port the socket is bound to
      * @exception  IOException  if an I/O error occurs when creating the socket.
      * @exception SSLException could not create the session context for this factory
      * @exception SSLeayIdentityException could not set the identity of the user
      */
    public
    Socket createSocket(InetAddress address, int port, InetAddress clientAddr,
		  int clientPort) throws IOException
    {
      // first of ALL, enable SSLeay
      initContext();
      setUserIdentity();

      // create the socket using Reflection
      SSLeaySocket s     = null;
      Class[] paramTypes =
      {INETADDRESS_CLASS,Integer.TYPE,INETADDRESS_CLASS,Integer.TYPE,CONTEXT_CLASS};
      Object[] params =
      {address,new Integer(port),clientAddr,new Integer(clientPort),context};
      String errorMsg =
	"SSLeaySocketFactory.createSocket(InetAddress,int,InetAddress,int):";

      try
	{
	  s = (SSLeaySocket)socketType.getConstructor(paramTypes).newInstance(params);
     	}
      catch (NoSuchMethodException e1)
	{
	  throw new Error(errorMsg + e1.getMessage());
	}
            catch (InstantiationException e2)
	{
	  throw new Error(errorMsg + e2.getMessage());
	}
      catch (InvocationTargetException e3)
	{
	  handleInvocationException(e3);
	}
      catch (IllegalAccessException e4)
	{
	  throw new Error(errorMsg + e4.getMessage());
	}


      setSSLPolicy(s);
      return (Socket)s;
    };

  /** This is a template method called in the <code>createSocket</code>
    * methods <em>after</em> creating and connecting a
    * <code>SSLeaySocket</code>. Connecting a <code>SSLeaySocket</code> does
    * not necessarily imply a SSL handshake. Without a Handshake, no
    * communication with the peer is possible. The handshake can be
    * initiated either explicitly by calling the
    * <code>startHandshake</code> method, or implicitly by the time either
    * of the <code>getInputStream</code> or <code>getOutputStream</code> are
    * called.
    *
    * <p>
    *
    * By default, this template method performs no configuration of the
    * created sockets. The created SSLeaySockets have the same
    * characteristics as if they would have been created using their
    * constructors.
    *
    * <p>
    *
    * Subclasses of <code>SSLeaySocketFactory</code> can redefine this
    * method to specify alternative SSL configuration policies. SSL policies
    * can be spefified using methods of <code>SSLSocket</code> i.e.
    * <ul>
    * <li> <code>setEnabledCipherSuites</code>
    * <li> <code>setNeedClientAuth</code>
    * <li> <code>setUseClientMode</code>
    * <li> <code>setEnableSessionCreation</code>
    * </ul>
    *
    * <p>
    *
    * If you decide to initiate the handshake as part of the policy
    * configuration, make sure this is the <em>last</em> action of your
    * <code>setSSLSocketPolicy</code> method.
    *
    * <p>
    * The default action of this method is to start a handshake.
    *
    * @exception IOException couldn't set the SSL Policy
    */
  protected void setSSLPolicy(SSLSocket s) throws IOException
    {
      s.startHandshake();
    }

  /** Specify the list of cipher suites which are enabled by default.
   * Handshaking on an SSL connection will use one of these cipher
   *suites. The minimum quality of service for these defaults
   * requires confidentiality protection and server  authentication.
   *
   * @param ciphers <code>String</code> array consisting of the new
   * default ciphers, in order of their preference
   */
  public void setDefaultCipherSuites(String[] ciphers)
    {
      initContext();
      context.setDefaultCipherSuites(ciphers);
    }

  /** Returns the list of cipher suites which are enabled by default.
   * Unless a different list is enabled, handshaking on an SSL connection
   *  will use one of these cipher suites. The minimum quality of service
   * for these defaults requires confidentiality protection and server
   * authentication.
   *
   * @return a <code>String</code> representing the supported ciphers
   */
  public    String[] getSupportedCipherSuites()
    {
      initContext();
      return context.getSupportedCipherSuites();
    }

  /** Returns the names of the cipher suites which could be enabled for use
   * on an SSL connection created by this factory. Normally, only a subset
   * of these will actually be enabled by default, since this list may
   * include cipher suites which do not meet quality of service
   * requirements for those defaults. Such cipher suites are useful in
   * specialized applications.
   *
   * @return a <code>String</code> array representing the default ciphers,
   * in order of their preference
   */
  public String[] getDefaultCipherSuites()
    {
      initContext();
      return context.getDefaultCipherSuites();
    }

  /** Specify the type of sockets (class) returned by the <code>createSocket</code>
    * methods. The constructors for <code>proto</code>'s class have to
    * be a superset of the <code>SSLeaySocket</code>'s constructors.
    *
    * @param proto a prototype for the sockets created by the <code>createSocket</code>
    * methods
    * @exception IllegalArgumentException <code>proto</code> is <code>null</code>
    */
  protected synchronized void setSocketPrototype(SSLeaySocket proto)
    {
      if (proto == null)
	{
	  throw new IllegalArgumentException("setSocketPrototype(null)");
	}
      socketType = proto.getClass();
    }

  /** Return the type of sockets created by this factory. The returned
    * <code>Class</code> object will be a subclass of <code>SSLeaySocket</code>.
    *
    * @return the class of the objects returned by <code>createSocket</code>
    */
  public Class getSocketPrototypeClass()
    {
      return socketType;
    }


  private void readObject(java.io.ObjectInputStream in)
     throws IOException, ClassNotFoundException
    {
      in.defaultReadObject();
      context=null;
      userIdentitySet=false;
    }


}





//======================================================================
//
// $Log: SSLeaySocketFactory.java,v $
// Revision 1.7  1999/04/23 09:03:32  apopovic
// check of property 'clientIdentityRequired' added
//
// Revision 1.6  1999/02/21 14:20:47  apopovic
// system properties renamed to initial 'iti.ssl..' insted of the new 'de.tu_darmstadt..'
//
// Revision 1.5  1999/02/20 19:39:58  apopovic
// - serialization support added
// - initContext split into initContext, setUserIdentity and moved into
//   the createSocket Procedures
// - 'setSocketType' replaced with 'setSocketPrototype'
// - 'getSocketType' replaced with 'getSocketPrototypeClass'
// - Bug Fix: INT_CLASS replaced with Integer.TYPE -- was not serializable
// - Documentation update
//
// Revision 1.4  1999/02/13 15:31:20  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.3  1999/01/27 18:52:52  apopovic
// - 'createSocket' methods changed to use reflection when creating object
// - support for dynamic specification of socket types added
// - methods 'setSocketType' and 'getSocketType' added
// - documentation spell-check
//
// Revision 1.2  1999/01/22 09:41:05  apopovic
// Documentation enhancemnents and fixes;
// - SocketPrototype added for later use;
// - context.setUserIdentity() called inconstructor.
// - startHandshake added in 'setSSLPolicy' -> sockets are now connected
//
// Revision 1.1  1999/01/08 10:42:40  apopovic
// Initial Revision
//
