/* itissl.c -- ITISSL Program */
/* ITISSL - a Java 2 implementation for  Sun's reference SSL API  using SSLeay
 * Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* $Id: itissl.c,v 1.5 1999/03/22 15:28:07 apopovic Exp $
 *----------------------------------------------------------------------
 * Written by Andrei Popovici.
 */

/*
 * Utility functions and global variables for the iti.ssl package
 *
 * (history at end)
 */

#include "itissl.h"
#include <pem.h>



/*
 * Throw an exception of class `exception' with message 'msg'.
 */
void ITISSLThrowException( JNIEnv *env, const char* exception, const char* msg )
{
    unsigned long sslerr;
    jclass exc = (*env)->FindClass( env, exception );
    TRACE2("Throwing exception %s: %s\n", exception, msg);
    sslerr=ERR_get_error();
    if (sslerr != 0)
	{
	    TRACE1("Exception reason: %s\n",ERR_error_string(sslerr,NULL));
	}
    (*env)->ThrowNew( env, exc, msg);
}

/** Throw an exception with of class `exception' and construct a message by
  * appendging `msg' with the last SSLeay error message
  */
void ITISSLThrowDetailException( JNIEnv *env, const char* exception, const char* msg, const char* detail )
{

  /** Create a buffer for the error string, BECAUSE SSLeay does not
    * does not support multi-threaded `ERR_error_string'.
    */
  char *longmsg;

  longmsg = (char*)malloc( strlen(msg) + strlen(detail) + 4);
  sprintf( longmsg, "%s: %s", msg, detail );
  TRACE2("Throwing exception %s: %s\n", exception, longmsg);
  ITISSLThrowException( env, exception, longmsg );
  free(longmsg);
}

/** Check if the condition 'condition' is true; if not, throw a
  * 'iti.ssl.SSLeayRuntimeException'
  * with the given message.
  *
  * condition - the condition to be checked
  * message   - message of the thrown exception
  */
int ITISSL_runtime_check(JNIEnv* env,int condition,const char* message)
{
  if (!condition)
    {
      ITISSLThrowException(env,"de/tu_darmstadt/sp/ssl/SSLeayRuntimeException",message);
    }
  return condition;
}
/** Check if the condition 'condition' is true; if not, throw a 'java.lang.Error'
  * with the given message. This kind of assertion should be used for JNI checks,
  * and other assertion which do not reflect runtime problems but rather wrong
  * irreparable problems like wrong JNI use, etc..
  *
  * condition - the condition to be checked
  * message   - message of the thrown error
  */
int ITISSL_error_check(JNIEnv* env,int condition, const char* message)
{
  if (!condition)
    {
      ITISSLThrowException(env,"java/lang/Error",message);
    }
  return condition;
}

jobjectArray  ITISSL_ciphers_to_jstringarr(JNIEnv* env, STACK *cipher_stack)
{

  /* variables */
  int         stack_size;
  jobjectArray      result;
  jclass      string_class;
  SSL_CIPHER* cipher;
  char*       cipher_name;
  jstring     jcipher_name;
  int         i;

  TRACE("ITISSL_ciphers_to_jstringar:Entering\n");
  /* initialisation */
  stack_size     = sk_num(cipher_stack);
  jcipher_name   = (*env)->NewStringUTF(env,"None");
  string_class   = (*env)->GetObjectClass(env, jcipher_name);
  string_class   = (*env)->FindClass(env,"java/lang/String");
  result         = (*env)-> NewObjectArray(env,
					  stack_size,
					  string_class,
					  jcipher_name);
  TRACE2("String array created. Value: %d, size %d \n", (jint)result,stack_size);
  /** fill the result (string array) with the cipher names */
  for(i=0; i<stack_size; i++)
   {
      TRACE1("Seting result[%d]\n",i);
      cipher       = (SSL_CIPHER*)sk_value(cipher_stack,i);
      TRACE1("   Getting SSL_CIPHER structure. Value: %d\n",(jint)cipher);
      cipher_name  = SSL_CIPHER_get_name(cipher);
      TRACE1("   Extracting Cipher name. Value: %s\n",cipher_name);
      jcipher_name = (*env)->NewStringUTF(env,cipher_name);
      TRACE1("   jstring created. Value: %d\n",(jint)jcipher_name);
      (*env)->SetObjectArrayElement(env,
				    result,
				    i,
				    jcipher_name);
    }

  TRACE("ITISSL_ciphers_to_jstringarr:Exiting\n");
  /* return */
  return result;
}



/** convert  the certificate 'cert' to a string representation */
jstring ITISSL_X509_to_jstring(JNIEnv *env, X509 *cert)
{
  jstring  result;
  BIO*     membio;
  int      certlength;
  char*    utfresult;

  TRACE("ITISSL_X509_to_jstring: Entering\n");
  membio = BIO_new(BIO_s_mem());
  PEM_write_bio_X509(membio,cert);
  certlength=BIO_pending(membio);
  TRACE1("  Certificate Length %d\n",certlength);
  utfresult  = (char*) malloc(certlength * sizeof(char));
  BIO_read(membio,utfresult,certlength);
  result = (*env)->NewStringUTF(env,utfresult);
  TRACE("ITISSL_X509_to_jstring: Leaving\n");
  BIO_free(membio);
  return result;
}


SSL_CTX* ITISSL_get_Handle_context(JNIEnv* env, jobject this)
{
  jclass    cls_handle;
  jfieldID  id_context;
  jobject   field_context;
  SSL_CTX*  result;

  TRACE("ITISSL_get_Handle_context: Entering\n");
  cls_handle     = (*env) -> GetObjectClass(env,this);
  id_context    = (*env) -> GetFieldID(env,cls_handle,"context",
					 "Lde/tu_darmstadt/sp/ssl/SSLeaySessionContext;");
  field_context = (*env) -> GetObjectField(env,this,id_context);
  TRACE1("  Context object: %d\n", (jint)field_context);
  result        = ITISSL_get_Context_context(env,field_context);
  TRACE("ITISSL_get_Handle_context: Leaving\n");
  return result;
}

/** Return a pointer to the ssl structure used by the
   'ssleayHandle'  */
SSL *ITISSL_get_Handle_connection(JNIEnv * env, jobject this)
{
  jclass     clsSSLeayHandle    =
      (*env) -> GetObjectClass(env,this);
  jfieldID  idSSLeayHandle_connection =
      (*env) -> GetFieldID(env,clsSSLeayHandle,CONNECTION_FIELD,"I");
  jint opaque_con =
      (*env) -> GetIntField(env,this,idSSLeayHandle_connection);
  return (SSL*)opaque_con;
}

/** Every SSLeaySession java class has a private field (called session).
  *   The private field is an integer being an opaque handle to the
  *  actual SSL_SESSION structure. This method returns the
  *  SSL_SESSION structure corresponding to a SSLeaySession java
  *  object (this)
  */

SSL_SESSION *ITISSL_get_Session_session(JNIEnv* env,jobject this)
{
    jclass clsSSLeaySession;
    jfieldID idSSLeaySession_session;
    jint opaque_session;

    TRACE("ITISSL_get_Session_session: Entering\n");

    clsSSLeaySession    =
	(*env) -> GetObjectClass(env,this);
    idSSLeaySession_session =
	(*env) -> GetFieldID(env,clsSSLeaySession,SESSION_FIELD,"I");
    opaque_session =
	(*env) -> GetIntField(env,this,idSSLeaySession_session);

    TRACE("ITISSL_get_Session_session: Leaving\n");
    return (SSL_SESSION*)opaque_session;
}



/** Set the identity of the user as described by the user's
  * certificate `cert-file' and his private key file `key-file'
  *
  * Parameters
  *  - cert_file location of the certificate file of this user
  *  - key_file  location of the private key file of this user
  * Returns
  *  - (NULL) on succes; error string (error cause) if identity setting failed
  */
char * ITISSL_setIdentity(env,ctx, cert_file, key_file)
JNIEnv *env;
SSL_CTX *ctx;
char *cert_file;
char *key_file;
{
    TRACE("ITISSL_setIdentity: Entering \n");
  if (cert_file != NULL)
    {
      SSL *ssl;
      X509 *x509;

      if ( cert_file == NULL ||
	   SSL_CTX_use_certificate_file(ctx,cert_file,
					SSL_FILETYPE_PEM) <= 0)
	{
	    TRACE("ITISSL_setIdentity: Leaving(abnormal) \n");
	  return("unable to set certificate file");
	}
      TRACE1("  Using %s certificate file\n",cert_file);
      if (key_file == NULL) key_file=cert_file;
      if (SSL_CTX_use_PrivateKey_file(ctx,key_file,
				      SSL_FILETYPE_PEM) <= 0)
	  {
	      TRACE("ITISSL_setIdentity: Leaving (abnormal) \n");
	      return("unable to set private key file");
	  }
      TRACE1("  Using %s private key file\n",key_file);
      ssl=SSL_new(ctx);
      x509=SSL_get_certificate(ssl);

      EVP_PKEY_copy_parameters(X509_get_pubkey(x509),
			       SSL_get_privatekey(ssl));
      SSL_free(ssl);

      /* If we are using DSA, we can copy the parameters from
       * the private key */


      /* Now we know that a key and cert have been set against
       * the SSL context */
      if (!SSL_CTX_check_private_key(ctx))
	{
	    return("Private key does not match the certificate public key");
	}
    }
  TRACE("ITISSL_setIdentity: Leaving \n");
  return(NULL);
}


SSL_CTX* ITISSL_get_Context_context(JNIEnv* env, jobject this)
{
  jclass cls_sessionContext;
  jfieldID id_context;
  jint   field_context;
  SSL_CTX* result;

  TRACE("ITISSL_get_Context_context: Entering\n");
  cls_sessionContext = (*env) -> GetObjectClass(env,this);
  id_context         = (*env) -> GetFieldID(env,cls_sessionContext,CONTEXT_FIELD,"I");
  field_context      = (*env) -> GetIntField(env,this,id_context);

  result = (SSL_CTX*)field_context;
  TRACE("ITISSL_get_Context_context: Leaving\n");
  return result;
}

SSL_CTX* ITISSL_get_Session_context(JNIEnv* env, jobject this)
{
  jclass   cls_Session;
  jfieldID id_context;
  jobject  field_context;
  SSL_CTX* result;

  cls_Session   = (*env) -> GetObjectClass(env,this);
  id_context    = (*env) -> GetFieldID(env,cls_Session,CONTEXT_FIELD,
					    "Lde/tu_darmstadt/sp/ssl/SSLeaySessionContext;");
  field_context = (*env) -> GetObjectField(env,this,id_context);
  result = ITISSL_get_Context_context(env,field_context);
  return result;
}



/*======================================================================
 *
 * $Log: itissl.c,v $
 * Revision 1.5  1999/03/22 15:28:07  apopovic
 *  variable 'reason' removed (used to produce warning)
 *
 * Revision 1.4  1999/02/20 19:35:30  apopovic
 * Bug fixes: exception and class files adapted for renaming iti ->de.tu_darmstadt.sp
 *
 * Revision 1.3  1999/01/27 18:46:59  apopovic
 * - ITISSL_ThrowException displays now in Debug-mode the last ssleay error
 * - signature for ITISSL_setIdentity changed according to header file
 * - error throws removed from ITISSL_setIdentity (used to cause core dump)
 *
 * Revision 1.2  1999/01/22 10:11:32  apopovic
 * - Bug fixes (used to cause core dumps!) when throwing exceptions.
 * - DEBUG statements improved
 * - Inconsistencies in exception throwing fixed.
 *
 * Revision 1.1  1999/01/08 10:41:14  apopovic
 * Initial Revision
 *
 *
 */
