/* itissl.h -- ITISSL Program */
/* ITISSL - a Java 2 implementation for  Sun's reference SSL API  using SSLeay
 * Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* $Id: itissl.h,v 1.4 1999/03/22 11:40:36 apopovic Exp $
 *----------------------------------------------------------------------
 */

/*
 * Definition used by SSLeayHandle
 *
 * (history at end)
 */

#ifndef	ITISSL_H_INCLUDED
#define	ITISSL_H_INCLUDED



/* java includes */
#include <jni.h>
/* ssl includes */
#include <ssl.h>
#include <err.h>
#include <lhash.h>
/* sll fixes */
#include <ssl_locl.h>

/* environment & file access */
#ifdef WIN32

#include <stdio.h>
#include <stdlib.h>

#ifndef	ITISSL_FILE_OK
#define	ITISSL_FILE_OK	0
#endif

#ifndef ITISSL_ACCESS
#define ITISSL_ACCESS _access
#endif

#else /* no windows */

#include <unistd.h>
#include <stdlib.h>

#ifndef	ITISSL_FILE_OK
#define	ITISSL_FILE_OK	F_OK
#endif

#ifndef ITISSL_ACCESS
#define ITISSL_ACCESS  access
#endif


#endif
/* #define DEBUG 1 */

/* for debugging */
#ifdef DEBUG
#define TRACE3(format,a1,a2,a3) fprintf(stderr,format,a1,a2,a3)
#else
#define TRACE3(format,a1,a2,a3)
#endif
#define TRACE2(format,a1,a2) TRACE3(format,a1,a2,NULL)
#define TRACE1(format,a1) TRACE3(format,a1,NULL,NULL)
#define TRACE(format) TRACE3(format,NULL,NULL,NULL)



/* --------------------------------------------------------------------------- */
/* ITISSL global variables and definitions                                     */
/* --------------------------------------------------------------------------- */



/** field names in the 'SSLeayHandle' class */
#define CONNECTION_FIELD "connection"
#define CONTEXT_FIELD    "context"
#define SESSION_FIELD    "session"

/*** ---------------------------------------------------------------------------
 ***  ITISSL Utility Routines
 *** ----------------------------------------------------------------------------
 *** These routines are used indirectly by the JNI implementations of this
 *** package in order to throw exceptions, make assertions about the success
 *** of function calls or do conversions from between types (i.e. between
 *** Java strings and normal UTF strings).
 ***
 ***/



/*
 * Throw an exception of class `exception' with message `msg'.
 *
 * Parameters:
 *  env       - the java environment
 *  exception - the exception class, e.g. "java/lang/IOException"
 *  msg       - the exception message

 */
extern
void ITISSLThrowException( JNIEnv *env, const char* exception, const char* msg );


/*
 * Throw an exception of class `exception' with message `msg', appended by
 * the last error generated by SSLeay. SSLeay manages its exceptions using
 * a stack model. This functions pops the last message from the error stack,
 * determines its SSLeay error message and appends it to `msg' .
 *
 * Parameters:
 *  env       - the java environment
 *  exception - the exception class, e.g. "java/lang/IOException"
 *  msg       - the exception message
 *  detail    - the detailed exception
 */
extern
void ITISSLThrowDetailException( JNIEnv *env, const char* exception, const char* msg,
				 const char* detail );

/** Check if the condition 'condition' is true; if not, throw a
  * 'iti.ssl.SSLeayRuntimeException'
  * with the given message.
  *
  * Parameters:
  *   env        - the java environment
  *   condition  - the condition to be checked
  *   message    - message of the thrown exception
  * Returns:
  *   the value of the 'condition' parameter
  */
extern int ITISSL_runtime_check(JNIEnv* env,int condition,const char* message);

/* assertion 'statement' which leaves the current scope if the corresponding
 *  'ITISSL_runtime_check' failed. This macro is intended to be used if
 *  it doesn't make sense to continue the execution of a function after some
 *  exception condition occurred.
 */
#define ITISSL_assert(env,cond,message) \
	if(!ITISSL_runtime_check((env),(cond),(message))) return;

/** Check if the condition 'condition' is true; if not, throw a 'java.lang.Error'
  * with the given message. This kind of assertion should be used for JNI checks,
  * and other assertion which do not reflect runtime problems but rather
  * irreparable problems like wrong JNI use, etc..
  *
  * Parameters:
  *   env        - the java environment
  *   condition  - the condition to be checked
  *   message    - message of the thrown error
  * Returns:
  *    the value of the 'condition' parameter
  */
extern int ITISSL_error_check(JNIEnv* env,int condition,const char* message);

/* Fatal assertion 'statement' which leaves the current scope if the corresponding
 * 'ITISSL_error_check' failed. This macro is intended to be used if
 *  it doesn't make sense to continue the execution of a function after the
 *  error condition occurred.
 */
#define ITISSL_error(env,cond,message)  if(!ITISSL_error_check((env),(cond),(message))) return;

/** Set the user identity defined by his certificate in `cert_file' and
  *  his private-key in `key_file' into the given context (`ctx')
  *
  * Parameters:
  *   env       - the java environment
  *   ctx       - the ssl context  of this application
  *   key-file  - file containing the private key
  *   cert-file - file containing the certificate
  * Returns
  *   NULL for a successful identity setting operation, error message otherwise
  */
char* ITISSL_setIdentity(JNIEnv *env, SSL_CTX *ctx, char *cert_file,char *key_file);


/** convert  the certificate 'cert' to a java string representation.
 *
 * Parameters:
 *   env         - the java environment
 *   cert        - the certificate to be converted to string
 * Returns:
 *   a java string representation 0of  'cert'
 */
extern jstring ITISSL_X509_to_jstring(JNIEnv *env, X509 *cert);

/** Transform a cipher list (SSLeay representation as stack) into a list of
  * Strings (java strings), each string representing a cipher name.
  *
  * Parameters:
  *  env          - the java environment
  *  cipher_stack - the cipher list
  */
extern jarray  ITISSL_ciphers_to_jstringarr(JNIEnv* env, STACK *cipher_stack);

/*** ---------------------------------------------------------------------------
 ***  ITISSL Access Routines
 *** ----------------------------------------------------------------------------
 *** These routines are just 'sintactic sugar' to facilitate access to the private
 *** opaque pointers hold by java objects. The opaques pointers are integers hiding
 *** pointers to SSLeay structures such as `SSL', `SSL_CTX', `SSL_SESSION'.
 ***
 *** The function naming uses the  ITISSL_get_<objname>_<fieldname> pattern, where
 *** <objname> in {'Handle','Session','Context'}
 *** <fieldname> in {'connection','context','session'}.
 ***
 *** For example, the function `ITISSL_get_Handle_context(1env,handleObject)'
 *** returns the context structure (in this case indirectly) used by
 *** `handleObject'
 ***/

/** Every SSLeaySession java class has a private field (called session).
  *   The private field is an integer being an opaque handle to the
  *  actual SSL_SESSION structure. This method returns the
  *  SSL_SESSION structure corresponding to a SSLeaySession java
  *  object (this)
  *
  * Parameters:
  *  env         - the java environment
  *  this        - the SSLeaySession java object
  */

extern SSL_SESSION *ITISSL_get_Handle_session(JNIEnv env,jobject this);


/** Return a pointer to the SSL_CTX context structure used by the
  *   `ssleayHandle'. This corresponds to the SSL_CTX structured used by the
  * `SSLeaySessionContext' of this handle.
  *
  * Parameters:
  *  env          - the java environment
  *  ssleayHandle - the ssleayHandle java Object
  * Returns:
  *  a pointer to the SSL_CTX used by the SSL* con corresponding to this handle.
  */
extern SSL_CTX *ITISSL_get_Handle_context(JNIEnv * env, jobject ssleayHandle);

/** Return a pointer to the SSL structure used  by 'ssleayHandle'
  *
  * Parameters:
  *  env          - the java environment
  *  ssleayHandle - the ssleayHandle java Object
  */
extern SSL     *ITISSL_get_Handle_connection(JNIEnv *env, jobject ssleayHandle);


/** Return a pointer to the `SSL_SESSION' structure corresponding to the `ssleaySession'
  * java object.
  *
  * Parameters:
  *  env           - the java environment
  *  ssleaySession - the `SSLeaySession' object to be mapped into the corresponding SSLeay
  *  structure
  * Returns:
  *  a pointer to the SSL_SESSION structure corresponding to the given `ssleaySession'
  */
extern SSL_SESSION *ITISSL_get_Session_session(JNIEnv* env,jobject ssleaySession);

/** Return a pointer to the `SSL_CTX' structure (indirectly) corresponding
  * to the `ssleaySession' java object. This structure is obtained by the
  * following indirection:
  *  1. get the `SSLeaySessionContext' jobject hold by `ssleaySession'
  *  2. get the `SSL_CTX' structure hold opaque by the context jobject
  * THE RESULT SHOULD *ALWAYS* be the SSL_CTX in which the SSL_SESSION to which
  * `ssleaySession' is opaquely mapped is stored in.
  *
  * Parameters:
  *   env - the Java environment
  *   ssleaySession - the sedssion object
  * Returns:
  *   a pointer to the `SSL_CTX' in which the session information is stored
  */
extern SSL_CTX* ITISSL_get_Session_context(JNIEnv* env, jobject ssleaySession);

/** Return a pointer to the `SSL_CTX' structure corresponding to the given java
  * context object.
  *
  * Parameters:
  *   env           - the java environment
  *   ssleayContext - the (Java) context object
  * Returns
  *   a pointers the `SSL_CTX' corresponding to ssleayContext
  */
extern SSL_CTX* ITISSL_get_Context_context(JNIEnv* env, jobject ssleayContext);

/** Set the user identity defined by his certificate in `cert_file' and
  *  his private-key in `key_file' into the given context (`ctx')
  *
  * Parameters:
  *  env       - the java environment
  *  ctx       - the ssl context  of this application
  *  key-file  - file containing the private key
  *  cert-file - file containing the certificate
  */
char* ITISSL_setIdentity(JNIEnv *env, SSL_CTX *ctx, char *cert_file,char *key_file);

/* --------------------------------------------------------------------------- */
/* SSLeay ADDONS                                                               */
/* --------------------------------------------------------------------------- */

/** additional access methods to the SSL internal fields */
/* SSL_CHIPER* SSL_SESSION_get_chipher(SSL_SESSION*) */
#define SSL_SESSION_get_cipher(sess) ((sess)->cipher)

/* X509* SSL_SESSION_get_peer_certificate(SSL_SESSION*) */
#define SSL_SESSION_get_peer_certificate(sess) ((sess)->peer)

/* STACK *SSL_SESSION_get_cert_chain(SSL_SESSION*) */
#define SSL_SESSION_get_cert_chain(sess) ((sess)->cert->cert_chain)

/* char * SSL_SESSION_get_app_data(SSL_SESSION*)
#define SSL_SESSION_get_app_data(sess) ((sess)->app_data);

 void SSL_SESSION_set_app_data(SSL_SESSION*,char*)
#define SSL_SESSION_set_app_data(sess,data) (((sess)->appdata)=(data));
 */
/* unsigned char * SSL_SESSION_get_session_id(SSL_SESSION*) */
#define SSL_SESSION_get_session_id(sess) ((sess)->session_id)

/* int SSL_SESSION_get_session_id_len(SSL_SESSION*) */
#define SSL_SESSION_get_session_id_len(sess) ((sess)->session_id_length)

/* STACK* SSL_CTX_get_cipher_list(SSL_CTX* ctx) */
#define SSL_CTX_get_cipher_list(ctx) ((ctx)->cipher_list)

#endif /* already imported */

/*======================================================================
 *
 * $Log: itissl.h,v $
 * Revision 1.4  1999/03/22 11:40:36  apopovic
 * Os-dependant includes and definitions added
 *
 * Revision 1.3  1999/01/27 18:44:11  apopovic
 * - Documentation spell-check
 * - ITISSL_setIdentity returns now 'char*'
 *
 * Revision 1.2  1999/01/22 10:08:27  apopovic
 * Debug Removed
 *
 * Revision 1.1  1999/01/08 10:43:16  apopovic
 * Initial Revision
 *
 *
 */

