// RMISSLSocket.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: RMISSLSocket.java,v 1.1 1999/02/19 14:23:47 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.rmi;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.net.*;
import java.rmi.*;
import java.rmi.server.*;
import de.tu_darmstadt.sp.ssl.*;
import javax.net.ssl.*;


/**
 * Class RMISSLSocket extends the <code>SSLeaySocket</code> and provides
 * an acoounting for all communications thorugh the created sockets: the most
 * recent thread which used a socket is associated to it. Conversely,
 * every thread can find out the last <code>RMISSLSocket</code> it has used using
 * the methods <code>getCurrentServerSideSocket</code> and
 * <code>getCurrentClientSideSocket</code>.
 *
 * @version	$Revision: 1.1 $
 * @author	Andrei Popovici
 */
public
class RMISSLSocket extends SSLeaySocket {

  /** Accounting table for all connections accepted by  sockets;
    */
  static Map incommingcns = Collections.synchronizedMap(new WeakHashMap());

  /** Accounting table for current */
  static Map outgoingcns  = Collections.synchronizedMap(new WeakHashMap());


    /** Create an unconnected socket using the client mode. Though unconnected,
    * SSL options like preferred ciphers a.o. can be specified.
    *
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    */
  public  RMISSLSocket(SSLeaySessionContext ctx)
    {
      super(ctx);
    }

  /**
    * Constructs a TCP connection to a named host at a specified port. This
    * (tipically) acts as the SSL client, but SSL policy is usually controlled
    * by the corresponding factory.
    * <em>The constructor does not start the handshake.</em>
    *
    * @param host   the host name.
    * @param port   the port number.
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception  IOException  if an I/O error occurs when creating the socket.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    */
  public  RMISSLSocket(String host, int port, SSLeaySessionContext ctx)
	throws java.net.UnknownHostException, IOException
    {
	super(host, port, ctx);
   }

  /** Constructs a TCP connection to a server at a
    * specified address and port.This
    * (usually) acts as the SSL client, but SSL policy is usually controlled
    * by the corresponding factory.
    * <em>The constructor does not start the handshake.</em>
    *
    *
    * @param      address   the IP address.
    * @param      port      the port number.
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception IOException  if an I/O error occurs when creating the socket.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
   * could be created
   */
  public  RMISSLSocket(InetAddress address, int port, SSLeaySessionContext ctx) throws IOException
    {
      super(address, port,ctx);
    }

  /** Constructs an SSL connection to a named host
    * at a specified port, binding the client side of the connection a given
    * address and port. This typically acts as the SSL client, but SSL
      policy is usually controlled  by the corresponding factory.
    * <em>The constructor does not start the handshake.</em>
    *
    * @param host the name of the remote host
    * @param port the remote port
    * @param clientAddr the local address the socket is bound to
    * @param clientPort the local port the socket is bound to
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception IOException  if an I/O error occurs when creating the socket.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    */
  public  RMISSLSocket(String host, int port, InetAddress clientAddr,
		    int clientPort, SSLeaySessionContext ctx) throws IOException
    {
      super(host,port,clientAddr,clientPort,ctx);
    }

    /**
     * Returns a socket connected to a ServerSocket on the named host,
     * at the given port. The client address address is the specified
     * host and port. This socket is configured using the socket options
     * established for this factory.
     *
     * @param address the remote address
     * @param port the remote port
     * @param clientAddr the local address the socket is bound to
     * @param clientPort the local port the socket is bound to
     * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
     * used by this connection.
     * @exception IOException  if an I/O error occurs when creating the socket.
     * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
     * could be created.
     */
    protected RMISSLSocket(InetAddress address, int port, InetAddress clientAddr,
		  int clientPort, SSLeaySessionContext ctx) throws IOException
    {
      super(address, port, clientAddr, clientPort,ctx);
    };



  /** Returns an input stream for this socket. The input stream  uses
    * the underlying SSL session. If there is no SSL session, a
    * <code>startHandshake</code> will be performed.
    * <p>
    * Additionaly, this method associates the current thread to this socket.
    *
    * @return an input stream for reading bytes from this socket
    * @exception IOException if an I/O error occurs when creating the
    * input stream <em>or</em> the underlying SSLHandle is not
    * properly initialized and connected.
    */
  public InputStream getInputStream() throws IOException
    {
      updateCurrentConnections();
      return super.getInputStream();
    }

  /** Returns an output stream for this socket. The output stream  uses
    * the underlying SSL session. If there is no SSL session, a
    * <code>startHandshake</code> will be performed.
    * <p>
    * Additionaly, this method associates the current thread to this socket.
    *
    * @return an output stream for writing bytes to this socket
    * @exception IOException if an I/O error occurs when creating the
      output stream <em>or</em> the underlying SSLHandle is not
    * properly initialized and connected.
    */
  public OutputStream getOutputStream() throws IOException
    {
      updateCurrentConnections();
      return super.getOutputStream();
    }

  /** Return the last <code>RMISSLSocket</code>connetion in the currrent
    * thread.
    */
  public static SSLSocket getCurrentServerSideSocket()
    {
      SSLSocket result = (SSLSocket)incommingcns.get(Thread.currentThread());
      return result;
    }

  /** Return the last <code>RMISSLSocket</code>connetion in the currrent
    * thread.
    */
  public static SSLSocket getCurrentClientSideSocket()
    {
      SSLSocket result = (SSLSocket)outgoingcns.get(Thread.currentThread());
      return result;
    }

  private void updateCurrentConnections()
    {
      if (getUseClientMode())
	{
	  outgoingcns.put(Thread.currentThread(),this);
	}
      else
	{
	  incommingcns.put(Thread.currentThread(),this);
	}
    }

}


//======================================================================
//
// $Log: RMISSLSocket.java,v $
// Revision 1.1  1999/02/19 14:23:47  apopovic
// Initial Revision
//
