// SSLeayHandle.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeayHandle.java,v 1.5 1999/02/13 15:31:17 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.net.ssl.*;

/**
 * Class SSLeayHandle is the object performing the SSL protocol. Every
 * <code>SSLeayHandle</code> is created and owned by a <code>SSLeaySocket</code>.
 * Every SSLeayHandle corresponds to a <code>SSL</code> structure in the
 * underlying SSLeay layer, to which it delegates most of the calls.
 * <p>
 * The SSLeayHandle "steals" the File descriptor used by the owner socket
 * and passes it to the underlying SSLeay structure(method
 *  <code>enableCommunication</code>). Instead, it exports
 * to the owner socket input and output streams which go through the
 * SSL structure of the lower level.
 *
 * @version	$Revision: 1.5 $
 * @author	Andrei Popovici
 */
public
class SSLeayHandle  {


  private int connection;
  private Socket sock;

  private SSLeaySessionContext context;

  /** Create an SSL handle based on the (connected) <code>sock</code>
    * socket.
    */
  public SSLeayHandle(Socket sock, SSLeaySessionContext context) throws SSLException
    {
      this.context = context;
      this.sock    = sock;
      setup();
    }


  protected native int doGetSession();

  /** Return the <code>SSLeaySession</code> cprresponding to the
    * <code>SSL_SESSION</code>of the <code>SSL</code>
    * represented by this handle.
    */
  public SSLeaySession getSession()
    {
      int opaque = doGetSession();
      return new SSLeaySession(opaque,context);
    }

  /** Return the cipher suites enabled for this conection, in the order
    * of preference.
    */
  public native String[] getEnabledCipherSuites();

  /** Specify the cipher suites to be used by default for this connection,
    * in order of preference.
    * If <code>ciphers</code> is <code>null</code>, use the default cipher
    * suites of the <em>context</em> in which the connection of this handle
    * was created.
    *
    * @param ciphers a list of strings representing the ciphers to be enabled.
    * The cipher names should <em>NOT</em> contain the ':' character.
    */
  public void setEnabledCipherSuites(String ciphers[])
    {
      if (ciphers == null)
	ciphers = context.getDefaultCipherSuites();


      StringBuffer strbuf = new StringBuffer();
      for (int i = 0; i < ciphers.length; i++)
	{
	  if (i != 0)
	    strbuf.append(":");
	  strbuf.append(ciphers[i]);
	}
      doSetEnabledCipherSuites(strbuf.toString());
    }

  /** Set the enabled cipher suites */
  native void doSetEnabledCipherSuites(String cipher_list);

  /** Tell the SSLeay layer to grab the filedescriptor used by the java socket
    * <code>sock</code> and enable TCP transport thorugh this socket. This
    * action has to be perfomed before handhakeing. The <code><javaMinorVersion/code>
    *  parameter tells the native layer how to find the file descriptior, since
    * there are differences between java-1.1.X and java 1.2.X.
    *
    * @param sock a connected sockte
    * @param javaMinorVersion the minor version of the java vm. Possible values:
    * 1 or 2.
    *
    */
  native void doEnableTransport(Socket sock, int javaMinorVersion)
    throws SSLException;

  /** Enable the SSL layer to communicate. This method should be called
    * after the owning socket is connected, but prior to the handshake.
    */
  void enableTransport() throws SSLException
    {

      int java_minor_version;
      String java_version = System.getProperty("java.specification.version");
      if (java_version == null || !java_version.equals("1.2"))
	java_minor_version=1;
      else
	java_minor_version=2;
      doEnableTransport(sock,java_minor_version);
    }

  /** Initialization sequence for this handle: create a
     * <code>SSL</code> structure
     */
  native void setup() throws SSLException;

  /** Client handshake.
    */
  native void clientHandshake() throws SSLException;

  /** Server handshake.
    */
  native void serverHandshake() throws SSLException;

  /** Set the verification mode for this handle. This
    * should be performed <em>before</em> handshake. If <code>param</code>
    * is <code>true</code> the peer will be verified during
    * handshake and the handshake will fail if the verification
    * process fails.
    *
    * @param verifyPeer if <code>true</code> verify peer
    * @param isClient   <code>true</code>if the socket works in client mode
    */
  native void setVerificationPolicy(boolean isClient,boolean verifyPeer);

  /**
    * Writes data from a byte array to this SSL  connection.
    * @param buf         buffer to write from
    * @param offset      starting offset within <code>buf</code>
    * @param len         maximum number of bytes to write
    * @exception IOException if an I/O error has occurred
    */
  native
  void writeBytes( byte buf[], int offset, int len ) throws IOException,SSLException;

  /**
   * Reads data from SSL connection into a byte array.
   * @param buf         buffer to read into
   * @param offset      starting offset within <code>buf</code>
   * @param len         maximum number of bytes to read
   * @return            actual number of bytes read, -1 for EOF
   * @exception IOException if an I/O error has occurred
   */
  native int readBytes(byte buf[],  int offset, int len) throws IOException,SSLException;


  /** Close the connection */
  native void close();


  /** Return the Distinguished name of the peer or <code>null</code> if
    * handshake not performed yet.
    */
  native String getPeerDN();





}


//======================================================================
//
// $Log: SSLeayHandle.java,v $
// Revision 1.5  1999/02/13 15:31:17  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.4  1999/02/12 08:39:35  apopovic
// 'doEnableTransport' enhanced with 'javaMinorVersion' paramter
//
// Revision 1.3  1999/01/27 18:42:19  apopovic
// Documentation spell-check
//
// Revision 1.2  1999/01/22 09:20:48  apopovic
// Documentation enhancements.
// Methods renamings: 'get_session' in 'doGetSession', 'bind_ssleay_handle' in
// 'doBindSSleayHandle'.
// Methods 'setup' and 'enableTransport' added.
//
// Revision 1.1  1999/01/08 10:42:34  apopovic
// Initial Revision
//:
//
