// SSLeaySession.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeaySession.java,v 1.8 1999/02/21 14:23:44 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.net.ssl.*;
import javax.security.cert.*;

/**
 * Class SSLeaySession is a stateless, <em>lightweight</em> object which
 * interracts with a <code>SSL_SESSION</code> Structure inside SSLeay. Most of the
 * methods here are native methods, exporting the state of the
 *  <code>SSL_SESSION</code> Structure as Java objects.
 * <p>
 * <code>SSLeaySession</code>  -- uses reference
 * to a Java Hashtable which represents the application data of this Session. However
 * this hashtable is referenced from the <code>SSL_SESSION</code> Structure and not
 * from <code>SSLeaySession</code>.
 * <p>
 * At the time being, it is not entirely clear to me if SSLeay exchanges the application
 * data of a client-session and its corresponging server-session (and
 * vice-versa) or if the application data is just meant for exchanging
 * application information among several connections on the same host.
 * <p>
 *
 *
 * @version	$Revision: 1.8 $
 * @author	Andrei Popovici
 */

public
class SSLeaySession implements SSLSession {

  // opaque pointer to the session structure
  protected  int session;



  protected SSLeaySessionContext context;

   /** Create a new SSLeaySession which is a facade of the <code>SSL_SESSION</code>
     * structure identified by the <code>session</code> opaque pointer. Use
     * the <code>context</code> java object to return information about the context.
     * The specified <code>context</code> object should correspond to the context
     * used internally by the underlying <code>SSL_SESSION</code>.
     *
     * @param session opaque pointer of the <code>SSL_SESSION</code> ssleay structure
     * @param context context object to use
     */
  protected  SSLeaySession(int session, SSLeaySessionContext context)
   {
     this.context = context;
     this.session = session;
   }

  /** Return the Dictionary we use to store application data in this session.
    * Since <code>SSLeaySession</code> is only a facade, the dictionary object
    * is referenced directly from the <code>SSL_SESSION</code>, to prevent
    * garbage collection of application data.
    */
   protected native synchronized Dictionary applicationData();

  /** Bind a object to the application data of this session. If the object
     * implements the <code><SSLSessionBindingListener/code> interfaced, it will
     * be notified via its <code>valueBound</code> method.
     *
     * The application data of a session is common to all  and thus SSL
     * sockets on one side of a connection (either in the client or in the server).
     * The current implementation of this interface to SSLeay does not interchange
     * application data of corresponding
     *
     * @param name the name to which <code>value</code> should be bound
     * @param value the object to be bound
     * @exception IllegalArgumentException (RuntimeException) if either <code>name</code>
     * or <code>value</code> are <code>NULL</code>
     */
   public void putValue(String name,Object value)
     {

       if (value == null || name == null)
 	throw new IllegalArgumentException();

       // to prevent bad things happen, we synchronize the acces to this object.
       // this means the event order corresponds to the real bounding/unbounding
       // of objects
       synchronized (value)
 	{
 	  // bound the object
 	  applicationData().put(name,value);

 	  // notify the bound object
 	  try
 	    {
 	      if (Class.forName("SSLSessionBindingListener").isInstance(value))
 		{
 		  SSLeaySessionBindingEvent event =
		    new SSLeaySessionBindingEvent(this,name);
 		  ((SSLSessionBindingListener)value).valueBound(event);
 		}
 	    }
 	  catch (ClassNotFoundException e)
 	    {
 	      // do nothing the class name is all right..
 	    }
 	}
     }
   /** Return the object previously bound to <code>name</code>. If no object
     * was bound to <code>name</code>, return <code>null</code>
     *
     * @param name the name of the binding to find
     * @return the object bound to <code>name</code>
     */
   public Object getValue(String name)
     {
       return applicationData().get(name);
     }

   /** Return the list of names in the application data
    *
    * @return a <code>String</code> array containing the keys
    */
   public String[] getValueNames()
     {
       String[] result;
       synchronized(applicationData())
 	{
 	  int size         = applicationData().size();
 	  Enumeration enum = applicationData().keys();
 	  result  = new String[size];
 	  int i = 0;
 	  while (enum.hasMoreElements())
 	    {
 	      String name = (String)enum.nextElement();
 	      result[i++] = name;
 	    }
 	}
       return result;
     }


   /** Removes the object bound to the given name in the session's
     * application layer data. Does nothing if there is no object bound to
     * the given name. If the value implements the <code>SessionBindingListener</code>
     * interface, it is notified appropriately.
     *
     * @param  name the name to remove the binding with
     * @exception IllegalArgumentException (RuntimeException) if <code>name</code>
     * is <code>NULL</code>
     */
   public void removeValue(String name)
     {
       if (name == null)
 	throw new IllegalArgumentException();

       Object value = applicationData().remove(name);

        // notify the bound object
        try
 	 {
 	   if (Class.forName("SSLSessionBindingListener").isInstance(value))
 	     {
 	       SSLeaySessionBindingEvent event
		 = new SSLeaySessionBindingEvent(this,name);
 	       ((SSLSessionBindingListener)value).valueUnbound(event);
 	     }
 	 }
        catch (ClassNotFoundException e)
 	 {
 	   // do nothing -- the class name is all right..
 	 }
     }

  /** Returns the context in which this session is bound.
    */
  public SSLSessionContext getSessionContext()
    {
      return context;
    }
  /** Return the certificate chain of the peer. The first certificate is the certificate
   * of the peer itself.
   *
   * @return array of <code>X509Certificate</code>, peer identity first
   */
  public
     X509Certificate[] getPeerCertificateChain() throws SSLPeerUnverifiedException
    {
      String[] certificates = getPeerCertificateChainString();
      if ( certificates == null)
	certificates = new String[0];

      Vector v = new Vector();

      try
	{
	  for (int i = 0; i< certificates.length; i++)
	    {
	      try
		{
		  Object obj = X509Certificate.getInstance(certificates[i].getBytes());
		  v.addElement(obj);
		}
	      catch (CertificateException e)
		{
		  // DIRTY FIX: it seems that the c Impl provides EMPTY CERTS
		}
	    }
	}
      catch(Exception e)
	{ throw new SSLPeerUnverifiedException(e.getMessage());}

      X509Certificate[] result = new X509Certificate[v.size()];

      for (int i = 0; i<v.size(); i++)
	result[i]=(X509Certificate)v.elementAt(i);
      return result;

    }

  /** Return a list of Strings, each string containing the textual
    * form of a <code>X509Certificate</code>.
    *
    * <em>Note:</em> the current implementation does not throw the Exception
    */
  protected native
    String[] getPeerCertificateChainString() throws SSLPeerUnverifiedException;


  /** Returns the host name of the peer in this session. That is, for
    * the server, this is the client's host, and for the client it is the
    * server's host
    *
    * <em>NOTE:Not yet implemented </em>
    */
  public String getPeerHost()
    {
      return null;
    }

  /** Returns the identifier assigned to this Session.
    */
  public native byte[] getId();

  /** Returns the time at which this Session representation was
    * created, in milliseconds since midnight, January 1, 1970 UTC.
    */
  public long getCreationTime()
    {
      return doGetCreationTime()*1000L;
    }

  private native long doGetCreationTime();

  /**
    * Returns the last time this Session representation was accessed by
    * the session level infrastructure, in milliseconds since midnight,
    * January 1, 1970 UTC. Access indicates a new connection being
    * established using session data. Application level operations,
    * such as getting or setting a value associated with the session,
    * are not reflected in this access time.
    * <p>
    * This information is particularly useful in session management
    * policies. For example, a session manager thread could leave all
    * sessions in a given context which haven't been used in a long
    * time; or, the sessions might be sorted according to age to
    * optimize some task.
    * <p>
    * <em>NOTE:not implemented yet</em>
    */
  public long getLastAccessedTime()
    {
      return doGetLastAccessedTime()*1000L;
    }

  private native long doGetLastAccessedTime();



  /**
    * Invalidates the session. Future connections will not be able to resume
    * or join this session. Removes this session object from the Session Context.
    */
  public void invalidate()
    {
      doInvalidate();
      context.update();
    }

  /** Removes the corresponding <code>SSL_SESSION</code> structure form the
    * respective <code>SSL_CTX</code>
    */
  protected native void doInvalidate();

  /**
    * Returns the name of the SSL cipher suite which is used for all
    * connections in the session. This defines the level of protection
    * provided to the data sent on the connection, including the kind of
    * encryption used and most aspects of how authentication is done..
    *
    * @return a string representing the cipher suite used for all connections
    * of this session
    */
  public native String getCipherSuite();

  /** Return all information available about this session in string forma */
  public String toString()
    {
      StringBuffer buf = new StringBuffer();
      buf.append("Cipher suite for all connections:");
      buf.append(getCipherSuite() + "\n");
      buf.append("Session creation time:");
      buf.append(new Date(getCreationTime()) + "\n");
      buf.append("Certificate of the peer:\n\n");

      try
	{
	  X509Certificate[] cert = getPeerCertificateChain();
	  if ( cert.length == 0)
	    buf.append("Peer sent no certificates\n");
	  else
	    buf.append(cert[0] + "\n");
	}
      catch (SSLPeerUnverifiedException e1)
	{
	  buf.append("Peer unverified (exception)\n");
	}

      return buf.toString();

    }

}


//======================================================================
//
// $Log: SSLeaySession.java,v $
// Revision 1.8  1999/02/21 14:23:44  apopovic
// Dirty fix in 'getPeerCertificateChain' if the native code
// produces garbage
//
// Revision 1.7  1999/02/20 19:46:18  apopovic
// Time methods fixed to return the real time (was:bug)
//
// Revision 1.6  1999/02/13 15:31:19  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.5  1999/02/04 18:45:17  apopovic
// Using javax.security instead of java security
//
// Revision 1.4  1999/02/01 13:34:48  apopovic
// 'toString' Method added
//
// Revision 1.3  1999/01/27 18:53:48  apopovic
// Documentation spell-check
//
// Revision 1.2  1999/01/22 09:31:28  apopovic
// Method 'getSessionContext' added. Method 'do_invalidate' renamed to 'doInvalidate'
//
// Revision 1.1  1999/01/08 10:42:38  apopovic
// Initial Revision
//
