/* SSLeaySessionContext.c -- ITISSL Program */
/* ITISSL - a Java 2 implementation for  Sun's reference SSL API  using SSLeay
 * Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* $Id: SSLeaySessionContext.c,v 1.6 1999/03/22 11:43:36 apopovic Exp $
 *----------------------------------------------------------------------
 * Written by Andrei Popovici.
 */

/*
 * Implementation of the native routines of SSLeaySessionContext
 *
 * (history at end)
 */

#include "itissl.h"
#include "SSLeaySessionContext.h"


/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    getSupportedCipherSuites
 * Signature: ()[Ljava/lang/String;
 */
JNIEXPORT jobjectArray JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_getSupportedCipherSuites
(JNIEnv * env, jobject this)
{
  /* variables */
  SSL_CTX* ctx;
  SSL*     tmpcon;
  STACK*   ciphers;
  jobjectArray   result;
  TRACE("SSLeaySessionContext_getSupportedCipherSuites:Entering\n");

  /* create a dummy context and a dummy connection and ask them about ciphers */
  ctx = SSL_CTX_new(SSLv3_method());
  ITISSL_assert(env,(ctx!=NULL),"cannot create context structure");

  SSL_CTX_set_cipher_list(ctx,SSL_TXT_ALL);
  tmpcon= SSL_new(ctx);
  ITISSL_assert(env,(tmpcon!=NULL),"cannot create temporary connection");

  ciphers = SSL_get_ciphers(tmpcon);
  result  =  ITISSL_ciphers_to_jstringarr(env,ciphers);

  SSL_free(tmpcon);
  SSL_CTX_free(ctx);

  TRACE("SSLeaySessionContext_getSupportedCipherSuites:Exiting\n");
  return result;
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    getDefaultCipherSuites
 * Signature: ()[Ljava/lang/String;
 */
JNIEXPORT jobjectArray JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_getDefaultCipherSuites
(JNIEnv* env, jobject this)
{
  STACK* ciphers;
  jobjectArray result;
  SSL_CTX* context;

  context = ITISSL_get_Context_context(env,this);
  ciphers = SSL_CTX_get_cipher_list(context);
  result = ITISSL_ciphers_to_jstringarr(env,ciphers);

  return result;

}
/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    doSetDefaultCiphers
 * Signature: (Ljava/lang/String;I)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_doSetDefaultCiphers

 (JNIEnv *env, jobject this, jstring ciphers, jint cipher_no)
{
  char *cipher_list;
  SSL_CTX* context;
  SSL_CTX* tmpctx;
  STACK*    tmpciphers;

  cipher_list  = (char*)(*env)->GetStringUTFChars(env, ciphers, 0);

  /* create a temporary context and check if *ALL* the specified ciphers can
   * be set
   */
  tmpctx=SSL_CTX_new(SSLv3_method());
  SSL_CTX_set_cipher_list(tmpctx,cipher_list);
  tmpciphers=SSL_CTX_get_cipher_list(tmpctx);
  if (tmpciphers == NULL || sk_num(tmpciphers) != cipher_no)
    {
      ITISSLThrowException(env,"java/lang/IllegalArgumentException","Unsupported cipher suite");
      return;
    }

  /* set the specified ciphers -- in case of success */
  context = ITISSL_get_Context_context(env,this);
  SSL_CTX_set_cipher_list(context,cipher_list);

  (*env) -> ReleaseStringUTFChars(env, ciphers, cipher_list);
}

int stack_push(char*,char*);
int stack_push(char* data, char* stack)
{
    TRACE("stack_push: Entering\n");
    return sk_push((STACK*)stack,data);
    TRACE("stack_push: Leaving\n");
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    getSessionOpaquePointers
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_getSessionOpaquePointers
(JNIEnv *env, jobject this)
{
  /* variables */
  jintArray result;
  jint*     result_elements;
  jsize     session_no;
  LHASH*    session_hash;
  STACK*    session_stack;
  SSL_CTX*  context;
  int       i;

  TRACE("SSLeaySessionContext_getSessionOpaquePointers: Entering\n");
  /* move the sessions from the hash into a stack */
  context       = ITISSL_get_Context_context(env,this);
  session_hash  = SSL_CTX_sessions(context);
  session_stack = sk_new(NULL);
  lh_doall_arg(session_hash,(void (*)())stack_push, (char*)session_stack);
  session_no    = sk_num(session_stack);

  /* create a jintarray and move the sessions from the stack into this jintarray */
  result = (*env)->NewIntArray(env,session_no);
  result_elements = (*env) -> GetIntArrayElements(env,result,NULL);
  for (i=0; i<session_no; i++)
      result_elements[i]=(jint)sk_value(session_stack,i);
  (*env)->ReleaseIntArrayElements( env, result, result_elements, 0 );

  /* return the result */
  TRACE("SSLeaySessionContext_getSessionOpaquePointers: Leaving\n");
  return result;
}
/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    setup
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_setup
 (JNIEnv * env, jobject this, jstring ciphers)
{
    /* variables */
    char* cipher_list = NULL;

    SSL_CTX* context;
    jclass cls_sessionContext;
    jfieldID id_context;
    TRACE("SSLeaySessionContext_setup: Entering\n");

    /* add algorithms, load error strings..     */
    SSL_load_error_strings();
    ERR_load_ERR_strings();
    SSLeay_add_all_algorithms();
    ERR_load_crypto_strings();
    TRACE("  Error-strings and algorithms added\n");

    /* create context */
    context = SSL_CTX_new(SSLv3_method());
    if ( context == NULL)
	{
	    ITISSLThrowException(env,"javax/net/ssl/SSLException","cannot create context");
	    return;
	}

    TRACE1("  Context structure created. Value %d\n",context);

    /** set cipher list from environment */
    if (ciphers != NULL)
      {
	cipher_list  = (char*)(*env)->GetStringUTFChars(env, ciphers, 0);
	SSL_CTX_set_cipher_list(context,cipher_list);
	(*env) -> ReleaseStringUTFChars(env, ciphers, cipher_list);
	TRACE1("  Cipher set from property. Value: %s \n",cipher_list);
      }
    else
      {
	cipher_list = getenv("SSL_CIPHER");

	if (cipher_list != NULL)
	  {
	    SSL_CTX_set_cipher_list(context,cipher_list);
	    TRACE1("  Cipher set from environment. Value: %s \n",cipher_list);
	  }
      }
    /** set opaque pointer in it's place..*/

    TRACE("  Attempting to set the opaque 'context' pointer\n");
    cls_sessionContext = (*env) -> GetObjectClass(env,this);
    id_context         = (*env) -> GetFieldID(env,cls_sessionContext,
					      CONTEXT_FIELD,"I");
    (*env) -> SetIntField(env,this,id_context,(jint)context);
    TRACE1("  Context opaque pointer set; Value: %d \n",(jint)context);
    TRACE("SSLeaySessionContext_setup: Leaving\n");
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext_addTrustedCACerts
 * Method:    setup
 * Signature: (Ljava/lang/String;Ljava/lang/string)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_addTrustedCACerts
 (JNIEnv * env, jobject this, jstring CAfile,jstring CApath)
{
    char* ca_file     = NULL;
    char* ca_path     = NULL;
    int bad_ca_file;
    int bad_ca_path;
    SSL_CTX* context;

    TRACE("SSLeaySessionContext_addTrustedCACerts: Entering\n");

    context = ITISSL_get_Context_context(env,this);

    /** set ca file  from environment */
    if (CAfile != NULL)
      ca_file = (char*)(*env)->GetStringUTFChars(env, CAfile, 0);
    else
      ca_file = getenv("CA_FILE");

    /** set ca path from environment */
    if (CApath != NULL)
	ca_path = (char*)(*env)->GetStringUTFChars(env, CApath, 0);
    else
	ca_path = getenv("CA_PATH");


    bad_ca_file=ITISSL_ACCESS(ca_file,ITISSL_FILE_OK);
    bad_ca_path=ITISSL_ACCESS(ca_path,ITISSL_FILE_OK);

    if (  (bad_ca_file && bad_ca_path) ||
	 (!SSL_CTX_load_verify_locations(context,ca_file,ca_path)) ||
	 (!SSL_CTX_set_default_verify_paths(context)))


	{
	    ITISSLThrowException(env,"java/io/IOException",
				 "Cannot set CA and verify locations");
	    if(CApath != NULL)
		(*env) -> ReleaseStringUTFChars(env, CApath, ca_path);

	    if(CAfile != NULL)
		(*env) -> ReleaseStringUTFChars(env, CAfile, ca_file);
	    return;
	}

    if(CAfile != NULL)
      (*env) -> ReleaseStringUTFChars(env, CAfile, ca_file);
    if(CApath != NULL)
      (*env) -> ReleaseStringUTFChars(env, CApath, ca_path);


    TRACE("  Using  CA file/path, tests ok.\n");
    TRACE("SSLeaySessionContext_addTrustedCACerts: Leaving\n");
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    doSetUserIdentity
 * Signature: (Ljava/lang/String;Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_doSetUserIdentity

  (JNIEnv * env, jobject this, jstring cert, jstring key)
{
  char* key_file    = NULL;
  char* cert_file   = NULL;
  SSL_CTX* context  = NULL;
  char* succ        = NULL;
  char* errorMsg    = "ok";

  TRACE("SSLeaySessionContext_setIdentity: Entering\n");
  context = ITISSL_get_Context_context(env,this);

  /** optional load the private key, own certificate, etc -> this is the case
    * if we are a server application.
    */
  if (cert != NULL)
    cert_file = (char*)(*env)->GetStringUTFChars(env, cert, 0);
  else
    cert_file = getenv("CERT_FILE");
  if (cert_file!=NULL)
      TRACE1("  Attempting to use %s certfile\n",cert_file);

  if (key != NULL)
    key_file  = (char*)(*env)->GetStringUTFChars(env, key, 0);
  else
    key_file  = getenv("KEY_FILE");

  if ( key_file!=NULL )
   TRACE1("  Attempting to use %s keyfile\n", key_file);

  TRACE("  Attempting to set user Identity using mentioned files\n");
  succ = ITISSL_setIdentity(env,context, cert_file, key_file);
  if (succ != NULL)
      errorMsg = succ;
  TRACE1("  Success in setting the identity:%s\n",errorMsg);

/* now we should check if the identity setting was successful . In a typical
 * client application, it should not be..
 */

  if ( cert_file == NULL || key_file == NULL || succ)
    {
      ITISSLThrowException(env,"de/tu_darmstadt/sp/ssl/SSLeayIdentityException",errorMsg);
      if (cert)
	(*env) -> ReleaseStringUTFChars(env, cert, cert_file);
      if (key)
	(*env) -> ReleaseStringUTFChars(env, key, key_file);
      return;
    }

  TRACE("  Attempting to set client CA list\n");
  SSL_CTX_set_client_CA_list(context,SSL_load_client_CA_file(cert_file));
  TRACE("  Client CA list set\n");
  if (cert)
    (*env) -> ReleaseStringUTFChars(env, cert, cert_file);
  if (key)
    (*env) -> ReleaseStringUTFChars(env, key, key_file);
  TRACE("SSLeaySessionContext_setIdentity: Leaving\n");
}

/*
 * Class:     de_tu_0005fdarmstadt_sp_ssl_SSLeaySessionContext
 * Method:    cleanup
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeaySessionContext_cleanup
  (JNIEnv *env, jobject this)
{
  SSL_CTX* context;
  context = ITISSL_get_Context_context(env,this);

  SSL_CTX_free(context);
  TRACE("SSL Context freed");
  context = NULL;
}

/*======================================================================
 *
 * $Log: SSLeaySessionContext.c,v $
 * Revision 1.6  1999/03/22 11:43:36  apopovic
 * Unix-specific 'access' replaced with 'ITISSL_ACCESS' macro
 *
 * Revision 1.5  1999/02/20 19:45:14  apopovic
 *  - setup changed; addTrustedCACerts added
 *  - suport for CApath added
 *
 * Revision 1.4  1999/02/05 10:02:02  apopovic
 * Bug fix 'doSetUserIdentity'. Used to throw a NULL message exception
 *
 * Revision 1.3  1999/01/27 18:56:45  apopovic
 * Documentation improvement; minor changes due to new ITISSL_setIdentity signature
 *
 * Revision 1.2  1999/01/22 10:18:11  apopovic
 * Trace message enhancements and method renamings.
 *
 * Revision 1.1  1999/01/08 10:41:13  apopovic
 * Initial Revision
 *
 */
