// SSLeaySocket.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: SSLeaySocket.java,v 1.6 1999/02/20 19:42:07 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.ssl;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.net.ssl.*;

/**
 * Class SSLeaySocket implements Sun's abstract <code>SSLSocket</code> definition
 * using the SSLeay package.  SSLeaySockets can be created either connected or
 * unconnected. <em>connected</em> means that there exists a connection on the
 * transport layer.
 * <p>
 * After creation (connected or unconnected) <code>SSLeaySocket</code>
 * objects have an <em>own</em> <code>SSLeayHandle</code> object. The handle
 * will actually perform the SSL protocol for this  object. The SSL protocol
 * can be initiated or restarted using the <code>startHandshake</code> method.
 * For the <code>startHandshake</code> method to succeed, a socket has to be
 * connected. Therefore, the initiation of transport layer connection
 * is mandatory before handshake start for unconnected sockets.
 * <p>
 * The handshake start is supposed to be performed in the factory which creates
 * a socket. If this is not the case, the connection will be initiated when
 * the communication starts, this is, by the first use of the <code>getInputStream</code>
 * or <code>getOutputStream</code> methods.
 * <p>
 * SSLeaySocket require by default peer authentication and use the client mode.
 * @version	$Revision: 1.6 $
 * @author	Andrei Popovici
 */

public
class SSLeaySocket extends SSLSocket {


  protected SSLeayHandle         handle = null;
  protected SSLeaySessionContext context= null;
  private InputStream    in             = null;
  private OutputStream   out            = null;
  private boolean        useClientMode  = true;
  private boolean        needClientAuth = true;
  private Vector         listeners      = null;
  private boolean        enableSessionCreation = true;
  private boolean        handshakeStarted = false;


  /** Create an unconnected socket using the client mode. Though unconnected,
    * SSL options like preferred ciphers a.o. can be specified. If the
    * <code>ctx</code> parameter is <code>null</code>, the socket cannot
    * be used for communication. A <code>SSLeaySocket</code> created with
    * a <code>null</code> ctx parameter can be used to specify the prototype
    * socket a factory should produce.
    *
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection. A <code>null</code> value yields a prototype
    * socket, unable to communicate.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    */
  public  SSLeaySocket(SSLeaySessionContext ctx)
    {
      super();
      if (ctx == null)
	return;
      useClientMode = true;
      listeners = new Vector();
      context = ctx;
      try
	{setupHandle();}
      catch(IOException e)
	{throw new SSLeayRuntimeException(e.getMessage());};
    }

  /**
    * Constructs a TCP connection to a named host at a specified port. This
    * (tipically) acts as the SSL client, but SSL policy is usually controlled
    * by the corresponding factory.
    * <em>The constructor does not start the handshake.</em>
    *
    * @param host   the host name.
    * @param port   the port number.
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception  IOException  if an I/O error occurs when creating the socket.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>
    */
  public  SSLeaySocket(String host, int port, SSLeaySessionContext ctx)
	throws UnknownHostException, IOException
    {
	this(InetAddress.getByName(host), port, ctx);
   }

  /** Constructs a TCP connection to a server at a
    * specified address and port.This
    * (usually) acts as the SSL client, but SSL policy is usually controlled
    * by the corresponding factory.
    * <em>The constructor does not start the handshake.</em>
    *
    *
    * @param      address   the IP address.
    * @param      port      the port number.
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception IOException  if an I/O error occurs when creating the socket.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>
    */
  public  SSLeaySocket(InetAddress address, int port, SSLeaySessionContext ctx) throws IOException
    {
      this(address, port, null, 0,ctx);
    }

  /** Constructs an SSL connection to a named host
    * at a specified port, binding the client side of the connection a given
    * address and port. This typically acts as the SSL client, but SSL
      policy is usually controlled  by the corresponding factory.
    * <em>The constructor does not start the handshake.</em>
    *
    * @param host the name of the remote host
    * @param port the remote port
    * @param clientAddr the local address the socket is bound to
    * @param clientPort the local port the socket is bound to
    * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
    * used by this connection.
    * @exception IOException  if an I/O error occurs when creating the socket.
    * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
    * could be created
    * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>
    */
  public  SSLeaySocket(String host, int port, InetAddress clientAddr,
		    int clientPort, SSLeaySessionContext ctx) throws IOException
    {
      this(InetAddress.getByName(host), port, clientAddr, clientPort,ctx);
    }

    /**
     * Returns a socket connected to a ServerSocket on the named host,
     * at the given port. The client address address is the specified
     * host and port. This socket is configured using the socket options
     * established for this factory.
     *
     * @param address the remote address
     * @param port the remote port
     * @param clientAddr the local address the socket is bound to
     * @param clientPort the local port the socket is bound to
     * @param ctx the <code>SSLeaySessionContext</code> for the session(s)
     * used by this connection.
     * @exception IOException  if an I/O error occurs when creating the socket.
     * @exception SSLeayRuntimeException if no <code>SSLeayHandle</code>
     * could be created.
     * @exception IllegalArgumentException <code>ctx</code> is <code>null</code>
     */
    protected SSLeaySocket(InetAddress address, int port, InetAddress clientAddr,
		  int clientPort, SSLeaySessionContext ctx) throws IOException
    {
      super(address, port, clientAddr, clientPort);

      if (ctx == null)
	{ throw new IllegalArgumentException("SSLeaySocket.init(): ctx is null"); }

      context=ctx;

      // the real initialization
      setNeedClientAuth(true);
      setUseClientMode(true);
      listeners = new Vector();
      try
	{ setupHandle(); }
      catch(IOException e)
	{ throw new SSLeayRuntimeException(e.getMessage());};
    };


  /** Returns the the SSL Session in use by this connection. These can
    * be long lived, and frequently correspond to an entire login session
    * for some user. The session specifies a particular cipher suite which
    * is being actively used by all connections in that session, as well as
    * the identities of the session's client and server.
    *
    */
  public SSLSession getSession()
    {
      return handle.getSession();
    }

  /** Close the SSL connection and this socket .
    *
    * @exception IOException if an I/O error occurs when closing this socket.
    */
  public void close() throws IOException
    {
      handle.close();
      super.close();
    }

  /** Returns an input stream for this socket. The input stream  uses
    * the underlying SSL session. If there is no SSL session, a
    * <code>startHandshake</code> will be performed.
    *
    * @return an input stream for reading bytes from this socket
    * @exception IOException if an I/O error occurs when creating the
    * input stream <em>or</em> the underlying SSLHandle is not
    * properly initialized and connected.
    */
  public InputStream getInputStream() throws IOException
    {
      if (!handshakeStarted)
	startHandshake();
      if (in == null)
	in = new SSLeayInputStream(handle);
      return in;
    }

  /** Returns an output stream for this socket. The output stream  uses
    * the underlying SSL session. If there is no SSL session, a
    * <code>startHandshake</code> will be performed.
    *
    * @return an output stream for writing bytes to this socket
    * @exception IOException if an I/O error occurs when creating the
      output stream <em>or</em> the underlying SSLHandle is not
    * properly initialized and connected.
    */
  public OutputStream getOutputStream() throws IOException
    {
      if (!handshakeStarted)
	startHandshake();
      if (out == null)
	out = new SSLeayOutputStream(handle);
      return out;
    }


  /** Configures the socket to use client (or server) mode in its
    * <em>first</em> handshake. Calling this method makes sense only if the
    * socket was created with the <code>SSLSocket()</code>
    * constructor.
    *
    * @param mode if <code>true</code> connect -- client mode, else accept connection
    */
  public  void setUseClientMode(boolean mode)
    {
      useClientMode = mode;
    }
  /** Returns <code>true</code> if the socket is set to use client
    * mode in its first handshake.
    *
    * @return <code>true</code> if the socket should start its first handshake in "client" mode
    */
  public boolean getUseClientMode()
    {
      return useClientMode;
    }


  /** Starts an SSL handshake on this connection. The current
    * implementation does not support this operation except for the first
    * handshake. This means the initial handshake will be performed according
    * to the current <code>useClientMode</code> mode. For further calls
    * to this method the behavior is not specified.
    * (for the moment)
    *
    */
  public synchronized
    void startHandshake() throws IOException, javax.net.ssl.SSLException
    {
      if(!handshakeStarted)
	handle.enableTransport();
      handshakeStarted=true;
      try
	{
	  if (getNeedClientAuth())
	    handle.setVerificationPolicy(getUseClientMode(),
					 getNeedClientAuth());

	  if ( useClientMode)
	    handle.clientHandshake();
	  else
	    handle.serverHandshake();
	}
      catch (SSLException e)
	{ throw new IOException("SSL:" + e.getMessage()); }

      notifyListeners();
    }

  private void notifyListeners()
    {
      SSLeayHandshakeCompletedEvent event =
	new SSLeayHandshakeCompletedEvent(this,this.getSession());

      Enumeration listenersEnum = listeners.elements();
      while (listenersEnum.hasMoreElements())
	{
	  HandshakeCompletedListener l =
	    (HandshakeCompletedListener)(listenersEnum.nextElement());

	  l.handshakeCompleted(event);
	}
    }

  /** Returns true if the socket will require client authentication.
    * The socket must <em>not necessarily</em> be a server socket.
    * This method returns the peer verification policy,
    *
    * @return <code>true</code> if the socket will require client authentication
    */
  public boolean getNeedClientAuth()
    {
      return needClientAuth;
    }

  /** Configures the socket to require client authentication. The
    * socket must <em>not necessarily</em> be a server mode socket.
    * This method sets the peer verification policy.
    *
    * @param  flag if <code>true</code> authenticate peer during handshake
    */
  public void setNeedClientAuth(boolean flag)
    {
      needClientAuth = flag;
    }

  /** Create a <code>SSLeayHandle</code>SSLeayHandle for this socket.
    * The <code>SSLeayHandle</code>
    * will be used for for the SSL handshake protocol
    * and data encryption, after a communication channel is established
    * on the transport layer or to set SSL options prior to this step.
    * <p>
    * Note that <code>setupHandle</code> is mandatory in every constructor
    * of this class, but does not initiate a transport layer connection.
    *
    * @exception IOException if the SSL data structure could not be created
    * for this socket (Encapsulation for a <code>SSLException</code>)
    */
  void setupHandle() throws IOException
    {
      if (handle !=null)
	return;
      try { handle = new SSLeayHandle(this,context); }
      catch ( SSLException e)
	{ throw new  IOException("SSL:" + e.getMessage()); }
    }

  /*
   * Registers an event listener to receive notifications that an SSL
   * handshake has completed on this connection.
   *
   * @param listener the listener to be added
   */
  public
    void addHandshakeCompletedListener(HandshakeCompletedListener listener)
    {
      synchronized(listeners)
	{
	  // maybe it's there
	  if (listeners.contains(listener))
	    return;

	  // add the component
	  listeners.addElement(listener);
	}
    }

  /**
    * Removes a previously registered handshake completion listener.
    *
    * @param  listener the listener to be removed
    */
  public
    void removeHandshakeCompletedListener(HandshakeCompletedListener listener)
    {
      synchronized(listeners)
	{
	  // maybe it's not there
	  if (!listeners.contains(listener))
	    return;

	  // remove the component
	  listeners.removeElement(listener);
	}
    }

  /**
    * Returns the names of the cipher suites which could be enabled for use
    * on an SSL connection. Normally, only a subset of these will actually
    * be enabled by default, since this list may include cipher suites which
    * do not meet quality of service requirements for those defaults. Such
    * cipher suites are useful in specialized applications.
    *
    * @return a <code>String</code> array representing the suported ciphers
    */
  public  String[] getSupportedCipherSuites()
    {
      return context.getSupportedCipherSuites();
    }

  /** Returns the names of the SSL cipher suites which are currently enabled
    * for use on this connection. When an SSL socket is first created, all
    * enabled cipher suites support a minium quality of service. Thus, in
    * some environments this value might be empty.
    *
    * @return a <code>String</code> array representing the suported ciphers
    */
  public String[] getEnabledCipherSuites()
    {
      return handle.getEnabledCipherSuites();
    }

/** Controls which particular cipher suites are enabled for use on
  * this connection. The cipher suites must have been listed by
  * getSupportedCipherSuites() as being supported. Even if a suite has
  * been enabled, it might never be used if no peer supports it, or the
  * requisite certificates (and private keys) are not available.
  *
  * @param ciphers the cipher list to be enabled. If <code>null</code>, use
  * the default ciphers
  * @exception IllegalArgumentException if one of the cipher suites is not supported
  */
  public void setEnabledCipherSuites(String[] ciphers) throws IllegalArgumentException
    {
      handle.setEnabledCipherSuites(ciphers);
    }

  /** Return peer Distinguished name. If no handshake yet, throw
    * <code>SSLException</code>.
    *
    * @exception SSLException no handshake yet
    */
  public  String getPeerName() throws SSLException
    {
      if ( handle == null)
	throw new SSLException("Handshake not performed");
      return handle.getPeerDN();
    }

  /**
    * Controls whether new SSL sessions may be established by this socket.
    * Not implemented yet.
    *
    * @param flag if <code>true</code>,new SSL sessions may be
    *  established by this socket
    */
  public void setEnableSessionCreation(boolean flag)
    {
      enableSessionCreation = flag;
    }

  /**
   * Returns <code>true</code> if new SSL sessions may be established by this socket.
   * Not implemented yet.
   *
   * @return <code>true</code> if new SSL sessions may be established by this socket.
   */
  public boolean getEnableSessionCreation()
    {
      return enableSessionCreation;
    }

  /** Return all information about this SSL socket in string Format: */
  public String toString()
    {
      StringBuffer buf = new StringBuffer();
      buf.append("\nSOCKET PARAMETERS\n\n");
      buf.append("Cipher suites enabled for this socket:\n  ");
      String[] eciphers = getEnabledCipherSuites();
      for (int i=0; i<eciphers.length; i++)
	{ buf.append("\n" + eciphers[i]);}
      buf.append("\nSocket uses client mode (true/false):");
      buf.append(getUseClientMode() + "\n");
      buf.append("\nSocket requires peer authentication (true/false):");
      buf.append(getNeedClientAuth() + "\n");
      SSLSession sess  = getSession();
      if (sess == null)
	buf.append("\nSocket not connected (no sesion)\n");
      else
	buf.append("\nSESSION INFORMATION\n\n");
      buf.append(sess.toString());
      return buf.toString();
    }
}


//======================================================================
//
// $Log: SSLeaySocket.java,v $
// Revision 1.6  1999/02/20 19:42:07  apopovic
// Exact specification for constructors in case of a null ctx added
//
// Revision 1.5  1999/02/13 15:31:20  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.4  1999/02/01 13:35:41  apopovic
// 'toString' Method added
//
// Revision 1.3  1999/01/27 18:57:18  apopovic
// Documentation spell-check
//
// Revision 1.2  1999/01/22 09:13:12  apopovic
// - Documentation enhancements;
// - setupHandle in Constructors -- throws RuntimeException on failure
// - startHandshake is now synchronized, and sets just the fd in SSL (enables Communication)
// - initSSLHandle replaced with setupHandle.
//
// Revision 1.1  1999/01/08 10:42:39  apopovic
// Initial Revision
//
//
//
