/* compress.c - Compression handling for data
 *  Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <zlib.h>

#include "opencdk.h"
#include "main.h"

#ifdef HAVE_LIBZ
static int
do_compress( int algo, int flush, byte *inbuf, size_t insize,
             byte *outbuf, size_t *ret_outsize )
{
    byte buf[4096];
    int pos = 0;
    int zrc = 0;
    z_stream *zs;
    
    zs = cdk_alloc( sizeof *zs );
    zrc = (algo==1)? deflateInit2( zs, Z_DEFAULT_COMPRESSION, Z_DEFLATED,
                                   -13, 8, Z_DEFAULT_STRATEGY ):
        deflateInit( zs, Z_DEFAULT_COMPRESSION );
    if ( zrc )
        goto leave;
    
    zs->next_in = inbuf;
    zs->avail_in = insize;

    do {
        zs->next_out = buf;
        zs->avail_out = sizeof(buf);
        zrc = deflate( zs, flush );
        if ( zrc == Z_STREAM_END ) {
            deflateEnd( zs );
            if ( zs->avail_out ) {
                memcpy( outbuf + pos, buf, sizeof(buf) - zs->avail_out );
                pos += (sizeof(buf) - zs->avail_out);
                break;
            }
        }
        if ( zrc == Z_OK ) {
            memcpy( outbuf + pos, buf, sizeof(buf) - zs->avail_out );
            pos += (sizeof(buf) - zs->avail_out);
        }        
        
        if ( zrc ) {
            if ( zs->msg )
                fprintf( stderr,"zlib error `%s'.\n", zs->msg );
            fprintf( stderr,"compress: deflate returned %d.\n", zrc );
        }            
    } while ( zs->avail_out == 0 );
    
leave:
    *ret_outsize = pos;
    cdk_free( zs );
    if ( zrc == Z_OK || zrc == Z_STREAM_END )
        zrc = 0;
    return zrc;
} /* do_compress */

int
do_compress_iobuf( int algo, int flush, CDK_IOBUF inp, CDK_IOBUF *out )
{
    byte *outbuf;
    size_t outlen = 0;
    CDK_IOBUF a;
    CDK_BSTRING inb;
    int rc = 0;

    inb = cdk_iobuf_read_mem( inp, 0 );
    if ( !inb )
        return CDKERR_OUT_OF_CORE;
    outbuf = cdk_alloc_clear( inb->len );
    if ( !outbuf ) {
        cdk_free( inb );
        return CDKERR_OUT_OF_CORE;
    }
    rc = do_compress( algo, Z_NO_FLUSH, inb->d, inb->len, outbuf, &outlen );
    if ( !rc )
        rc = do_compress( algo, Z_FINISH, inb->d, inb->len, outbuf, &outlen );
    if ( rc ) {
        *out = NULL;
        goto fail;
    }
    if ( flush == Z_FINISH ) {
        if ( inp == *out )
            cdk_iobuf_write_mem( inp, outbuf, outlen );
        else {
            cdk_iobuf_from_mem( &a, outbuf, outlen );
            *out = a;
        }
    }
    
fail:
    cdk_free( inb );
    cdk_free( outbuf );

    return rc;
} /* do_compress_iobuf */

static int
do_uncompress( int algo, byte *inbuf, size_t insize,
               byte *outbuf, size_t *ret_outsize )
{
    z_stream *zs;
    byte buf[4096];
    int pos = 0;
    int zrc = 0;
    
    zs = cdk_alloc( sizeof *zs );
    zrc = (algo==1)? inflateInit2( zs, -13 ) : inflateInit( zs );
    if ( zrc )
        goto leave;

    zs->next_in = inbuf;
    zs->avail_in = insize;

    do {
        zs->next_out = buf;
        zs->avail_out = sizeof(buf);

        zrc = inflate( zs, Z_SYNC_FLUSH );
        if ( zrc == Z_STREAM_END ) {
            inflateEnd( zs );
            if ( zs->avail_out ) {
                memcpy( outbuf + pos, buf, sizeof(buf) - zs->avail_out );
                pos += (sizeof(buf) - zs->avail_out);
            }
            break;
        }
        if ( zrc == Z_OK ) {
            memcpy( outbuf + pos, buf, sizeof(buf) - zs->avail_out );
            pos += (sizeof(buf) - zs->avail_out);
        }
        if ( zrc ) {            
            if ( zs->msg )
                fprintf( stderr,"zlib error `%s'.\n", zs->msg );
            fprintf( stderr, "uncompress: inflate returned %d.\n", zrc );
            break;
        }
    } while ( 1 );

leave:
    *ret_outsize = pos;
    cdk_free( zs );
    if ( zrc == Z_OK || zrc == Z_STREAM_END )
        zrc = 0;
    return zrc;
} /* do_uncompress */

int
do_uncompress_iobuf( int algo, CDK_IOBUF inp, CDK_IOBUF *out )
{
    byte *outbuf;
    size_t  outlen = 0;
    CDK_BSTRING inb;
    CDK_IOBUF a;
    int rc = 0;
    
    inb = cdk_iobuf_read_mem( inp, 0 );
    if ( !inb )
        return CDKERR_INV_VALUE;
    outbuf = cdk_alloc_clear( 5 * inb->len ); /* fixme */
    if ( !outbuf ) {
        cdk_free( inb );
        return CDKERR_OUT_OF_CORE;
    }

    rc = do_uncompress( algo, inb->d, inb->len, outbuf, &outlen );
    if ( rc ) {
        *out = NULL;
        goto fail;
    }

    if ( inp == *out )
        cdk_iobuf_write_mem( inp, outbuf, outlen );
    else {
        cdk_iobuf_from_mem( &a, outbuf, outlen );
        *out = inp;
    }
    
fail:
    cdk_free( inb );
    cdk_free( outbuf );
    
    return rc;
} /* do_uncompress_iobuf */

int
cdk_compress_filter( compress_filter_s *zfx, int control, CDK_IOBUF buf )
{
    cdkPKT_compressed cd;
    CDK_PACKET pkt;
    CDK_IOBUF zip, a;
    int rc = 0;
  
    if ( !zfx || !buf )
        return CDKERR_INV_VALUE;

    if ( control == IOBUF_CTRL_UNDERFLOW ) {
        rc = do_uncompress_iobuf( zfx->algo, buf, &buf );        
    }
    else if ( control == IOBUF_CTRL_FLUSH ) {
        if ( !zfx->algo )
            zfx->algo = ZIP_ZIP;
        rc = do_compress_iobuf( zfx->algo, Z_NO_FLUSH, buf, &a );
        if ( !rc )
            do_compress_iobuf( zfx->algo, Z_FINISH, buf, &a );
        cdk_iobuf_new( &zip, cdk_iobuf_get_length( a ) + 4 );
        init_packet( &cd );
        cd.len = cdk_iobuf_get_length( a );
        cd.buf = a;
        cd.algorithm = zfx->algo;
        pkt.pkttype = PKT_COMPRESSED;
        pkt.pkt.compressed = &cd;
        rc = cdk_pkt_build( zip, &pkt );
        if ( rc ) {
            cdk_iobuf_close( a );
            return rc;
        }
        if ( !rc )
            cdk_iobuf_copy( buf, zip, 1 );
        cdk_iobuf_close( zip );
        cdk_iobuf_close( a );
    }
    return rc;
} /* cdk_compress_filter */
#else
int
cdk_compress_filter( void *ctx, int control, IOBUF buf )
{
    return CDKERR_NOT_IMPLEMENT;
}
#endif /*HAVE_LIBZ*/

  
     


  



