/* iobuf.c - File input/output routines.
 *  Copyright (C) 2001, 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "opencdk.h"
#include "iobuf.h"

#define MAX_FILE_SIZE 16777216 /* 16 MB is the limit */
#define IOBUF_DEFSIZE 8192

static const char*
_iobuf_get_mode_str( iobuf_mode_t mode )
{
    switch (mode) {
    case IOBUF_MODE_RD: return "r";
    case IOBUF_MODE_WR: return "w+";
    case IOBUF_MODE_WR|IOBUF_MODE_EXP: return "w+";
	case IOBUF_MODE_RDWR: return "r+";      
	default: return "r";
    }
} /* get_mode_str */

CDK_BSTRING
cdk_iobuf_read_mem( CDK_IOBUF buf, size_t pos )
{
    CDK_BSTRING a;
    
    if ( buf && !pos ) {
        a = cdk_bstring_new( buf->data->d, buf->size );
        if ( a )
            return a;
    }
    else if ( buf && pos < buf->size ) {
        a = cdk_bstring_new( buf->data->d + pos, buf->size - pos );
        if ( a )
            return a;
    }
    return NULL;
} /* cdk_iobuf_read_mem */

int
cdk_iobuf_write_mem( CDK_IOBUF buf, const byte *d, size_t dlen )
{
    if ( !buf || !d )
        return CDKERR_INV_VALUE;

    if ( buf->data ) {
        cdk_free( buf->data );
        buf->data = NULL;
    }
    buf->data = cdk_bstring_new( d, dlen );
    if ( !buf->data )
        return CDKERR_OUT_OF_CORE;
    buf->size = dlen;

    return 0;
} /* cdk_iobuf_write_mem */

void
_cdk_iobuf_hexdump( CDK_IOBUF buf, size_t pos, size_t len )
{
    CDK_BSTRING a;
    int i = 0;
  
    if ( !buf )
        return;

    a = cdk_iobuf_read_mem( buf, 0 );
    if ( !a )
        return;
    if ( !len ) {
        for ( i = 0; i < a->len - pos; i++ )
            printf("%4X", a->d[i] );
    }
    else {
        for ( i = 0; i< (len > a->len? a->len: len); i++ )
            printf("%4X", a->d[i] );
    }
    printf("\n");
} /* _cdk_iobuf_hexdump */    

int
cdk_iobuf_store( CDK_IOBUF buf, const char *file, int release )
{
    FILE *fp;
    CDK_BSTRING a = NULL;
    int rc = 0;
  
    if ( !buf || !file )
        return CDKERR_INV_VALUE;

    fp = fopen( file, "w+" );
    if ( fp == NULL )
        return CDKERR_FILE_CREATE;
    a = cdk_iobuf_read_mem( buf, 0 );
    if ( a ) {
        size_t buflen = cdk_iobuf_get_length( buf );
        if ( a->len > 0 )
            fwrite( a->d, 1, buflen, fp );
        cdk_free( a );
    }
    fclose( fp );
    if ( release )
        cdk_iobuf_close( buf );
    return rc;
} /* cdk_iobuf_store */

int
cdk_iobuf_open( CDK_IOBUF *ret_buf, const char *fname, iobuf_mode_t mode )
{
    CDK_IOBUF c = NULL;
    FILE *fp = NULL;
    const char *fmode;
    int fd, rc = 0;
    struct stat statbuf;
	
    fmode = _iobuf_get_mode_str( mode );
    if ( !fmode )
        return CDKERR_INV_VALUE;
    fp = fopen( fname, fmode );
    if ( fp == NULL )
        return CDKERR_FILE_OPEN;
		
    c = cdk_calloc( 1, sizeof *c );
    if ( !c ) {
        rc = CDKERR_OUT_OF_CORE;
        goto fail;
    }
    fd = fileno( fp );
    if ( fstat( fd, &statbuf ) == -1 ) {
        rc = CDKERR_GENERAL;
        goto fail;
    }
    else if ( statbuf.st_size == 0 ) { /* empty file */
        *ret_buf = NULL;
        rc = CDKERR_GENERAL;
        goto fail;
    }
    else if ( statbuf.st_size > MAX_FILE_SIZE ) {
        *ret_buf = NULL;
        rc = CDKERR_GENERAL;
        goto fail;
    }
    c->name = cdk_strdup( fname );
    if ( !c->name ) {
        rc = CDKERR_OUT_OF_CORE;
        goto fail;
    }
    c->mode = mode;
    c->size = statbuf.st_size;
    c->read_pos = 0;
    c->write_pos = statbuf.st_size;
    c->data = cdk_bstring_new( NULL, c->size );
    if ( !c->data ) {
        rc = CDKERR_OUT_OF_CORE;
        goto fail;
    }
    rc = fread( c->data->d, 1, c->data->len, fp );
    if ( !rc || rc != statbuf.st_size ) {
        rc = CDKERR_FILE_READ;
        goto fail;
    }

    *ret_buf = c;
    rc = 0;
fail:
    if ( rc && c ) {
        cdk_free( c->name );
        cdk_free( c->data );
        cdk_free( c );
    }
    fclose( fp );
    return rc;
} /* cdk_iobuf_open */

CDK_IOBUF
cdk_iobuf_mopen( const char *fname, int mode, int ext )
{
    CDK_IOBUF a = NULL;
    char *buf = NULL;
    size_t len = 0;
    int rc = 0;

    if ( !fname )
        return NULL;

    if ( mode == IOBUF_MODE_SIG ) {
        len = strlen( fname );
        if ( len > 4
             && ( !strcmp(fname + len - 4, ".sig")
                  || ( len > 5 && !strcmp(fname + len - 5, ".sign") )
                  || !strcmp(fname + len - 4, ".asc") )) {
            buf = cdk_strdup( fname );
            buf[ len - (buf[len-1]=='n'? 5 : 4) ] = 0 ;
            cdk_iobuf_open( &a,  buf, IOBUF_MODE_RD );
            cdk_free( buf );
        }
    }
    else if ( mode == IOBUF_MODE_OUT ) {
        buf = cdk_alloc( strlen( fname ) + 4 + 1 );
        if ( !buf )
            return NULL;
        strcpy( stpcpy( buf, fname ),
                ext == 1 ? ".asc" :
                ext == 2 ? ".sig" : ".gpg");
        rc = cdk_iobuf_create( &a, buf );
        if ( rc )
            a = NULL;
    }
  
    return a;
} /* cdk_iobuf_mopen */

int
cdk_iobuf_new( CDK_IOBUF *ret_buf, size_t size )
{
    CDK_IOBUF c = NULL;
    int rc = 0;
  
    if ( !size || size > MAX_FILE_SIZE )
        return CDKERR_INV_VALUE;
  
    c = cdk_calloc( 1, sizeof *c );
    if ( !c )
        return CDKERR_OUT_OF_CORE;
    c->mode = IOBUF_MODE_RDWR;
    c->eof = 0;
    c->size = 0;
    c->read_pos = 0;
    c->write_pos = 0;
    c->data = cdk_bstring_new( NULL, size );
    if ( !c->data ) {
        rc = CDKERR_OUT_OF_CORE;
        goto fail;
    }
    *ret_buf = c;

fail:
    if ( rc )
        cdk_free( c );
    return rc;
} /* cdk_iobuf_new */

int
cdk_iobuf_from_mem( CDK_IOBUF *ret_buf, byte *buf, size_t buflen )
{
    CDK_IOBUF a;
    int rc = 0;

    rc = cdk_iobuf_new( &a, buflen );
    if ( !rc )
        cdk_iobuf_write( a, buf, buflen );
    return rc;
} /* cdk_iobuf_from_mem */    

int
cdk_iobuf_create( CDK_IOBUF *ret_buf, const char *fname )
{
    CDK_IOBUF c = NULL;

    if ( !ret_buf )
        return CDKERR_INV_VALUE;
  
    c = cdk_calloc( 1, sizeof *c );
    if ( !c )
        return CDKERR_OUT_OF_CORE;
    c->mode = IOBUF_MODE_WR | IOBUF_MODE_EXP;
    c->eof = 0;
    c->size = 0;
    c->read_pos = 0;
    c->write_pos = 0;
    c->data = NULL;
    c->name = fname? cdk_strdup( fname ) : NULL;
    *ret_buf = c;

    return 0;
} /* cdk_iobuf_create */

int
cdk_iobuf_expand( CDK_IOBUF buf, size_t size )
{
    if ( !buf )
        return CDKERR_INV_VALUE;

    buf->data = cdk_bstring_realloc( buf->data, buf->data->len + size );
  
    return 0;
} /* cdk_iobuf_expand */

CDK_IOBUF
cdk_iobuf_temp( void )
{
    CDK_IOBUF c = NULL;
	
    c = cdk_calloc( 1, sizeof *c );
    if ( !c )
        return NULL;
    c->mode = IOBUF_MODE_RDWR | IOBUF_FLAG_TMP;
    c->eof = 0;
    c->size = 0;
    c->read_pos = 0;
    c->write_pos = 0;
    c->data = cdk_bstring_new( NULL, IOBUF_DEFSIZE );
    if ( !c->data ) {
        cdk_free( c );
        return NULL;
    }
	
    return c;
} /* cdk_iobuf_temp */

void
cdk_iobuf_close( CDK_IOBUF buf )
{
    const char *fmode;
    FILE *fp = NULL;
	
    if ( !buf )
        return;

    if ( !(buf->mode & IOBUF_FLAG_TMP) && buf->name ) {
        fmode = _iobuf_get_mode_str( buf->mode );
        fp = fopen( buf->name, fmode );
        if ( fp ) {
            CDK_BSTRING a = buf->data;
            size_t len = cdk_iobuf_get_length( buf ) - 1;
            fwrite( a->d, 1, len, fp );
            fclose( fp );
        }
    }
    buf->size = 0;
    buf->read_pos = 0;
    buf->write_pos = 0;
    cdk_free( buf->data ); buf->data = NULL;
    cdk_free( buf->name ); buf->name = NULL;
    cdk_free( buf );
} /* cdk_iobuf_close */

int
cdk_iobuf_copy( CDK_IOBUF d, CDK_IOBUF s, int add )
{
    if ( !d || !s )
        return CDKERR_INV_VALUE;

    if ( !add ) {
        if ( d->data ) {
            cdk_free( d->data );
            d->data = NULL;
        }
        d->size = s->size;
    }
    d->eof = 0;
    d->read_pos = 0;

    if ( !add ) {
        d->data = cdk_bstring_new( NULL, s->data->len );
        if ( !d->data )
            return CDKERR_OUT_OF_CORE;
        memcpy( d->data->d, s->data->d, s->data->len );
    }
    else {
        d->data = cdk_bstring_realloc( d->data, s->data->len + d->data->len );
        if ( !d->data )
            return CDKERR_OUT_OF_CORE;
        memcpy( d->data->d + d->size, s->data->d, s->data->len );
        d->size += s->size;
    }

    return 0;
} /* cdk_iobuf_copy */

int
cdk_iobuf_setmode( CDK_IOBUF buf, int mode )
{
    if ( !buf )
        return CDKERR_INV_VALUE;

    if ( buf->mode < 0 || buf->mode > 4 )
        return CDKERR_INV_MODE;
    buf->mode = mode;

    return 0;
} /* cdk_iobuf_setmode */

int
cdk_iobuf_eof( CDK_IOBUF buf )
{
    if ( !buf )
        return -1;

    return buf->eof;
} /* cdk_iobuf_eof */

size_t
cdk_iobuf_get_length( CDK_IOBUF buf )
{
    if ( !buf )
        return 0;

    return buf->size;
} /* cdk_iobuf_get_length */    

size_t
cdk_iobuf_tell( CDK_IOBUF buf )
{
    if ( !buf )
        return 0;

    return buf->read_pos;
} /* cdk_iobuf_tell */

void
cdk_iobuf_rewind( CDK_IOBUF buf )
{
    if ( !buf )
        return;

    buf->read_pos = 0;
    buf->write_pos = 0;
    buf->eof = 0;
} /* cdk_iobuf_rewind */

int
cdk_iobuf_skip( CDK_IOBUF buf, size_t nbytes )
{
    size_t fsize = 0;
  
    if ( !buf )
        return CDKERR_INV_VALUE;

    if ( nbytes == 0 )
        return 0; /* Nothing to do */

    fsize = cdk_iobuf_get_length( buf );
    if ( fsize == 0 || nbytes > fsize
         || (buf->read_pos + nbytes) > fsize )
        return CDKERR_INV_VALUE;
	
    if ( buf->eof )
        return 0;
  
    buf->read_pos += nbytes;
    if ( buf->read_pos == cdk_iobuf_get_length( buf ) )
        buf->eof = 1;
	
    return 0;
} /* cdk_iobuf_skip */

int
cdk_iobuf_goback( CDK_IOBUF buf, size_t nbytes )
{
    int rc = 0;
  
    if ( !buf )
        return CDKERR_INV_VALUE;

    if ( nbytes == 0 )
        return 0; /* Nothing to do */

    if ( buf->size == 0 || nbytes > cdk_iobuf_get_length( buf )
         || ( buf->read_pos - nbytes ) < 0 )
        rc = CDKERR_INV_VALUE;

    if ( !rc ) {
        buf->read_pos -= nbytes;
        buf->eof = 0;
    }
	
    return rc;
} /* cdk_iobuf_goback */

int
cdk_iobuf_seek( CDK_IOBUF buf, size_t nbytes )
{
    int rc = 0;
  
    if ( !buf )
        return CDKERR_INV_VALUE;

    if ( buf->size == 0 || nbytes > cdk_iobuf_get_length( buf ) )
        rc = CDKERR_INV_VALUE;

    if ( !rc ) {
        buf->read_pos = nbytes;
        buf->write_pos = nbytes;
        if ( ( buf->read_pos < cdk_iobuf_get_length( buf ) && buf->eof ) )
            buf->eof = 0;
        else if ( buf->read_pos == cdk_iobuf_get_length( buf ) )
            buf->eof = 1;
    }    
	
    return rc;
} /* cdk_iobuf_seek */

int
cdk_iobuf_peek( CDK_IOBUF buf, byte *buffer, size_t buflen )
{
    size_t nread = 0;
    int rc = 0;
  
    if ( !buf || !buf )
        return CDKERR_INV_VALUE;

    if ( cdk_iobuf_get_length( buf ) < buflen )
        return CDKERR_GENERAL;

    rc = cdk_iobuf_read( buf, buffer, buflen, &nread );
    if ( rc != 0 )
        nread = -1;
    if ( !rc )
        cdk_iobuf_goback( buf, nread ); /* unget */

    return nread;
} /* cdk_iobuf_peek */

int
cdk_iobuf_get( CDK_IOBUF buf )
{
    size_t fsize = 0;
    int c = 0;

    if ( !buf )
        return -1;
    if ( !(buf->mode & IOBUF_MODE_EXP) && !(buf->mode & IOBUF_MODE_RD) ) {
        buf->eof = -1;
        return -1;
    }    

    fsize = cdk_iobuf_get_length( buf );
    if ( !fsize || buf->read_pos > fsize ) {
        buf->eof = -1;
        return -1;
    }
    else if ( buf->read_pos == fsize ) {
        buf->eof = 1;
        return 0;
    }
    c = buf->data->d[buf->read_pos++];
    return c;
} /* cdk_iobuf_get */

int
cdk_iobuf_put( CDK_IOBUF buf, byte b )
{
    if ( !buf )
        return CDKERR_INV_VALUE;

    if ( !(buf->mode & IOBUF_MODE_WR) ) {
        return CDKERR_INV_MODE;
    }

    if ( !buf->data && (buf->mode & IOBUF_MODE_EXP) )
        buf->data = cdk_bstring_new( NULL, IOBUF_DEFSIZE );
        
    if ( buf->data->len == 0 || buf->write_pos < 0
         || buf->write_pos > buf->data->len ) {
        return CDKERR_INV_VALUE;
    }
    else if ( buf->write_pos == buf->data->len
              && !(buf->mode & IOBUF_MODE_EXP) ) {
        buf->eof = 1;
        return 0;
    }
    else if ( buf->write_pos == buf->data->len
              && (buf->mode & IOBUF_MODE_EXP ) ) {
        buf->data = cdk_bstring_realloc( buf->data, buf->data->len + 4096 );
    }
    buf->data->d[buf->write_pos++] = b;
    buf->size++;
    return 0;
} /* cdk_iobuf_put */	

/* Note that this function only makes sense for text data. The function
   itself returns when a '\n' was detected. The returned data does not
   contain the line ending. */
int
cdk_iobuf_read_line( CDK_IOBUF buf, byte *buffer, size_t buflen,
                     size_t *r_nread)
{
    int b;
    size_t nbytes = 0;

    if ( !buf || !buffer )
        return CDKERR_INV_VALUE;

    while ( buflen-- && !cdk_iobuf_eof(buf) ) {
        if ( (b = cdk_iobuf_get( buf )) == -1 )
            return CDKERR_FILE_READ;
        if ( b == '\r' ) {
            int n = cdk_iobuf_get( buf );
            if ( n != '\n' )
                return -1;
            buffer[ nbytes++ ] = '\0';
            break;
        }
        else if ( b == '\n' ) {
            buffer[nbytes++] = '\0';
            break;
        }
        buffer[nbytes++] = b;
    }
	
    if ( cdk_iobuf_eof( buf ) )
        nbytes--;
    if ( r_nread )
        *r_nread = nbytes;
  
    return 0;
} /* cdk_iobuf_read_line */

/* Same as read_line but if the buffer is too short a larger one will
   be allocated up to some limit *max_length. */
int
cdk_iobuf_get_line( CDK_IOBUF buf, byte **addr_of_buffer,
                    unsigned *length_of_buffer, unsigned *max_length )
{
    int c = 0;
    char *buffer = *addr_of_buffer;
    unsigned length = *length_of_buffer;
    unsigned nbytes = 0;
    unsigned maxlen = *max_length;
    char *p = NULL;

    if ( !buf )
        return CDKERR_INV_VALUE;
  
    if ( !buffer ) {
        /* must allocate a new buffer */
        length = 256;
        buffer = cdk_alloc_clear( length );
        if ( !buffer )
            return CDKERR_OUT_OF_CORE;
        *addr_of_buffer = buffer;
        *length_of_buffer = length;  
    }
    length -= 3; /* reserve 3 bytes (cr,lf,eol) */
    p = buffer;
  
    while ( !cdk_iobuf_eof( buf ) ) {
        if ( (c = cdk_iobuf_get( buf )) == -1 )
            break;
        if ( nbytes == length ) {
            /* increase the buffer */
            if ( length > maxlen  ) {
                /* this is out limit */
                /* skip the rest of the line */
                while ( c != '\n' ) {
                    if ( (c = cdk_iobuf_get( buf )) == -1 )
                        break;
                }
                *p++ = '\n';
                /* always append a LF (we have reserved space) */
                nbytes++;
                *max_length = 0; /* indicate truncation */
                break;
            }
            length += 3; /* correct for the reserved byte */
            length += length < 1024? 256 : 1024;
            buffer = cdk_realloc( buffer, length );
            if ( !buffer )
                return CDKERR_OUT_OF_CORE;
            *addr_of_buffer = buffer;
            *length_of_buffer = length;
            length -= 3; /* and reserve again */
            p = buffer + nbytes;      
        }
        *p++ = c;
        nbytes++;
        if ( c == '\n' )
            break;
    }
    *p = 0; /* make sure the line is a string */
  
    return nbytes;
} /* cdk_iobuf_get_line */

int
cdk_iobuf_read( CDK_IOBUF buf, byte *buffer, size_t buflen, size_t *r_nread )
{
    size_t nbytes = 0;
    int b = 0;
    int rc = 0;

    if ( !buf || !buffer )
        return CDKERR_INV_VALUE;

    while ( buflen-- && !cdk_iobuf_eof( buf ) ){
        if ( (b = cdk_iobuf_get(buf)) == -1 ) { /* error */
            rc = CDKERR_FILE_READ;
            goto fail;
        }
        buffer[nbytes++] = b;
    }

    if ( cdk_iobuf_eof(buf) && nbytes )
        nbytes--;
    if ( r_nread )
        *r_nread = nbytes;

fail:
    return rc;
} /* cdk_iobuf_read */

int
cdk_iobuf_write( CDK_IOBUF buf, const byte *buffer, size_t buflen )
{
    byte b = 0;
    int rc = 0;

    if ( !buf || !buffer )
        return CDKERR_INV_VALUE;

    while ( buflen-- ) {
        b = *buffer++;
        rc = cdk_iobuf_put( buf, b );
        if ( rc || buf->eof == -1 ) {
            rc = CDKERR_FILE_WRITE;
            break;
        }
    }
	
    return rc;
} /* cdk_iobuf_write */

