/* memory.c - Gcrypt memory wrappers
 *  Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <gcrypt.h>

#include "opencdk.h"

void cdk_printf( const char *fmt, ... );

static void *(*alloc_func)(size_t n) = gcry_xmalloc;
static void *(*alloc_secure_func)(size_t n) = gcry_malloc_secure;
static void *(*realloc_func)(void *p, size_t n) = gcry_realloc;
static void *(*calloc_func)(size_t m, size_t n) = gcry_xcalloc;
static void (*free_func)(void*) = gcry_free;
static int malloc_hooks = 0;

static void
out_of_core(void)
{
    cdk_printf( "\n ** fatal error: out of memory **\n" );
} /* out_of_core */

void
cdk_set_malloc_hooks( void *(*new_alloc_func)(size_t n),
                      void *(*new_alloc_secure_func)(size_t n),
                      void *(*new_realloc_func)(void *p, size_t n),
                      void *(*new_calloc_func)(size_t m, size_t n),
                      void (*new_free_func)(void*) )
{
    alloc_func = new_alloc_func;
    alloc_secure_func = new_alloc_secure_func;
    realloc_func = new_realloc_func;
    calloc_func = new_calloc_func;
    free_func = new_free_func;
    malloc_hooks = 1;
} /* cdk_set_malloc_hooks */

int
cdk_malloc_hook_initialized( void )
{
    return malloc_hooks;
} /* cdk_malloc_hook_initialized */

void*
cdk_alloc( size_t size )
{
    void *p = alloc_func( size );
    if ( !p )
        out_of_core( );
    return p;
} /* cdk_alloc */

void*
cdk_alloc_clear( size_t size )
{
    void *p = alloc_func( size );
    if ( !p )
        out_of_core( );
    memset( p, 0, size );
    return p;
} /* cdk_alloc_clear */

/**
 * cdk_secmem_init - Control secure memory.
 * @size: The size of the secure memory pool.
 *
 * Initialize the secure memory in gcrypt. This function is static
 * in the way that you only need to call it once and then the pool
 * will be available the next time you request secure memory. For
 * the special value "0" (size) the privileges are dropped!
 **/
void
cdk_secmem_init( size_t size )
{
    static int init = 0;

    if ( size == 0xffffffff ) {
        init = 0;
        return; /* reset */
    }
    if ( size == 0 ) {
        gcry_control( GCRYCTL_DROP_PRIVS );
        return;
    }
    if ( init == 1 )
        return;
    if ( size >= 65536 )
        size = 16384;
    gcry_control( GCRYCTL_INIT_SECMEM, 16384, 0 );
    gcry_control( GCRYCTL_DISABLE_SECMEM_WARN );
    init = 1;
} /* cdk_secmem_init */

void
cdk_secmem_end( void )
{
    gcry_control( GCRYCTL_TERM_SECMEM );
    cdk_secmem_init( 0xffffffff );
} /* cdk_secmem_end */  

void*
cdk_alloc_secure( size_t size )
{
    void *p = alloc_secure_func( size );
    if ( !p )
        out_of_core( );
    return p;
} /* cdk_alloc_secure */

void*
cdk_alloc_secure_clear( size_t size )
{
    void *p = cdk_alloc_secure( size );
    memset( p, 0, size );
    return p;
} /* cdk_alloc_secure_clear */

void*
cdk_calloc( size_t nmemb, size_t size )
{
    void *p = calloc_func( nmemb, size );
    if ( !p )
        out_of_core( );
    return p;
} /* cdk_calloc */
            
void*
cdk_realloc( void *ptr, size_t size )
{
    void *p = realloc_func( ptr, size );
    if ( !p )
        out_of_core( );
    return p;
} /* cdk_realloc */

void*
cdk_strdup( const char *ptr )
{
    size_t n;
    char *p = NULL;
  
    if ( !ptr )
        return NULL;
    n = strlen( ptr );
    p = cdk_alloc( n+1 );
    strcpy( p, ptr );
    return p;
} /* cdk_strdup */
         
void
cdk_free( void *ptr )
{
    if ( ptr )
        free_func( ptr );
} /* cdk_free */

CDK_BSTRING
cdk_bstring_new( const byte *buffer, size_t length )
{
    CDK_BSTRING a;
    
    a = cdk_alloc_clear( sizeof *a + length - 1  );
    a->len = length;
    if ( buffer )
        memcpy( a->d, buffer, length );
    return a;
} /* cdk_bstring_new */

CDK_BSTRING
cdk_bstring_realloc( CDK_BSTRING src, int length )
{
    CDK_BSTRING t;
    size_t len = 0;

    len = (src? src->len : 0) + length;
    t = cdk_alloc_clear( sizeof *t + len - 1 );
    t->len = len;
    if ( src && src->len ) {
        memcpy( t->d, src->d, src->len );
        cdk_free( src );   
    }
    return t;    
} /* cdk_bstring_realloc */

