/* read-packet.c - Read OpenPGP packets
 *  Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#include <string.h>
#include <stdio.h>

#include "opencdk.h"
#include "iobuf.h"
#include "main.h"

#define BUFSIZE 2048

typedef struct {
    int ctb;
    int type;
    int rfc2440;
    int len;
    int size;
    int partial;
} PKTHEAD;

u32
read_32( CDK_IOBUF buf, int *ret_err )
{
    u32 u = 0;
    int tmp[4], err = 1;
    
    if ( buf ) {
        if ( (tmp[0] = cdk_iobuf_get(buf)) == -1 )
            goto fail;
        if ( (tmp[1] = cdk_iobuf_get(buf)) == -1 )
            goto fail;
        if ( (tmp[2] = cdk_iobuf_get(buf)) == -1 )
            goto fail;
        if ( (tmp[3] = cdk_iobuf_get(buf)) == -1 )
            goto fail;
        u = (tmp[0] << 24)|(tmp[1] << 16)|(tmp[2] << 8)|(tmp[3]);
        err = 0;
    }
fail:
    if ( ret_err ) *ret_err = err;
    return u;
} /* read_u32 */

u16
read_16( CDK_IOBUF buf, int *ret_err )
{
    u16 u = 0;
    int tmp[2], err = 1;
    
    if ( buf ) {
        if ( (tmp[0] = cdk_iobuf_get(buf)) == -1 )
            goto fail;
        if ( (tmp[1] = cdk_iobuf_get(buf)) == -1 )
            goto fail;
        u = (tmp[0] << 8)|(tmp[1]);
        err = 0;
    }
fail:
    if ( ret_err ) *ret_err = err;
    return u;
} /* read_16 */

static int
read_s2k( CDK_IOBUF inp, CDK_STRING2KEY *s2k )
{
    size_t nread = 0;
    int rc = 0;

    if ( !inp || !s2k )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read S2K part\n");

    s2k->mode = cdk_iobuf_get( inp );
    if ( s2k->mode == -1 || s2k->mode > 3 )
        return CDKERR_INV_PACKET;
    s2k->hash_algo = cdk_iobuf_get( inp );
    if ( s2k->mode == 0 )
        ;  /* nothing to do */
    else if ( s2k->mode == 1 || s2k->mode == 3 ) { 
        rc = cdk_iobuf_read( inp, s2k->salt, sizeof(s2k->salt), &nread );
        if ( !rc && nread != sizeof(s2k->salt) )
            return CDKERR_INV_PACKET;
        if ( !rc && s2k->mode == 3 ) {
            if ( (s2k->count = cdk_iobuf_get( inp )) == -1 )
                return CDKERR_INV_PACKET;
        }
    }
  
    return rc;
} /* read_s2k */

static int
read_mpi( CDK_IOBUF inp, CDK_MPI **ret_m, int secure )
{
    CDK_MPI *m = NULL;
    size_t nread = 0, nbits = 0, nbytes = 0;
    int rc = 0;

    if ( !inp || !ret_m )
        return CDKERR_INV_VALUE;

#if 0
    if ( DEBUG_PKT )
        cdk_printf("** read MPI part\n");
#endif
    nbits = read_16( inp, &rc ); nbytes = ( nbits + 7 ) / 8;
    if ( rc || nbits > MAX_MPI_BITS || nbits == 0 )
        return CDKERR_MPI; /* sanity check */
    if ( secure )
        m = cdk_alloc_secure_clear( sizeof *m + nbytes + 2 );
    else
        m = cdk_alloc_clear( sizeof *m + nbytes + 2 );
    if ( !m )
        return CDKERR_OUT_OF_CORE;
    m->bytes = nbytes;
    m->bits = nbits;
    cdk_iobuf_goback( inp, 2 );
    /* we read two additional bytes, because this is the prefix which
       encode the length of the following MPI data. */
    rc = cdk_iobuf_read( inp, m->data, nbytes + 2, &nread );
    if ( !rc && nread != nbytes + 2 )
        rc = CDKERR_MPI;
    *ret_m = m;
    
    return rc;
} /* read_mpi */

static int
read_pkt_length( CDK_IOBUF inp, int *ret_partial )
{
    int c1 = 0, c2 = 0;
    size_t pktlen = 0;
    int rc = 0;
  
    if ( (c1 = cdk_iobuf_get( inp )) == -1 )
        return -1;
    if ( c1 < 224 || c1 == 255 )
        *ret_partial = 0; /* end of partial data */
    if ( c1 < 192 )
        pktlen = c1;
    else if ( c1 >= 192 && c1 <= 223 ) {
        if ( (c2 = cdk_iobuf_get( inp )) == -1 )
            return -1;
        pktlen = ((c1 - 192) << 8) + c2 + 192;
    }
    else if ( c1 == 255 ) {
        pktlen = read_32( inp, &rc );
        if ( rc ) return -1;
    }
    else
        pktlen = 1 << (c1 & 0x1f);
  
    return pktlen;
} /* read_pkt_length */

static int
read_encrypted( CDK_IOBUF inp, size_t pktlen, cdkPKT_encrypted *enc,
                int partial, int mdc )
{
    CDK_IOBUF a;
    byte data[BUFSIZE];
    size_t nread = 0;
    int version = 0 ;
    int rc = 0;
    
    if ( !inp || !enc )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read encrypted packet %d bytes\n", pktlen);

    nread = cdk_iobuf_get_length( inp );
    if ( !nread )
        return CDKERR_GENERAL;
    cdk_iobuf_new( &a, nread );
    enc->buf = a;

    if ( mdc ) {
        version = cdk_iobuf_get( inp );
        if ( version != 1 )
            return CDKERR_INV_PACKET;
        enc->mdc_method = GCRY_MD_SHA1;
        if ( partial ) pktlen--;
    }
    while ( !cdk_iobuf_eof( inp ) ) {
        while ( pktlen > BUFSIZE && !cdk_iobuf_eof( inp ) ) {
            rc = cdk_iobuf_read( inp, data, BUFSIZE, &nread );
            if ( rc )
                goto leave;
            pktlen -= nread;
            if ( nread )
                cdk_iobuf_write( a, data, nread );
        }
        if ( pktlen > 0 && !cdk_iobuf_eof( inp ) ) {
            rc = cdk_iobuf_read( inp, data, pktlen, &nread );
            if ( rc )
                goto leave;
            pktlen -= nread;
            if ( nread )
                cdk_iobuf_write( a, data, nread );
        }
        if ( !partial )
            break;
        else {
            pktlen = read_pkt_length( inp, &partial );
            if ( pktlen == -1 )
                return CDKERR_INV_PACKET;
        }
    }
    enc->len = cdk_iobuf_get_length( a );
    
leave:
    return rc;
} /* read_encrypted */

static int
read_symkey_enc( CDK_IOBUF inp, size_t pktlen, cdkPKT_symkey_enc *ske )
{
    int rc = 0;
    size_t nread = 0;
    size_t minlen = 0;
  
    if ( !inp || !ske )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read symmetric key encrypted packet\n");
  
    ske->version = cdk_iobuf_get( inp );
    if ( ske->version != 4 ) {
        return CDKERR_INV_PACKET;
    }
    
    ske->cipher_algo = cdk_iobuf_get( inp );
    ske->s2k.mode = cdk_iobuf_get( inp );
    switch ( ske->s2k.mode ) {
    case 0: minlen = 0; break;
    case 1: minlen = 8; break;
    case 3: minlen = 9; break;
    }
    ske->s2k.hash_algo = cdk_iobuf_get( inp );
    if ( ske->s2k.mode == 0 )
        ; /* nothing to do */
    if ( ske->s2k.mode == 1 || ske->s2k.mode == 3 ) {
        rc = cdk_iobuf_read( inp, ske->s2k.salt, DIM(ske->s2k.salt), &nread );
        if ( !rc && nread != sizeof(ske->s2k.salt) )
            return CDKERR_INV_PACKET;
        if ( !rc && ske->s2k.mode == 3 )
            ske->s2k.count = cdk_iobuf_get( inp );
    }
    ske->seskeylen = pktlen - 4 - minlen;
    if ( ske->seskeylen > sizeof(ske->seskey) )
        return CDKERR_INV_PACKET;
    for ( nread=0; nread < ske->seskeylen; nread++ ) {
        ske->seskey[nread]= cdk_iobuf_get( inp );
        if ( cdk_iobuf_eof( inp ) ) break;
    }
    return rc;
} /* read_symkey_enc */

static int
read_pubkey_enc( CDK_IOBUF inp, cdkPKT_pubkey_enc *pke )
{
    int rc = 0;
    int i, nenc = 0;

    if ( !inp || !pke )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read public key encrypted packet\n");
  
    pke->version = cdk_iobuf_get( inp );
    if ( pke->version < 2 || pke->version > 3 )
        return CDKERR_INV_PACKET;
    pke->keyid[0] = read_32( inp, NULL );
    pke->keyid[1] = read_32( inp, NULL );
    if ( !pke->keyid[0] && !pke->keyid[1] )
        pke->throw_keyid = 1; /* RFC2440 "speculative" keyID */
    pke->pubkey_algo = cdk_iobuf_get( inp );
    nenc = cdk_pk_get_nenc( pke->pubkey_algo );
    for ( i = 0; i < nenc && !rc; i++ )
        rc = read_mpi( inp, &pke->mpi[i], 0 );
    
    return rc;
} /* read_pubkey_enc */

static int
read_mdc( CDK_IOBUF inp, cdkPKT_mdc *mdc )
{
    size_t n = 0;
    int rc = 0;

    if ( !inp || !mdc )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read MDC packet\n");
  
    rc = cdk_iobuf_read( inp, mdc->hash, 20, &n );
    if ( !rc && n != 20 )
        rc = CDKERR_INV_PACKET;
    return rc;
} /* read_mdc */

static int
read_compressed( CDK_IOBUF inp, size_t pktlen, cdkPKT_compressed *c )
{
    size_t nread = 0, size = 0;
    byte data[BUFSIZE], *buf = NULL;
    int rc = 0;

    if ( !inp || !c )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read compressed packet\n");
  
    c->algorithm = cdk_iobuf_get( inp );
    if ( c->algorithm > 2 )
        return CDKERR_INV_PACKET;
    if ( pktlen ) {
        c->len = pktlen;
        cdk_iobuf_new( &c->buf, pktlen+1 );
        buf = cdk_alloc( pktlen+1 );
        if ( !buf )
            return CDKERR_OUT_OF_CORE;
        rc = cdk_iobuf_read( inp, buf, pktlen, &nread );
        if ( !rc && nread )
            cdk_iobuf_write( c->buf, buf, nread );
        cdk_free( buf );
    }
    else {
        /* don't know the size, so we read until EOF */
        size = cdk_iobuf_get_length( inp );
        cdk_iobuf_new( &c->buf, size+1 );
        while ( !cdk_iobuf_eof( inp ) && !rc ) {
            rc = cdk_iobuf_read( inp, data, BUFSIZE-1, &nread );
            if ( !rc && nread )
                cdk_iobuf_write( c->buf, data, nread );
        }
    }
    return rc;
} /* read_compressed */

static int
read_public_key( CDK_IOBUF inp, cdkPKT_public_key *pk )
{
    int i = 0;
    int rc = 0;
	
    if ( !inp || !pk )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read public key packet\n");
  
    pk->is_invalid = 1; /* default to detect missing self signatures */
    pk->is_revoked = 0;
    pk->has_expired = 0;
	
    pk->version = cdk_iobuf_get( inp );
    if ( pk->version < 2 || pk->version > 4 )
        return CDKERR_INV_PACKET;
    if ( pk->version < 4 ) {
        pk->expiredate = read_16( inp, &rc );
        if ( rc )
            return CDKERR_INV_PACKET;
    }
    pk->timestamp = read_32( inp, &rc );
    if ( rc )
        return CDKERR_INV_PACKET;
    pk->pubkey_algo = cdk_iobuf_get( inp );
    for ( i = 0; i < cdk_pk_get_npkey( pk->pubkey_algo ) && !rc; i++ )
        rc = read_mpi( inp, &pk->mpi[i], 0 );
    pk->pubkey_usage = openpgp_pk_algo_usage( pk->pubkey_algo );
    
    return rc;
} /* read_public_key */

static int
read_public_subkey( CDK_IOBUF inp, cdkPKT_public_key *pk )
{
    if ( !inp || !pk )
        return CDKERR_INV_VALUE;
  
    return read_public_key( inp, pk );
} /* read_public_subkey */

static int
read_secret_key( CDK_IOBUF inp, size_t pktlen, cdkPKT_secret_key *sk )
{
    size_t p1 = 0, p2 = 0, nread = 0;
    int i = 0, blklen = 0, nskey = 0;
    int rc = 0;

    if ( !inp || !sk || !sk->pk )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read secret key\n");
  
    p1 = cdk_iobuf_tell( inp );
    rc = read_public_key( inp, sk->pk );
    if ( rc )
        return rc;
  
    sk->s2k_usage = cdk_iobuf_get( inp );
    sk->protect.sha1chk = 0;
    if ( sk->s2k_usage == 254 || sk->s2k_usage == 255 ) {
        sk->protect.sha1chk = (sk->s2k_usage == 254);
        sk->protect.algo = cdk_iobuf_get( inp );
        rc = read_s2k( inp, &sk->protect.s2k );
        if ( rc )
            return rc;
        blklen = gcry_cipher_get_algo_blklen(sk->protect.algo);
        sk->protect.ivlen = blklen;
        rc = cdk_iobuf_read( inp, sk->protect.iv, sk->protect.ivlen, &nread );
        if ( !rc && nread != sk->protect.ivlen )
            return CDKERR_INV_PACKET;
    }
    else
        sk->protect.algo = sk->s2k_usage;
    if ( sk->protect.algo == GCRY_CIPHER_NONE ) {
        sk->csum = 0;
        nskey = cdk_pk_get_nskey( sk->pk->pubkey_algo );
        for ( i = 0; i < nskey && !rc; i++)
            rc = read_mpi( inp, &sk->mpi[i], 1 );
        if ( !rc ) {
            sk->csum = read_16( inp, &rc );
            sk->is_protected = 0;
            if ( rc )
                return CDKERR_INV_PACKET;
        }
    }
    else if ( sk->pk->version < 4 ) {
        /* mpi size isn't encrypted! */
        nskey = cdk_pk_get_nskey( sk->pk->pubkey_algo );
        for ( i = 0; i < nskey && !rc; i++ )
            rc = read_mpi( inp, &sk->mpi[i], 1 );
        if ( !rc ) {
            sk->csum = read_16( inp, &rc );
            sk->is_protected = 1;
            if ( rc )
                return CDKERR_INV_PACKET;
        }
    }
    else {
        /* we need to read the rest of the packet because we don't
           have any information how long the encrypted mpi's are */
        p2 = cdk_iobuf_tell( inp );
        p2 -= p1;
        sk->enclen = pktlen-p2;
        if ( sk->enclen < 2 )
            return CDKERR_INV_PACKET; /* at least 16 bits for the checksum! */
        sk->encdata = cdk_alloc( sk->enclen+1 );
        if ( !sk->encdata )
            return CDKERR_OUT_OF_CORE;
        cdk_iobuf_read( inp, sk->encdata, sk->enclen, &nread );
        nskey = cdk_pk_get_nskey( sk->pk->pubkey_algo );
        for ( i = 0; i < nskey; i++ ) {
            sk->mpi[i] = cdk_alloc_secure( sizeof **sk->mpi + MAX_MPI_BYTES );
            if ( !sk->mpi[i] )
                return CDKERR_OUT_OF_CORE;
        }
        sk->is_protected = 1;
    }
    sk->is_primary = 1;
    copy_pk_parts_to_sk( sk->pk, sk );
  
    return 0;
} /* read_secret_key */

static int
read_secret_subkey( CDK_IOBUF inp, size_t pktlen, cdkPKT_secret_key *sk )
{
    int rc = 0;
  
    if ( !inp || !sk || !sk->pk )
        return CDKERR_INV_VALUE;
        
    rc = read_secret_key( inp, pktlen, sk );
    sk->is_primary = 0;
  
    return rc;
} /* read_secret_subkey */

static int
read_attribute( CDK_IOBUF inp, size_t pktlen, cdkPKT_user_id *attr )
{
    size_t nread = 0;
    byte *buf;
    const byte *p;
    int len = 0;
    int rc = 0;

    if ( !inp || !attr || !pktlen )
        return CDKERR_INV_VALUE;
    
    strcpy( attr->name, "[attribute]" );
    attr->len = strlen( attr->name );
    buf = cdk_alloc_clear( pktlen );
    rc = cdk_iobuf_read( inp, buf, pktlen, &nread );
    if ( rc ) {
        cdk_free( buf );
        return CDKERR_INV_PACKET;
    }
    p = buf;
    len = *p++;
    if ( len == 255 ) {
        len = p[0] << 24 | p[1] << 16 | p[2] << 8 | p[3];
        p += 4;
        pktlen -= 4;
    }
    else if ( len >= 192 ) {
        if ( pktlen < 2 ) {
            cdk_free( buf );
            return CDKERR_INV_PACKET;
        }
        len = ( (len - 192) << 8 ) + *p + 192;
        p++;
        pktlen--;
    }
    if ( *p != 1 ) { /* ATTRIBUTE IMAGE */
        cdk_free( buf );
        return CDKERR_INV_PACKET;
    }
    p++;
    
    attr->attrib_img = cdk_alloc_clear( len );
    attr->attrib_len = len;
    memcpy( attr->attrib_img, p, len );
    cdk_free( buf );
    
    return rc;
} /* read_attribute */

static int
read_user_id( CDK_IOBUF inp, size_t pktlen, cdkPKT_user_id *user_id )
{
    size_t nread = 0;
    int rc = 0;
	
    if ( !inp || !user_id || !pktlen )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read user ID packet\n");
  
    user_id->len = pktlen;
    rc = cdk_iobuf_read( inp, user_id->name, pktlen, &nread );
    if ( !rc && nread != pktlen )
        return CDKERR_INV_PACKET;
    user_id->name[ nread ]= '\0';

    return rc;
} /* read_user_id */

static int
read_subpkt( CDK_IOBUF inp, CDK_SUBPKT *r_ctx )
{
    byte c, c1;
    size_t size = 0, nread = 0;
    CDK_SUBPKT ctx = NULL;
    int rc = 0;
	
    if ( !inp || !r_ctx )
        return 0;

#if 0
    if ( DEBUG_PKT )
        cdk_printf("** read sub packet");
#endif
  
    ctx = *r_ctx;
    c = cdk_iobuf_get( inp );
    if ( c == 255 ) {
        ctx = cdk_subpkt_add( ctx );
        size = read_32( inp, NULL );
        ctx->size = size - 4;
        size += 4;
    }
    else if (c >= 192 && c < 255) {
        c1 = cdk_iobuf_get( inp );
        if (c1 == 0)
            return 0;
        ctx = cdk_subpkt_add( ctx );
        size = ((c - 192) << 8) + c1 + 192;
        ctx->size = size - 2;
        size += 2;
    }
    else if ( c < 192 ) {
        ctx = cdk_subpkt_add( ctx );
        ctx->size = c-1;
        size = c += 1;
	}
    else
        return 0;

    if ( DEBUG_PKT )
        cdk_printf(" `%d' bytes\n", size);

    ctx->type = cdk_iobuf_get( inp );
    ctx->data = cdk_alloc( ctx->size+1 );
    if ( !ctx->data )
        return CDKERR_OUT_OF_CORE;
    rc = cdk_iobuf_read( inp, ctx->data, ctx->size, &nread );
    if ( rc )
        return 0;
    *r_ctx = ctx;

    return size;
} /* read_subpkt */

static int
read_onepass_sig( CDK_IOBUF inp, cdkPKT_onepass_sig *sig )
{
    if ( !inp || !sig )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read one pass signature packet\n");
  
    sig->version = cdk_iobuf_get( inp );
    if ( sig->version != 3 )
        return CDKERR_INV_PACKET;
    sig->sig_class = cdk_iobuf_get( inp );
    sig->digest_algo = cdk_iobuf_get( inp );
    sig->pubkey_algo = cdk_iobuf_get( inp );
    sig->keyid[0] = read_32( inp, NULL );
    sig->keyid[1] = read_32( inp, NULL );
    sig->last = cdk_iobuf_get( inp );
  
    return 0;
} /* read_onepass_sig */

static int
read_signature( CDK_IOBUF inp, cdkPKT_signature *sig )
{
    struct cdk_subpkt_s *p = NULL;
    int i, size;
    int rc = 0;

    if ( !inp || !sig )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read signature packet\n");
  
    sig->version = cdk_iobuf_get( inp );
    if (sig->version < 2 || sig->version > 4)
        return CDKERR_INV_PACKET;
    if (sig->version < 4) {
        if ( cdk_iobuf_get( inp ) != 5 )
            return CDKERR_INV_PACKET;
        sig->sig_class = cdk_iobuf_get( inp );
        sig->timestamp = read_32( inp, NULL );
        sig->keyid[0] = read_32( inp, NULL );
        sig->keyid[1] = read_32( inp, NULL );
        sig->pubkey_algo = cdk_iobuf_get( inp );
        sig->digest_algo = cdk_iobuf_get( inp );
        sig->digest_start[0] = cdk_iobuf_get( inp );
        sig->digest_start[1] = cdk_iobuf_get( inp );
        for (i=0; i<cdk_pk_get_nsig(sig->pubkey_algo) && !rc; i++) 
            rc = read_mpi( inp, &sig->mpi[i], 0 );
    }
    else {
        if ( !sig->hashed )
            cdk_subpkt_new(&sig->hashed);
        if ( !sig->unhashed )
            cdk_subpkt_new(&sig->unhashed);		    
        sig->sig_class = cdk_iobuf_get( inp );
        sig->pubkey_algo = cdk_iobuf_get( inp );
        sig->digest_algo = cdk_iobuf_get( inp );
        sig->hashed_size = read_16( inp, NULL );
        size = sig->hashed_size;
        while ( size ) {
            i = read_subpkt( inp, &sig->hashed );
            if (i == 0) return CDKERR_INV_PACKET;
            else size -= i;
        }
        sig->unhashed_size = read_16( inp, NULL );
        size = sig->unhashed_size;
        while ( size ) {
            i = read_subpkt( inp, &sig->unhashed );
            if (i == 0) return CDKERR_INV_PACKET;
            else size -= i;
        }
        /* Setup the standard packet entries, so we can use V4
           signatures similar to V3. */
        for ( p = sig->unhashed; p && p->size; p = p->next) {
            if (p->type == SIGSUBPKT_ISSUER) {
                sig->keyid[0] = buffer_to_u32( p->data );
                sig->keyid[1] = buffer_to_u32( p->data + 4 );
            }
        }
        for ( p = sig->hashed; p && p->size; p = p->next ) {
            if ( p->type == SIGSUBPKT_SIG_CREATED )
                sig->timestamp = buffer_to_u32(p->data);
            else if ( p->type == SIGSUBPKT_SIG_EXPIRE ) {
                sig->expiredate = buffer_to_u32(p->data);
                if ( sig->expiredate>0 && sig->expiredate < make_timestamp() )
                    sig->flags.expired = 1;                         
            }
            else if ( p->type == SIGSUBPKT_POLICY )
                sig->flags.policy_url = 1;
            else if ( p->type == SIGSUBPKT_NOTATION )
                sig->flags.notation = 1;
            else if ( p->type == SIGSUBPKT_REVOCABLE )
                sig->flags.revocable = 1;
            else if ( p->type == SIGSUBPKT_EXPORTABLE && p->data[0] == 1 )
                sig->flags.exportable = 1;
        }
        sig->digest_start[0] = cdk_iobuf_get( inp );
        sig->digest_start[1] = cdk_iobuf_get( inp );
        for (i=0; i<cdk_pk_get_nsig(sig->pubkey_algo) && !rc; i++) 
            rc = read_mpi( inp, &sig->mpi[i], 0 );
    }
	
    return rc;
} /* read_signature */

static int
read_plaintext( CDK_IOBUF inp, size_t pktlen, cdkPKT_plaintext *pt )
{
    byte *buf = NULL;
    size_t nread = 0;
    int rc = 0;
  
    if ( !inp || !pt )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf("** read plaintext packet\n");
  
    pt->mode = cdk_iobuf_get( inp );
    pt->namelen = cdk_iobuf_get( inp );
    if ( pt->namelen  ) {
        rc = cdk_iobuf_read( inp, pt->name, pt->namelen, &nread );
        if ( !rc && nread != pt->namelen )
            return CDKERR_INV_PACKET;
        pt->name[ pt->namelen ] = '\0';
    }
    pt->timestamp = read_32( inp, NULL );
    pktlen -= (6+pt->namelen);
    if ( pktlen ) {
        buf = cdk_alloc_clear( pktlen+1 );
        if ( !buf )
            return CDKERR_OUT_OF_CORE;
        rc = cdk_iobuf_read( inp, buf, pktlen, &nread );
        if ( rc ) {
            cdk_free( buf );
            return rc;
        }
        pt->len = pktlen;
        cdk_iobuf_from_mem( &pt->buf, buf, pt->len );
        cdk_free( buf );
    }

    return rc;
} /* read_plaintext */

int
cdk_pkt_parse( CDK_IOBUF inp, CDK_PACKET *pkt )
{
    PKTHEAD pkth = {0};
    byte c = 0, c1 = 0;
    int use_mdc = 0;
    int rc = 0;
  
    if ( !inp || !pkt )
        return CDKERR_INV_VALUE;

    pkth.ctb = cdk_iobuf_get( inp );
    if ( cdk_iobuf_eof( inp ) )
        return CDKERR_EOF;
    else if ( !pkth.ctb || pkth.ctb == -1 )
        return CDKERR_INV_PACKET;
    pkth.size++;
    if ( !(pkth.ctb & 0x80) ) {
        size_t pos = 0;
        pos = cdk_iobuf_tell( inp );
        cdk_printf("No valid openpgp data ctb=%02X; pos=%d\n", pkth.ctb, pos);
        return CDKERR_INV_PACKET;
    }
    if ( pkth.ctb & 0x40 ) {
        pkth.type = pkth.ctb & 0x3f;
        pkth.rfc2440 = 1;
    }
    else {
        pkth.type = pkth.ctb & 0x3f;
        pkth.type >>= 2;
        pkth.rfc2440 = 0; /* RFC1991 message format */
    }
    if ( pkth.type > 63 ) {
        cdk_printf( "Unknown or experimental packet type (%d)\n", pkth.type );
        return CDKERR_INV_PACKET;
    }
    if ( pkth.rfc2440 ) {
        c = cdk_iobuf_get( inp );
        pkth.size++;
        if ( c < 192 )
            pkth.len = c;
        else if ( c >= 192 && c <= 223 ) {
            c1 = cdk_iobuf_get( inp );
            pkth.size++;
            pkth.len = ((c - 192) << 8) + c1 + 192;               
        }
        else if ( c == 255 ) {
            pkth.len = read_32( inp, NULL );
            pkth.size += 4;
        }
        else {
            pkth.len = 1 << (c & 0x1f);
            pkth.partial = 1;
        }
    }
    else {
        if ( (pkth.ctb & 0x03) == 0 ) { 
            pkth.len = cdk_iobuf_get( inp );
            pkth.size++;
        }
        else if ( (pkth.ctb & 0x03) == 1 ) {
            pkth.len = read_16( inp, NULL );
            pkth.size += 2;
        }
        else if ( (pkth.ctb & 0x03) == 2 ) {
            pkth.len = read_32( inp, NULL );
            pkth.size += 4;
        }
        else {
            pkth.len = 0;
            pkth.size = 0;
        }
    }
  
    pkt->pkttype = pkth.type;
    pkt->pktlen = pkth.len;
    pkt->pktsize = pkth.size + pkth.len;

    switch (pkt->pkttype) {
    case PKT_ATTRIBUTE:
        pkt->pkt.user_id = cdk_alloc_clear( sizeof *pkt->pkt.user_id
                                            + pkt->pktlen );
        if ( !pkt->pkt.user_id )
            return CDKERR_OUT_OF_CORE;
        rc = read_attribute( inp, pkth.len, pkt->pkt.user_id );
        pkt->pkttype = PKT_USER_ID; /* treated as an user id */
        break;
        
    case PKT_USER_ID:
        pkt->pkt.user_id = cdk_alloc_clear( sizeof *pkt->pkt.user_id
                                            + pkt->pktlen );
        if ( !pkt->pkt.user_id )
            return CDKERR_OUT_OF_CORE;
        rc = read_user_id( inp, pkth.len, pkt->pkt.user_id );
        break;

    case PKT_PUBLIC_KEY:
        pkt->pkt.public_key = cdk_alloc_clear( sizeof *pkt->pkt.public_key );
        if ( !pkt->pkt.public_key )
            return CDKERR_OUT_OF_CORE;
        rc = read_public_key( inp, pkt->pkt.public_key );
        break;
      
    case PKT_PUBLIC_SUBKEY:
        pkt->pkt.public_key = cdk_alloc_clear( sizeof *pkt->pkt.public_key );
        if ( !pkt->pkt.public_key )
            return CDKERR_OUT_OF_CORE;
        rc = read_public_subkey( inp, pkt->pkt.public_key );
        break;

    case PKT_SECRET_KEY:
        pkt->pkt.secret_key = cdk_alloc_clear( sizeof *pkt->pkt.secret_key );
        if ( !pkt->pkt.secret_key )
            return CDKERR_OUT_OF_CORE;
        pkt->pkt.secret_key->pk =
            cdk_alloc_clear( sizeof *pkt->pkt.secret_key->pk );
        if ( !pkt->pkt.secret_key->pk )
            return CDKERR_OUT_OF_CORE;
        rc = read_secret_key( inp, pkth.len, pkt->pkt.secret_key );
        break;

    case PKT_SECRET_SUBKEY:
        pkt->pkt.secret_key = cdk_alloc_clear( sizeof *pkt->pkt.secret_key );
        if ( !pkt->pkt.secret_key )
            return CDKERR_OUT_OF_CORE;
        pkt->pkt.secret_key->pk =
            cdk_alloc_clear( sizeof *pkt->pkt.secret_key->pk );
        if ( !pkt->pkt.secret_key->pk )
            return CDKERR_OUT_OF_CORE;
        rc = read_secret_subkey( inp, pkth.len, pkt->pkt.secret_key );
        break;

    case PKT_PLAINTEXT:
        pkt->pkt.plaintext = cdk_alloc_clear( sizeof *pkt->pkt.plaintext
                                              + pkt->pktlen );
        if ( !pkt->pkt.plaintext )
            return CDKERR_OUT_OF_CORE;
        rc = read_plaintext( inp, pkth.len, pkt->pkt.plaintext );
        break;

    case PKT_ONEPASS_SIG:
        pkt->pkt.onepass_sig = cdk_alloc_clear( sizeof *pkt->pkt.onepass_sig );
        if ( !pkt->pkt.onepass_sig )
            return CDKERR_OUT_OF_CORE;
        rc = read_onepass_sig( inp, pkt->pkt.onepass_sig );
        break;
      
    case PKT_SIGNATURE:
        pkt->pkt.signature = cdk_alloc_clear( sizeof *pkt->pkt.signature );
        if ( !pkt->pkt.signature )
            return CDKERR_OUT_OF_CORE;
        rc = read_signature( inp, pkt->pkt.signature );
        break;

    case PKT_ENCRYPTED_MDC:
    case PKT_ENCRYPTED:
        pkt->pkt.encrypted = cdk_alloc_clear( sizeof *pkt->pkt.encrypted );
        if ( !pkt->pkt.encrypted )
            return CDKERR_OUT_OF_CORE;
        use_mdc = (pkt->pkttype == PKT_ENCRYPTED_MDC)? 1 : 0;
        rc = read_encrypted( inp, pkth.len, pkt->pkt.encrypted,
                             pkth.partial, use_mdc );
        break;

    case PKT_SYMKEY_ENC:
        pkt->pkt.symkey_enc = cdk_alloc_clear( sizeof *pkt->pkt.symkey_enc );
        if ( !pkt->pkt.symkey_enc )
            return CDKERR_OUT_OF_CORE;
        rc = read_symkey_enc( inp, pkth.len, pkt->pkt.symkey_enc );
        break;

    case PKT_PUBKEY_ENC:
        pkt->pkt.pubkey_enc = cdk_alloc_clear( sizeof *pkt->pkt.pubkey_enc );
        if ( !pkt->pkt.pubkey_enc )
            return CDKERR_OUT_OF_CORE;
        rc = read_pubkey_enc( inp, pkt->pkt.pubkey_enc );
        break;

    case PKT_COMPRESSED:
        pkt->pkt.compressed = cdk_alloc_clear( sizeof *pkt->pkt.compressed );
        if ( !pkt->pkt.compressed )
            return CDKERR_OUT_OF_CORE;
        rc = read_compressed( inp, pkth.len, pkt->pkt.compressed );
        break;

    case PKT_MDC:
        pkt->pkt.mdc = cdk_alloc_clear( sizeof *pkt->pkt.mdc );
        if ( !pkt->pkt.mdc )
            return CDKERR_OUT_OF_CORE;
        rc = read_mdc( inp, pkt->pkt.mdc );
        break;
      
    default:
        /* skip all packets we don't understand */
        cdk_iobuf_skip( inp, pkth.len );
        break;
    }
  
    return rc;
} /* cdk_pkt_parse */  

int
cdk_pkts_parse( CDK_IOBUF inp, CDK_KBNODE *r_pkt )
{
    CDK_PACKET *pkt = NULL;
    CDK_KBNODE p = NULL, t = NULL;
    int rc = 0;
    
    if ( !inp || !r_pkt )
        return CDKERR_INV_VALUE;

    cdk_iobuf_rewind( inp );
    while ( !cdk_iobuf_eof( inp )  ) {
        pkt = cdk_alloc_clear( sizeof *pkt );
        rc = cdk_pkt_parse( inp, pkt );
        if ( rc )
            break;
        t = cdk_kbnode_new( pkt );
        if ( !p ) p = t;
        else cdk_kbnode_add( p, t );
    }
    *r_pkt = p;
  
    return rc;
} /* cdk_pkts_parse */
