/* seskey.c - Session key routines
 *  Copyright (C) 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <assert.h>
#include <stdio.h>

#include "opencdk.h"
#include "main.h"

#define REPEAT_KEY_LOOP 16

#define gcry_md_get_asnoid( algo, asnbuf, asnlen ) \
gcry_md_algo_info( (algo), GCRYCTL_GET_ASNOID, (asnbuf), (asnlen) )

/* We encode the MD in this way:
 *
 * 0  1 PAD(n bytes)   0  ASN(asnlen bytes)  MD(len bytes)
 *
 * PAD consists of FF bytes.
 */
static int
do_encode_md( byte **r_frame, size_t *r_flen, const byte *md, int algo,
              size_t len, unsigned nbits, const byte *asn, size_t asnlen )
{
    int i, nframe = (nbits+7)/8;
    byte *frame = NULL;
    size_t n = 0;

    if ( !asn || !md )
        return CDKERR_INV_VALUE;
  
    if ( len + asnlen + 4 > nframe )
        return CDKERR_GENERAL;

    frame = cdk_alloc( nframe );
    if ( !frame )
        return CDKERR_OUT_OF_CORE;
    frame[n++] = 0;
    frame[n++] = 1;
    i = nframe - len - asnlen -3;
    if ( i < 0 ) {
        cdk_free( frame );
        return CDKERR_INV_VALUE;
    }
    memset( frame+n, 0xff, i );
    n += i;
    frame[n++] = 0;
    memcpy( frame+n, asn, asnlen );
    n += asnlen;
    memcpy( frame+n, md, len );
    n += len;
    if ( n != nframe ) {
        cdk_free( frame );
        return CDKERR_INV_VALUE;
    }
    if ( r_frame )
        *r_frame = frame;
    if ( r_flen )
        *r_flen = n;
  
    return 0;
} /* do_encode_md */

/* RFC2437 format:
 *  
 *  0  2  RND(n bytes)  0  [A  DEK(k bytes)  CSUM(2 bytes)]
 *  
 *  RND - randomized bytes for padding.
 *  A - cipher algorithm.
 *  DEK - random session key.
 *  CKSUM - algebraic checksum of the DEK.
 */
int
_cdk_encode_sesskey( GCRY_MPI *esk, CDK_DEK *dek, int nbits )
{
    int rc = 0;
    int i = 0;
    int nframe;
    byte *p, *frame;
    u16 chksum = 0;
    size_t n = 0;
    GCRY_MPI a = NULL;

    if ( !esk || !dek )
        return CDKERR_INV_VALUE;

    for ( i = 0; i < dek->keylen; i++ )
        chksum += dek->key[i];
    nframe = ( nbits+7 ) / 8;
    frame = cdk_alloc_secure( nframe+1 );
    if ( !frame )
        return CDKERR_OUT_OF_CORE;
    n = 0;
    frame[n++] = 0x00;
    frame[n++] = 0x02;
    i = nframe - 6 - dek->keylen;
    p = gcry_random_bytes( i, GCRY_STRONG_RANDOM );
    /* replace zero bytes by new values */
    for (;;) {
        int j, k;
        byte *pp;
        
        /* count the zero bytes */
        for ( j = k = 0; j < i; j++ ) { 
            if( !p[j] )
                k++;
        }
        if( !k )
            break; /* okay: no zero bytes */
        k += k/128; /* better get some more */
        pp = gcry_random_bytes( k, GCRY_STRONG_RANDOM );
        for( j = 0; j < i && k ; j++ ) { 
            if( !p[j] )
                p[j] = pp[--k];
        }
        cdk_free( pp );
    }
    memcpy( frame+n, p, i );
    cdk_free( p );
    n += i;
    frame[n++] = 0;
    frame[n++] = dek->algo;
    memcpy( frame+n, dek->key, dek->keylen );
    n += dek->keylen;
    frame[n++] = chksum >> 8;
    frame[n++] = chksum     ;
    rc = gcry_mpi_scan( &a, GCRYMPI_FMT_USG, frame, &nframe );
    if ( rc )
        rc = CDKERR_GCRY;
    cdk_free( frame );
    if ( !rc )
        *esk = a;
  
    return rc;
} /* _cdk_encode_sesskey */

/* Do some tests before it calls do_encode_md that depends on the
   public key algorithm that is used. */
int
_cdk_encode_digest( byte **r_md, size_t *r_mdlen, int pk_algo,
                    const byte *md, int digest_algo, unsigned nbits )
{
    int rc = 0;
  
    if ( !md || !r_md || !r_mdlen )
        return CDKERR_INV_VALUE;

    if ( is_DSA(pk_algo) ) {
        size_t n = gcry_md_get_algo_dlen( digest_algo );
        if ( !n )
            return CDKERR_INV_ALGO;
        *r_md = cdk_alloc( n + 1 );
        if ( !*r_md )
            return CDKERR_OUT_OF_CORE;
        *r_mdlen = n;
        memcpy( *r_md, md, n );
        return 0;
    }
    else {
        byte *asn = NULL;
        size_t asnlen = 0, digest_size = 0;

        rc = gcry_md_get_asnoid( digest_algo, NULL, &asnlen );
        if ( rc || !asnlen )
            return CDKERR_GCRY;
        asn = cdk_alloc( asnlen+1 );
        if ( !asn )
            return CDKERR_OUT_OF_CORE;
        rc = gcry_md_get_asnoid( digest_algo, asn, &asnlen );
        if ( rc )
            return CDKERR_GCRY;
        digest_size = gcry_md_get_algo_dlen( digest_algo );
        rc = do_encode_md( r_md, r_mdlen, md, digest_algo, digest_size,
                           nbits, asn, asnlen);
        cdk_free( asn );
        return rc;
    }
  
    return 0;
} /* _cdk_encode_digest */

/**
 * _cdk_dek_create - Create session key.
 * @dek - the structure to store the data encryption key.
 *
 * Creates a random session key with the algo set in the DEK
 * structure. The routine also tests for weak keys.
 **/
int
_cdk_dek_create( CDK_DEK *dek )
{
    GCRY_CIPHER_HD chd;
    int i, rc;
  
    if ( !dek )
        return CDKERR_INV_VALUE;

    if ( !dek->algo )
        dek->algo = OPENPGP_DEF_CIPHER;
    dek->keylen = gcry_cipher_get_algo_keylen( dek->algo );
    chd = gcry_cipher_open( dek->algo, GCRY_CIPHER_MODE_CFB, 1 );
    if ( !chd )
        return CDKERR_GCRY;
    gcry_randomize( dek->key, dek->keylen, GCRY_STRONG_RANDOM );
    for ( i=0; i<REPEAT_KEY_LOOP; i++ ) {
        rc = gcry_cipher_setkey( chd, dek->key, dek->keylen );
        if ( !rc ) {
            gcry_cipher_close( chd );
            return 0;
        }
        cdk_printf( "weak key created - retrying\n" );
        gcry_randomize( dek->key, dek->keylen, GCRY_STRONG_RANDOM );
    }
    cdk_printf( "cannot avoid weak key!!!\n" );
  
    return CDKERR_WEAK_KEY;
} /* _cdk_dek_create */


