/* trustdb.c - High level interface for ownertrust handling
 *   Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"

#define RECORD_SIZE 40

typedef enum {
    TDB_RECORD_TRUST = 12,
    TDB_RECORD_VALID = 13,
} tdb_records_t;

struct trustdb_rec_s {
    int recno;  
    union {
        struct {
            byte reserved;
            byte fpr[20];
            int ownertrust;
            byte depth;
            u32 validlist;
        } trust;
    
        struct {
            byte reserved;
            byte fpr[20];
            u32 next;
            int valid;
        } valid;
    } u; 
};

typedef struct trustdb_rec_s *TRUSTREC;

static void
trustdb_rec_release( TRUSTREC rec )
{
    if (!rec)
        return;
    rec->recno = 0;
    cdk_free(rec);
} /* cdk_trustdb_rec_release */

static TRUSTREC
trustdb_rec_new( void )
{
    TRUSTREC rec;
    
    rec = cdk_alloc_clear( sizeof *rec );
    if (!rec)
        return NULL;
    rec->recno = 0;
  
    return rec;
} /* trustdb_rec_new */

int
cdk_trustdb_check( const char *file, int req_ver )
{
    CDK_IOBUF a;
    int rc = 0;
    int c = 0;
    byte magic[3];
    size_t nread = 0;

    rc = cdk_iobuf_open( &a, file, IOBUF_MODE_RD );
    if ( rc )
        return rc;
    c = cdk_iobuf_get( a );
    if ( c == -1 || c != 1 )
        return CDKERR_GENERAL;
    rc = cdk_iobuf_read( a, magic, 3, &nread );
    if ( rc || nread != 3 )
        goto fail;
    c = cdk_iobuf_get( a );
    if ( c == -1 || c < req_ver )
        return CDKERR_GENERAL;

fail:
    cdk_iobuf_close( a );
    return rc;
} /* cdk_trustdb_check */

static int
trustdb_rec_parse( CDK_IOBUF buf, TRUSTREC r )
{
    size_t n;
    int recno;
	
    if ( !buf || !r )
        return CDKERR_INV_VALUE;

    if ( (recno = cdk_iobuf_get( buf )) == -1 )
        return -1;
	
    switch (recno) {
    case TDB_RECORD_TRUST: /* trust record: new */
        r->recno = 12;
        r->u.trust.reserved = cdk_iobuf_get( buf );
        cdk_iobuf_read( buf, r->u.trust.fpr, 20, &n );
        r->u.trust.ownertrust = cdk_iobuf_get( buf );
        r->u.trust.depth = cdk_iobuf_get( buf );
        r->u.trust.validlist = 0;
        cdk_iobuf_skip( buf, 4 ); /* validlist */
        cdk_iobuf_skip( buf, RECORD_SIZE-28 );
        if ( r->u.trust.ownertrust == -1 )
            return CDKERR_GENERAL;
        break;
        
    case TDB_RECORD_VALID: /* valid record: new */
        r->recno = 13;
        r->u.valid.reserved = cdk_iobuf_get( buf );
        cdk_iobuf_read( buf, r->u.valid.fpr, 20, &n );
        r->u.valid.valid = cdk_iobuf_get( buf );
        r->u.valid.next = 0;
        cdk_iobuf_skip( buf, 4 ); /* next */
        cdk_iobuf_skip( buf, RECORD_SIZE-27 );
        if ( r->u.valid.valid == -1 )
            return CDKERR_GENERAL;
        break;
		
    default:
        cdk_iobuf_skip( buf, RECORD_SIZE - 1 );
        break;
    }	
    r->recno = recno;
		
    return 0;
} /* trustdb_rec_parse */

TRUSTREC
trustdb_rec_byfpr( CDK_IOBUF buf, int type, const byte *fpr, size_t fprlen )
{
    TRUSTREC rec;
  
    if ( !fpr || !buf )
        return NULL;

    rec = trustdb_rec_new();

    while ( trustdb_rec_parse( buf, rec ) != -1 ) {
        if ( rec->recno != type )
            continue;
        switch ( type ) {
        case TDB_RECORD_VALID:
            if ( !memcmp( fpr, rec->u.valid.fpr, fprlen ) )
                return rec;
            break;
          
        case TDB_RECORD_TRUST:
            if ( !memcmp( rec->u.trust.fpr, fpr, fprlen ) )
                return rec;
            break;
        }
    }
  
    trustdb_rec_release( rec );
    return NULL;
}

int
cdk_trustdb_get_ownertrust( CDK_IOBUF buf, cdkPKT_public_key *pk,
                            int *r_val, int *r_flags )
{
    TRUSTREC rec = NULL;
    byte fpr[20];
    int rc = CDKERR_GENERAL;
    int flags = 0;
  
    if (!buf || !r_val || !r_flags || !pk)
        return CDKERR_INV_VALUE;

    *r_val = TRUST_UNKNOWN;
    cdk_pk_get_fingerprint( pk, fpr );
    cdk_iobuf_rewind( buf );

    rec = trustdb_rec_byfpr( buf, TDB_RECORD_TRUST, fpr, 20 );
    if ( rec ) {
        *r_val = rec->u.trust.ownertrust & TRUST_MASK;
        if (*r_val & TRUST_FLAG_REVOKED)
            flags |= TRUST_FLAG_REVOKED;
        if (*r_val & TRUST_FLAG_SUB_REVOKED)
            flags |= TRUST_FLAG_SUB_REVOKED;
        if (*r_val & TRUST_FLAG_DISABLED)
            flags |= TRUST_FLAG_DISABLED;
        *r_flags = flags;
        rc = 0;
    }
  
    trustdb_rec_release(rec);
  
    return rc;
} /* cdk_trustdb_get_ownertrust */
  
int
cdk_trustdb_get_validity( CDK_IOBUF buf, cdkPKT_user_id *id, int *r_val )
{
    GCRY_MD_HD rmd;
    TRUSTREC rec;
    byte *fpr;
    int rc = CDKERR_GENERAL;
    
    if ( !buf || !r_val || !id )
        return CDKERR_INV_VALUE;

    *r_val = TRUST_UNKNOWN;
    rmd = gcry_md_open( GCRY_MD_RMD160, 0 );
    if ( !rmd )
        return CDKERR_GCRY;
            
    gcry_md_write( rmd, id->name, id->len );
    gcry_md_final( rmd );
    fpr = gcry_md_read( rmd, GCRY_MD_RMD160 );

    cdk_iobuf_rewind( buf );
    rec = trustdb_rec_byfpr( buf, TDB_RECORD_VALID, fpr, 20 );
    if ( rc ) {
        *r_val = rec->u.valid.valid;
        rc = 0;
    }
  
    trustdb_rec_release( rec );
    gcry_md_close( rmd );
  
    return rc;
} /* cdk_trustdb_get_validity */












