/* t-armor.c
 *  Copyright (C) 2001, 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <gcrypt.h>

#include "opencdk.h"

void
test_armor(void)
{
    CDK_IOBUF s;
    int rc;
    CDK_KBNODE pkt, p;
    armor_filter_s afx = {0};
    
    rc = cdk_iobuf_open( &s, "Makefile.asc", IOBUF_MODE_RD );
    if ( rc ) {
        printf( "`%s': Can't open\n", "Makefile.asc" );
        return;
    }

    cdk_set_logging( 1 );
    memset(&afx, 0, sizeof afx);
    rc = cdk_armor_filter( &afx, IOBUF_CTRL_UNDERFLOW, s );
    if ( rc )
        printf("error: `%s'\n", cdk_strerror(rc));
    else {
        /*
          {
          int i;        
          for (i=0; i<cdk_iobuf_get_length(s); i++) 
          printf("%c", cdk_iobuf_get(s));
          }
          */
        cdk_pkts_parse( s, &pkt );
        for (p=pkt; p; p=p->next) {
            printf("packet id `%d'\n", p->pkt->pkttype);
            if (p->pkt->pkttype==18 || p->pkt->pkttype==9)
                printf("size=`%d'\n",
                       cdk_iobuf_get_length(p->pkt->pkt.encrypted->buf));
        }
        cdk_kbnode_release( pkt );
        cdk_iobuf_close( s );
    }
}

static void
dump_iobuf( CDK_IOBUF a )
{
    CDK_BSTRING bs;
  
    if ( !a )
        return;
    bs = cdk_iobuf_read_mem( a, 0 );
    printf("%s\n", bs->d );
    cdk_free( bs );
}
  
  
void
test_armor_filter2( const char *fname )
{
    CDK_IOBUF buf;
    armor_filter_s afx;
    int rc = 0;
    int n = 128;

    cdk_iobuf_new( &buf, 128 );
    while ( n > 0 ) { 
        cdk_iobuf_write( buf, "Hallo Welt!", 11 );
        n -= 12;
    }
    memset( &afx, 0, sizeof afx );
    afx.idx = afx.idx2 = 0;
    afx.hdrlines = "Comment: Hi, I'm busy with hacking on OpenCDK :-)\r\n";

    rc = cdk_armor_filter( &afx, IOBUF_CTRL_FLUSH, buf );
    if ( rc )
        printf("push_filter: `%s'\n", cdk_strerror( rc ) );
    dump_iobuf( buf );
    cdk_iobuf_close( buf );
}
  

void
test_armor_filter(const char *fname)
{
    CDK_IOBUF buf, out;
    armor_filter_s a;
    byte data[4096], c;
    size_t n;
    int rc = 0;
    
    rc = cdk_iobuf_open(&buf, fname, IOBUF_MODE_RD);
    if ( rc ) {
        printf("%s: `%s'\n", fname, cdk_strerror(CDKERR_FILE_OPEN));
        return;
    }
    cdk_iobuf_new(&out, cdk_iobuf_get_length(buf));
    memset( &a, 0, sizeof(armor_filter_s) );
    rc = cdk_armor_filter( &a, IOBUF_CTRL_UNDERFLOW, buf );
    while ( !cdk_iobuf_eof(buf) ) {
        cdk_iobuf_read( buf, data, sizeof(data), &n );
        if ( n )
            cdk_iobuf_write( out, data, n );
    }
    
    cdk_iobuf_close(buf);
    cdk_iobuf_rewind(out);
    while( !cdk_iobuf_eof(out) ) {
        c = cdk_iobuf_get(out);
        printf("%c", c);
    }
    cdk_iobuf_close(out);
}

void
test_armor_file( const char *file )
{
    int rc = 0;
    GCRY_SEXP opts;

    cdk_set_logging (2);
    gcry_sexp_build( &opts, NULL, "(armor-file((compress%d)))", 0 );
    rc = cdk_armor_file( opts, file, "foo.asc" );
    if ( rc )
        printf("`%s'\n", cdk_strerror( rc ) );
    gcry_sexp_release( opts );
}

void
test_armor_ctrl( const char *file )
{
  int rc = 0;
  CDK_IOBUF inp = NULL;

  rc = cdk_iobuf_open( &inp, file, IOBUF_MODE_RD );
  if ( rc )
    printf("open `%s': %s\n", file, cdk_strerror( rc ) );
  if ( !rc ) {
      char line[ 500 ];
      size_t n = 0;
      rc = cdk_armor_filter_use( inp );
      printf("is armored %d\n", rc );
      if ( rc ) { 
          rc = cdk_iobuf_read_line( inp, line, sizeof(line)-1, &n );
          if ( !rc && n )
            printf("%s\n", line);
        }
    }
  
  cdk_iobuf_close( inp );
}
  
int
main(int argc, char **argv)
{
    /*test_armor();*/
    /*test_armor_filter2("Makefile");*/
    test_armor_file( "foo" );
    /*test_armor_filter("Makefile.asc");*/
    if ( argc > 1 )
        test_armor_ctrl( argv[1] );
    return 0;    
}









