/* t-key.c
 *  Copyright (C) 2001, 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <stdio.h>
#include <assert.h>
#include <gcrypt.h>

#include "opencdk.h"
#include "main.h"

const char*
pk_algo( int algo )
{
    if ( is_RSA( algo ) )
        return "RSA";
    else if ( is_ELG( algo ) )
        return "ELG";
    else if ( is_DSA( algo ) )
        return "DSA";
    else
        return "???";
}

void
test_list_pubkeys(const char *fname)
{
    CDK_KBNODE key, p;
    int rc = 0, eof = 0;
    CDK_IOBUF buf;
	
    rc = cdk_iobuf_open(&buf, fname, IOBUF_MODE_RD);
    if ( rc ) {
        printf("can't open `%s'\n", fname);
        return;
    }

    do {
        rc = cdk_keydb_get_keyblock(buf, &key, &eof);
        if ( rc ) {
            printf("Error `%s'\n", cdk_strerror( rc ) );
            break;
        }
        for (p=key; p; p=p->next) {
            if (p->pkt->pkttype == PKT_PUBLIC_KEY) {
                byte md[24], i;
                cdkPKT_public_key *pk = p->pkt->pkt.public_key;
                printf("pub %s %d/%08X revoked %d invalid %d\n",
                       pk_algo( p->pkt->pkt.public_key->pubkey_algo ),
                       p->pkt->pkt.public_key->mpi[0]->bits,
                       p->pkt->pkt.public_key->keyid[1],
                       p->pkt->pkt.public_key->is_revoked,
                       p->pkt->pkt.public_key->is_invalid);
                cdk_pk_get_fingerprint(p->pkt->pkt.public_key, md);
                for (i=0; i<20/2; i++)
                    printf("%02X%02X ", md[2*i], md[2*i+1]);
                printf("\n");

                for ( i = 0; i < cdk_pk_get_npkey( pk->pubkey_algo ); i++ ) {
                    printf("mpi %d bits=%d\n", i, pk->mpi[i]->bits );
                }
            }
            if (p->pkt->pkttype == PKT_PUBLIC_SUBKEY) {
                int i = 0;
                cdkPKT_public_key *pk = p->pkt->pkt.public_key;
                printf("sub %d/%08X revoked %d [main key %08X]\n", 
                       p->pkt->pkt.public_key->mpi[0]->bits,
                       p->pkt->pkt.public_key->keyid[1],
                       p->pkt->pkt.public_key->is_revoked,
                       p->pkt->pkt.public_key->main_keyid[1]);
                if ( p->pkt->pkt.public_key->expiredate )
                    printf("Expire in `%d' days!\n",
                           p->pkt->pkt.public_key->expiredate );
                for ( i = 0; i < cdk_pk_get_npkey( pk->pubkey_algo ); i++ ) {
                    printf("mpi %d bits=%d (hex %x)\n", i, pk->mpi[i]->bits,
                           pk->mpi[i]->data[2] );
                }
            }            
            else if (p->pkt->pkttype == PKT_USER_ID) {
                printf( "uid [r%d] %s\n",
                        p->pkt->pkt.user_id->is_revoked,
                        p->pkt->pkt.user_id->name );
            }
            else if (p->pkt->pkttype == PKT_SIGNATURE) {
                /*printf("sig %08X: %08X type %X\n", 
                  p->pkt->pkt.signature->key[1],
                  p->pkt->pkt.signature->keyid[1],
                  p->pkt->pkt.signature->sig_class);*/
                printf("sig v%d %s\n", p->pkt->pkt.signature->version,
                       p->pkt->pkt.signature->sig_class==0x13?"[self-sig]":"");
            }
            else
                printf( "packetID `%d'\n", p->pkt->pkttype );
        }
        cdk_kbnode_release( key );
    } while (!rc && !eof);    
    
    if (rc)
        cdk_kbnode_release( key );
    cdk_iobuf_close(buf);
}

void
test_key_uid(const char *fname, const char *uid)
{
    CDK_KBNODE key, p;
#if 0
    cdkPKT_public_key *pk;
    cdkPKT_user_id *id;
#endif
    int rc = 0;
    CDK_KEYDB_SEARCH ks;
    CDK_IOBUF a = NULL;
    CDK_KEYDB_HD khd;
    byte md[20] = {
        0x1D, 0x75, 0x81, 0x08, 0x5B, 0xC9, 0xD9, 0xFB, 0xE7, 0x8B,
        0x20, 0x78, 0xED, 0x46, 0x81, 0xC9, 0xBF, 0x3D, 0xF9, 0xB4};

    cdk_keydb_add_resource(fname, 0);
    khd = cdk_keydb_get_ctx(0);
    printf("keyring `%s'\n", khd->name);

    /*
      rc = cdk_iobuf_open(&a, fname, IOBUF_MODE_RD);
      if ( rc ) {
      printf("can't open `%s'\n", fname);
      return;
      }
      khd = cdk_alloc_clear( sizeof *khd );
      khd->buf = a;
      */


    ks.type = KEYDB_SEARCH_SUBSTR;
    ks.u.pattern = uid;
    rc = cdk_keydb_search( khd, &ks, &key );
    if ( rc ) {
        printf("can't find key with userID `%s'\n", uid);
        return;
    }

#if 0
    ks.type = KEYDB_SEARCH_EXACT;
    ks.u.pattern = uid;
    rc = cdk_keydb_search( khd, &ks, &key );
    if ( rc ) {
        printf("can't find key with userID `%s'\n", uid);
        return;
    }
#endif

#if 0
    ks.type = KEYDB_SEARCH_SHORT_KEYID;
    ks.u.keyid[0] = 0;
    ks.u.keyid[1] = 0xBF3DF9B4;
    rc = cdk_keydb_search( khd, &ks, &key );
    if ( rc ) {
        printf("can't find key with keyID `%08X'\n", ks.u.keyid[1]);
        return;
    }
#endif

#if 0
    ks.type = KEYDB_SEARCH_FPR;
    memcpy(ks.u.fpr, md, 20);
    rc = cdk_keydb_search( khd, &ks, &key );
    if ( rc ) {
        printf("can't find key with fingerpinrt ID ");
        {
            int i;        
            for (i=0; i<20; i++)
                printf("%02X", md[i]);
        }
        printf("\n");      
        return;      
    }
#endif
  
    for (p=key; p; p=p->next) {
        if (p->pkt->pkttype == PKT_PUBLIC_KEY) {
            printf("%08X r%d/i%d [caps %d]\n",
                   p->pkt->pkt.public_key->keyid[1], 
                   p->pkt->pkt.public_key->is_revoked, 
                   p->pkt->pkt.public_key->is_invalid,
                   p->pkt->pkt.public_key->pubkey_usage);
        }
        else if (p->pkt->pkttype == PKT_USER_ID) {
            printf("%s [r%d]\n",
                   p->pkt->pkt.user_id->name, 
                   p->pkt->pkt.user_id->is_revoked);
        }
    }
    
    cdk_kbnode_release( key );
    cdk_iobuf_close(a);
}    

void
test_sig_check(const char *fname)
{
    CDK_IOBUF buf;
    CDK_KBNODE key, p;
    int rc, eof = 0;
    
    rc = cdk_iobuf_open(&buf, fname, IOBUF_MODE_RD);
    if ( rc ) {
        printf("can't open `%s'\n", fname);
        return;
    }
    
    for (;;) {
        rc = cdk_keydb_get_keyblock(buf, &key, &eof);
        if (rc)
            break;
        for (p=key; p; p=p->next) {
            if (p->pkt->pkttype == PKT_PUBLIC_KEY
                || p->pkt->pkttype == PKT_PUBLIC_SUBKEY) {
                printf("%08X i%d r%d\n",
                       p->pkt->pkt.public_key->keyid[1], 
                       p->pkt->pkt.public_key->is_invalid,
                       p->pkt->pkt.public_key->is_revoked);
            }
        }
        cdk_kbnode_release( key );
    }
    
    cdk_iobuf_close(buf);
}                          

int
test_seckey(void)
{
    CDK_IOBUF buf;
    CDK_KBNODE key, p;
    int rc, eof = 0;
    
    cdk_iobuf_open(&buf, "secring.gpg", IOBUF_MODE_RD);
    rc = cdk_keydb_get_keyblock(buf, &key, &eof);
    printf("rc=%d\n", rc);
    
    for (p=key; p; p=p->next) {
        if (p->pkt->pkttype == PKT_SECRET_KEY) 
            printf("%08X\n", p->pkt->pkt.secret_key->pk->keyid[1]);
    }
    
    cdk_iobuf_close(buf);
    cdk_kbnode_release( key );
    return 0;
}    

void
test_key_sig(const char *keyring)
{
    CDK_KBNODE key = NULL, p;
    u32 keyid[2] = {0, 0xBF3DF9B4};
    /*u32 keyid[2] = {0, 0xCCC07C35};*/
    CDK_KEYDB_SEARCH ks;
    CDK_KEYDB_HD khd;
    int rc = 0;
    int key_stat = 0;

    cdk_set_logging(1);
    rc = cdk_keydb_add_resource(keyring, 0);
    if ( rc ) {
        printf("Error: `%s'\n", cdk_strerror(rc));
        return;
    }
    khd = cdk_keydb_get_ctx(0);
    assert(khd);

    ks.type = KEYDB_SEARCH_SHORT_KEYID;
    ks.u.keyid[0] = keyid[0];
    ks.u.keyid[1] = keyid[1];
    rc = cdk_keydb_search(khd, &ks, &key );
    if ( rc ) {
        printf("can't find key `%08X'\n", keyid[1]);
        cdk_kbnode_release( key );
        return;
    }

    cdk_keydb_remove_resource(0);
    cdk_keydb_add_resource("/home/twoaday/.gnupg/pubring.gpg", 0);
    rc = cdk_key_check_sigs(key, NULL, &key_stat);
    if ( rc )
        printf("Error `%s'\n", cdk_strerror( rc ) );
    printf("rc=%d (key status `%d')\n", rc, key_stat);
    
    cdk_kbnode_release( key );
}    

void
test_key_count_sigs(const char *file)
{
    CDK_IOBUF a;
    CDK_KBNODE key, p;
    int n = 0, eof = 0;
    int rc = 0;
    u32 keyid = 0;
    
    rc = cdk_iobuf_open(&a, file, IOBUF_MODE_RD);
    if ( rc ) {
        printf("can't open `%s'\n", file);
        return;
    }
    
    cdk_keydb_get_keyblock(a, &key, &eof);
    for (p=key; p; p=p->next) {
        if (p->pkt->pkttype == PKT_SIGNATURE
            && p->pkt->pkt.signature->sig_class == 0x10) 
            n++;
        if (p->pkt->pkttype == PKT_PUBLIC_KEY)
            keyid = p->pkt->pkt.public_key->keyid[1];
    }
    
    if (keyid)
        printf("`%08X' has %d signatures\n", keyid, n);
    cdk_kbnode_release( key );
}

void
test_unprotect_key( void )
{
    CDK_KBNODE key, p;
    CDK_IOBUF a;
    cdkPKT_secret_key *sk = NULL;
    int eof, rc;
    const char *s = "sec-with-pwd.gpg";
    /*const char *s = "/home/twoaday/tls/pgp/romeo.gpg";*/
    char *pwd = "abc";
    /*char *pwd = "test";*/

    cdk_secmem_init( 16384 );
    
    rc = cdk_iobuf_open(&a, s, IOBUF_MODE_RD);
    if ( rc  ) {
        printf("Can't open `%s'\n", s);
        return;
    }

    cdk_keydb_add_resource( s, 1 );
    cdk_keydb_get_keyblock( a, &key, &eof );
    if ( !key )
        printf("No key was found!");
    else {
        for (p=key; p; p=p->next) {
            if (p->pkt->pkttype == PKT_SECRET_KEY) { 
                sk = p->pkt->pkt.secret_key;
                printf("seckey: enclen=%d\n", p->pkt->pkt.secret_key->enclen);
            }
            else if (p->pkt->pkttype == PKT_SECRET_SUBKEY)
                printf("secsubkey: enclen=%d\n",
                       p->pkt->pkt.secret_key->enclen);
        }
        if (sk) {
            if ( sk->protect.sha1chk )
                printf("Key is SHA1 protected!\n");
            rc=cdk_seckey_unprotect( sk,  pwd );
            if (rc)
                printf("Key decrypt key: `%s'\n", cdk_strerror(rc));
        }
    }
    cdk_kbnode_release( key );
    cdk_secmem_end( );
} /* test_unprotect_key */

void
test_sec_key(void)
{
    CDK_MPI *m[4];

    cdk_secmem_init( 16384 );
    m[0] = cdk_alloc_secure(sizeof *m + 512);
    m[0]->bits = 1000;
    m[1] = cdk_alloc_secure(sizeof *m);
    m[1]->bits = 2000;
    m[2] = cdk_alloc_secure(sizeof *m);
    m[2]->bits = 3000;  
    m[3] = cdk_alloc_secure(sizeof *m);
    m[3]->bits = 4000;

    m[0]->data[0] = 111;
  
    printf("[%d]: 1: %d; 2: %d; 3: %d 4: %d\n",
           sizeof *m,
           m[0]->bits,
           m[1]->bits,
           m[2]->bits,
           m[3]->bits);  
    cdk_free(m[0]);
    cdk_secmem_end( );
}

void
test_key_prefs(void)
{
    CDK_KEYDB_HD khd;
    CDK_KBNODE key, p;
    u32 keyid[2] = {0, 0xCCC07C35};
    int rc = 0, i;

    cdk_keydb_add_resource("pub.gpg", 0);
    khd = cdk_keydb_get_ctx(0);
  
    rc=cdk_keydb_get_bykeyid( khd, keyid, &key );
    if (rc) {
        printf("Could not find `%08X'\n", keyid[1]);
        return;
    }

    for (p=key; p && p->pkt->pkttype; p=p->next) {
        if (p->pkt->pkttype == PKT_PUBLIC_KEY) {
            cdkPKT_public_key *pk;
            pk = p->pkt->pkt.public_key;
            printf("key prefs `%p'\n", pk->prefs);
            if (!pk->prefs)
                continue;
            for (i=0; pk->prefs[i].type != PREFTYPE_NONE; i++)
                fprintf(stderr, "%d[%d] ",
                        pk->prefs[i].type,
                        pk->prefs[i].value);
        }
    }
    cdk_kbnode_release(key);
}

void
test_trustdb(void)
{
    CDK_IOBUF a;
    cdkPKT_user_id *id;
    cdkPKT_public_key *pk = NULL;
    u32 keyid[2] = {0, 0xBF3DF9B4};
    byte fpr[20];
    int rc = 0;
    int val = 0;
    int flags = 0;
    const char *s;

    rc = cdk_iobuf_open(&a, "/home/twoaday/.gnupg/trustdb.gpg", IOBUF_MODE_RD);
    if ( rc ) {
        printf("Can't open trustdb.gpg\n");
        return;
    }

    /*s = "Timo Schulz <ts@winpt.org>";*/
    s = "Timo Schulz <twoaday@gmx.net>";
    id = cdk_alloc_clear( sizeof *id + strlen(s) -1 );
    memcpy( id->name, s, strlen(s) );
    id->name[ strlen(s) ] = '\0';
    id->len = strlen(s);
    cdk_trustdb_get_validity(a, id, &val);
    printf("Validity for `%s' = %d\n", s, val);

    cdk_keydb_add_resource("/home/twoaday/.gnupg/pubring.gpg", 0);
    rc = cdk_keydb_get_pk( NULL, keyid, &pk );
    if (!rc) {
        cdk_trustdb_get_ownertrust(a, pk, &val, &flags);
        printf("Ownertrust for key `%08X' = %d (Flags `%d')\n",
               pk->keyid[1], val, flags);
    }
    else
        printf("keydb_get_pk `%s'\n", cdk_strerror(rc));
  
    cdk_pk_get_fingerprint( pk, fpr );
    cdk_pk_fingerprint_get_keyid( fpr, 20, keyid );
    {
        int i = 20;
        for (i=0; i<20; i++)
            printf("%02X", fpr[i]);
        printf("\n");
        printf("%08X%08X\n", keyid[0], keyid[1]);
    }
  
    cdk_iobuf_close(a);
    cdk_free( id );
}

void
test_keydb( void )
{
    CDK_IOBUF a;
    int rc = 0;
    const char *s = "ts.gpg";
    CDK_KBNODE k, p, f;
    int eof = 0;

    cdk_set_logging( 1 );
    rc = cdk_iobuf_open( &a, s, IOBUF_MODE_RD );
    if ( rc ) {
        printf(" %s: `%s'\n", s, cdk_strerror( rc ) );
        return;
    }

    cdk_secmem_init( 16384 );
    while ( (rc = cdk_keydb_get_keyblock( a, &k, &eof )) == 0 ) {
        printf( "get_keyblock: rc `%d' k `%p' eof `%d'\n", rc, k, eof );
        f = cdk_kbnode_find( k, PKT_PUBLIC_KEY );
        if ( f ) {
            cdkPKT_public_key *pk = f->pkt->pkt.public_key;
            printf(" pub %d%c/%08X\n", pk->mpi[0]->bits,
                   is_DSA(pk->pubkey_algo)? 'D' : '?',
                   cdk_pk_get_keyid(pk, NULL) );
        }
        f = cdk_kbnode_find( k, PKT_SECRET_KEY );
        if ( f ) {
            cdkPKT_public_key *pk = f->pkt->pkt.secret_key->pk;
            printf(" sec %d%c/%08X\n", pk->mpi[0]->bits,
                   is_DSA(pk->pubkey_algo)? 'D' : '?',
                   cdk_pk_get_keyid(pk, NULL) );
        }
        if ( k )
            cdk_kbnode_release( k );
        if ( eof )
            break;
    }
    if ( rc )
        printf( "Error `%s'\n", cdk_strerror( rc ) );
    cdk_iobuf_close( a );
    cdk_secmem_end( );

#if 0
    {
        CDK_KEYDB_HD khd;
        u32 keyid[2] = { 0, 0xBF3DF9B4 };
        CDK_KBNODE kb = NULL, p;
    
        cdk_keydb_add_resource( "/home/twoaday/.gnupg/pubring.gpg", 0 );
        khd = cdk_keydb_get_ctx( 0 );
        rc = cdk_keydb_get_bykeyid( khd, keyid, &kb );
        printf("rc `%d' k `%p'\n", rc, kb );
        if ( rc )
            printf( "Error `%s'\n", cdk_strerror( rc ) );
        for ( p=kb; p; p=p->next ) {
            if ( p->pkt->pkttype == PKT_PUBLIC_KEY )
                printf("%08X\n", cdk_pk_get_keyid( p->pkt->pkt.public_key,
                                                   NULL ) );
        }
        if ( kb )
            cdk_kbnode_release( kb );
    }
#endif
  
}

void
test_sec_keylist( void )
{
    CDK_IOBUF sk;
    int rc = 0, eof = 0;
    const char *s = "secring.gpg";
    CDK_KBNODE key = NULL;

    cdk_secmem_init( 16384 );
    rc = cdk_iobuf_open( &sk, s, IOBUF_MODE_RD );
    if ( rc ) {
        printf(" Can't open: `%s'\n", s);
        return;
    }

    do {
        rc = cdk_keydb_get_keyblock( sk, &key, &eof );
        if ( !rc ) {
            CDK_KBNODE p = cdk_kbnode_find( key, PKT_USER_ID );
            if ( p )
                printf("%s\n", p->pkt->pkt.user_id->name );
            cdk_kbnode_release( key );
        }
    } while ( !rc && !eof );
  
    cdk_iobuf_close( sk );
} 

CDK_KBNODE _cdk_keydb_get_pkblock( u32 *keyid );
     
void
basic_foo( void )
{
    u32 keyid[2] = { 0xED4681C9, 0xBF3DF9B4 };
    CDK_KBNODE key, k;

    cdk_keydb_add_resource( "/home/twoaday/.gnupg/pubring.gpg", 0 );
    key = _cdk_keydb_get_pkblock( keyid );
    if ( key ) {
        k = cdk_kbnode_find( key, PKT_USER_ID );
        if ( k )
            printf("`%s'\n", k->pkt->pkt.user_id->name);
        cdk_kbnode_release( key );
    }
}

void
test_sec_getkey( void )
{
    const char *kr = "sec.gpg";
    cdkPKT_secret_key *sk = NULL;
    CDK_KEYLIST sk_list, r;
    CDK_STRLIST locusr = NULL;
    int rc = 0;

    cdk_secmem_init( 8192 );
    cdk_keydb_add_resource(kr, 1);
#if 0
    rc = cdk_keydb_get_sk_byname( "opencdk@foo-bar.org", &sk, NULL, 0, 0 );
    if ( rc )
        printf("Can't get secret key: `%s'\n", cdk_strerror( rc ) );
    if ( !rc ) {
        u32 keyid[2] = {0};
        cdk_sk_get_keyid( sk, keyid );
        printf("`%08X' success\n", keyid[1] );
        cdk_pkt_sk_release( sk );
    }
#endif
    cdk_strlist_add( &locusr, "<opencdk@foo-bar.org>" );
    rc = cdk_sklist_build( locusr, &sk_list, 0, GCRY_PK_USAGE_SIGN );
    if ( rc )
        printf("sklist_build: `%s'\n", cdk_strerror (rc) );
  
    for ( r = sk_list; r; r = r->next ) {
        u32 kid[2];
        cdk_sk_get_keyid( r->key.sk, kid );
        printf("KeyID `%08X%08X'\n", kid[0], kid[1]);
    }
    cdk_sklist_release( sk_list );
    cdk_strlist_free( locusr );  
}

void
test_export( void )
{
    CDK_STRLIST remusr = NULL;
    CDK_IOBUF out;
    int rc = 0;

    cdk_strlist_add( &remusr, "opencdk@foo-bar.org" );
    cdk_keydb_add_resource( "/home/twoaday/.gnupg/pubring.gpg", 0 );
    rc = cdk_iobuf_create( &out, "foo.key" );
    if ( rc ) {
        printf("iobuf_create `%s'\n", cdk_strerror( rc ) );
        return;
    }
    rc = cdk_keydb_export( out, remusr );
    if ( rc )
        printf("keydb_export `%s'\n", cdk_strerror( rc ) );
    cdk_strlist_free( remusr );
    cdk_iobuf_close( out );
}

int
main(int argc, char **argv)
{
    const char *kr = "/home/twoaday/.gnupg/pubring.gpg";
    
    if (argc == 2) {
        test_list_pubkeys(argv[1]);
        return 0;
    }

    /*test_unprotect_key();*/
    /*test_sec_key();*/
    /*test_key_uid(kr, "Timo Schulz <ts@winpt.org>");*/
    /*test_key_sig("ts.gpg");*/
    test_export();
    /*test_trustdb();*/
    /*basic_foo();*/
    /*test_sec_keylist();*/
    /*test_key_prefs();*/
    /*if (argc == 2)
      test_key_uid("ts.gpg", argv[1]);*/
    /*test_keydb( );*/
    /*test_sec_getkey();*/
    return 0;
}











