/* read-packet.c - Read OpenPGP packets
 *        Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <string.h>
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"

#define BUFSIZE 2048

typedef struct {
  int ctb;
  int type;
  unsigned rfc2440:1;
  int len;
  int size;
  unsigned partial:1;
} PKTHEAD;


static int
stream_getc (CDK_STREAM s)
{
  return cdk_stream_getc (s);
}



static int
stream_read (CDK_STREAM s, void *buf, size_t count, size_t *r_nread)
{
  int nread = cdk_stream_read (s, buf, count);
  if (!nread)
    return CDK_File_Error;
  if (r_nread)
    *r_nread = nread;
  return 0;
}


u32
read_32 (CDK_STREAM buf, int *ret_err)
{
  u32 u = 0;
  int tmp[4], err = 1;

  if (buf)
    {
      if ((tmp[0] = stream_getc (buf)) == EOF)
	goto fail;
      if ((tmp[1] = stream_getc (buf)) == EOF)
	goto fail;
      if ((tmp[2] = stream_getc (buf)) == EOF)
	goto fail;
      if ((tmp[3] = stream_getc (buf)) == EOF)
	goto fail;
      u = (tmp[0] << 24) | (tmp[1] << 16) | (tmp[2] << 8) | tmp[3];
      err = 0;
    }
 fail:
  if (ret_err)
    *ret_err = err;
  return u;
}


u16
read_16 (CDK_STREAM buf, int *ret_err)
{
  u16 u = 0;
  int tmp[2], err = 1;

  if (buf)
    {
      if ((tmp[0] = stream_getc (buf)) == EOF)
	goto fail;
      if ((tmp[1] = stream_getc (buf)) == EOF)
	goto fail;
      u = tmp[0] << 8 | tmp[1];
      err = 0;
    }
 fail:
  if (ret_err)
    *ret_err = err;
  return u;
}


static int
read_s2k (CDK_STREAM inp, CDK_S2K_HD * s2k)
{
  size_t nread = 0;
  int rc = 0;

  if (!inp || !s2k)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read S2K part\n");

  s2k->mode = stream_getc (inp);
  if (s2k->mode == EOF || s2k->mode > 3)
    return CDK_Inv_Packet;
  s2k->hash_algo = stream_getc (inp);
  if (s2k->mode == 0)
    ;				/* nothing to do */
  else if (s2k->mode == 1 || s2k->mode == 3)
    {
      rc = stream_read (inp, s2k->salt, sizeof s2k->salt, &nread);
      if (!rc && nread != sizeof s2k->salt)
	return CDK_Inv_Packet;
      if (!rc && s2k->mode == 3)
	{
	  if ((s2k->count = stream_getc (inp)) == EOF)
	    return CDK_Inv_Packet;
	}
    }
  else
    return CDK_Inv_Mode;

  return rc;
}


static int
read_mpi (CDK_STREAM inp, CDK_MPI * ret_m, int secure)
{
  CDK_MPI m = NULL;
  size_t nread = 0, nbits = 0, nbytes = 0;
  int rc = 0;

  if (!inp || !ret_m)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read MPI part\n");

  nbits = read_16 (inp, &rc);
  nbytes = (nbits + 7) / 8;
  if (rc || nbits > MAX_MPI_BITS || nbits == 0)
    return CDK_MPI_Error;		/* sanity check */
  if (secure)
    m = cdk_scalloc (sizeof *m + nbytes + 2);
  else
    m = cdk_calloc (1, sizeof *m + nbytes + 2);
  if (!m)
    return CDK_Out_Of_Core;
  m->bytes = nbytes;
  m->bits = nbits;

  /* this is the prefix which encodes the length of the following
     MPI data. */
  m->data[0] = nbits >> 8;
  m->data[1] = nbits;
  rc = stream_read (inp, m->data + 2, nbytes, &nread);
  if (!rc && nread != nbytes)
    rc = CDK_MPI_Error;
  *ret_m = m;

  return rc;
}


static int
read_pkt_length (CDK_STREAM inp, int *ret_partial)
{
  int c1 = 0, c2 = 0;
  size_t pktlen = 0;
  int rc = 0;

  if ((c1 = stream_getc (inp)) == EOF)
    return EOF;
  if (c1 < 224 || c1 == 255)
    *ret_partial = 0;		/* end of partial data */
  if (c1 < 192)
    pktlen = c1;
  else if (c1 >= 192 && c1 <= 223)
    {
      if ((c2 = stream_getc (inp)) == EOF)
	return EOF;
      pktlen = ((c1 - 192) << 8) + c2 + 192;
    }
  else if (c1 == 255)
    {
      pktlen = read_32 (inp, &rc);
      if (rc)
	return EOF;
    }
  else
    pktlen = 1 << (c1 & 0x1f);
  return pktlen;
}


static int
read_encrypted (CDK_STREAM inp, size_t pktlen, cdkPKT_encrypted * enc,
		int partial, int mdc)
{
  byte buf[BUFSIZE];
  size_t nread = 0;
  int rc = 0, version;

  if (!inp || !enc)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read encrypted packet %d bytes\n", pktlen);

  enc->buf = cdk_stream_tmp ();
  if (!enc->buf)
    return CDK_Out_Of_Core;

  if (mdc)
    {
      version = stream_getc (inp);
      if (version != 1)
	return CDK_Inv_Packet;
      enc->mdc_method = GCRY_MD_SHA1;
      pktlen--;
    }
  while (!cdk_stream_eof (inp))
    {
      rc = stream_read (inp, buf, sizeof buf-1, &nread);
      if (rc)
        goto leave;
      pktlen -= nread;
      cdk_stream_write (enc->buf, buf, nread);
      if (!partial)
	break;
      else
	{
	  pktlen = read_pkt_length (inp, &partial);
	  if (pktlen == EOF)
	    return CDK_Inv_Packet;
	}
    }
  if (pktlen)
    return CDK_Inv_Packet;
  cdk_stream_seek (enc->buf, 0);
  enc->len = cdk_stream_get_length (enc->buf);

leave:
  return rc;
}


static int
read_symkey_enc (CDK_STREAM inp, size_t pktlen, cdkPKT_symkey_enc * ske)
{
  size_t nread = 0, minlen = 0;
  int rc = 0;

  if (!inp || !ske)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read symmetric key encrypted packet\n");

  ske->version = stream_getc (inp);
  if (ske->version != 4)
    return CDK_Inv_Packet;

  ske->cipher_algo = stream_getc (inp);
  ske->s2k.mode = stream_getc (inp);
  switch (ske->s2k.mode)
    {
    case 0: minlen = 0; break;
    case 1: minlen = 8; break;
    case 3: minlen = 9; break;
    }
  ske->s2k.hash_algo = stream_getc (inp);
  if (ske->s2k.mode == 0)
    ; /* nothing to do */
  if (ske->s2k.mode == 1 || ske->s2k.mode == 3)
    {
      rc = stream_read (inp, ske->s2k.salt, DIM (ske->s2k.salt), &nread);
      if (!rc && nread != DIM (ske->s2k.salt))
	return CDK_Inv_Packet;
      if (!rc && ske->s2k.mode == 3)
	ske->s2k.count = stream_getc (inp);
    }
  else
    return CDK_Inv_Packet;
  ske->seskeylen = pktlen - 4 - minlen;
  if (ske->seskeylen > sizeof (ske->seskey))
    return CDK_Inv_Packet;
  for (nread = 0; nread < ske->seskeylen; nread++)
    {
      ske->seskey[nread] = stream_getc (inp);
      if (cdk_stream_eof (inp))
	break;
    }
  return rc;
}


static int
read_pubkey_enc (CDK_STREAM inp, cdkPKT_pubkey_enc * pke)
{
  int rc = 0;
  int i, nenc = 0;

  if (!inp || !pke)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read public key encrypted packet\n");

  pke->version = stream_getc (inp);
  if (pke->version < 2 || pke->version > 3)
    return CDK_Inv_Packet;
  pke->keyid[0] = read_32 (inp, NULL);
  pke->keyid[1] = read_32 (inp, NULL);
  if (!pke->keyid[0] && !pke->keyid[1])
    pke->throw_keyid = 1;	/* RFC2440 "speculative" keyID */
  pke->pubkey_algo = stream_getc (inp);
  nenc = cdk_pk_get_nenc (pke->pubkey_algo);
  for (i = 0; i < nenc; i++)
    {
      rc = read_mpi (inp, &pke->mpi[i], 0);
      if (rc)
	break;
    }
  return rc;
}


static int
read_mdc (CDK_STREAM inp, cdkPKT_mdc * mdc)
{
  size_t n = 0;
  int rc = 0;

  if (!inp || !mdc)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read MDC packet\n");

  rc = stream_read (inp, mdc->hash, 20, &n);
  if (!rc && n != 20)
    rc = CDK_Inv_Packet;
  return rc;
}


static int
read_compressed (CDK_STREAM inp, size_t pktlen, cdkPKT_compressed * c)
{
  if (!inp || !c)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read compressed packet\n");

  c->algorithm = stream_getc (inp);
  if (c->algorithm > 2)
    return CDK_Inv_Packet;

  /* don't know the size, so we read until EOF */
  c->len = 0;
  c->buf = inp;

  return 0;
}


static int
read_public_key (CDK_STREAM inp, cdkPKT_public_key * pk)
{
  int i = 0;
  int rc = 0;

  if (!inp || !pk)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read public key packet\n");

  pk->is_invalid = 1; /* default to detect missing self signatures */
  pk->is_revoked = 0;
  pk->has_expired = 0;

  pk->version = stream_getc (inp);
  if (pk->version < 2 || pk->version > 4)
      return CDK_Inv_Packet_Ver;
  if (pk->version < 4)
    pk->expiredate = read_16 (inp, &rc);
  pk->timestamp = read_32 (inp, &rc);
  pk->pubkey_algo = stream_getc (inp);
  for (i = 0; i < cdk_pk_get_npkey (pk->pubkey_algo); i++)
    {
      rc = read_mpi (inp, &pk->mpi[i], 0);
      if (rc)
	break;
    }
  pk->pubkey_usage = _cdk_pk_algo_usage (pk->pubkey_algo);
  return rc;
}


static int
read_public_subkey (CDK_STREAM inp, cdkPKT_public_key * pk)
{
  if (!inp || !pk)
    return CDK_Inv_Value;
  return read_public_key (inp, pk);
}


static int
read_secret_key (CDK_STREAM inp, size_t pktlen, cdkPKT_secret_key * sk)
{
  size_t p1 = 0, p2 = 0, nread = 0;
  int i = 0, blklen = 0, nskey = 0;
  int rc = 0;

  if (!inp || !sk || !sk->pk)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read secret key\n");

  p1 = cdk_stream_tell (inp);
  rc = read_public_key (inp, sk->pk);
  if (rc)
    return rc;

  sk->s2k_usage = stream_getc (inp);
  sk->protect.sha1chk = 0;
  if (sk->s2k_usage == 254 || sk->s2k_usage == 255)
    {
      sk->protect.sha1chk = (sk->s2k_usage == 254);
      sk->protect.algo = stream_getc (inp);
      rc = read_s2k (inp, &sk->protect.s2k);
      if (rc)
	return rc;
      blklen = gcry_cipher_get_algo_blklen (sk->protect.algo);
      sk->protect.ivlen = blklen;
      rc = stream_read (inp, sk->protect.iv, sk->protect.ivlen, &nread);
      if (!rc && nread != sk->protect.ivlen)
	return CDK_Inv_Packet;
    }
  else
    sk->protect.algo = sk->s2k_usage;
  if (sk->protect.algo == GCRY_CIPHER_NONE)
    {
      sk->csum = 0;
      nskey = cdk_pk_get_nskey (sk->pk->pubkey_algo);
      for (i = 0; i < nskey; i++)
	{
	  rc = read_mpi (inp, &sk->mpi[i], 1);
	  if (rc)
	    break;
	}
      if (!rc)
	{
	  sk->csum = read_16 (inp, &rc);
	  sk->is_protected = 0;
	}
    }
  else if (sk->pk->version < 4)
    {
      /* mpi size isn't encrypted! */
      nskey = cdk_pk_get_nskey (sk->pk->pubkey_algo);
      for (i = 0; i < nskey; i++)
	{
	  rc = read_mpi (inp, &sk->mpi[i], 1);
	  if (rc)
	    break;
	}
      if (!rc)
	{
	  sk->csum = read_16 (inp, &rc);
	  sk->is_protected = 1;
	}
    }
  else
    {
      /* we need to read the rest of the packet because we don't
         have any information how long the encrypted mpi's are */
      p2 = cdk_stream_tell (inp);
      p2 -= p1;
      sk->enclen = pktlen - p2;
      if (sk->enclen < 2)
	return CDK_Inv_Packet; /* at least 16 bits for the checksum! */
      sk->encdata = cdk_calloc (1, sk->enclen + 1);
      if (!sk->encdata)
	return CDK_Out_Of_Core;
      rc = stream_read (inp, sk->encdata, sk->enclen, &nread);
      if (rc)
	return CDK_Inv_Packet;
      nskey = cdk_pk_get_nskey (sk->pk->pubkey_algo);
      for (i = 0; i < nskey; i++)
	{
	  sk->mpi[i] = cdk_salloc (sizeof **sk->mpi + MAX_MPI_BYTES);
	  if (!sk->mpi[i])
	    return CDK_Out_Of_Core;
	}
      sk->is_protected = 1;
    }
  sk->is_primary = 1;
  _cdk_copy_pk_to_sk (sk->pk, sk);
  return rc;
}


static int
read_secret_subkey (CDK_STREAM inp, size_t pktlen, cdkPKT_secret_key * sk)
{
  int rc = 0;

  if (!inp || !sk || !sk->pk)
    return CDK_Inv_Value;

  rc = read_secret_key (inp, pktlen, sk);
  sk->is_primary = 0;
  return rc;
}


static int
read_attribute (CDK_STREAM inp, size_t pktlen, cdkPKT_user_id * attr)
{
  size_t nread = 0;
  byte *buf;
  const byte *p;
  int len = 0;
  int rc = 0;

  if (!inp || !attr || !pktlen)
    return CDK_Inv_Value;

  strcpy (attr->name, "[attribute]");
  attr->len = strlen (attr->name);
  buf = cdk_calloc (1, pktlen);
  rc = stream_read (inp, buf, pktlen, &nread);
  if (rc)
    {
      cdk_free (buf);
      return CDK_Inv_Packet;
    }
  p = buf;
  len = *p++;
  if (len == 255)
    {
      len = buffer_to_u32 (p);
      p += 4;
      pktlen -= 4;
    }
  else if (len >= 192)
    {
      if (pktlen < 2)
	{
	  cdk_free (buf);
	  return CDK_Inv_Packet;
	}
      len = ((len - 192) << 8) + *p + 192;
      p++;
      pktlen--;
    }
  if (*p != 1)
    { /* ATTRIBUTE IMAGE */
      cdk_free (buf);
      return CDK_Inv_Packet;
    }
  p++;

  attr->attrib_img = cdk_calloc (1, len);
  if (!attr->attrib_img)
    return CDK_Out_Of_Core;
  attr->attrib_len = len;
  memcpy (attr->attrib_img, p, len);
  cdk_free (buf);
  return rc;
}


static int
read_user_id (CDK_STREAM inp, size_t pktlen, cdkPKT_user_id * user_id)
{
  size_t nread = 0;
  int rc = 0;

  if (!inp || !user_id || !pktlen)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read user ID packet\n");

  user_id->len = pktlen;
  rc = stream_read (inp, user_id->name, pktlen, &nread);
  if (!rc && nread != pktlen)
    return CDK_Inv_Packet;

  user_id->name[nread] = '\0';
  return rc;
}


static int
read_subpkt (CDK_STREAM inp, CDK_SUBPKT * r_ctx, size_t *r_nbytes)
{
  byte c, c1;
  size_t size = 0, nread, n = 0;
  CDK_SUBPKT node;
  int rc = 0;

  if (!inp || !r_nbytes)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read sub packet");
  
  *r_nbytes = 0;
  c = stream_getc (inp);
  n++;
  if (c == 255)
    {
      size = read_32 (inp, NULL);
      n += 4;
      node = cdk_subpkt_new (size);
    }
  else if (c >= 192 && c < 255)
    {
      c1 = stream_getc (inp);
      n++;
      if (c1 == 0)
	return 0;
      size = ((c - 192) << 8) + c1 + 192;
      node = cdk_subpkt_new (size);
    }
  else if (c < 192)
    {
      size = c;
      node = cdk_subpkt_new (size);
    }
  else
    return CDK_Inv_Packet;

  if (DEBUG_PKT)
    _cdk_log_debug (" `%d' bytes\n", size);

  if (!node)
    return CDK_Out_Of_Core;
  node->size = size;
  node->type = stream_getc (inp);
  n++;
  node->size--;
  rc = stream_read (inp, node->d, node->size, &nread);
  n += nread;
  if (rc)
    return rc;
  *r_nbytes = n;
  if (!*r_ctx)
    *r_ctx = node;
  else
    _cdk_subpkt_add (*r_ctx, node);
  return rc;
}


static int
read_onepass_sig (CDK_STREAM inp, cdkPKT_onepass_sig * sig)
{
  if (!inp || !sig)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read one pass signature packet\n");

  sig->version = stream_getc (inp);
  if (sig->version != 3)
    return CDK_Inv_Packet_Ver;
  sig->sig_class = stream_getc (inp);
  sig->digest_algo = stream_getc (inp);
  sig->pubkey_algo = stream_getc (inp);
  sig->keyid[0] = read_32 (inp, NULL);
  sig->keyid[1] = read_32 (inp, NULL);
  sig->last = stream_getc (inp);
  return 0;
}


static int
read_signature (CDK_STREAM inp, cdkPKT_signature * sig)
{
  struct cdk_subpkt_s *p = NULL;
  size_t nbytes;
  int i, size;
  int rc = 0;

  if (!inp || !sig)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read signature packet\n");

  sig->version = stream_getc (inp);
  if (sig->version < 2 || sig->version > 4)
      return CDK_Inv_Packet_Ver;

  if (sig->version < 4)
    {
      if (stream_getc (inp) != 5)
        return CDK_Inv_Packet;

      sig->sig_class = stream_getc (inp);
      sig->timestamp = read_32 (inp, NULL);
      sig->keyid[0] = read_32 (inp, NULL);
      sig->keyid[1] = read_32 (inp, NULL);
      sig->pubkey_algo = stream_getc (inp);
      sig->digest_algo = stream_getc (inp);
      sig->digest_start[0] = stream_getc (inp);
      sig->digest_start[1] = stream_getc (inp);
      for (i = 0; i < cdk_pk_get_nsig (sig->pubkey_algo); i++)
	{
	  rc = read_mpi (inp, &sig->mpi[i], 0);
	  if (rc)
	    break;
	}
    }
  else
    {
      sig->sig_class = stream_getc (inp);
      sig->pubkey_algo = stream_getc (inp);
      sig->digest_algo = stream_getc (inp);
      sig->hashed_size = read_16 (inp, NULL);
      size = sig->hashed_size;
      sig->hashed = NULL;
      while (size > 0)
	{
          rc = read_subpkt (inp, &sig->hashed, &nbytes);
          if (rc)
            break;
          size -= nbytes;
        }
      
      sig->unhashed_size = read_16 (inp, NULL);
      size = sig->unhashed_size;
      sig->unhashed = NULL;
      while (size > 0)
	{
          rc = read_subpkt (inp, &sig->unhashed, &nbytes);
          if (rc)
            break;
          size -= nbytes;
	}
      
      sig->flags.exportable = 1;
      sig->flags.revocable = 1;
      /* Setup the standard packet entries, so we can use V4
         signatures similar to V3. */
      for (p = sig->unhashed; p; p = p->next)
	{
	  if (p->type == SIGSUBPKT_ISSUER)
	    {
	      sig->keyid[0] = buffer_to_u32 (p->d);
	      sig->keyid[1] = buffer_to_u32 (p->d + 4);
	    }
	  else if (p->type == SIGSUBPKT_EXPORTABLE && p->d[0] == 0)
	    {
	      /* this packet might be also placed in the unhashed area */
	      sig->flags.exportable = 0;
	    }
	}
      for (p = sig->hashed; p && p->size; p = p->next)
	{
	  if (p->type == SIGSUBPKT_SIG_CREATED)
	    sig->timestamp = buffer_to_u32 (p->d);
	  else if (p->type == SIGSUBPKT_SIG_EXPIRE)
	    {
	      sig->expiredate = buffer_to_u32 (p->d);
	      if (sig->expiredate > 0 && sig->expiredate < make_timestamp ())
		sig->flags.expired = 1;
	    }
	  else if (p->type == SIGSUBPKT_POLICY)
	    sig->flags.policy_url = 1;
	  else if (p->type == SIGSUBPKT_NOTATION)
	    sig->flags.notation = 1;
	  else if (p->type == SIGSUBPKT_REVOCABLE && p->d[0] == 0)
	    sig->flags.revocable = 0;
	  else if (p->type == SIGSUBPKT_EXPORTABLE && p->d[0] == 0)
	    sig->flags.exportable = 0;
	}
      sig->digest_start[0] = stream_getc (inp);
      sig->digest_start[1] = stream_getc (inp);
      for (i = 0; i < cdk_pk_get_nsig (sig->pubkey_algo); i++)
	{
	  rc = read_mpi (inp, &sig->mpi[i], 0);
	  if (rc)
	    break;
	}
    }
  return rc;
}


static int
read_plaintext (CDK_STREAM inp, size_t pktlen, cdkPKT_plaintext * pt)
{
  size_t nread = 0;
  int rc = 0;

  if (!inp || !pt)
    return CDK_Inv_Value;

  if (DEBUG_PKT)
    _cdk_log_debug ("** read plaintext packet\n");

  pt->mode = stream_getc (inp);
  if (pt->mode != 0x62 && pt->mode != 0x74)
    return CDK_Inv_Packet;
  pt->namelen = stream_getc (inp);
  if (pt->namelen)
    {
      rc = stream_read (inp, pt->name, pt->namelen, &nread);
      if (!rc && nread != pt->namelen)
	return CDK_Inv_Packet;
      pt->name[pt->namelen] = '\0';
    }
  pt->timestamp = read_32 (inp, NULL);
  pktlen -= 6;
  pktlen -= pt->namelen;
  if (pktlen)
    { 
      pt->buf = inp;
      pt->len = pktlen;
    }
  return rc;
}


static void
read_old_length (PKTHEAD * ph, CDK_STREAM inp)
{
  if ((ph->ctb & 0x03) == 0)
    {
      ph->len = stream_getc (inp);
      ph->size++; 
    }
  else if ((ph->ctb & 0x03) == 1)
    {
      ph->len = read_16 (inp, NULL);
      ph->size += 2; 
    }
  else if ((ph->ctb & 0x03) == 2)
    {
      ph->len = read_32 (inp, NULL);
      ph->size += 4; 
    }
  else
    {
      ph->len = 0;
      ph->size = 0;
    }  
}


static void
read_new_length (PKTHEAD * ph, CDK_STREAM inp)
{
  int c, c1;
  
  c = stream_getc (inp);
  ph->size++;
  if (c < 192)
    ph->len = c;
  else if (c >= 192 && c <= 223)
    {
      c1 = stream_getc (inp);
      ph->size++;
      ph->len = ((c - 192) << 8) + c1 + 192; 
    }
  else if (c == 255)
    {
      ph->len = read_32 (inp, NULL);
      ph->size += 4; 
    }
  else
    {
      ph->len = 1 << (c & 0x1f);
      ph->partial = 1;       
    }  
}


int
cdk_pkt_parse (CDK_STREAM inp, CDK_PACKET * pkt)
{
  PKTHEAD pkth;
  int use_mdc = 0;
  int rc = 0;

  if (!inp || !pkt)
    return CDK_Inv_Value;

  memset (&pkth, 0, sizeof pkth);
  pkth.ctb = stream_getc (inp);
  if (cdk_stream_eof (inp) || pkth.ctb == EOF)
    return CDK_EOF;
  else if (!pkth.ctb)
    return CDK_Inv_Packet;

  pkth.size++;
  if (!(pkth.ctb & 0x80))
    {
      _cdk_log_info ("no valid openpgp data found. (ctb=%02X; fpos=%02X)\n",
                     pkth.ctb, cdk_stream_tell (inp));
      return CDK_Inv_Packet;
    }
  if (pkth.ctb & 0x40)
    {
      pkth.type = pkth.ctb & 0x3f;
      pkth.rfc2440 = 1;
    }
  else
    {
      pkth.type = pkth.ctb & 0x3f;
      pkth.type >>= 2;
      pkth.rfc2440 = 0;	/* RFC1991 message format */
    }
  if (pkth.type > 63)
    {
      _cdk_log_info ("unknown packet type (%d)\n", pkth.type);
      return CDK_Inv_Packet;
    }
  if (pkth.rfc2440)
    read_new_length (&pkth, inp);
  else
    read_old_length (&pkth, inp);

  pkt->pkttype = pkth.type;
  pkt->pktlen = pkth.len;
  pkt->pktsize = pkth.size + pkth.len;

  switch (pkt->pkttype)
    {
    case CDK_PKT_ATTRIBUTE:
      pkt->pkt.user_id = cdk_calloc (1, sizeof *pkt->pkt.user_id+ pkt->pktlen);
      if (!pkt->pkt.user_id)
	return CDK_Out_Of_Core;
      rc = read_attribute (inp, pkth.len, pkt->pkt.user_id);
      pkt->pkttype = CDK_PKT_USER_ID; /* treated as an user id */
      break;

    case CDK_PKT_USER_ID:
      pkt->pkt.user_id = cdk_calloc (1, sizeof *pkt->pkt.user_id+ pkt->pktlen);
      if (!pkt->pkt.user_id)
	return CDK_Out_Of_Core;
      rc = read_user_id (inp, pkth.len, pkt->pkt.user_id);
      break;

    case CDK_PKT_PUBLIC_KEY:
      pkt->pkt.public_key = cdk_calloc (1, sizeof *pkt->pkt.public_key);
      if (!pkt->pkt.public_key)
	return CDK_Out_Of_Core;
      rc = read_public_key (inp, pkt->pkt.public_key);
      break;

    case CDK_PKT_PUBLIC_SUBKEY:
      pkt->pkt.public_key = cdk_calloc (1, sizeof *pkt->pkt.public_key);
      if (!pkt->pkt.public_key)
	return CDK_Out_Of_Core;
      rc = read_public_subkey (inp, pkt->pkt.public_key);
      break;

    case CDK_PKT_SECRET_KEY:
      pkt->pkt.secret_key = cdk_calloc (1, sizeof *pkt->pkt.secret_key);
      if (!pkt->pkt.secret_key)
	return CDK_Out_Of_Core;
      pkt->pkt.secret_key->pk =cdk_calloc (1, sizeof *pkt->pkt.secret_key->pk);
      if (!pkt->pkt.secret_key->pk)
	return CDK_Out_Of_Core;
      rc = read_secret_key (inp, pkth.len, pkt->pkt.secret_key);
      break;

    case CDK_PKT_SECRET_SUBKEY:
      pkt->pkt.secret_key = cdk_calloc (1, sizeof *pkt->pkt.secret_key);
      if (!pkt->pkt.secret_key)
	return CDK_Out_Of_Core;
      pkt->pkt.secret_key->pk =
	cdk_calloc (1, sizeof *pkt->pkt.secret_key->pk);
      if (!pkt->pkt.secret_key->pk)
	return CDK_Out_Of_Core;
      rc = read_secret_subkey (inp, pkth.len, pkt->pkt.secret_key);
      break;

    case CDK_PKT_PLAINTEXT:
      pkt->pkt.plaintext = cdk_calloc (1, sizeof *pkt->pkt.plaintext
				       + pkt->pktlen);
      if (!pkt->pkt.plaintext)
	return CDK_Out_Of_Core;
      rc = read_plaintext (inp, pkth.len, pkt->pkt.plaintext);
      break;

    case CDK_PKT_ONEPASS_SIG:
      pkt->pkt.onepass_sig = cdk_calloc (1, sizeof *pkt->pkt.onepass_sig);
      if (!pkt->pkt.onepass_sig)
	return CDK_Out_Of_Core;
      rc = read_onepass_sig (inp, pkt->pkt.onepass_sig);
      break;

    case CDK_PKT_SIGNATURE:
      pkt->pkt.signature = cdk_calloc (1, sizeof *pkt->pkt.signature);
      if (!pkt->pkt.signature)
	return CDK_Out_Of_Core;
      rc = read_signature (inp, pkt->pkt.signature);
      break;

    case CDK_PKT_ENCRYPTED_MDC:
    case CDK_PKT_ENCRYPTED:
      pkt->pkt.encrypted = cdk_calloc (1, sizeof *pkt->pkt.encrypted);
      if (!pkt->pkt.encrypted)
	return CDK_Out_Of_Core;
      use_mdc = (pkt->pkttype == CDK_PKT_ENCRYPTED_MDC) ? 1 : 0;
      rc = read_encrypted (inp, pkth.len, pkt->pkt.encrypted,
			   pkth.partial, use_mdc);
      break;

    case CDK_PKT_SYMKEY_ENC:
      pkt->pkt.symkey_enc = cdk_calloc (1, sizeof *pkt->pkt.symkey_enc);
      if (!pkt->pkt.symkey_enc)
	return CDK_Out_Of_Core;
      rc = read_symkey_enc (inp, pkth.len, pkt->pkt.symkey_enc);
      break;

    case CDK_PKT_PUBKEY_ENC:
      pkt->pkt.pubkey_enc = cdk_calloc (1, sizeof *pkt->pkt.pubkey_enc);
      if (!pkt->pkt.pubkey_enc)
	return CDK_Out_Of_Core;
      rc = read_pubkey_enc (inp, pkt->pkt.pubkey_enc);
      break;

    case CDK_PKT_COMPRESSED:
      pkt->pkt.compressed = cdk_calloc (1, sizeof *pkt->pkt.compressed);
      if (!pkt->pkt.compressed)
	return CDK_Out_Of_Core;
      rc = read_compressed (inp, pkth.len, pkt->pkt.compressed);
      break;

    case CDK_PKT_MDC:
      pkt->pkt.mdc = cdk_calloc (1, sizeof *pkt->pkt.mdc);
      if (!pkt->pkt.mdc)
	return CDK_Out_Of_Core;
      rc = read_mdc (inp, pkt->pkt.mdc);
      break;

    default:
      /* skip all packets we don't understand */
      while (pkth.len--)
        stream_getc (inp);
      break;
    }

  return rc;
}
