/* minpg.c - Minimal Privacy Guard
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * This is only a regression test and should _NOT_ used for real work.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef HAVE_LIBPOPT
#include <stdio.h>
#include <popt.h>
#include <opencdk.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

enum
{
  cArmor = 'a',
  cDearmor = 'u',
  cHelp = 'h',
  cVersion = 'V',
  cSymmetric = 'c',
  cKeyList = 'k',
  cSigList = 'l',
  cEncr = 'e',
  cDecr = 'd',
  cExport = 'x',
  oCompress = 'z',
  oRecipient = 'r',
  oVerbose = 'v',
  oOutput = 'o',
  oCipherAlg = 'n',
  oDigestAlgo = 'm',
  oFetchKey = 'f',
};


static struct
{
  const char *name;
  int algid;
} algo_table[] =
{
  {"3DES", GCRY_CIPHER_3DES},
  {"CAST5", GCRY_CIPHER_CAST5},
  {"BLOWFISH", GCRY_CIPHER_BLOWFISH},
  {"TWOFISH", GCRY_CIPHER_TWOFISH},
  {"AES", GCRY_CIPHER_AES128},
  {"MD5", GCRY_MD_MD5},
  {"SHA1", GCRY_MD_SHA1},
  {"RIPEMD160", GCRY_MD_RMD160},
  {NULL},
};

char *file = "(none)";
char *user = "(none)";
char *outname = "(none)";
char *cipher_algo = "(none)";
char *digest_algo = "(none)";
char *ks_keyid = "(none)";
int cipher_algid = 0;
int digest_algid = 0;
int use_outname = 0;


struct poptOption arguments[] =
{
  {"fetch-key", oFetchKey, POPT_ARG_STRING, &ks_keyid, oFetchKey,
   "Fetch key from keyserver", 0},
  {"cipher-algo", oCipherAlg, POPT_ARG_STRING, &cipher_algo, oCipherAlg,
   "Cipher Algo", 0},
  {"digest-algo", oDigestAlgo, POPT_ARG_STRING, &digest_algo, oDigestAlgo,
   "Digest Algo", 0},
  {"list-sigs", cSigList, POPT_ARG_STRING, &user, cSigList,
   "List signatures", 0},
  {"output", oOutput, POPT_ARG_STRING, &outname, oOutput,
   "Output filename", 0},
  {"export", cExport, POPT_ARG_STRING, &user, cExport,
   "Export keys", 0},
  {"list-keys", cKeyList, POPT_ARG_STRING, &user, cKeyList,
   "List keys", 0},
  {"help", cHelp, POPT_ARG_NONE, 0, cHelp, "Show help message", 0},
  {"version", cVersion, POPT_ARG_NONE, 0, cVersion,
   "Show program version", 0},
  {"armor", cArmor, POPT_ARG_STRING, &file, cArmor,
   "Protect a file with ASCII-Armor", 0},
  {"dearmor", cDearmor, POPT_ARG_STRING, &file, cDearmor,
   "Remove ASCII-Armor from a file", 0},
  {"compress", oCompress, POPT_ARG_NONE, 0, oCompress,
   "Use compression", 0},
  {"symmetric", cSymmetric, POPT_ARG_STRING, &file, cSymmetric,
   "Symmetrically encrypt a file", 0},
  {"recipient", oRecipient, POPT_ARG_STRING, &user, oRecipient,
   "Use this recipient", 0},
  {"encrypt", cEncr, POPT_ARG_STRING, &file, cEncr, "Encrypt a file", 0},
  {"decrypt", cDecr, POPT_ARG_STRING, &file, cDecr, "Decrypt a file", 0},
  {"verbose", oVerbose, POPT_ARG_NONE, 0, oVerbose, "Verbosity", 0},
  {NULL}
};


static void
out_of_core (void)
{
  fputs ("\n** out of core **\n", stderr);
  exit (1);
}


static char *
do_add_ext (const char *filename, const char *ext)
{
  char *output = NULL;

  if (!filename)
    return NULL;
  output = cdk_calloc (1, strlen (filename) + 5);
  if (!output)
    out_of_core ();
  strcpy (output, filename);
  strcat (output, ext ? ext : ".gpg");
  return output;
}


static char *
do_remove_ext (const char *filename)
{
  char *output = NULL;

  if (!filename)
    return NULL;
  if (!strstr (filename, ".gpg")
      && !strstr (filename, ".sig") && !strstr (filename, ".asc"))
    {
      output = cdk_calloc (1, strlen (filename) + 5);
      if (!output)
        out_of_core ();
      strcpy (output, filename);
      strcat (output, ".dec");
      return output;
    }
  output = cdk_strdup (filename);
  if (!output)
    out_of_core ();
  output[strlen (output) - 4] = '\0';
  return output;
}


static char
key_get_letter (int pubkey_algo)
{
  switch (pubkey_algo)
    {
    case GCRY_PK_RSA:   return 'R';
    case GCRY_PK_RSA_E: return 'r';
    case GCRY_PK_ELG_E: return 'g';
    case GCRY_PK_ELG:   return 'G';
    case GCRY_PK_DSA:   return 'D';
    default:            return '?';
    }
  return '?';
}


static const char *
key_get_date (long timestamp)
{
  static char kdate[32];
  struct tm *tm;

  tm = localtime (&timestamp);
  snprintf (kdate, sizeof kdate - 1, "%04d-%02d-%02d",
	    tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday);
  return kdate;
}


static char
sig_get_letter (cdkPKT_signature * sig)
{
  if (sig->flags.missing_key)
    return '%';
  if (sig->flags.checked)
    {
      if (sig->flags.valid == 1)
	return sig->flags.valid ? '!' : '-';
    }
  return '?';
}


static char
sig_get_status (cdkPKT_signature * sig)
{
  if (!sig->flags.exportable)
    return 'L';
  if (sig->sig_class == 0x13)
    return '3';
  else if (sig->sig_class == 0x12)
    return '2';
  else if (sig->sig_class == 0x11)
    return '1';
  return ' ';
}


static int
algo_get_id (const char *algo, int what)
{
  int i = 0;

  for (i = 0; algo_table[i].name; i++)
    {
      if (!strcmp (algo, algo_table[i].name))
	return algo_table[i].algid;
    }
  if (what)
    i = GCRY_CIPHER_3DES;
  else
    i = GCRY_MD_SHA1;
  return i;
}


static void
do_show_key (CDK_KBNODE pk, int with_sigs)
{
  CDK_KBNODE p, ctx = NULL;
  CDK_PACKET *pkt;
  
  while ((p = cdk_kbnode_walk (pk, &ctx, 0)))
    {
      pkt = cdk_kbnode_get_packet (p);
      if (pkt->pkttype == CDK_PKT_PUBLIC_KEY)
	{
	  cdkPKT_public_key *key = pkt->pkt.public_key;
	  fprintf (stdout, "pub  %04d%c/%08lX %s\n",
		   cdk_pk_get_nbits (key),
		   key_get_letter (key->pubkey_algo),
		   (unsigned long)cdk_pk_get_keyid (key, NULL),
		   key_get_date (key->timestamp));
	}
      else if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
	{
	  cdkPKT_public_key *key = pkt->pkt.public_key;
	  fprintf (stdout, "sub  %04d%c/%08lX %s\n",
		   cdk_pk_get_nbits (key),
		   key_get_letter (key->pubkey_algo),
		   (unsigned long)cdk_pk_get_keyid (key, NULL),
		   key_get_date (key->timestamp));
	}
      else if (pkt->pkttype == CDK_PKT_USER_ID)
	{
	  cdkPKT_user_id *id = pkt->pkt.user_id;
	  fprintf (stdout, "uid  %s  %s\n",
		   id->is_revoked ? "[revoked]" : "         ", id->name);
	}
      else if (with_sigs && pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  cdkPKT_signature *sig = pkt->pkt.signature;
	  fprintf (stdout, " sig%c %c %08lX %s\n",
		   sig_get_letter (sig),
		   sig_get_status (sig),
		   (unsigned long)cdk_sig_get_keyid (sig, NULL),
		   key_get_date (sig->timestamp));
	}
    }
}


int
do_list_keys (CDK_STRLIST keys)
{
  const char * ring, * s;
  CDK_STREAM inp;
  CDK_KBNODE pk = NULL;
  CDK_KEYDB_HD khd = NULL;
  CDK_STRLIST ctx = NULL;
  int rc = 0;

  if (keys)
    {
      khd = cdk_keydb_get_ctx (0, -1);
      while ((s = cdk_strlist_walk (keys, &ctx)))
        {
	  rc = cdk_keydb_get_bypattern (khd, s, &pk);
	  if (rc)
	    fprintf (stdout, "get key `%s': %s\n", s, cdk_strerror (rc));
	  if (!rc)
	    do_show_key (pk, 0);
	  cdk_kbnode_release (pk);
	  pk = NULL;
	}
      return 0;
    }
  /* Show all keys */
  ring = cdk_keydb_get_name (-1);
  if (!ring)
    {
      fprintf (stderr, "No keyring was found.\n");
      return CDK_General_Error;
    }
  rc = cdk_stream_open (ring, &inp);
  if (rc)
    {
      fprintf (stderr, "`%s': %s\n", ring, cdk_strerror (rc));
      return rc;
    }
  while (!(rc = cdk_keydb_get_keyblock (inp, &pk) && !cdk_stream_eof (inp)))
    {
      do_show_key (pk, 0);
      cdk_kbnode_release (pk);
      pk = NULL;
    }
  cdk_stream_close (inp);
  return 0;
}


int
do_export_keys (CDK_STRLIST keys, int secret)
{
  CDK_STREAM outp = NULL;
  int rc = 0;

  if (!use_outname)
    {
      fprintf (stdout, "Please use --output for a file.\n");
      return CDK_General_Error;
    }

  rc = cdk_stream_new (outname, &outp);
  if (rc)
    {
      fprintf (stdout, "can't open `%s': %s\n", outname, cdk_strerror (rc));
      return rc;
    }
  if (keys)
    {
      rc = cdk_keydb_export (outp, keys, 0);
      if (rc)
	fprintf (stdout, "key export `%s'\n", cdk_strerror (rc));
    }
  cdk_stream_close (outp);
  return rc;
}


int
do_list_sigs (CDK_STRLIST keys)
{
  CDK_KEYDB_HD hd = NULL;
  CDK_KBNODE pk = NULL;
  CDK_STRLIST ctx = NULL;
  const char *s;
  int kstat = 0;
  int rc = 0;

  if (keys)
    {
      hd = cdk_keydb_get_ctx (0, -1);
      while ((s = cdk_strlist_walk (keys, &ctx)))
	{
	  rc = cdk_keydb_get_bypattern (hd, s, &pk);
	  if (rc)
	    fprintf (stdout, "get key `%s': %s\n", s, cdk_strerror (rc));
	  rc = cdk_key_check_sigs (pk, hd, &kstat);
	  if (rc)
	    fprintf (stdout, "check sigs `%s': %s\n", s, cdk_strerror (rc));
	  do_show_key (pk, 1);
	}
    }

  return 0;
}


char *
get_passphrase (const char *prompt)
{
  char *p = NULL;
  int maxlen = 255;

  fprintf (stdout, "%s", prompt);
  p = cdk_calloc (1, 256);
  if (p)
    {
      fgets (p, maxlen, stdin);
      p[strlen (p) - 1] = '\0';
      return p;
    }
  return NULL;
}


int
ks_fetch_key (const char * kid)
{
  unsigned long keyid;
  CDK_KBNODE pk = NULL;
  CDK_PACKET * pkt;
  int rc;
  
  if (!strncmp (kid, "0x", 2))
    kid += 2;
  if (strlen (kid) != 8 && strlen (kid) != 16)
    {
      printf ("wrong keyID usage\n");
      return -1;
    }
  if (strlen (kid) == 16)
    keyid = strtoul (kid + 8, NULL, 16);
  else
    keyid = strtoul (kid, NULL, 16);

  printf ("fetch %08lX from the keyserver...", keyid);
  rc = cdk_keyserver_recv_key ("http://horowitz.surfnet.nl", 11371,
                               &keyid, &pk);
  if (rc)
      printf ("%s\n", cdk_strerror (rc));
  else
      printf ("OK\n");
  pkt = cdk_kbnode_find_packet (pk, CDK_PKT_USER_ID);
  if (pkt)
    printf ("%s\n", pkt->pkt.user_id->name);

  cdk_kbnode_release (pk);
  return 0;
}


int
main (int argc, char **argv)
{
  CDK_HD hd;
  CDK_STRLIST remusr = NULL, klist = NULL;
  poptContext opt_ctx;
  char *output = NULL;
  char u[128];
  int c = 0, verbose = 0, i = 0;
  int rc = 0;

  cdk_keydb_add_resource ("/home/twoaday/.gnupg/pubring.gpg", 0);
  cdk_passphrase_cb_set (get_passphrase);

  cdk_handle_new (&hd);
  opt_ctx = poptGetContext ("pred", argc, argv, arguments, 0);
  while ((c = poptGetNextOpt (opt_ctx)) >= 0)
    {
      switch (c)
	{
	case 'v':
	  verbose++;
	  break;

	case 'o':
	  if (verbose)
	    fprintf (stdout, "Output filename `%s'\n", outname);
	  use_outname = 1;
	  break;

	case 'h':
	  fprintf (stderr, "opencdk (MinPG) %s\n\n", VERSION);
	  poptPrintHelp (opt_ctx, stdout, 0);
	  return 0;

	case 'V':
	  fprintf (stderr, "Minimal Privacy Guard (minpg) %s\n\n", VERSION);
	  fprintf (stderr, "Supported algorithms:\n");
	  fprintf (stderr, "Cipher: ");
	  for (i = 0; algo_table[i].name; i++)
	    fprintf (stderr, "%s ", algo_table[i].name);
	  fprintf (stderr, "\n");
	  return 0;

        case 'm':
          digest_algid = algo_get_id (digest_algo, 0);
          if (verbose)
            fprintf (stdout, "Use %s as the message digest (%d)\n",
                     digest_algo, digest_algid);
          break;
          
	case 'n':
	  cipher_algid = algo_get_id (cipher_algo, 1);
	  if (verbose)
	    fprintf (stdout, "Use %s for encryption (%d)\n",
		     cipher_algo, cipher_algid);
	  break;

	case 'u':
	  output = cdk_calloc (1, strlen (file));
	  if (strstr (file, ".asc"))
	    strncpy (output, file, strlen (file) - 4);
	  else
	    strcpy (output, file);
	  if (verbose)
	    fprintf (stdout, "dearmor `%s' -> %s\n", file, output);
	  rc = cdk_file_dearmor (file, output);
	  if (rc)
	    fprintf (stderr, "minpg: dearmor file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;

	case 'l':
	  cdk_strlist_add (&klist, user);
	  do_list_sigs (klist);
	  break;

	case 'a':
	  output = do_add_ext (file, ".asc");
	  if (verbose)
	    fprintf (stdout, "armor `%s' -> %s\n", file, output);
	  rc = cdk_file_armor (hd, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: armor file `%s': %s\n", file,
		     cdk_strerror (rc));
	  break;

	case 'z':
	  fprintf (stderr, "minpg: Compression is not available YET!\n");
	  break;

	case 'c':
	  output = do_add_ext (file, ".gpg");
	  if (verbose)
	    fprintf (stdout, "symencrypt `%s' -> %s\n", file, output);
	  if (cipher_algid)
            cdk_handle_set_cipher (hd, cipher_algid);
          if (digest_algid)
            {
              cdk_handle_set_digest (hd, digest_algid);
              cdk_handle_set_s2k (hd, 3, digest_algid, cipher_algid);
            }
	  rc = cdk_file_encrypt (hd, NULL, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: symmetric file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;

	case 'k':
	  cdk_strlist_add (&klist, user);
	  do_list_keys (klist);
	  break;

	case 'x':
	  cdk_strlist_add (&klist, user);
	  do_export_keys (klist, 0);
	  break;

        case 'f':
          ks_fetch_key (ks_keyid);
          break;

	case 'r':
	  cdk_strlist_add (&remusr, user);
	  break;

	case 'e':
	  if (!remusr)
	    {
	      fprintf (stdout, "No recipient was chosen.\n\n");
	      fprintf (stdout, "Recipient: ");
	      fgets (u, sizeof u, stdin);
	      cdk_strlist_add (&remusr, user);
	    }
	  output = do_add_ext (file, ".gpg");
	  if (verbose)
	    fprintf (stdout, "encrypt `%s' -> %s\n", file, output);
	  rc = cdk_file_encrypt (hd, remusr, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: encrypt file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;

	case 'd':
	  output = do_remove_ext (file);
	  if (verbose)
	    fprintf (stdout, "decrypt `%s' -> %s\n", file, output);
	  rc = cdk_file_decrypt (hd, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: decrypt file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;
	}
    }

  cdk_handle_free (hd);
  cdk_strlist_free (remusr);
  return 0;
}
#endif /* HAVE_LIBPOPT */
