/* plaintext.c - Literal packet filters
 *       Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "filters.h"


int
plaintext_decode (void * opaque, FILE * in, FILE * out)
{
  plaintext_filter_s * pfx = opaque;
  CDK_STREAM si, so;
  CDK_PACKET pkt;
  cdkPKT_plaintext * pt;
  byte buf[4096];
  size_t nread;
  int rc, bufsize;

  if (!pfx || !in || !out)
    return CDK_Inv_Value;

  si = cdk_stream_fpopen (in, 0);
  if (!si)
    return CDK_Out_Of_Core;
  so = cdk_stream_fpopen (out, 1);
  if (!so)
    {
      cdk_stream_close (si);
      return CDK_Out_Of_Core;
    }
  init_packet (&pkt);
  rc = cdk_pkt_parse (si, &pkt);
  if (pkt.pkttype != CDK_PKT_PLAINTEXT)
    {
      if (pkt.pkttype)
        cdk_pkt_free (&pkt);
      return rc;
    }
  pt = pkt.pkt.plaintext;
  pfx->mode = pt->mode;
  pfx->filename = cdk_strdup (pt->name? pt->name : " ");
  if (!pfx->filename)
    {
      cdk_pkt_free (&pkt);
      return CDK_Out_Of_Core;
    }
  while (pt->len > 0)
    {
      bufsize = pt->len < sizeof buf-1? pt->len : sizeof buf-1;
      nread = cdk_stream_read (pt->buf, buf, bufsize);
      if (nread == EOF)
        {
          rc = CDK_File_Error;
          break;
        }
      if (pfx->md)
        gcry_md_write (pfx->md, buf, nread);
      cdk_stream_write (so, buf, nread);
      pt->len -= nread;
    }
  cdk_stream_close (si);
  cdk_stream_close (so);
  cdk_pkt_free (&pkt);
  return rc;
}


int
plaintext_encode (void * opaque, FILE * in, FILE * out)
{
  plaintext_filter_s * pfx = opaque;
  cdkPKT_plaintext * pt;
  CDK_STREAM si, so;
  CDK_PACKET pkt;
  size_t filelen;
  int rc;

  if (!pfx || !in || !out)
    return CDK_Inv_Value;
  
  if (!pfx->filename)
    pfx->filename = cdk_strdup ("_CONSOLE");

  si = cdk_stream_fpopen (in, 0);
  if (!si)
    return CDK_Out_Of_Core;
  so = cdk_stream_fpopen (out, 1);
  if (!so)
    {
      cdk_stream_close (si);
      return CDK_Out_Of_Core;
    }
  filelen = strlen (pfx->filename);
  pt = cdk_calloc (1, sizeof *pt + filelen - 1);
  if (!pt)
    return CDK_Out_Of_Core;
  memcpy (pt->name, pfx->filename, filelen);
  pt->namelen = filelen;
  pt->name[pt->namelen] = '\0';
  pt->timestamp = _cdk_timestamp ();
  pt->mode = pfx->mode ? 't' : 'b';
  pt->len = cdk_stream_get_length (si);
  pt->buf = si;
  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_PLAINTEXT;
  pkt.pkt.plaintext = pt;
  rc = cdk_pkt_build (so, &pkt);

  cdk_free (pt);
  cdk_stream_close (si);
  cdk_stream_close (so);

  return rc;
}


int
text_encode (void * opaque, FILE * in, FILE * out)
{
  const char * s;
  char buf[1024];

  while (!feof (in))
    {
      s = fgets (buf, sizeof buf-1, in);
      if (!s)
        break;
      _cdk_trim_string (buf, 1);
      fwrite (buf, 1, strlen (buf), out);
    }

  return 0;
}

      
int
text_decode (void * opaque, FILE * in, FILE * out)
{
  const char * s;
  char buf[1024];
  text_filter_s * tfx = opaque;

  if (!tfx)
    return CDK_Inv_Value;

  while (!feof (in))
    {
      s = fgets (buf, sizeof buf-1, in);
      if (!s)
        break;
      _cdk_trim_string (buf, 0);
      fwrite (buf, 1, strlen (buf), out);
      fwrite (tfx->lf, 1, strlen (tfx->lf), out);
    }

  return 0;
}

