/* minpg.c - Minimal Privacy Guard
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * This is only a regression test and should _NOT_ used for real work.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef HAVE_LIBPOPT
#include <stdio.h>
#include <popt.h>
#include <opencdk.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

enum
{
  cSignEncr = 'b',
  cEnarmor = 'a',
  cDearmor = 'u',
  cHelp = 'h',
  cVersion = 'V',
  cSymmetric = 'c',
  cKeyList = 'k',
  cVerify = 'y',
  cGenKey = 'w',
  cSigList = 'l',
  cEncr = 'e',
  cDecr = 'd',
  cExport = 'x',
  cImport = 'i',
  oCompress = 'z',
  oRecipient = 'r',
  oVerbose = 'v',
  oOutput = 'o',
  oCipherAlg = 'n',
  oDigestAlgo = 'm',
  oFetchKey = 'f',
  cSign = 's',
  cSignClear = 't',
  oArmor = 'p',
  oCompat = 'g',
  oNoCompress = '0',
};


static struct
{
  const char * name;
  int algid;
} algo_table[] =
{
  {"3DES", GCRY_CIPHER_3DES},
  {"CAST5", GCRY_CIPHER_CAST5},
  {"BLOWFISH", GCRY_CIPHER_BLOWFISH},
  {"TWOFISH", GCRY_CIPHER_TWOFISH},
  {"AES", GCRY_CIPHER_AES128},
  {"AES192", GCRY_CIPHER_AES192},
  {"AES256", GCRY_CIPHER_AES256},
  {"MD5", GCRY_MD_MD5},
  {"SHA1", GCRY_MD_SHA1},
  {"RIPEMD160", GCRY_MD_RMD160},
  {NULL},
};

char * file = "(none)";
char * user = "(none)";
char * outname = "(none)";
char * cipher_algo = "(none)";
char * digest_algo = "(none)";
char * ks_keyid = "(none)";
int cipher_algid = 0;
int digest_algid = 0;
int use_outname = 0;


struct poptOption arguments[] =
{
  {"gen-key", cGenKey, POPT_ARG_NONE, 0, cGenKey, "Generate a key", 0},
  {"no-compression", oNoCompress, POPT_ARG_NONE, 0, oNoCompress,
   "Turns off compression", 0},
  {"compat", oCompat, POPT_ARG_NONE, 0, oCompat, "PGP compat mode", 0},
  {"verify", cVerify, POPT_ARG_STRING, &file, cVerify,
   "Verify a signature", 0},
  {"sign", cSign, POPT_ARG_STRING, &file, cSign, "Sign a file", 0},
  {"sign-encrypt", cSignEncr, POPT_ARG_STRING, &file, cSignEncr,
   "Sign and encrypt a file", 0},
  {"clearsign", cSignClear, POPT_ARG_STRING, &file, cSignClear,
   "Sign a text file", 0},
  {"fetch-key", oFetchKey, POPT_ARG_STRING, &ks_keyid, oFetchKey,
   "Fetch key from keyserver", 0},
  {"cipher-algo", oCipherAlg, POPT_ARG_STRING, &cipher_algo, oCipherAlg,
   "Cipher Algo", 0},
  {"digest-algo", oDigestAlgo, POPT_ARG_STRING, &digest_algo, oDigestAlgo,
   "Digest Algo", 0},
  {"list-sigs", cSigList, POPT_ARG_STRING, &user, cSigList,
   "List signatures", 0},
  {"output", oOutput, POPT_ARG_STRING, &outname, oOutput,
   "Output filename", 0},
  {"export", cExport, POPT_ARG_STRING, &user, cExport,
   "Export keys", 0},
  {"import", cImport, POPT_ARG_STRING, &file, cImport, "Import keys", 0},
  {"list-keys", cKeyList, POPT_ARG_STRING, &user, cKeyList,
   "List keys", 0},
  {"help", cHelp, POPT_ARG_NONE, 0, cHelp, "Show help message", 0},
  {"version", cVersion, POPT_ARG_NONE, 0, cVersion,
   "Show program version", 0},
  {"enarmor", cEnarmor, POPT_ARG_STRING, &file, cEnarmor,
   "Protect a file with ASCII-Armor", 0},  
  {"dearmor", cDearmor, POPT_ARG_STRING, &file, cDearmor,
   "Remove ASCII-Armor from a file", 0},
  {"compress", oCompress, POPT_ARG_NONE, 0, oCompress,
   "Use compression", 0},
  {"symmetric", cSymmetric, POPT_ARG_STRING, &file, cSymmetric,
   "Symmetrically encrypt a file", 0},
  {"recipient", oRecipient, POPT_ARG_STRING, &user, oRecipient,
   "Use this recipient", 0},
  {"encrypt", cEncr, POPT_ARG_STRING, &file, cEncr, "Encrypt a file", 0},
  {"decrypt", cDecr, POPT_ARG_STRING, &file, cDecr, "Decrypt a file", 0},
  {"verbose", oVerbose, POPT_ARG_NONE, 0, oVerbose, "Verbosity", 0},
  {"armor", oArmor, POPT_ARG_NONE, 0, oArmor, "Armor output", 0},
  {NULL}
};


static void
out_of_core (void)
{
  fputs ("\n** out of core **\n", stderr);
  exit (1);
}


static void
callback (void * opaque, int type, const char * s)
{
  fprintf (stdout, "type=%d val=%s\n", type, s);
}


static void
progr_callback (void * opaque, unsigned off, unsigned len)
{
  printf ("off=%d len=%d\n", off, len);
}


static char *
do_add_ext (const char * filename, const char * ext)
{
  char * output = NULL;

  if (!filename)
    return NULL;
  output = cdk_calloc (1, strlen (filename) + 5);
  if (!output)
    out_of_core ();
  strcpy (output, filename);
  strcat (output, ext ? ext : ".gpg");
  return output;
}


static char *
do_remove_ext (const char * filename)
{
  char * output = NULL;

  if (!filename)
    return NULL;
  if (!strstr (filename, ".gpg") && !strstr (filename, ".pgp")
      && !strstr (filename, ".sig") && !strstr (filename, ".asc"))
    {
      output = cdk_calloc (1, strlen (filename) + 5);
      if (!output)
        out_of_core ();
      strcpy (output, filename);
      strcat (output, ".dec");
      return output;
    }
  output = cdk_strdup (filename);
  if (!output)
    out_of_core ();
  output[strlen (output) - 4] = '\0';
  return output;
}


static char
key_get_letter (int pubkey_algo)
{
  switch (pubkey_algo)
    {
    case GCRY_PK_RSA:   return 'R';
    case GCRY_PK_RSA_E: return 'r';
    case GCRY_PK_ELG_E: return 'g';
    case GCRY_PK_ELG:   return 'G';
    case GCRY_PK_DSA:   return 'D';
    default:            return '?';
    }
  return '?';
}


static const char *
key_get_date (long timestamp)
{
  static char kdate[32];
  struct tm * tm;

  tm = localtime (&timestamp);
  snprintf (kdate, sizeof kdate - 1, "%04d-%02d-%02d",
	    tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday);
  return kdate;
}


static char
sig_get_letter (cdkPKT_signature * sig)
{
  if (sig->flags.missing_key)
    return '%';
  if (sig->flags.checked)
    {
      if (sig->flags.valid == 1)
	return sig->flags.valid ? '!' : '-';
    }
  return '?';
}


static char
sig_get_status (cdkPKT_signature * sig)
{
  if (!sig->flags.exportable)
    return 'L';
  if (sig->sig_class == 0x13)
    return '3';
  else if (sig->sig_class == 0x12)
    return '2';
  else if (sig->sig_class == 0x11)
    return '1';
  return ' ';
}


static int
algo_get_id (const char *algo, int what)
{
  int i = 0;

  for (i = 0; algo_table[i].name; i++)
    {
      if (!strcmp (algo, algo_table[i].name))
	return algo_table[i].algid;
    }
  if (what)
    i = GCRY_CIPHER_3DES;
  else
    i = GCRY_MD_SHA1;
  return i;
}


static char *
get_keyvalid_char (cdkPKT_public_key * pk)
{
  static char flags[2];
  memset (flags, 0, 2);
  if (pk->has_expired)
    strcat (flags, "e");
  else
    strcat (flags, " ");
  if (pk->is_revoked)
    strcat (flags, "r");
  else
    strcat (flags, " ");
  if (!pk->has_expired && !pk->is_revoked)
    strcat (flags, "!");
  else
    strcat (flags, " ");
  return flags;
}

static void
do_show_key (CDK_KBNODE pk, int with_sigs)
{
  CDK_KBNODE p, ctx = NULL;
  CDK_PACKET * pkt;
  
  while ((p = cdk_kbnode_walk (pk, &ctx, 0)))
    {
      pkt = cdk_kbnode_get_packet (p);
      if (pkt->pkttype == CDK_PKT_PUBLIC_KEY)
	{
	  cdkPKT_public_key * key = pkt->pkt.public_key;
	  fprintf (stdout, "pub[%s]  %04d%c/%08lX %s",
                   get_keyvalid_char (key),
		   cdk_pk_get_nbits (key),
		   key_get_letter (key->pubkey_algo),
		   (unsigned long)cdk_pk_get_keyid (key, NULL),
		   key_get_date (key->timestamp));
          if (!key->expiredate)
            fputc ('\n', stdout);
          else
            fprintf (stdout, "  [expires: %s]\n",
                     key_get_date (key->expiredate));
	}
      else if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
	{
	  cdkPKT_public_key * key = pkt->pkt.public_key;
	  fprintf (stdout, "sub[%s]  %04d%c/%08lX %s",
                   get_keyvalid_char (key),
		   cdk_pk_get_nbits (key),
		   key_get_letter (key->pubkey_algo),
		   (unsigned long)cdk_pk_get_keyid (key, NULL),
		   key_get_date (key->timestamp));
          if (!key->expiredate)
            fputc ('\n', stdout);
          else
            fprintf (stdout, "  [expires: %s]\n",
                     key_get_date (key->expiredate));
	}
      else if (pkt->pkttype == CDK_PKT_USER_ID)
	{
	  cdkPKT_user_id * id = pkt->pkt.user_id;
          char * uid = cdk_utf8_decode (id->name, id->len, 0);
          fprintf (stdout, "uid %s %s\n",
                   id->is_revoked ? "[revoked]" : "         ", uid);
          cdk_free (uid);
	}
      else if (with_sigs && pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  cdkPKT_signature * sig = pkt->pkt.signature;
	  fprintf (stdout, " sig%c %c %08lX %s\n",
		   sig_get_letter (sig),
		   sig_get_status (sig),
		   (unsigned long)cdk_sig_get_keyid (sig, NULL),
		   key_get_date (sig->timestamp));
	}
    }
}


int
do_list_keys (CDK_STRLIST keys)
{
  const char * ring, * s;
  unsigned char fpr[20];
  CDK_PACKET * pkt;
  CDK_STREAM inp;
  CDK_KBNODE pk = NULL;
  CDK_KEYDB_HD khd = NULL;
  CDK_STRLIST ctx = NULL;
  int rc = 0, i;

  if (keys)
    {
      khd = cdk_keydb_get_ctx (0, -1);
      while ((s = cdk_strlist_walk (keys, &ctx)))
        {
	  rc = cdk_keydb_get_bypattern (khd, s, &pk);
	  if (rc)
	    fprintf (stdout, "get key `%s': %s\n", s, cdk_strerror (rc));
	  if (!rc)
            {
              do_show_key (pk, 0);
              pkt = cdk_kbnode_get_packet (pk);
              printf ("Fingerprint ");
              cdk_pk_get_fingerprint (pkt->pkt.public_key, fpr);
              for (i = 0; i < 20/2; i++)
                printf ("%02X%02X ", fpr[2*i], fpr[2*i+1]);
              printf ("\n");
            }
              
	  cdk_kbnode_release (pk);
	  pk = NULL;
	}
      return 0;
    }
  /* Show all keys */
  ring = cdk_keydb_get_name (-1);
  if (!ring)
    {
      fprintf (stderr, "No keyring was found.\n");
      return CDK_General_Error;
    }
  rc = cdk_stream_open (ring, &inp);
  if (rc)
    {
      fprintf (stderr, "`%s': %s\n", ring, cdk_strerror (rc));
      return rc;
    }

  while (!(rc = cdk_keydb_get_keyblock (inp, &pk)))
    {
      do_show_key (pk, 0);
      fputc ('\n', stdout);
      cdk_kbnode_release (pk);
      pk = NULL;
      rc = cdk_stream_eof (inp);
      if (rc)
        break;
    }
  cdk_stream_close (inp);
  return 0;
}


int
do_export_keys (CDK_STRLIST keys, int secret)
{
  CDK_STREAM outp = NULL;
  int rc = 0;

  if (!use_outname)
    {
      fprintf (stdout, "Please use --output for a file.\n");
      return CDK_General_Error;
    }

  rc = cdk_stream_new (outname, &outp);
  if (rc)
    {
      fprintf (stdout, "can't open `%s': %s\n", outname, cdk_strerror (rc));
      return rc;
    }
  if (keys)
    {
      rc = cdk_keydb_export (outp, keys, 0);
      if (rc)
	fprintf (stdout, "key export `%s'\n", cdk_strerror (rc));
    }
  cdk_stream_close (outp);
  return rc;
}


int
do_import_keys (const char * keyfile)
{
  CDK_KEYDB_HD hd;
  CDK_KBNODE node = NULL, n, ctx = NULL;
  CDK_STREAM s;
  int rc, res[4];

  hd = cdk_keydb_get_ctx (0, -1);
  if (!hd)
    {
      fprintf (stderr, "No public keyring available.\n");
      return -1;
    }
  rc = cdk_stream_open (keyfile, &s);
  if (!s)
    {
      fprintf (stderr, "open `%s' failed: %s\n", keyfile, cdk_strerror (rc));
      return -1;
    }
  rc = cdk_keydb_get_keyblock (s, &node);
  if (!node)
    {
      cdk_stream_close (s);
      fprintf (stderr, "get keyblock failed: %s\n", cdk_strerror (rc));
      return -1;
    }
  rc = cdk_keydb_import (hd, node, res);
  if (rc)
    fprintf (stderr, "import failed: %s\n", cdk_strerror (rc));
  else
    {
      while ((n = cdk_kbnode_walk (node, &ctx, 0)))
        {
          CDK_PACKET * pkt = cdk_kbnode_get_packet (n);
          if (pkt->pkttype == CDK_PKT_USER_ID)
            {
              printf ("public keys read/updated: %d/%d\n"
                      "secret keys read/updated: %d/%d\n",
                      res[0], res[2], res[1], res[3]);
              printf ("`%s' successfully imported.\n",
                      pkt->pkt.user_id->name);
              break;
            }
        }
    }
  cdk_stream_close (s);
  cdk_kbnode_release (node);
  return 0;
}
  

int
do_list_sigs (CDK_STRLIST keys)
{
  CDK_KEYDB_HD hd = NULL;
  CDK_KBNODE pk = NULL;
  CDK_STRLIST ctx = NULL;
  const char * s;
  int kstat = 0;
  int rc = 0;

  if (keys)
    {
      hd = cdk_keydb_get_ctx (0, -1);
      while ((s = cdk_strlist_walk (keys, &ctx)))
	{
	  rc = cdk_keydb_get_bypattern (hd, s, &pk);
	  if (rc)
	    fprintf (stdout, "get key `%s': %s\n", s, cdk_strerror (rc));
	  rc = cdk_key_check_sigs (pk, hd, &kstat);
	  if (rc)
	    fprintf (stdout, "check sigs `%s': %s\n", s, cdk_strerror (rc));
	  do_show_key (pk, 1);
	}
    }

  return 0;
}


char *
get_passphrase (const char *prompt)
{
  char * p = NULL;
  int maxlen = 255;

  fprintf (stdout, "%s", prompt);
  p = cdk_calloc (1, 256);
  if (p)
    {
      fgets (p, maxlen, stdin);
      p[strlen (p) - 1] = '\0';
      return p;
    }
  return NULL;
}


int
ks_fetch_key (const char * kid)
{
  unsigned long keyid;
  CDK_KBNODE pk = NULL;
  CDK_PACKET * pkt;
  int rc;
  
  if (!strncmp (kid, "0x", 2))
    kid += 2;
  if (strlen (kid) != 8 && strlen (kid) != 16)
    {
      printf ("wrong keyID usage\n");
      return -1;
    }
  if (strlen (kid) == 16)
    keyid = strtoul (kid + 8, NULL, 16);
  else
    keyid = strtoul (kid, NULL, 16);

  printf ("fetch %08lX from the keyserver...", keyid);
  rc = cdk_keyserver_recv_key ("http://horowitz.surfnet.nl", 11371,
                               &keyid, &pk);
  if (rc)
      printf ("%s\n", cdk_strerror (rc));
  else
      printf ("OK\n");
  pkt = cdk_kbnode_find_packet (pk, CDK_PKT_USER_ID);
  if (pkt)
    printf ("%s\n", pkt->pkt.user_id->name);

  cdk_kbnode_release (pk);
  return 0;
}


static const char *
get_sig_status (int id)
{
  const char * s = "unknown error";

  switch (id)
    {
    case CDK_SIGSTAT_NONE : s = "no signature"; break;
    case CDK_SIGSTAT_GOOD : s = "good signature"; break;
    case CDK_SIGSTAT_BAD  : s = "bad signature"; break;
    case CDK_SIGSTAT_NOKEY: s = "no key for checking the signature"; break; 
    }
  return s;
}

static const char *
get_sig_algo (int pubkey_algo)
{
  switch (pubkey_algo)
    {
    case GCRY_PK_RSA_E:
    case GCRY_PK_RSA_S:
    case GCRY_PK_RSA  : return "RSA";
    case GCRY_PK_ELG  :
    case GCRY_PK_ELG_E: return "ELG";
    case GCRY_PK_DSA  : return "DSA";
    default           : return "???"; 
    }
  return NULL; 
}

static const char *
get_sig_time (unsigned long timestamp)
{
  static char buf[128];
  struct tm * tmbuf;

  tmbuf = localtime (&timestamp);
  snprintf (buf, sizeof buf-1, "%04d-%02d-%02d",
            tmbuf->tm_year+1900, tmbuf->tm_mon+1, tmbuf->tm_mday);
  return buf;  
}


void
print_verify_stats (CDK_HD hd)
{
  const unsigned long * keyid;
  unsigned long created;
  int sigstat, pkalgo;

  sigstat = cdk_sig_get_ulong_attr (hd, 0, CDK_ATTR_STATUS);
  if (sigstat)
    {
      keyid = cdk_sig_get_data_attr (hd, 0, CDK_ATTR_KEYID);
      created = cdk_sig_get_ulong_attr (hd, 0, CDK_ATTR_CREATED);
      pkalgo = cdk_sig_get_ulong_attr (hd, 0, CDK_ATTR_ALGO_PK);
      printf ("(%s) keyid %08lX%08lX algo %s created %s\n",
              get_sig_status (sigstat), keyid[0], keyid[1],
              get_sig_algo (pkalgo), get_sig_time (created));
    }
}


int
genkey (void)
{
  int c, algo = 0, rc =0;
  CDK_KEYGEN_CTX hd;
  
  printf ("(1) DSA (sign only)\n");
  c = fgetc (stdin);
  switch (c)
    {
    case '1': algo = GCRY_PK_DSA; break;
    default: printf ("invalid pk algorithm\n"); break;
    }
  if (!algo)
    return CDK_Inv_Algo;
  cdk_salloc (1);
  cdk_keygen_new (&hd);
  cdk_keygen_set_algo_info (hd, 0, algo, 768);
  cdk_keygen_set_algo_info (hd, 1, GCRY_PK_ELG_E, 768);
  cdk_keygen_set_name (hd, "Otto Bismarck");
  cdk_keygen_set_mdc_feature (hd, 1);
  cdk_keygen_set_expire_date (hd, 1040820537);
  rc = cdk_keygen_start (hd);
  if (!rc)
    rc = cdk_keygen_save (hd, "pub.test", "sec.test");
  cdk_keygen_free (hd);
  return rc;
}


int
main (int argc, char ** argv)
{
  CDK_HD hd;
  CDK_STRLIST remusr = NULL, klist = NULL;
  poptContext opt_ctx;
  char * output = NULL;
  char u[128];
  int c = 0, verbose = 0, i = 0;
  int rc = 0;

  cdk_keydb_add_resource ("/home/twoaday/.gnupg/pubring.cdk", 0);
  /*cdk_keydb_add_resource ("pubring.gpg", 0);*/
  cdk_keydb_add_resource ("sec-with-pwd.gpg", 1);
  /*cdk_keydb_add_resource ("secring.gpg", 1);*/
  /*cdk_keydb_add_resource ("sec2.gpg", 1);*/
  /*cdk_passphrase_cb_set (get_passphrase);*/
  /*cdk_keydb_add_resource ("/home/twoaday/.pgp/pubring.pgp", 0);
    cdk_keydb_add_resource ("/home/twoaday/.pgp/secring.pgp", 1);*/

  cdk_handle_new (&hd);
  cdk_handle_set_callback (hd, callback, NULL);
  cdk_set_progress_handler (progr_callback, NULL);
  opt_ctx = poptGetContext ("pred", argc, argv, arguments, 0);
  while ((c = poptGetNextOpt (opt_ctx)) >= 0)
    {
      switch (c)
	{
	case 'v':
	  verbose++;
	  break;

        case oNoCompress:
          cdk_handle_set_compress (hd, 0);
          break;
          
        case 'p':
          cdk_handle_set_armor (hd, 1);
          break;

	case 'o':
	  if (verbose)
	    fprintf (stdout, "Output filename `%s'\n", outname);
	  use_outname = 1;
	  break;

        case 'g':
          cdk_handle_set_compat (hd, -1);
          break;

	case 'h':
	  fprintf (stderr, "opencdk (MinPG) %s\n\n", VERSION);
	  poptPrintHelp (opt_ctx, stdout, 0);
	  return 0;

	case 'V':
	  fprintf (stderr, "Minimal Privacy Guard (minpg) %s\n\n", VERSION);
	  fprintf (stderr, "Supported algorithms:\n");
	  fprintf (stderr, "Cipher: ");
	  for (i = 0; algo_table[i].name; i++)
	    fprintf (stderr, "%s ", algo_table[i].name);
	  fprintf (stderr, "\n");
	  return 0;

        case 'm':
          digest_algid = algo_get_id (digest_algo, 0);
          if (verbose)
            fprintf (stdout, "Use %s as the message digest (%d)\n",
                     digest_algo, digest_algid);
          break;
          
	case 'n':
	  cipher_algid = algo_get_id (cipher_algo, 1);
	  if (verbose)
	    fprintf (stdout, "Use %s for encryption (%d)\n",
		     cipher_algo, cipher_algid);
	  break;

        case 'y':
          rc = cdk_file_verify_clearsign (hd, file, NULL);
          if (rc)
            fprintf (stderr, "verify `%s' failed: %s\n", file,
                     cdk_strerror (rc));
          else
            print_verify_stats (hd);
          break;
          
	case 'u':
	  output = cdk_calloc (1, strlen (file));
	  if (strstr (file, ".asc"))
	    strncpy (output, file, strlen (file) - 4);
	  else
	    strcpy (output, file);
	  if (verbose)
	    fprintf (stdout, "dearmor `%s' -> %s\n", file, output);
	  rc = cdk_file_dearmor (file, output);
	  if (rc)
	    fprintf (stderr, "minpg: dearmor file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;

	case 'l':
	  cdk_strlist_add (&klist, user);
	  do_list_sigs (klist);
	  break;

	case 'a':
	  output = do_add_ext (file, ".asc");
	  if (verbose)
	    fprintf (stdout, "armor `%s' -> %s\n", file, output);
	  rc = cdk_file_armor (hd, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: armor file `%s': %s\n", file,
		     cdk_strerror (rc));
	  break;

	case 'z':
	  fprintf (stderr, "minpg: Compression is not available YET!\n");
	  break;

	case 'c':
	  output = do_add_ext (file, ".gpg");
	  if (verbose)
	    fprintf (stdout, "symencrypt `%s' -> %s\n", file, output);
	  if (cipher_algid)
            cdk_handle_set_cipher (hd, cipher_algid);
          if (digest_algid)
            {
              cdk_handle_set_digest (hd, digest_algid);
              cdk_handle_set_s2k (hd, 3, digest_algid, cipher_algid);
            }
	  rc = cdk_file_encrypt (hd, NULL, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: symmetric file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;

	case 'k':
	  cdk_strlist_add (&klist, user);
	  do_list_keys (klist);
	  break;

	case 'x':
	  cdk_strlist_add (&klist, user);
	  do_export_keys (klist, 0);
	  break;

        case 'i':
          do_import_keys (file);
          break;

        case 'f':
          ks_fetch_key (ks_keyid);
          break;

	case 'r':
	  cdk_strlist_add (&remusr, user);
	  break;

	case 'e':
	  if (!remusr)
	    {
	      fprintf (stdout, "No recipient was chosen.\n\n");
	      fprintf (stdout, "Recipient: ");
	      fgets (u, sizeof u, stdin);
	      cdk_strlist_add (&remusr, user);
	    }
          cdk_set_log_level (CDK_LOG_DEBUG);
	  output = do_add_ext (file, ".gpg");
	  if (verbose)
	    fprintf (stdout, "encrypt `%s' -> %s\n", file, output);
	  rc = cdk_file_encrypt (hd, remusr, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: encrypt file `%s': %s\n",
		     file, cdk_strerror (rc));
	  break;

	case 'd':
          cdk_set_log_level (CDK_LOG_DEBUG);
	  output = do_remove_ext (file);
	  if (verbose)
	    fprintf (stdout, "decrypt `%s' -> %s\n", file, output);
	  rc = cdk_file_decrypt (hd, file, output);
	  if (rc)
	    fprintf (stdout, "minpg: decrypt file `%s' failed: %s\n",
		     file, cdk_strerror (rc));
	  break;

        case 'w':
          rc = genkey ();
          if (rc)
            printf ("key generation failed `%s'\n", cdk_strerror (rc));
          break;
          
        case 's':
          if (!remusr)
            {
              fprintf (stdout, "No user was chosen.\n");
              exit (0);
            } 
          output = do_add_ext (file, ".sig");
          if (verbose)
            fprintf (stdout, "sign `%s' -> %s\n", file, output);
          rc = cdk_file_sign (hd, remusr, NULL, file, output, 1, 0);
          if (rc)
            fprintf (stderr, "minpg: sign file `%s' failed: %s\n",
                     file, cdk_strerror (rc));
          break;

        case 't':
          if (!remusr)
            {
              fprintf (stdout, "No user was chosen.\n");
              exit (0);
            }
          output = do_add_ext (file, ".asc");
          rc = cdk_file_clearsign (hd, remusr, file, output);
          if (rc)
            fprintf (stderr, "minpg: clearsign file `%s': failed: %s\n",
                     file, cdk_strerror (rc));
          break;

        case 'b':
          if (!remusr)
            {
              fprintf (stderr, "No recipient was chosen.\n");
              exit (0);
            }
          output = do_add_ext (file, ".gpg");
          cdk_strlist_add (&klist, "opencdk");
          rc = cdk_file_sign (hd, klist, remusr, file, output, 0, 1);
          if (rc)
            fprintf (stderr, "minpg: sign and encrypt `%s' failed: %s\n",
                     file, cdk_strerror (rc));
          break;
	}
    }

  cdk_handle_free (hd);
  cdk_strlist_free (remusr);
  cdk_free (output);
  return 0;
}
#else
int
main (int argc, char ** argv)
{
  printf ("MinPG example is not available.\n");
  return 0;
}
#endif /* HAVE_LIBPOPT */
