/* -*- Mode: C; c-file-style: "bsd" -*- */
/* encrypt.c
 *       Copyright (C) 2002, 2003 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <stdarg.h>
#include <malloc.h>
#include <assert.h>

#include "opencdk.h"
#include "main.h"
#include "filters.h"
#include "stream.h"
#include "packet.h"


struct mainproc_ctx_s {
    CDK_DEK dek;
    CDK_STREAM s;
    CDK_KBNODE node;
    struct {
        unsigned present:1;
        unsigned one_pass:1;
        GCRY_MD_HD md;
        int digest_algo;
        int is_expired;
        cdkPKT_public_key * pk;
        unsigned pt_offset;
    } sig;
    unsigned reset:1;
    unsigned eof_seen:1;
    unsigned key_seen:1;
    char * file; /* for detached signatures */
    const char * output;
};
typedef struct mainproc_ctx_s *CTX;


static void
write_marker_packet( CDK_STREAM out )
{
    byte buf[5];

    buf[0] = (0x80 | (10<<2));
    buf[1] = 3;
    buf[2] = 0x50;
    buf[3] = 0x47;
    buf[4] = 0x50;
    cdk_stream_write( out, buf, 5 );
}


static void
plaintext_set_rfc1991( CDK_STREAM out )
{
    plaintext_filter_s * pfx;
    pfx = _cdk_stream_get_opaque( out, fPLAINTEXT );
    if( pfx )
        pfx->rfc1991 = 1;
}


static int
sym_stream_encrypt (CDK_HD hd, CDK_STREAM inp, CDK_STREAM out)
{
    CDK_PACKET pkt;
    cdkPKT_symkey_enc * enc;
    char * pw = NULL;
    int rc = 0;

    if (!hd || !inp || !out)
        return CDK_Inv_Value;

    pw = _cdk_passphrase_get ("Enter Passphrase: ");
    if( !pw )
        goto fail;

    cdk_free( hd->s2k );
    rc = cdk_s2k_new( &hd->s2k, hd->_s2k.mode, hd->_s2k.digest_algo, NULL );
    if( rc )
        goto fail;
    
    cdk_dek_free( hd->dek );
    rc = cdk_dek_from_passphrase( &hd->dek, hd->cipher_algo, hd->s2k, 2, pw );
    if( rc )
        goto fail;

    if( hd->opt.rfc1991 ) {
        hd->dek->rfc1991 = 1;
        goto start; /* skip the pkt_symkey_enc packet. */
    }
  
    cdk_stream_set_cache (out, 1);
    if (hd->opt.compat)
        write_marker_packet (out);
      
    enc = cdk_calloc (1, sizeof *enc);
    if (!enc) {
        rc = CDK_Out_Of_Core;
        goto fail;
    }
    enc->version = 4;
    enc->cipher_algo = hd->dek->algo;
    enc->s2k = hd->s2k;
    cdk_pkt_init (&pkt);
    pkt.pkttype = CDK_PKT_SYMKEY_ENC;
    pkt.pkt.symkey_enc = enc;
    rc = cdk_pkt_build (out, &pkt);
    cdk_free (enc);
    if( rc )
        goto fail;
    cdk_stream_set_cache (out, 0);

 start:
    if (hd->opt.armor)
        cdk_stream_set_armor_flag (out, 0);
    cdk_stream_set_cipher_flag (out, hd->dek, hd->opt.mdc);
    if( hd->opt.compress )
        cdk_stream_set_compress_flag( out, hd->compress.algo,
                                      hd->compress.level );
    cdk_stream_set_literal_flag (out, 0, _cdk_stream_get_fname (inp));
    if (hd->opt.rfc1991)
        plaintext_set_rfc1991 (out);

    rc = cdk_stream_kick_off (inp, out);

 fail:
    _cdk_passphrase_free (pw);
    return rc;
}


static int
use_rfc1991_format( CDK_HD hd, CDK_KEYLIST kl )
{
    CDK_KEYLIST l;
  
    if( hd->opt.rfc1991 )
        return 1;
    for( l = kl; l; l = l->next ) {
        if( l->type == CDK_PKT_PUBLIC_KEY && l->key.pk->version == 3 )
            return 1;
    }
    return 0;
}


static int
select_cipher_algo( int rfc1991, CDK_KEYLIST kl )
{
    int pgp2 = _cdk_is_idea_available( );
    int def_cipher = pgp2 && rfc1991? GCRY_CIPHER_IDEA : GCRY_CIPHER_CAST5;
    return rfc1991? def_cipher : cdk_pklist_select_algo( kl );
} 


int
_cdk_check_args( int overwrite, const char * in, const char * out )
{
    if( !in || !out )
        return CDK_Inv_Value;
    if( !_cdk_strcmp( in, out ) )
        return CDK_Inv_Mode;
    if( !overwrite && !_cdk_check_file( out ) )
        return CDK_Inv_Mode;
    return 0;
}


/**
 * cdk_stream_encrypt: Encrypt a stream.
 * @hd: Handle
 * @remusr: List of recipients
 * @inp: Input stream handle
 * @out: Output stream handle
 *
 * If remusr is NULL, then symmetric encryption is used. Via the
 * handle the caller can set or unset multiple options.
 */
int
cdk_stream_encrypt( CDK_HD hd, CDK_STRLIST remusr,
                    CDK_STREAM inp, CDK_STREAM out )
{
    CDK_KEYLIST pkl = NULL;
    int cipher_algo, compress_algo = 0;
    int use_rfc1991 = 0;
    int rc = 0;

    if (!hd || !inp || !out)
        return CDK_Inv_Value;

    if (!remusr)
        return sym_stream_encrypt (hd, inp, out);

    rc = cdk_pklist_build (hd->db.pub, remusr, &pkl, GCRY_PK_USAGE_ENCR);
    if (rc)
        return rc;

    use_rfc1991 = use_rfc1991_format (hd, pkl);
    cipher_algo = select_cipher_algo (use_rfc1991, pkl);    
    cdk_dek_free( hd->dek );
    rc = cdk_dek_new( &hd->dek );
    if( !rc )
        rc = cdk_dek_set_cipher( hd->dek, cipher_algo );
    if( !rc )
        rc = cdk_dek_set_key( hd->dek, NULL, 0 );
    if( rc ) {
        cdk_pklist_release (pkl);
        return rc;
    }
    compress_algo = use_rfc1991? CDK_COMPRESS_ZIP : hd->compress.algo;

    if (!hd->opt.rfc1991 && !hd->opt.compat)
        cdk_dek_set_mdc_flag( hd->dek, cdk_pklist_use_mdc (pkl) );
    hd->dek->rfc1991 = use_rfc1991;

    cdk_stream_set_cache (out, 1);
    if (hd->opt.compat && !hd->opt.rfc1991)
        write_marker_packet (out);
  
    rc = cdk_pklist_encrypt( pkl, hd->dek, out );
    cdk_pklist_release( pkl );
    if( rc )
        return rc;

    cdk_stream_set_cache (out, 0);

    if (hd->opt.armor)
        cdk_stream_set_armor_flag (out, 0);
    cdk_stream_set_cipher_flag (out, hd->dek, 0);
    if (hd->opt.compress)
        cdk_stream_set_compress_flag( out, compress_algo, hd->compress.level );
    cdk_stream_set_literal_flag (out, 0, _cdk_stream_get_fname (inp));
    if (hd->dek->rfc1991)
        plaintext_set_rfc1991 (out);

    rc = cdk_stream_kick_off (inp, out);
    return rc;
}


/**
 * cdk_file_encrypt: Encrypt a file.
 * @hd: Handle
 * @remusr: List of recipient
 * @file: Input file
 * @output: Output file
 *
 **/
int
cdk_file_encrypt (CDK_HD hd, CDK_STRLIST remusr,
                  const char * file, const char * output)
{
    CDK_STREAM inp = NULL, out = NULL;
    int rc;

    rc = _cdk_check_args( hd->opt.overwrite, file, output );
    if( !rc )
        rc = cdk_stream_open( file, &inp );
    if( !rc )
        rc = cdk_stream_new( output, &out );
    if( !rc )
        rc = cdk_stream_encrypt( hd, remusr, inp, out );
    cdk_stream_close( inp );
    cdk_stream_close( out );
    return rc;
}


static void
set_reset (CTX c, CDK_STREAM inp)
{
    if (!c->reset) {
        cdk_stream_control (inp, CDK_STREAMCTL_RESET, 1);
        c->reset = 1;
    }
}


static void
write_status (CDK_HD hd, int type, const char * fmt, ...)
{
    va_list arg_ptr;
    char * buf;
    int n;
  
    if (!hd->callback)
        return;

    va_start (arg_ptr, fmt);
    n = vasprintf (&buf, fmt, arg_ptr);
    buf[n] = '\0';
    hd->callback (hd->callback_value, type, buf);
    free (buf);
    va_end (arg_ptr);
}


static int
is_openpgp_ext (const char * file)
{
    int yes = 0;

    if (strstr (file, ".asc") || strstr (file, ".sig")
        || strstr (file, ".gpg") || strstr (file, ".pgp"))
        yes = 1;

    return yes;
}
  

static int
hash_data_file( char * file, int digest_algo, GCRY_MD_HD * r_md )
{
    md_filter_s * mfx;
    CDK_STREAM s;
    int rc;

    if( file && is_openpgp_ext( file ) )
        file[strlen( file ) - 4] = '\0';
    else
        return CDK_General_Error;

    rc = cdk_stream_open( file, &s );
    if( rc )
        return rc;
  
    cdk_stream_set_hash_flag( s, digest_algo );
    cdk_stream_read( s, NULL, 0 );
    mfx = _cdk_stream_get_opaque( s, fHASH );
    if( mfx && mfx->md )
        *r_md = gcry_md_copy( mfx->md );
    cdk_stream_close( s );
    return 0;
}


static int
handle_symkey_enc (CTX c, CDK_PACKET * pkt)
{
    char * pw;
    cdkPKT_symkey_enc * key;
    int rc = 0;

    assert (pkt->pkttype == CDK_PKT_SYMKEY_ENC);

    c->key_seen = 1;
    if (c->dek)
        return 0; /* we already decrypted the session key */
  
    pw = _cdk_passphrase_get ("Enter Passphrase: ");
    if (!pw)
        return CDK_Out_Of_Core;

    key = pkt->pkt.symkey_enc;
    rc = cdk_dek_from_passphrase( &c->dek, key->cipher_algo, key->s2k, 0, pw );
    _cdk_passphrase_free (pw);
    return rc;
}


static int
handle_pubkey_enc (CTX c, CDK_HD hd, CDK_PACKET * pkt)
{
    cdkPKT_pubkey_enc * enc;
    cdkPKT_secret_key * sk = NULL;
    int rc = 0;

    assert( pkt->pkttype == CDK_PKT_PUBKEY_ENC );

    c->key_seen = 1;
    enc = pkt->pkt.pubkey_enc;
    write_status( hd, CDK_CB_PUBKEY_ENC, "%08lX%08lX %d %d",
                  enc->keyid[0], enc->keyid[1], enc->pubkey_algo,
                  (enc->mpi[0]->bits+7)/8*8 );
  
    if( c->dek )
        return 0; /* we already decrypted the session key */

    /* we checked before that there is at least one secret key so we
       skip this packet and continue without errors */
    if( cdk_keydb_check_sk( hd->db.sec, enc->keyid ) )
        return 0;
    rc = cdk_keydb_get_sk( hd->db.sec, enc->keyid, &sk );
    if( !rc )
        rc = cdk_dek_extract( &c->dek, enc, sk );
    _cdk_free_seckey( sk );
    return rc;
}


static int
rfc1991_get_sesskey (CDK_DEK * r_dek)
{
    CDK_S2K_HD s2k;
    char * pw;
    int rc;

    if (!r_dek)
        return CDK_Inv_Value;
  
    pw = _cdk_passphrase_get ("Enter Passphrase: ");
    if (!pw)
        return CDK_Out_Of_Core;
    rc = cdk_s2k_new( &s2k, 0, GCRY_MD_MD5, NULL );
    if( rc ) {
        _cdk_passphrase_free (pw);
        return CDK_Out_Of_Core;
    }
    rc = cdk_dek_from_passphrase( r_dek, GCRY_CIPHER_IDEA, s2k, 0, pw );
    _cdk_passphrase_free( pw );
    cdk_free( s2k );
    return rc;
}

  
static int
handle_encrypted (CTX c, CDK_PACKET * pkt, int use_mdc)
{
    cdkPKT_encrypted * enc;
    int pgp2_compat = _cdk_is_idea_available ();
    int rc = 0, pkttype = pkt->pkttype;

    assert (CDK_PKT_IS_ENCRYPTED (pkttype));

    if (!c->dek) {
        if (!pgp2_compat)
            return CDK_Error_No_Key;
        else if (!c->key_seen) {
            _cdk_log_debug ("RFC1991 message was detected.\n");
            rc = rfc1991_get_sesskey (&c->dek);
            if (rc)
                return rc;
        }
        else
            return CDK_Error_No_Key;
    }
  
    enc = pkt->pkt.encrypted;
    set_reset (c, enc->buf);
    cdk_stream_set_cipher_flag (enc->buf, c->dek, use_mdc);
    rc = cdk_stream_read (enc->buf, NULL, 0);
    if (!rc)
        c->s = enc->buf;
    else
        rc = _cdk_stream_get_errno (enc->buf);
    return rc;
}


static int
handle_compressed( CTX c, CDK_PACKET * pkt )
{
    cdkPKT_compressed * zip;
    int rc;

    assert( pkt->pkttype == CDK_PKT_COMPRESSED );
  
    zip = pkt->pkt.compressed;
    cdk_stream_set_compress_flag( c->s, zip->algorithm, 0 );
    rc = cdk_stream_read( c->s, NULL, 0 );
    if( rc )
        rc = _cdk_stream_get_errno( c->s );
    return rc;
}


static int
handle_onepass_sig( CTX c, CDK_PACKET * pkt )
{
    int rc = 0;
    
    assert (pkt->pkttype == CDK_PKT_ONEPASS_SIG);

    if( c->sig.md )
        return 0; /* already open */       
    c->sig.digest_algo = pkt->pkt.onepass_sig->digest_algo;
    if( _cdk_md_test_algo( c->sig.digest_algo ) )
        return CDK_Inv_Algo;
    c->sig.md = gcry_md_open( c->sig.digest_algo, 0 );
    if( !c->sig.md )
        rc = CDK_Gcry_Error;
    return rc;
}


static int
handle_plaintext( CTX c, CDK_PACKET * pkt, CDK_STREAM * ret_out )
{
    plaintext_filter_s * pfx;
    cdkPKT_plaintext * pt;
    CDK_STREAM out;
    const char * s;
    int rc = 0;

    assert( pkt->pkttype == CDK_PKT_PLAINTEXT );

    if( !ret_out )
        return CDK_Inv_Value;

    s = c->output? c->output : pkt->pkt.plaintext->name;
    rc = cdk_stream_create( s, ret_out );
    if( rc )
        return rc;
    out = *ret_out;
    pt = pkt->pkt.plaintext;
    cdk_stream_seek( c->s, c->sig.present? c->sig.pt_offset : 0 );
    cdk_stream_set_literal_flag( c->s, 0, NULL );
    if( c->sig.present ) {
        pfx = _cdk_stream_get_opaque( c->s, fPLAINTEXT );
        if( pfx )
            pfx->md = c->sig.md;
    }
    return cdk_stream_kick_off( c->s, out );
}


static byte *
mpi_encode( cdkPKT_signature * sig )
{
    CDK_MPI a;
    byte * p;
    size_t len, i, nsig = 0, pos = 0;
    
    nsig = cdk_pk_get_nsig( sig->pubkey_algo );
    for( i = 0, len = 0; i < nsig; i++ )
        len += sig->mpi[i]->bytes + 2;
    p = cdk_calloc( 1, len + 1 );
    for( i = 0; i < nsig; i++ ) {
        a = sig->mpi[i];
        p[pos++] = a->bits >> 8;
        p[pos++] = a->bits;
        memcpy( p + pos, a->data, a->bytes );
        pos += a->bytes;
    }
    return p;
}


static void
set_verify_result( cdkPKT_signature * sig, Verify_Result res )
{
    res->sig_len = sig->mpi[0]->bits;
    res->sig_ver = sig->version;
    res->keyid[0] = sig->keyid[0];
    res->keyid[1] = sig->keyid[1];
    res->created = sig->timestamp;
    res->pubkey_algo = sig->pubkey_algo;
    res->digest_algo = sig->digest_algo;
    if( sig->flags.expired )
        res->sig_flags |= CDK_FLAG_SIG_EXPIRED;
    res->sig_data = mpi_encode( sig );
}

    
static int
handle_signature (CDK_HD hd, CTX c, CDK_PACKET * pkt)
{
    Verify_Result res;
    cdkPKT_signature * sig;
    u32 keyid[2];
    int rc;

    assert (pkt->pkttype == CDK_PKT_SIGNATURE);

    if( !c->sig.present )
        return CDK_Inv_Packet;

    _cdk_result_verify_free (hd->result.verify);
    res = hd->result.verify = _cdk_result_verify_new ();
    if (!hd->result.verify)
        return CDK_Out_Of_Core;
  
    sig = pkt->pkt.signature;
    if( !c->sig.one_pass && !c->sig.md ) {
        if( _cdk_md_test_algo( sig->digest_algo ) )
            return CDK_Inv_Algo;
        rc = hash_data_file( c->file, sig->digest_algo, &c->sig.md );
        if( rc )
            return rc;
    }
  
    cdk_sig_get_keyid( sig, keyid );
    set_verify_result( sig, res );
  
    rc = cdk_keydb_get_pk( hd->db.pub, keyid, &c->sig.pk );
    if( rc ) {
        res->sig_status = CDK_SIGSTAT_NOKEY;
        return rc;
    }
    if( c->sig.pk->is_revoked )
        res->sig_flags |= CDK_FLAG_KEY_REVOKED;
    if( c->sig.pk->has_expired )
        res->sig_flags |= CDK_FLAG_KEY_EXPIRED;

    rc = _cdk_sig_check( c->sig.pk, sig, c->sig.md, &c->sig.is_expired );
    res->sig_status = !rc? CDK_SIGSTAT_GOOD : CDK_SIGSTAT_BAD;
    if( !rc )
        _cdk_log_debug("Good Signature from %08lX%08lX (expired %d)\n",
                       keyid[0], keyid[1], c->sig.is_expired );
    return rc;
}


static void
free_mainproc( CTX c )
{
    if( !c )
        return;
    cdk_kbnode_release( c->node );
    c->node = NULL;
    if( c->sig.present ) {
        gcry_md_close( c->sig.md );
        c->sig.md = NULL;
        _cdk_free_pubkey( c->sig.pk );
        c->sig.pk = NULL;
    }
    cdk_free (c->file);
    c->file = NULL;
    cdk_free (c->dek);
    c->dek = NULL;
    cdk_free (c);
}


static int
do_proc_packets( CDK_HD hd, CTX c, CDK_STREAM inp, CDK_STREAM *ret_out )
{
    CDK_PACKET * pkt = NULL;
    CDK_KBNODE n = NULL, node;
    const char * s;
    int rc = 0, npos, with_mdc = 0;

    if( !hd || !c )
        return CDK_Inv_Value;

    s = _cdk_stream_get_fname (inp);
    c->file = cdk_strdup (s? s : " ");
    if (!c->file) {
        cdk_free (c);
        return CDK_Out_Of_Core;
    }
  
    while (!cdk_stream_eof (inp)) {
        pkt = cdk_calloc (1, sizeof *pkt);
        if (!pkt)
            return CDK_Out_Of_Core;
        rc = cdk_pkt_parse (inp, pkt);
        _cdk_log_debug ("type=%d old_ctb=%d len=%d (%d)\n",
                        pkt->pkttype, pkt->old_ctb, pkt->pktlen, rc);
        if (rc == CDK_EOF)
            c->eof_seen = 1;
        if (rc)
            break;
      
        n = cdk_kbnode_new (pkt);
        if (!c->node)
            c->node = n;
        else
            _cdk_kbnode_add (c->node, n);

        switch (pkt->pkttype) {
        case CDK_PKT_SYMKEY_ENC:
            rc = handle_symkey_enc (c, pkt);
            _cdk_log_debug (" handle_symkey_enc (%d)\n", rc);
            break;
          
        case CDK_PKT_PUBKEY_ENC:
            rc = handle_pubkey_enc (c, hd, pkt);
            _cdk_log_debug (" handle_pubkey_enc (%d)\n", rc); 
            break;
          
        case CDK_PKT_ENCRYPTED_MDC: 
        case CDK_PKT_ENCRYPTED:
            with_mdc = pkt->pkttype == CDK_PKT_ENCRYPTED_MDC;
            rc = handle_encrypted (c, pkt, with_mdc);
            _cdk_log_debug (" handle_encrypted (%d)\n", rc);
            if (!rc)
                inp = c->s;
            break;
          
        case CDK_PKT_COMPRESSED:
            if (!c->s)
                c->s = inp;
            set_reset (c, c->s);
            rc = handle_compressed (c, pkt);
            _cdk_log_debug (" handle_compressed (%d)\n", rc);
            break;

        case CDK_PKT_ONEPASS_SIG:
            if (!c->s)
                c->s = inp;
            _cdk_log_debug (" handle_onepass_sig (0)\n");
            c->sig.present = 1;
            c->sig.one_pass = 1;
            c->sig.pt_offset = cdk_stream_tell (c->s);
            break;

        case CDK_PKT_PLAINTEXT:
            /* skip rest of the packet */
            if (!c->s)
                c->s = inp;
            npos = cdk_stream_tell (c->s) + pkt->pkt.plaintext->len;
            cdk_stream_seek (c->s, npos);
            set_reset (c, c->s);
            break;
          
        case CDK_PKT_SIGNATURE:
            if (!c->sig.present)
                c->sig.present = 1;
            break; /* handle it later */

        case CDK_PKT_MDC:
            _cdk_log_debug( "MDC packet detected.\n" );
            break;

        case CDK_PKT_MARKER:
            _cdk_log_debug( "marker packet detected.\n" );
            break;

        default:
            rc = CDK_Inv_Packet;
            break; 
        }
        if (rc)
            break;
    }
    if (c->eof_seen == 1)
        rc = 0;
    for (node = c->node; !rc && node; node = node->next) {
        pkt = node->pkt;
        switch (pkt->pkttype) {
        case CDK_PKT_ONEPASS_SIG:
            rc = handle_onepass_sig (c, pkt);
            _cdk_log_debug (" _handle_onepass_sig (%d)\n", rc);
            break;
          
        case CDK_PKT_PLAINTEXT:
            rc = handle_plaintext (c, pkt, ret_out);
            _cdk_log_debug (" _handle_plaintext (%d)\n", rc);
            break;

        case CDK_PKT_SIGNATURE:
            rc = handle_signature (hd, c, pkt);
            _cdk_log_debug (" _handle_signature (%d)\n", rc);
            break;

        default:
            _cdk_log_debug ("skip packet type %d\n", pkt->pkttype);
            break;
        }
        if (rc)
            break;
    }
    if (rc == CDK_EOF)
        rc = CDK_Wrong_Seckey;
    return rc;
}


int
_cdk_proc_packets( CDK_HD hd, CDK_STREAM inp, const char * output,
                   GCRY_MD_HD md )
{
    CDK_STREAM out = NULL;
    CTX c;
    int rc;

    if( !inp )
        return CDK_Inv_Value;
  
    c = cdk_calloc( 1, sizeof *c );
    if( !c )
        return CDK_Out_Of_Core;
    if( output )
        c->output = output;
    if( md )
        c->sig.md = md;
    rc = do_proc_packets( hd, c, inp, &out );
    cdk_stream_close( out );
    free_mainproc( c );
    return rc;
}


static int
check_pubkey_enc_list( CDK_STREAM in, CDK_KEYDB_HD hd )
{
    CDK_PACKET pkt;
    int n = 0, nenc = 0;
    
    if( !in || !hd )
        return CDK_Inv_Value;

    if( cdk_armor_filter_use( in ) ) {
        cdk_stream_set_armor_flag( in, 0 );
        cdk_stream_control( in, CDK_STREAMCTL_RESET, 1 );
    }
    cdk_pkt_init( &pkt );
    while( !cdk_pkt_parse( in, &pkt ) ) {
        if( pkt.pkttype != CDK_PKT_PUBKEY_ENC ) {
            if( CDK_PKT_IS_ENCRYPTED( pkt.pkttype ) )
                cdk_free( pkt.pkt.encrypted );
            else
                cdk_pkt_free( &pkt );
            break;
        }
        nenc++;
        if( !cdk_keydb_check_sk( hd, pkt.pkt.pubkey_enc->keyid ) )
            n++;
        cdk_pkt_free( &pkt );
        cdk_pkt_init( &pkt );
    }
    cdk_stream_seek( in, 0 );
    if( !nenc )
        return 0;
    _cdk_log_debug( "found %d secret keys\n", n );
    return n? 0 : CDK_Error_No_Key;
}
    

/**
 * cdk_file_decrypt - Decrypt a file.
 * @hd: Handle.
 * @file: Name of the file to decrypt.
 * @output: Output filename.
 *
 * When the operation was successfull, hd can contain information about
 * the signature (when present) and more.
 **/
int
cdk_file_decrypt( CDK_HD hd, const char * file, const char * output )
{
    CDK_STREAM inp = NULL;
    int rc = 0;

    if( !file )
        return CDK_Inv_Value;
    
    if( file && output )
        rc = _cdk_check_args( hd->opt.overwrite, file, output );
    if( !rc )
        rc = cdk_stream_open( file, &inp );
    if( !rc )
        rc = check_pubkey_enc_list( inp, hd->db.sec );
    if( !rc )
        rc = _cdk_proc_packets( hd, inp, output, NULL );

    cdk_stream_close( inp );
    return rc;
}

