/**
   RSASignature.java

   Copyright (C) 1999, Claymore Systems, Inc.
   All Rights Reserved.

   ekr@rtfm.com  Sun Jun 20 19:27:27 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   $Id: RSASignature.java,v 1.1.1.1 2000/05/09 15:03:08 ekr Exp $

*/

package COM.claymoresystems.provider;

import COM.claymoresystems.crypto.PKCS1Pad;
import java.security.Signature;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.interfaces.CryptixRSAPrivateKey;
import java.security.interfaces.CryptixRSAPublicKey;
import java.security.interfaces.RSAFactors;
import java.security.InvalidKeyException;
import java.security.SignatureException;
import java.security.InvalidParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.KeyException;
import cryptix.provider.rsa.RSAAlgorithm;

import java.math.BigInteger;
/** This class implements RSA signature w/ PKCS#1 padding but
    no DigestInfo encoding because Cryptix doesn't
    and SSL requires it for Client Auth.
    <P>
    We assume that we already have an RSAEncryption primitive floating
    around.
*/
public class RSASignature extends Signature
{
     private byte[] data=null;
     private BigInteger n;
     private BigInteger exp;
     private BigInteger p;
     private BigInteger q;
     private BigInteger u;
     
     public RSASignature() {
       super("RawRSAPKCS#1");
     }

     /** <b>SPI: </b> Initializes this object for verification
	 using the given key

	 @param key the public key
	 @exception InvalidKeyExcaption if the key class doesn't
	 implement CryptixRSAPublicKey
     */
     protected void engineInitVerify(PublicKey key)
       throws InvalidKeyException  {
       if(!(key instanceof CryptixRSAPublicKey))
	 throw new InvalidKeyException(getAlgorithm() +
	   ": Not a RSA Public Key");

       CryptixRSAPublicKey rsa=(CryptixRSAPublicKey)key;
       n=rsa.getModulus();
       exp=rsa.getExponent();
     }

     /** <b>SPI: </b> Initializes this object for signing
	 using the private key

	 @param key the private key
	 @exception InvalidKeyExcaption if the key class doesn't
	 implement than CryptixRSAPrivateKey
     */
     protected void engineInitSign(PrivateKey key)
       throws InvalidKeyException {
       if(!(key instanceof CryptixRSAPrivateKey))
	 throw new InvalidKeyException(getAlgorithm() +
	   ": Not a RSA Private Key");

       CryptixRSAPrivateKey rsa=(CryptixRSAPrivateKey)key;
       n=rsa.getModulus();
       exp=rsa.getExponent();

       if(key instanceof RSAFactors){
	 RSAFactors factors=(RSAFactors)key;
	 p = factors.getP();
	 q = factors.getQ();
	 u = factors.getInverseOfQModP();
       }
     } 

     /** <b>SPI: </b> Single byte updates are forbidden

	 @exception SignatureException
     */
     protected void engineUpdate(byte b)
       throws SignatureException {
       throw new SignatureException(getAlgorithm() +
	 ": Must be called with a complete input");
     }
     
     /** <b>SPI: </b> Updates the data. This can
	 only be called once with a single data value
	 or we throw an error

	 @param b the buffer containing the data
	 @param off the offset into the buffer
	 @param len the length 

	 @exception SignatureException either if the algorithm
	 was improperly initialized or you try to call update
	 more than once
     */
     protected void engineUpdate(byte[] b,int off,int len)
       throws SignatureException{
       if(state != VERIFY && state != SIGN)
	 throw new SignatureException(getAlgorithm() + ": Not initialized");
       if(data!=null)
	 throw new SignatureException(getAlgorithm() +
	   ": Raw RSA may only be updated once");
       data=new byte[len];
       System.arraycopy(b,off,data,0,len);
     }

     /** <b> SPI: </b> Sign the input
	 
	 @return a signature as a byte string
     */
     protected byte[] engineSign()
       throws SignatureException {
       if(state != SIGN)
	 throw new SignatureException(getAlgorithm() + ": Not initialized");
       if(data == null)
	 throw new SignatureException(getAlgorithm() + ": Must supply input");

       byte[] toBeSigned=PKCS1Pad.pkcs1PadBuf(data,n,2);
//       COM.claymoresystems.util.Util.xdump("tobeSigned",toBeSigned);
       BigInteger padded=new BigInteger(1,toBeSigned);
       BigInteger sval=RSAAlgorithm.rsa(padded,n,exp,p,q,u);
       byte[] signed=toBytes(sval);
//       COM.claymoresystems.util.Util.xdump("signed",signed);

       return signed;
     }

     /** <b> SPI: </b> Verify the input
	 
	 @return true or false
     */
     protected boolean engineVerify(byte[] signature)
       throws SignatureException {
       if(state != VERIFY)
	 throw new SignatureException(getAlgorithm() + ": Not initialized");
       if(data == null)
	 throw new SignatureException(getAlgorithm() + ": Must supply input");

       BigInteger sval=new BigInteger(1,signature);
       byte[] buf=RSAAlgorithm.rsa(sval,n,exp,p,q,u).toByteArray();
       byte[] data_prime=PKCS1Pad.pkcs1UnpadBuf(buf);

       return(cryptix.util.core.ArrayUtil.areEqual(data_prime,data));
     }

     /**
	<b> SPI: </b> There aren't any parameters
     */
     protected void engineSetParameter(String param, Object value)
       throws InvalidParameterException {
       throw new InvalidParameterException(getAlgorithm() +
	 ": No settable parameters");
     }

     /**
	<b> SPI: </b> There aren't any parameters
     */
     protected Object engineGetParameter(String param)
       throws InvalidParameterException {
       throw new InvalidParameterException(getAlgorithm() +
	 ": No settable parameters");
     }

     
     protected byte[] toBytes(BigInteger num){
       byte[] tmp=num.toByteArray();

       if(tmp[0]==0){
	 byte[] trim=new byte[tmp.length-1];

	 System.arraycopy(tmp,1,trim,0,tmp.length-1);
	 return trim;
       }
       else
	 return tmp;
     }
}
