#include "general.h"

int initialise(int argc, char **argv) {
   struct sockaddr_in servaddr;
   char *file = NULL, *label = NULL;
   char *keyfile = NULL;
   char *ptr;
   int listenport, listenhost;
   int o, sd, opt = 1;
   long flags;

   opterr = 0;
   while((o = getopt(argc, argv, "?hvduisf:l:k:")) != EOF) {
      switch(o) {
	case '?':
	case 'h':
	   usage(argv[0]);
	   break;
	case 'v':
	   version();
	   break;
	case 'd':
	   debug++;
	   break;
	case 'u':
	   probe_all = 1;
	   break;
	case 'i':
	   skip_verification = 1;
	   break;
	case 's':
	   skip_ok = 1;
	   break;
	case 'f':
	   if(file)
              err_quit("File already specified");
           if((file = malloc(strlen(optarg)+1)) == NULL)
              err_sys("Memory allocation error");
           strcpy(file, optarg);
           break;
	case 'l':
	   if(label)
	      err_quit("You can only specify one label");
	   if((label = malloc(strlen(optarg)+1)) == NULL)
              err_sys("Memory allocation error");
	   strcpy(label, optarg);
	   break;
	case 'k':
	   if(keyfile)
	      err_quit("Public Keyfile already specified");
           if((keyfile = malloc(strlen(optarg)+1)) == NULL)
              err_sys("Memory allocation error");
           strcpy(keyfile, optarg);
           break;
	default:
	   fprintf(stderr, "Warning: invalid option %c\n", o);
	   usage(argv[0]);
	   break;
      }
   }

   if((argc - optind) < 2) {
      fprintf(stderr, "Error: Too few arguments\n");
      usage(argv[0]);
   }

   if(LoadPublicKey(keyfile ? keyfile : PUBLIC_KEY_FILE, &ServerPublicKey) != 0)
      exit(-1);         /* LoadPrivateKey prints error message */

   if((sd = socket(AF_INET, SOCK_STREAM, 0)) == -1)
      err_sys("initialise: Can not SOCK_STREAM create socket");

   if(setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(int)) == -1)
      err_sys("initialise: Can not set SO_REUSEADDR");

   if((flags = fcntl(sd, F_GETFL, 0)) == -1)
      err_sys("initialise: fcntl F_GETFL error");

   if(fcntl(sd, F_SETFL, flags | O_NONBLOCK) == -1)
      err_sys("initialise: fcntl F_SETFL error");

   if((ptr = strchr(argv[optind+1], ':'))) {
      struct hostent *host;

      *ptr++ = '\0';
      listenport = atoi(ptr);

      if((host = gethostbyname(argv[optind+1])) == NULL) {
         fprintf(stderr, "Error resolving %s: %s\n", argv[optind+1], hstrerror(h_errno));
         exit(-1);
      }
      listenhost = *(int *)host->h_addr_list[0];
   } else {
      listenport = atoi(argv[optind+1]);
      listenhost = INADDR_ANY;
   }

   bzero(&servaddr, sizeof(servaddr));
   servaddr.sin_family = AF_INET;
   servaddr.sin_addr.s_addr = listenhost;
   servaddr.sin_port = htons(listenport);

   if(bind(sd, (struct sockaddr *)&servaddr, sizeof(servaddr)) == -1)
      err_sys("initialise: Can not bind SOCK_STREAM socket");

   if(listen(sd, SOMAXCONN) == -1)
      err_sys("initialise: Can not listen");

   if(!sndbufsz) {
      socklen_t optlen = sizeof(int);

      if(getsockopt(sd, SOL_SOCKET, SO_SNDBUF, &sndbufsz, &optlen))
         err_sys("initialise: Can not get SO_SNDBUF");
   }

   if(!file)
      err_quit("No file specified");

   parse_file(file, label);

   if((ptr = strchr(argv[optind], ':'))) {
      *ptr++ = '\0';

      add_proxy(argv[optind], atoi(ptr), 0);
   } else
      add_proxy("127.0.0.1", atoi(argv[optind]), 0);

   proxy_final();

   if(!debug) {
      if(fork()) {
         exit(0);
      } else {
         setsid();
	 openlog(argv[0], LOG_PID, LOG_USER);
         daemon_proc = 1;
      }
   }

   free(file);
   free(label);

   return sd;
}

void usage(char *name) {
   fprintf(stderr, "aesoptunnel: Client Tunnel for Aesop \n");
   fprintf(stderr, "usage: %s [options] [destinationhost:]port [listenhost:]port\n", name);
   fprintf(stderr, "where options are:\n");
   fprintf(stderr, "\t-h\t\t: Show this help\n");
   fprintf(stderr, "\t-d\t\t: Run the tunnel in debug mode\n");
   fprintf(stderr, "\t-f file\t\t: Specify file to load routes from\n");
   fprintf(stderr, "\t-l label\t: Specify the label for the chain, default: first label\n");
   fprintf(stderr, "\t-k file\t\t: Specify the public key file of the proxy\n");
   fprintf(stderr, "\t-i\t\t: Ignore signature from endpoint: insecure\n");
   fprintf(stderr, "\t-u\t\t: Do an udp probe on all the proxies in the route\n");
   fprintf(stderr, "\t-s\t\t: Skip proxies which didn't respond to udp probe\n");
   fprintf(stderr, "\t-v\t\t: Show version information.\n");
   exit(0);
}

void version(void) {
   fprintf(stderr, "\tThis is aesoptunnel v%s\n", VERSION);
   fprintf(stderr, "webpage: http://www.kryptology.org/aesop/\n");
   fprintf(stderr, "author:  bighawk, bighawk@kryptology.org\n");
   exit(0);
}
