#include "general.h"

int w_listen(char *host, const char *serv, int socktype) {
   int listenfd;
   const int on = 1;

#ifdef IPV6
   int n;
   struct addrinfo hints, *res, *ressave;

   bzero(&hints, sizeof(struct addrinfo));
   hints.ai_flags = AI_PASSIVE;
   hints.ai_family = AF_UNSPEC;
   hints.ai_socktype = socktype;

   if(host)
      ipv6host(host);	/* Strip eventual epifix	*/

   if((n = getaddrinfo(host, serv, &hints, &res)) != 0)
      err_quit("w_listen: Can not resolve: %s:%s: %s", host, serv, gai_strerror(n));

   ressave = res;

   do {
      listenfd = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
      if (listenfd < 0)
	 continue;

      if(setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
         err_sys("w_listen: Can not set SO_REUSEADDR");

      if(bind(listenfd, res->ai_addr, res->ai_addrlen) == 0)
	 break;

      close(listenfd);
   } while((res = res->ai_next) != NULL);

   if (res == NULL)
      err_sys("w_listen: Couldn't create listening socket for %s:%s", host, serv);

   freeaddrinfo(ressave);

#else

   struct sockaddr_in servaddr;
   struct hostent *hst;
   int listenhost;

   if(host) {
      ipv6host(host);	/* Strip eventual epifix	*/
      if((hst = gethostbyname(host)) == NULL) {
         fprintf(stderr, "w_listen: error resolving %s: %d\n", host, h_errno);
            exit(-1);
      }
      listenhost = *(int *)hst->h_addr_list[0];
   } else
      listenhost = INADDR_ANY;

   if((listenfd = socket(AF_INET, socktype, 0)) == -1)
      err_sys("initialise: Can not create %s socket", socktype == SOCK_STREAM ? "SOCK_STREAM" : "SOCK_DGRAM");

   if(setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) == -1)
      err_sys("w_listen: Can not set SO_REUSEADDR");

   bzero(&servaddr, sizeof(servaddr));
   servaddr.sin_family = AF_INET;
   servaddr.sin_addr.s_addr = listenhost;
   servaddr.sin_port = htons(atoi(serv));

   if(bind(listenfd, (struct sockaddr *)&servaddr, sizeof(servaddr)) == -1)
      err_sys("w_listen: Can not bind %s socket", socktype == SOCK_STREAM ? "SOCK_STREAM" : "SOCK_DGRAM");

#endif

   if(socktype == SOCK_STREAM)
      if(listen(listenfd, SOMAXCONN) == -1)
         err_sys("w_listen: Can not listen");

   return(listenfd);
}

char *w_sock_ntop(const struct sockaddr *sa) {
   char portstr[7];
   static char str[40];

   switch(sa->sa_family) {
      case AF_INET: {
         struct sockaddr_in *sin = (struct sockaddr_in *)sa;

         strcpy(str, inet_ntoa(sin->sin_addr));
         snprintf(portstr, sizeof(portstr), "#%d", ntohs(sin->sin_port));
         strcat(str, portstr);
         return(str);
      }

#ifdef IPV6
      case AF_INET6: {
         struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) sa;

         inet_ntop(AF_INET6, &sin6->sin6_addr, str, sizeof(str));
         snprintf(portstr, sizeof(portstr), "#%d", ntohs(sin6->sin6_port));
         strcat(str, portstr);
         return(str);
      }
#endif
   }
   return NULL;	/* Never reached */
}

struct sockaddr *w_setsockaddr(void *destination, short port, socklen_t *destlen, int inet6) {
   struct sockaddr *dest;   

#ifdef IPV6
   *destlen = inet6 ? sizeof(struct sockaddr_in6) : sizeof(struct sockaddr_in);
#else
   *destlen = sizeof(struct sockaddr_in);
#endif

   if((dest = calloc(1, *destlen)) == NULL)
      err_quit("w_connect: Memory allocation error");

   if(inet6 == 0) {
      ((struct sockaddr_in *)dest)->sin_family = AF_INET;
      ((struct sockaddr_in *)dest)->sin_port = port;
      memcpy(&((struct sockaddr_in *)dest)->sin_addr.s_addr, destination, 4);
   }
#ifdef IPV6
   else {
      ((struct sockaddr_in6 *)dest)->sin6_family = AF_INET6;
      ((struct sockaddr_in6 *)dest)->sin6_port = port;
      memcpy(&((struct sockaddr_in6 *)dest)->sin6_addr.s6_addr, destination, 16);
   }
#endif
   return dest;
}

int ipv6host(char *host) {
   char *ptr;

   if((ptr = strchr(host, '/')) == NULL)
      return 0;

   *ptr++ = '\0';

   if(*ptr++ != 'v')
      err_quit("Invalid host specifier: valid specifiers are /v4 and /v6");

   switch(*ptr) {
      case '4':
         return 0;
      case '6':
         return 1;
      default:
         err_quit("Invalid host specifier: valid specifiers are /v4 and /v6");
   }
   return 0;    /* Never reached        */
}
