/*
 * Modified by bighawk, based on:
 */
/* kex.c - Signed DH key exchange for HQ.
 * anakata <anakata@anakata.dhs.org>
 */

#include "general.h"

/* Hardcoded DH parameters - these are 512 bits. */
static unsigned char PRIME1[64] = {
  0xd0, 0x45, 0x1f, 0xfe, 0x2c, 0x64, 0xc4, 0xed, 0x6b, 0x0a, 0xe6,
  0x36, 0x5b, 0x7f, 0xef, 0x9c, 0x15, 0x42, 0x5e, 0x40, 0xa3, 0x7c,
  0xa5, 0xf8, 0x39, 0x86, 0x5e, 0x2c, 0xfb, 0x41, 0x69, 0xa0, 0xd8,
  0x25, 0xc9, 0x13, 0x0f, 0x88, 0x64, 0xff, 0xfc, 0xf3, 0xbf, 0xbe,
  0xb0, 0x27, 0x36, 0x60, 0x67, 0xaa, 0x27, 0xe2, 0x7b, 0xfc, 0xaf,
  0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};
static unsigned char GENERATOR1[64] = {
  0x0a, 0xcf, 0x95, 0x8c, 0x40, 0xd3, 0x01, 0xef, 0xc5, 0x15, 0x3e,
  0x7d, 0xcd, 0x5e, 0xf7, 0x5f, 0xec, 0x9e, 0x8f, 0xb0, 0xfa, 0xe6,
  0xa8, 0x0e, 0xe5, 0xc3, 0xb8, 0x4b, 0x9c, 0x0e, 0x51, 0x30, 0x51,
  0xb2, 0xb7, 0x54, 0x2e, 0x66, 0xb8, 0xd3, 0xa2, 0x5e, 0x93, 0x89,
  0x11, 0xad, 0x6b, 0xe5, 0xc2, 0x43, 0x95, 0x09, 0x9c, 0x6d, 0xda,
  0xa8, 0x6e, 0x18, 0x94, 0x2f, 0x29, 0x84, 0x27, 0x5a
};

static R_DH_PARAMS PARAMS1 = {
  PRIME1, sizeof (PRIME1), GENERATOR1, sizeof (GENERATOR1)
};

/*
 * InitDH(): Function to initialize the DH key exchange.
 * Arguments:
 *   randomStruct: Seeded RSAREF random structure.
 *   server: 1 if this is the server, 0 if this is the client.
 *   privateKey: If server==1, private key of the server. Else NULL.
 * Returns a filled-in KexData structure, or NULL on failure.
 */
struct DHData *InitDH(R_RANDOM_STRUCT *randomStruct, int server, R_RSA_PRIVATE_KEY *privateKey)
{
  struct DHData *new;
  
  /* Set up the DH agreement and generate new public and private values. */
  new = Malloc(sizeof(struct DHData));
  new->server = server;
  if (R_SetupDHAgreement(new->publicValue, new->privateValue, DH_VAL_LEN, &PARAMS1, randomStruct) != 0) {
    free(new);
    return NULL;
  }
  /* If this is the server, sign the public value. */
  if (server == 1) {
    MD5_CTX MD5Ctx;
    unsigned char digest[16];
    unsigned int siglen;
    
    MD5Init(&MD5Ctx);
    MD5Update(&MD5Ctx, new->publicValue, DH_VAL_LEN);
    MD5Final(digest, &MD5Ctx);
    if (RSAPrivateEncrypt(new->signature, &siglen, digest, 16, privateKey) != 0) {
      free(new);
      return NULL;
    }
  }
  return new;
} 

/*
 * ComputeDH(): Function to get the agreed-upon key and in case of client mode
 * verify the signature from the server.
 * Arguments:
 *   data: String of data from the other end of the exchange.
 *   kexData: KexData structure from InitKex().
 *   publicKey: If server==0, public key of the server. Else NULL.
 *   agreedKey: Where to store key. Must be large enough (i.e. DH_VAL_LEN). 
 * Returns:
 *    0 if agreement succeeded,
 *   -1 if agreement failed due to invalid signature,
 *   -2 if agreement failed for some other reason (f.ex. invalid data).
 */
int ComputeDH(unsigned char *data, struct DHData *kexData, R_RSA_PUBLIC_KEY *publicKey, unsigned char *agreedKey)
{
  unsigned char otherpubval[DH_VAL_LEN], othersignature[MAX_SIGNATURE_LEN];
  
  /* If this is the client, save the server's signature. */
  if (kexData->server == 0)
    memcpy(othersignature, data+DH_VAL_LEN, 128);

  memcpy(otherpubval, data, DH_VAL_LEN);
  /* If this is the client, check the server's signature. */
  if (kexData->server == 0) {
    unsigned char digest[16], otherdigest[MAX_SIGNATURE_LEN];
    unsigned int otherdigestlen;
    MD5_CTX MD5Ctx;
    
    /* Decrypt the digest using the server's public key. 
     * XXX - Will RSAPublicDecrypt() ever return more output than input? */
    if (RSAPublicDecrypt(otherdigest, &otherdigestlen, othersignature, 128, publicKey) != 0) return -2;
    if (otherdigestlen != 16) return -2;
    /* Create our own digest of the server's public value and compare it with the decrypted digest. */
    MD5Init(&MD5Ctx);
    MD5Update(&MD5Ctx, otherpubval, DH_VAL_LEN);
    MD5Final(digest, &MD5Ctx);
    if (memcmp(digest, otherdigest, 16) != 0) return -1;
  }
  /* Compute the actual agreed key. */
  if (R_ComputeDHAgreedKey(agreedKey, otherpubval, kexData->privateValue, DH_VAL_LEN, &PARAMS1) != 0) return -2;
  return 0;
}
