/*
   CIPE - encrypted IP over UDP tunneling

   ciped.c - the user mode driver

   Copyright 1996 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: ciped.c,v 1.16.6.1 1998/11/14 20:17:21 olaf Exp $ */

#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <ctype.h>
#include <mntent.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#ifdef __GLIBC__      /* Ugh! the ehthernet address function needs this */
#include <net/if_arp.h>
#include <netinet/if_ether.h>
#else
#include <linux/if_ether.h>
#include <linux/if_arp.h>
#endif
#include "linux/ipx.h"
#include "ciped.h"
#include "ioctl.h"

static const char daemon_version[]=VERSION;

void usage(const char *p)
{
    fprintf(stderr, "usage: %s [-o file ] [options] device=name\n", p);
    exit(1);
}

/* FIXME: places marked *#* have byteorder conversion missing */
/* This is corrected in version 3 protocol */

/*** The global options structure ***/

struct options {
    const char * const oname;
    const char otyp;
    union {
	int ovint;
	char *ovstr;
	struct sockaddr_in *ovsaddr;
	struct sockaddr *ovhwaddr;
    } ov;
} opts[] = {
#define Opt(x,t) { #x, T##t, {0} },
#define Tbool	0
#define Tint	1
#define Tstr	2
#define Taddr	3
#define Tuaddr	4
#define Ttaddr	5
#define Thwaddr 6
#define Tipxnet 7

#define Odevice		0  /* name of the CIPE device, e.g. cip3b0 */
Opt(device,str)
#define Odebug		1  /* Debug mode - don't detach, verbose messages */
Opt(debug,bool)
#define Oipaddr		2  /* This CIPE device's IP address */
Opt(ipaddr,addr)
#define Optpaddr	3  /* The peer CIPE device's IP address */
Opt(ptpaddr,addr)
#define Omtu		4  /* Device MTU */
Opt(mtu,int)
#define Ometric		5  /* Device metric */
Opt(metric,int)
#define Ome		6  /* My UDP address */
Opt(me,uaddr)
#define Opeer		7  /* The peer's UDP address,
			      i.e. where to send packets */
Opt(peer,uaddr)
#define Okey		8  /* The interface secret key
			      - set via options file only */
Opt(key,str)
#define Onokey		9  /* Don't encrypt */
Opt(nokey,bool)
#define Osocks		10  /* SOCKS5 server (ip:port) */
Opt(socks,taddr)
#define Otokxc		11  /* Timeout for key exchange (secs) */
Opt(tokxc,int)
#define Otokey		12  /* Dynamic key lifetime (secs) */
Opt(tokey,int)
#define Oipup		13  /* Name of the ip-up script */
Opt(ipup,str)
#define Oipdown		14  /* Name of the ip-down script */
Opt(ipdown,str)
#define Oarg		15  /* Extra argument to ip-up script */
Opt(arg,str)
#define Octtl		16  /* TTL value for carrier */
Opt(cttl,int)
#define Omaxerr		17  /* Maximum ECONNREFUSED tolerance */
Opt(maxerr,int)
#define Onetmask        18  /* The device netmask address */
Opt(netmask,addr)
#define Ohwaddr         19  /* The device hardware address */
Opt(hwaddr,hwaddr)
#define Obroadcast      20  /* The device broadcast address */
Opt(broadcast,addr)
#define Obrdcstmode     21  /* An address for a static arp  for peer */
Opt(brdcstmode,bool)       
#define Oarphwaddr      22  /* the hardware address for the far end */
Opt(arphwaddr,hwaddr)      
#define Oipxnet         23  /* the CIPE links' IPX network address */
Opt(ipxnet,ipxnet)

#undef Opt
{ NULL, 0, {0} }};

#define OI(x)      (opts[O##x].ov.ovint)
#define OS(x)      (opts[O##x].ov.ovstr)
#define OA(x)      (opts[O##x].ov.ovsaddr)
#define OAaddr(x)  (OA(x)->sin_addr)
#define OAport(x)  (OA(x)->sin_port)
#define OH(x)      (opts[O##x].ov.ovhwaddr)

/* Statistics */
struct kstat {
    int rreq, req, ind, indb, ack, ackb, bogus;
} ks = { 0, 0, 0, 0, 0, 0, 0 };


/*** Option parser ***/

/* Parse a hardware address  - taken from ifconfig */
int gethwaddr(char *c, struct sockaddr *sap)
{
    unsigned char *ptr;
    char ch, *orig;
    int i, val;

    sap->sa_family = ARPHRD_ETHER;
    ptr = sap->sa_data;

    i = 0;
    orig = c;
    while((*c != '\0') && (i < ETH_ALEN)) {
        val = 0;
        ch = *c++;
        if (isdigit(ch)) val = ch - '0';
	else if (ch >= 'a' && ch <= 'f') val = ch - 'a' + 10;
	else if (ch >= 'A' && ch <= 'F') val = ch - 'A' + 10;
	else {
	    fprintf(stderr, "gethwaddr: '%s' invalid hardware address\n",
		    orig);
	    return(-1);
        }
        val <<= 4;
        ch = *c++;
        if (isdigit(ch)) val |= ch - '0';
	else if (ch >= 'a' && ch <= 'f') val |= ch - 'a' + 10;
	else if (ch >= 'A' && ch <= 'F') val |= ch - 'A' + 10;
	else {
	    fprintf(stderr, "gethwaddr: '%s' invalid hardware address\n",
		    orig);
	    return(-1);
        }
        *ptr++ = (unsigned char) (val & 0377);
        i++;

        /* We might get a semicolon here - not required. */
        if (*c == ':') {
	    if (i == ETH_ALEN) {
		; /* nothing */
	    }
	    c++;
        }
    }

    return(0);
}

/* Convert an IPX address */
int getipxnet(char *c, int *ipxnet) 
{
    if (sscanf(c, "%x", ipxnet)==0 || *ipxnet == 0) {
	fprintf(stderr, "getipxnet: '%s' invalid IPX network address\n", c);
	return -1;
    }

    return 0;
}

/* Resolve a host:port address */
int getaddr(char *c, struct sockaddr_in *sa, const char *prot)
{
    struct hostent *h;
    struct servent *s;
    char *q=strchr(c, ':');

    sa->sin_family=AF_INET;
    sa->sin_port=0;
    sa->sin_addr.s_addr=INADDR_ANY;
    if (q) {
	if (!prot) {
	    fprintf(stderr, "getaddr: raw IP address has no port number\n");
	    return -1;
	}
	*q++='\0';
	if (!(sa->sin_port=htons(atoi(q)))) {
	    if ((s=getservbyname(q, prot)))
		sa->sin_port=s->s_port;
	    else {
		fprintf(stderr, "getaddr: port '%s' invalid/not found\n", q);
		return -1;
	    }
	}
    }
    if (!inet_aton(c, &sa->sin_addr)) {
	if ((h=gethostbyname(c)))
	    memcpy(&sa->sin_addr.s_addr, h->h_addr_list[0],
		   sizeof(sa->sin_addr.s_addr));
	else {
	    fprintf(stderr, "getaddr: host '%s' invalid/not found\n", c);
	    return -1;
	}
    }
    return 0;
}

/* Set an option */
int setopt(const char *n, char *v)
{
    struct options *o=opts;
    static const char * const protos[]={NULL, "udp", "tcp"};

    while (o->oname) {
	if (!strcmp(o->oname, n)) {
	    if ((!v) && (o->otyp!=Tbool)) {
		fprintf(stderr, "missing value for %s\n", n); return -1;
	    }
	    switch(o->otyp) {
	    case Tbool:
		o->ov.ovint=1; break;
	    case Tint:
		o->ov.ovint=atoi(v); break;
	    case Tstr:
		if (!(o->ov.ovstr=strdup(v))) {
		    fprintf(stderr, "setopt: out of memory\n"); return -1;
		}
		break;
	    case Taddr:
	    case Tuaddr:
	    case Ttaddr:
		if (!(o->ov.ovsaddr=malloc(sizeof(*o->ov.ovsaddr)))) {
		    fprintf(stderr, "setopt: out of memory\n"); return -1;
		}
		if (getaddr(v, o->ov.ovsaddr, protos[o->otyp-Taddr])<0) {
		    free(o->ov.ovsaddr);
		    o->ov.ovsaddr = NULL;
		    return -1;
		}
		break;
	    case Thwaddr:
		if (!(o->ov.ovhwaddr=malloc(sizeof(*o->ov.ovhwaddr)))) {
		    fprintf(stderr, "setopt: out of memory\n"); return -1;
		}
		if (gethwaddr(v, o->ov.ovhwaddr)<0) {
		    free(o->ov.ovhwaddr);
		    o->ov.ovhwaddr = NULL;
		    return -1;
		}
		break;
	    case Tipxnet:
		if(getipxnet(v, &o->ov.ovint) < 0) {
		    return -1;
		}
		break;
	    default:
		fprintf(stderr, "internal error: %d\n", __LINE__); return -1;
	    }
	    return 0;
	}
	++o;
    }
    fprintf(stderr, "unknown option: %s\n", n);
    return -1;
}

/* Set options from command line */
void setopt_cmdline(int argc, char *argv[])
{
    int i;
    char *q;
    for (i=0; i<argc; ++i) {
	if ((q=strchr(argv[i], '=')))
	    *q++='\0';
	if (!strcmp(argv[i], "key")) {
	    fprintf(stderr, "Refused to set key from command line!\n");
	    continue;
	}
	(void) setopt(argv[i], q);
    }
}

/* Set options from file, report file not found if (v) */
void setopt_file(const char *n, int v)
{
    int d;
    FILE *f;
    char b[128];
    char *p, *q;
    struct stat s;

    if ((d=open(n, O_RDONLY))<0) {
	if (v)
	    perror("setopt_file: open");
	return;
    }
    if (fstat(d, &s)<0) {
	perror("setopt_file: stat");
	return;
    }
    if (!S_ISREG(s.st_mode) || (s.st_mode&(S_IRWXG|S_IRWXO)) ||
	(s.st_uid!=0)) {
	fprintf(stderr, "file %s has incorrect permissions\n", n);
	return;
    }

    if (!(f=fdopen(d, "r"))) {
	if (v)
	    perror("setopt_file: fdopen");
	return;
    }
    while (fgets(b, sizeof(b), f)) {
	p=b;
	if ((q=strchr(p, '\n')))
	    *q='\0';
	while ((*p) && strchr(" \t", *p))
	    ++p;
	if ((!*p) || strchr("#;:!", *p))
	    continue;
	if ((q=strpbrk(p, " \t="))) {
	    while ((*q) && strchr(" \t=", *q))
		*q++='\0';
	    if (!*q)
		q=NULL;
	}
	(void) setopt(p, q);
    }
    fclose(f);
}

/* Print out all options */
void dumpopt(void)
{
    char *hwaddr;
    struct options *o;
    for (o=opts; o->oname; ++o) {
	printf("%s ", o->oname);
	switch(o->otyp) {
	case Tbool: printf("%s\n", o->ov.ovint?"yes":"no"); break;
	case Tint:  printf("%d\n", o->ov.ovint); break;
	case Tstr:  printf("%s\n", o->ov.ovstr?o->ov.ovstr:"(none)"); break;
	case Taddr:
	    if (o->ov.ovsaddr)
		printf("%s\n", inet_ntoa(o->ov.ovsaddr->sin_addr));
	    else
		printf("(none)\n");
	    break;
	case Tuaddr:
	case Ttaddr:
	    if (o->ov.ovsaddr)
		printf("%s:%d\n", inet_ntoa(o->ov.ovsaddr->sin_addr),
		       ntohs(o->ov.ovsaddr->sin_port));
	    else
		printf("(none)\n");
	    break;
	case Thwaddr:
	    if (o->ov.ovhwaddr) {
		hwaddr = o->ov.ovhwaddr->sa_data;
		printf("%02X:%02X:%02X:%02X:%02X:%02X\n",
		       hwaddr[0],
		       hwaddr[1],
		       hwaddr[2],
		       hwaddr[3],
		       hwaddr[4],
		       hwaddr[5]);
	    }
	    else
		printf("(none)\n");
	    break;
	case Tipxnet:
	    if (o->ov.ovint) {
		printf("%08X\n", o->ov.ovint);
	    }
	    else
		printf("(none)\n");
	    break;
	default: printf("internal error: %d\n", __LINE__); break;
	}
    }
}

/*** Logging ***/

void log(int lev, const char *s)
{
    if (OI(debug))
	fprintf(stderr, s);
    else
	syslog(lev, s);
}

void log2(int lev, const char *f, const char *s)
{
    if (OI(debug))
	fprintf(stderr, f, s);
    else
	syslog(lev, f, s);
}

/* perror replacement */
void logerr(int lev, const char *s)
{
    if (OI(debug))
	perror(s);
    else
	syslog(lev, "%s: %s", s, strerror(errno));
}

/* missing option */
void argmiss(const char *s)
{
    if (OI(debug))
	fprintf(stderr, "missing argument: %s\n", s);
    else
	syslog(LOG_ERR, "missing argument: %s", s);
}

/* print if debugging */
#define dprintf0(f)    if(OI(debug)){printf(f);}
#define dprintf1(f,a)  if(OI(debug)){printf(f,a);}


/*** ARP code ***/

/* set an ARP for the far end */
int setarp (void) 
{
    struct arpreq arpreq;
/*
 * Sometime in the 1.3 series kernels, the arp request added a device name.
 */
#include <linux/version.h>
#if LINUX_VERSION_CODE < 66381
    char arpreq_arp_dev[32];
#else
#define  arpreq_arp_dev arpreq.arp_dev
#endif
    int sockfd = 0;

#define SET_SA_FAMILY(addr, family)                     \
    memset ((char *) &(addr), '\0', sizeof(addr));      \
    addr.sa_family = (family);

    memset (&arpreq, '\0', sizeof(arpreq));
    
    if (!OA(ptpaddr)) {
	return 0;
    }

    if (!OH(arphwaddr)) {
	return 0;
    }
    
    if ((sockfd = socket(AF_INET,SOCK_DGRAM,0)) <0) {
	perror("setarp: socket");
	return -1;
    }

    memcpy(&arpreq.arp_ha, OH(arphwaddr), sizeof(*OH(arphwaddr)));
    strncpy(arpreq_arp_dev, OS(device), sizeof(arpreq_arp_dev)-1);
    SET_SA_FAMILY(arpreq.arp_pa, AF_INET);
    ((struct sockaddr_in *) &arpreq.arp_pa)->sin_addr = OAaddr(ptpaddr);
    arpreq.arp_flags = ATF_PERM;

    if (ioctl(sockfd, SIOCSARP, (caddr_t)&arpreq) < 0) {
	logerr(LOG_ERR, "setarp: SIOCSARP");
	close(sockfd);
	return -1;
    }

    close(sockfd);
    return 0;
}

/*** IPX stuff ***/

/* path_to_procfs - find the path to the proc file system mount point */

char procpath[256];
int ipxconfigured = 0;

int path_to_procfs (void) {
    struct mntent *mntent;
    FILE *fp;
  
    fp = fopen (MOUNTED, "r");
    if (fp != 0) {
	mntent = getmntent (fp);
	while (mntent != (struct mntent *) 0) {
	    if (strcmp (mntent->mnt_type, MNTTYPE_IGNORE) != 0) {
		if (strcmp (mntent->mnt_type, "proc") == 0) {
		    strncpy (procpath, mntent->mnt_dir,
			     sizeof (procpath)-10);
		    procpath [sizeof (procpath)-10] = '\0';
		    fclose (fp);
		    return 1;
		}
	    }
	    mntent = getmntent (fp);
	}
	fclose (fp);
    }
    return 0;
}

/* Set the IPX network number on the interface */
int setipxnet(void) {
    int    result = 1;

    struct stat stat_buf;
    int    skfd; 
    struct ifreq         ifr;
    struct sockaddr_ipx *sipx = (struct sockaddr_ipx *) &ifr.ifr_addr;

    if (!OI(ipxnet)) {
	/* Do nothing if no net number given */
	return 0;
    }

    if (path_to_procfs()) {
	strcat (procpath, "/net/ipx_interface");
	if (lstat (procpath, &stat_buf) < 0) {
	    logerr (LOG_ERR, "setipxnet: IPX support is not present in the kernel\n");
	    return -1;
	}
    }

    skfd = socket (AF_IPX, SOCK_DGRAM, 0);
    if (skfd < 0) { 
	if (errno != EIO) {
	    logerr(LOG_DEBUG, "setipxnet: socket(AF_IPX)");
	}
	result = -1;
    }
    else {
	memset (&ifr, '\0', sizeof (ifr));
	strcpy (ifr.ifr_name, OS(device));

	memcpy (sipx->sipx_node, OH(hwaddr)->sa_data, IPX_NODE_LEN);
	sipx->sipx_family  = AF_IPX;
	sipx->sipx_port    = 0;
	sipx->sipx_network = htonl (OI(ipxnet));
	sipx->sipx_type    = IPX_FRAME_ETHERII;
	sipx->sipx_action  = IPX_CRTITF;
	/*
	 *  Set the IPX device
	 */
	if (ioctl(skfd, SIOCSIFADDR, (caddr_t) &ifr) < 0) {
	    result = -1;
	    if (errno != EEXIST) {
		if (errno != EIO) {
		    logerr (LOG_DEBUG,
			    "setipxnet: ioctl(SIOCAIFADDR, CRTITF)");
		}
	    }
	    else {
		logerr (LOG_WARNING,
			"setipxnet: ioctl(SIOCAIFADDR, CRTITF): Address already exists");
	    }
	}
    
	ipxconfigured = 1;

	close (skfd);
    }
    return result;
}

/* Clear the IPX network information */
int clearipxnet (void) {
    int    result = 0;
    int    skfd; 
    struct ifreq         ifr;
    struct sockaddr_ipx *sipx = (struct sockaddr_ipx *) &ifr.ifr_addr;

    if( !ipxconfigured ) {
	/* Do not attmept this if IPX is not configured */
	return 0;
    }

    skfd = socket (AF_IPX, SOCK_DGRAM, 0);
    if (skfd < 0){ 
	if (errno != EIO) {
	    logerr(LOG_DEBUG, "clearipxnet: socket(AF_IPX)");
	}
	result = -1;
    }
    else {
	memset (&ifr, '\0', sizeof (ifr));
	strcpy (ifr.ifr_name, OS(device));

	sipx->sipx_type    = IPX_FRAME_ETHERII;
	sipx->sipx_action  = IPX_DLTITF;
	sipx->sipx_family  = AF_IPX;
/*
 *  Set the IPX device
 */
	if (ioctl(skfd, SIOCSIFADDR, (caddr_t) &ifr) < 0) {
	    if (errno != EIO) {
		logerr (LOG_INFO, "clearipxnet: ioctl(SIOCAIFADDR, IPX_DLTITF)");
	    }
	    result = -1;
	}
	close (skfd);
    }

    return result;
}



/*** Initialize, open and attach the CIPE device. ***/

int opendev(void)
{
    int f, i;
    struct siocsifcippar p;
    struct siocsifcipkey k;
    struct siocsifcipatt a;
    struct ifreq r,s,t;

#define amiss(s) { argmiss(s); goto error; }
#define err(s)   { logerr(LOG_ERR, s); goto error; }

    if (!OS(device)) {
	argmiss("device");
	return -1;
    }

    /* Initialize, bind and connect the socket */
    if ((f=socket(AF_INET, SOCK_DGRAM, 0))<0) {
	logerr(LOG_ERR, "opendev: socket");
	return -1;
    }
    i=65536; /* we need much to receive fast */
    if (setsockopt(f, SOL_SOCKET, SO_RCVBUF, &i, sizeof(i)))
	/* not fatal */
	logerr(LOG_NOTICE, "opendev: setsockopt");
    if (!OA(me)) {
	if (!(OA(me)=malloc(sizeof(*OA(me)))))
	    err("opendev: malloc");
	OA(me)->sin_family=AF_INET;
	OAaddr(me).s_addr=INADDR_ANY;
	OAport(me)=0;
    }
    if (bind(f, (struct sockaddr *)OA(me), sizeof(*OA(me)))<0)
	err("opendev: bind");
    if (OA(peer)) {
	if (connect(f, (struct sockaddr *)OA(peer), sizeof(*OA(peer)))<0)
	    err("opendev: bind");
    } else
	amiss("peer");
    i=sizeof(*OA(me));
    if (getsockname(f, (struct sockaddr *)OA(me), &i)<0)
	/* not fatal */
	logerr(LOG_NOTICE, "opendev: getsockname");

    /* Set the CIPE operation parameters */
    memset(&p, 0, sizeof(p));
    if (OA(socks)) {
	p.sockshost=OAaddr(socks).s_addr;
	p.socksport=OAport(socks);
    }
    p.tmo_keyxchg=OI(tokxc);
    p.tmo_keylife=OI(tokey);
    p.mayclear=OI(nokey);
    p.cttl=OI(cttl);
    if (ioctl_setpar(f, OS(device), &p)<0)
	err("opendev: setpar");

    /* Set the key */
    if (OS(key)) {
	unsigned char *c=OS(key);
	unsigned char *x=(unsigned char *)&k.thekey;
	i=0;
	k.which=KEY_STATIC;
	/* primitive hex decoder */
	while(*c) {
	    x[i]=((*c)<<4)&0xF0;
	    *c++='\0'; /* clear after use */
	    if (!*c) break;
	    x[i++]|=(*c)&0x0F;
	    *c++='\0';
	    if (i>=userKeySize) break;
	}
	if (ioctl_setkey(f, OS(device), &k)<0)
	    err("opendev: setkey");
	memset(&k, 0, sizeof(k));
    } else {
	if (!OI(nokey))
	    amiss("key");
    }

    /* Perform attach */
    a.fd=f;
    if (ioctl_attach(f, OS(device), &a)<0)
	err("opendev: attach");

    /* Perform ifconfig */
    memset((char *)&r, 0, sizeof(r));
    memset((char *)&s, 0, sizeof(s));
    memset((char *)&t, 0, sizeof(t));
    strcpy(r.ifr_name, OS(device));
    strcpy(s.ifr_name, OS(device));
    if (OH(hwaddr)) {
	memcpy(&r.ifr_hwaddr, OH(hwaddr), sizeof(*OH(hwaddr)));
	if (ioctl(f, SIOCSIFHWADDR, &r)<0)
	    err("opendev: SIOCSIFHWADDR");
      
    }
    if (OA(ipaddr)) {
	memcpy(&r.ifr_addr, OA(ipaddr), sizeof(*OA(ipaddr)));
	if (ioctl(f, SIOCSIFADDR, &r)<0)
	    err("opendev: SIOCSIFADDR");
    } else
	amiss("ipaddr");
    if (!OI(brdcstmode)) {
	/* point to point mode - MUST have a point to point address */
	if (!OA(ptpaddr)) {
	    amiss("ptpaddr");
	}
	s.ifr_flags |= IFF_POINTOPOINT;
	memcpy(&r.ifr_dstaddr, OA(ptpaddr), sizeof(*OA(ptpaddr)));
	if (ioctl(f, SIOCSIFDSTADDR, &r)<0)
	    err("opendev: SIOCSIFDSTADDR");
	t = s;
	if (ioctl(f, SIOCSIFFLAGS, &t)<0)
	    err("opendev: SIOCSIFFLAGS");
    } 
    else {
	/* Broadcast mode - set IFF flags */ 
	s.ifr_flags |= IFF_BROADCAST;
#ifdef CIPE_MULTICAST      
	s.ifr_flags |= IFF_MULTICAST;
#endif
	t = s;
	if (ioctl(f, SIOCSIFFLAGS, &t)<0)
	    err("opendev: SIOCSIFFLAGS");
    }
    if (OA(broadcast)) {
	s.ifr_flags |= IFF_BROADCAST;
	memcpy(&r.ifr_broadaddr, OA(broadcast), sizeof(*OA(broadcast)));
	if (ioctl(f, SIOCSIFBRDADDR, &r)<0)
	    err("opendev: SIOCSIFBRDADDR");
	t = s;
	if (ioctl(f, SIOCSIFFLAGS, &t)<0)
	    err("opendev: SIOCSIFFLAGS");
    }
    if (OA(netmask)) {
	memcpy(&r.ifr_netmask, OA(netmask), sizeof(*OA(netmask)));
	if (ioctl(f, SIOCSIFNETMASK, &r)<0)
	    err("opendev: SIOCSIFNETMASK");
    }
    if (OI(mtu)) {
	r.ifr_mtu=OI(mtu);
	if (ioctl(f, SIOCSIFMTU, &r)<0)
	    err("opendev: SIOCSIFMTU");
    }
    if (OI(metric)) {
	r.ifr_metric=OI(metric);
	if (ioctl(f, SIOCSIFMETRIC, &r)<0)
	    err("opendev: SIOCSIFMETRIC");
    }
    s.ifr_flags |= IFF_UP|IFF_RUNNING|IFF_NOTRAILERS;
    if (ioctl(f, SIOCSIFFLAGS, &s)<0)
	err("opendev: SIOCSIFFLAGS");

    /* Not setting the ARP for the far end of the tunnel is not fatal */
    setarp();

    /* Set up IPX network */
    setipxnet();

    return f;

error:
    close(f);
    return -1;

#undef amiss
#undef err
}

/* Close the device. */

void closedev(int f)
{
    struct ifreq r;
    strcpy(r.ifr_name, OS(device));
    if (ioctl(f, SIOCGIFFLAGS, &r)<0)
	logerr(LOG_ERR, "closedev: SIOCGIFFLAGS");
    r.ifr_flags&=~(IFF_UP|IFF_RUNNING);
    if (ioctl(f, SIOCSIFFLAGS, &r)<0)
	logerr(LOG_ERR, "closedev: SIOCSIFFLAGS");
    close(f);
}

/*** Open a SOCKS5 connection ***/

int opensocks(void) {
    struct sockaddr_in a;
    int fd=socks5_open(OA(socks));
    if (fd>=0) {
	/* 3rd par of socks5_cmd is in: my address, out: relayer */
	a=*OA(me);
	fd=socks5_cmd(fd, S5_UDP, &a);
	/* Swap peer and socks addresses.
	   Use the SOCKS relayer as "peer" and the real peer as "socks"
	   parameter. See output.c why. */
	*OA(socks)=*OA(peer);
	*OA(peer)=a;
    }
    return fd;
}

/* Check whether the SOCKS5 server has closed the connection */

const char *eofsocks_str[] = {"no error", "exception", "read error", "EOF"};

int eofsocks(int fd) {
    fd_set ifd, efd;
    struct timeval t={0,0};
    FD_ZERO(&ifd);
    FD_ZERO(&efd);
    FD_SET(fd, &ifd);
    FD_SET(fd, &efd);
    if (select(fd+1, &ifd, NULL, &efd, &t)>0) {
	if (FD_ISSET(fd, &efd)) {
	    return 1;
	}
	if (FD_ISSET(fd, &ifd)) {
	    int e;
	    char c;
	    if ((e=read(fd, &c, 1))<=0) {
		return e<0 ? 2 : 3;
	    }
	}
    }
    return 0;    
}

/*** Signal handler ***/

sig_atomic_t gotsig=0;

void sighand(int s)
{
    if (OI(debug)) {
	printf("Signal %d\n", s);
    } else {
	syslog(LOG_NOTICE, "got signal %d", s);
    }
    gotsig=s;
}

void sigignore(int s)
{
    /* Dummy */
}

void setsig(int sig, void (*fun)(int))
{
  struct sigaction sa;
  sa.sa_handler = fun;
  sa.sa_flags = 0;
  sigemptyset(&sa.sa_mask);
  sigaction(sig, &sa, NULL);
}

/*** The key exchange procedure ***/

#ifdef VER_CRC32
typedef unsigned long crc;
#define CRCFORM "%08lX"
#else
typedef unsigned short crc;
#define CRCFORM "%04X"
#endif

#define REM 	256	/* How many keys to remember */

/* Cache last valid key CRCs. */

static int dupcheck(crc c)
{
    static crc ca[REM];
    static int caf=0, cal=0;
    int i;

    for (i=cal; i<caf; ++i)
	if (ca[i%REM]==c)
	    return 1;
    if (caf-cal>=REM)
	if (++cal>REM) {
	    caf-=REM;
	    cal-=REM;
	}
    ca[caf%REM]=c;
    ++caf;
    return 0;
}

int errcnt;

int kxchg(int f, int r)
{
    int x=0, e=0;
    unsigned char kx[KEYXCHGBLKMIN];
    #define kx_typ       (kx[0])
    #define kx_nkind_key (kx+1)
#ifdef VER_CRC32
    #define kx_nkind_crc (*(unsigned long *)(kx+1+userKeySize))
    #define kx_nkack_crc (*(unsigned long *)(kx+1))
#else
    #define kx_nkind_crc (*(unsigned short *)(kx+1+userKeySize))
    #define kx_nkack_crc (*(unsigned short *)(kx+1))
#endif
    static crc scrc=0;
    static unsigned char skey[userKeySize];

    if (recv(f, &kx, sizeof(kx), 0)<=0) {
	if (errno==EAGAIN || errno==EINTR)
	     /* This is normal, would be even after select */
	    return 0;
	logerr(LOG_INFO, "kxchg: recv");
	if (errno==ENXIO)
	    /* Device was closed */
	    return -1;
	/* allow for some amount of ECONNREFUSED etc. */
	return (OI(maxerr)<0 ? 0 : --errcnt);
    }
    errcnt=OI(maxerr);
    switch(kx_typ) {
    case NK_RREQ:
	kx_typ=NK_REQ;
	dprintf0("KX: [NK_RREQ] sending NK_REQ\n");
	x=send(f, &kx, sizeof(kx), 0); e=errno;
	++ks.rreq;
	break;
    case NK_REQ:
	kx_typ=NK_IND;
	if (read(r, &skey, userKeySize)!=userKeySize) {
	    logerr(LOG_ERR, "kxchg: read(r)");
	    return -1;
	}
	memcpy(kx_nkind_key, skey, userKeySize);
#ifdef VER_CRC32
	kx_nkind_crc=(scrc=htonl(crc32(skey, userKeySize)));
#else
	kx_nkind_crc=(scrc=block_crc(skey, userKeySize)); /*#*/
#endif
	dprintf1("KX: [NK_REQ] sending NK_IND " CRCFORM "\n", scrc);
	x=send(f, &kx, sizeof(kx), 0); e=errno;
	{
	    struct siocsifcipkey sk;
	    sk.which=KEY_INVAL+KEY_SEND;
	    if (ioctl_setkey(f, OS(device), &sk)<0)
		logerr(LOG_ERR, "setkey");
	    memset(&sk, 0, sizeof(sk));
	}
	++ks.req;
	break;
    case NK_IND:
	if (
#ifdef VER_CRC32
	    crc32(kx_nkind_key, userKeySize)==ntohl(kx_nkind_crc)
#else
	    block_crc(kx_nkind_key, userKeySize)==(kx_nkind_crc) /*#*/
#endif
	    ) {
	    struct siocsifcipkey sk;
	    if (dupcheck(kx_nkind_crc)) {
		dprintf0("KX: [NK_IND] duplicate\n");
		++ks.indb;
	    } else {
		sk.which=KEY_RECV;
		memcpy(&sk.thekey, kx_nkind_key, userKeySize);
		if (ioctl_setkey(f, OS(device), &sk)<0)
		    perror("setkey");
		else {
		    kx_typ=NK_ACK;
		    kx_nkack_crc=kx_nkind_crc; /* XX */
		    dprintf1("KX: [NK_IND] sending NK_ACK " CRCFORM "\n",
			     (kx_nkack_crc)); /*#*/
		    x=send(f, &kx, sizeof(kx), 0); e=errno;
		}
		++ks.ind;
	    }
	    memset(&sk, 0, sizeof(sk));
	} else {
	    dprintf0("KX: [NK_IND] invalid\n");
	    ++ks.indb;
	}
	break;
    case NK_ACK:
	if (scrc==(kx_nkack_crc)) { /*#*/
	    struct siocsifcipkey sk;
	    sk.which=KEY_SEND;
	    memcpy(&sk.thekey, &skey, userKeySize);
	    dprintf1("KX: [NK_ACK] got " CRCFORM "\n", scrc);
	    if (ioctl_setkey(f, OS(device), &sk)<0)
		perror("setkey");
	    ++ks.ack;
	    memset(&sk, 0, sizeof(sk));
	    memset(&skey, 0, sizeof(skey));
	} else {
	    dprintf1("KX: [NK_ACK] unknown " CRCFORM "\n", scrc);
	    ++ks.ackb;
	}
	break;
    default:
	dprintf1("KX: invalid type %02X\n", kx_typ);
	++ks.bogus;
    }
    memset(kx, 0, sizeof(kx));
    if (x<0) {
	errno=e;
	logerr(LOG_WARNING, "kxchg: send");
    }
    return x;
}

/*** Call the ip-up, ip-down scripts ***/

void pspawn(char *s)
{
    int p, c;
    p=getpid();
    if ((c=fork())<0) {
	logerr(LOG_ERR, "pspawn: fork");
	return;
    }
    if (c) {
	(void) wait(NULL); /* exactly one child is active */
    } else {
      	/* call <interface> <my-addr> <daemon-pid> <local> <remote> <network-address> <netmask> <arg>*/
        struct in_addr networkaddr;
	struct in_addr netmask;
	char bufa[32], bufb[32], bufc[32], bufd[32], bufe[32], buff[32];
	char *na[]={s, OS(device), bufa, bufb, bufc, bufd, 
		    bufe, buff, OS(arg), NULL};

	netmask.s_addr = (OA(netmask) ?  OAaddr(netmask).s_addr : 0xffffffff);
	networkaddr.s_addr = htonl(ntohl(OAaddr(ipaddr).s_addr) 
				   & ntohl(OAaddr(netmask).s_addr));
	sprintf(bufa, "%s:%d", inet_ntoa(OAaddr(me)), ntohs(OAport(me)));
	sprintf(bufb, "%d", p);
	sprintf(bufc, "%s", inet_ntoa(OAaddr(ipaddr)));
	sprintf(bufd, "%s", inet_ntoa(OAaddr(ptpaddr)));
	sprintf(bufe, "%s", inet_ntoa(networkaddr));
	sprintf(buff, "%s", inet_ntoa(netmask));
	close(0); close(1); close(2);
	(void) open("/dev/null", O_RDWR);
	(void) dup(0); (void) dup(0);
	execv(s, na);
	logerr(LOG_WARNING, "pspawn: exec");
	exit(1);
    }
}

/*** Main program ***/

int main(int argc, char *argv[])
{
    int i, ur, fd, sd=-1;
    char *pn=strrchr(argv[0], '/');
    if (pn)
	++pn;
    else
	pn=argv[0];

    if (getuid() || geteuid()) {
	fprintf(stderr, "This program must be run by root.\n");
	exit(1);
	/* DO NOT remove this check */
    }
    /* Snarf options from: standard options file, user-supplied file and
       finally, command line */
    setopt_file(CTLDIR "/options", 0);
    if ((argc>1) && (!strcmp(argv[1], "-o"))) {
	if (argc>2) {
	    setopt_file(argv[2], 1);
	    argc-=2; argv+=2;
	} else {
	    usage(pn);
	}
    }
    setopt_cmdline(--argc, ++argv);

    for (i=getdtablesize()-1; i>2; --i)
	(void) close(i);

    if (!OI(debug)) {
	/* daemonize */
	if ((i=fork())<0) {
	    perror("fork"); exit(1);
	}
	if (i)
	    exit(0);
	openlog(pn, LOG_PID, LOG_DAEMON);
	/* don't let the syslog socket get in the way of the standard fds */
	close(0); close(1); close(2);
	setsid();
	syslog(LOG_INFO, "%s: CIPE daemon vers %s (c) Olaf Titz 1996-1998",
	       OS(device), daemon_version);
    } else {
	printf("%s: CIPE daemon vers %s (c) Olaf Titz 1996-1998\n",
	       OS(device), daemon_version);
	dumpopt();
    }

    if ((ur=open("/dev/urandom", O_RDONLY))<0) {
	logerr(LOG_ERR, "open(/dev/urandom)"); exit(1);
    }
    if (OA(socks)) {
	struct itimerval t;
	if ((sd=opensocks())<0)
	    exit(1);
	/* Since the key exchange socket can't do select(), we make
	   the recv() return EINTR every now and then to get a chance
	   to check the socks5 control socket. */
	setsig(SIGALRM, sigignore);
	t.it_interval.tv_sec = 30;
	t.it_interval.tv_usec = 0;
	t.it_value.tv_sec = 60;
	t.it_value.tv_usec = 0;
	setitimer(ITIMER_REAL, &t, 0);
    }
    if ((fd=opendev())<0)
	exit(1);
    pspawn(OS(ipup) ? OS(ipup) : CTLDIR "/ip-up");
    setsig(SIGHUP, sighand);
    setsig(SIGINT, sighand);
    setsig(SIGTERM, sighand);
    if (!OI(maxerr))
	OI(maxerr)=8; /* compatibility */
    errcnt=OI(maxerr);

    /* Main loop */
    while ((!gotsig) && (kxchg(fd, ur)>=0)) {
	if ((sd>=0) && (i=eofsocks(sd))) {
	    log2(LOG_INFO, "%s on socks connection", eofsocks_str[i]);
	    break;
	}
    }

    /* Clear ipx configuration */
    clearipxnet();

    closedev(fd);
    if (OI(debug))
	printf("KX stats: rreq=%d, req=%d, ind=%d, indb=%d, ack=%d,"
	       " ackb=%d, unknown=%d\n", ks.rreq, ks.req, ks.ind, ks.indb,
	       ks.ack, ks.ackb, ks.bogus);
    else
	syslog(LOG_INFO, "KX stats: rreq=%d, req=%d, ind=%d, indb=%d, ack=%d,"
	       " ackb=%d, unknown=%d\n", ks.rreq, ks.req, ks.ind, ks.indb,
	       ks.ack, ks.ackb, ks.bogus);
    pspawn(OS(ipdown) ? OS(ipdown) : CTLDIR "/ip-down");
    if (!OI(debug))
	syslog(LOG_INFO, "%s: daemon exiting", OS(device));
    return 0;
}
