/* sig-check.c - Check signatures
 *  Copyright (C) 2001, 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <time.h>

#include "opencdk.h"
#include "main.h"

void
cdk_hash_public_key( cdkPKT_public_key *pk, GCRY_MD_HD hd )
{
    int i;
    u16 n = 0, a = 0;
    byte buf[4];
    
    if ( !pk || !hd )
        return;

    if ( pk->version < 4 && is_RSA(pk->pubkey_algo) ) {
        for (i = 0; i< cdk_pk_get_npkey(pk->pubkey_algo); i++ )
            gcry_md_write( hd, pk->mpi[i]->data + 2, pk->mpi[i]->bytes );
        return;
    }
    n = pk->version < 4? 8 : 6;
    for ( i = 0; i < cdk_pk_get_npkey(pk->pubkey_algo); i++ ) {
        if ( pk->version < 4 )
            n += pk->mpi[i]->bytes;
        else
            n += 2 + pk->mpi[i]->bytes;
    }
    gcry_md_putc( hd, 0x99);
    gcry_md_putc( hd, n >> 8 );
    gcry_md_putc( hd, n );
    gcry_md_putc( hd, 4);
    
    buf[0] = pk->timestamp >> 24;
    buf[1] = pk->timestamp >> 16;
    buf[2] = pk->timestamp >>  8;
    buf[3] = pk->timestamp;
    gcry_md_write( hd, buf, 4 );
    
    if ( pk->version < 4 ) { 
        a = pk->expiredate? pk->expiredate : 0;
        gcry_md_putc( hd, a >> 8 );
        gcry_md_putc( hd, a );
    } 
    gcry_md_putc(hd, pk->pubkey_algo);
    for ( i=0; i<cdk_pk_get_npkey(pk->pubkey_algo); i++ ) {
        if ( pk->version >= 4 ){
            gcry_md_putc( hd, pk->mpi[i]->bits >> 8 );
            gcry_md_putc( hd, pk->mpi[i]->bits );
        }
        gcry_md_write( hd, pk->mpi[i]->data + 2, pk->mpi[i]->bytes );
    }
} /* cdk_hash_public_key */

void
cdk_hash_user_id( cdkPKT_user_id *uid, int sig_version, GCRY_MD_HD hd )
{
    byte buf[5];
    const byte *data;
    u32 dlen = 0;
    
    if ( !uid || !uid->name || !hd )
        return;
    
    if ( sig_version >= 4 ) {
        if ( uid->attrib_img ) {
            buf[0] = 0xd1;
            buf[1] = uid->attrib_len >> 24;
            buf[2] = uid->attrib_len >> 16;
            buf[3] = uid->attrib_len >>  8;
            buf[4] = uid->attrib_len;
        }
        else {
            buf[0] = 0xb4;
            buf[1] = uid->len >> 24;
            buf[2] = uid->len >> 16;
            buf[3] = uid->len >>  8;
            buf[4] = uid->len;
        }
        gcry_md_write( hd, buf, 5 );
    }
    data = uid->attrib_img? uid->attrib_img : (byte *)uid->name;
    dlen = uid->attrib_img? uid->attrib_len : uid->len;
    gcry_md_write( hd, data, dlen );
} /* cdk_hash_user_id */

void
cdk_hash_sig_result( cdkPKT_signature *sig, GCRY_MD_HD hd )
{
    u32 n = 0;
    byte buf[4];

    if ( !sig || !hd )
        return;

    if ( sig->version >= 4 )
        gcry_md_putc( hd, sig->version );
    gcry_md_putc( hd, sig->sig_class );
    if ( sig->version < 4 ) {
        buf[0] = sig->timestamp >> 24;
        buf[1] = sig->timestamp >> 16;
        buf[2] = sig->timestamp >>  8;
        buf[3] =  sig->timestamp;
        gcry_md_write( hd, buf, 4 );
    }
    else {
        gcry_md_putc( hd, sig->pubkey_algo );
        gcry_md_putc( hd, sig->digest_algo );
        if ( sig->hashed ) {
            n = sig->hashed_size;
            cdk_subpkt_hash( sig->hashed, sig->hashed_size, hd );
            n += 6;
        }
        else {
            gcry_md_putc( hd, 0 );
            gcry_md_putc( hd, 0 );
            n = 6;
        }
        gcry_md_putc( hd, sig->version );
        gcry_md_putc( hd, 0xff );
        buf[0] = n >> 24;
        buf[1] = n >> 16;
        buf[2] = n >>  8;
        buf[3] = n;
        gcry_md_write( hd, buf, 4 );
    }
} /* cdk_hash_sig_result */

static void
cache_sig_result( cdkPKT_signature *sig, int res )
{
    if ( !res ) {
        sig->flags.checked = 1;
        sig->flags.valid = 1;
    }
    else if ( res == CDKERR_BAD_SIGNATURE ) {
        sig->flags.checked = 1;
        sig->flags.valid = 0;
    }
    else {
        sig->flags.checked = 0;
        sig->flags.valid = 0;
    
    }
} /* cache_sig_result */

int
signature_check( cdkPKT_public_key *pk, cdkPKT_signature *sig,
                 GCRY_MD_HD digest, int *r_expired )
{
    byte md[24];
    time_t cur_time = make_timestamp();
    int digest_algo;
    int rc = 0;
  
    if ( !pk || !sig || !digest )
        return CDKERR_INV_VALUE;

    if ( sig->flags.checked )
        return sig->flags.valid? 0 : CDKERR_BAD_SIGNATURE;
  
    if ( pk->version == 4 && pk->pubkey_algo == GCRY_PK_ELG_E )
        return CDKERR_INV_ALGO;
    if ( pk->timestamp > sig->timestamp
         || pk->timestamp > cur_time )
        return CDKERR_TIME_CONFLICT;

    digest_algo = sig->digest_algo;
    if ( pk->expiredate && pk->expiredate < cur_time ) {
        if ( r_expired ) *r_expired = 1;
    }
    
    cdk_hash_sig_result( sig, digest );
    gcry_md_final( digest );
    memcpy( md, gcry_md_read( digest, sig->digest_algo ),
            gcry_md_get_algo_dlen(sig->digest_algo) );
    if ( md[0] != sig->digest_start[0]
         || md[1] != sig->digest_start[1] )
        return CDKERR_BAD_SIGNATURE;
  
    rc = cdk_pk_verify( pk, sig, md );
    cache_sig_result( sig, rc );

    return rc;
} /* signature_check */

static int
signature_key_check( CDK_KEYDB_HD khd, CDK_KBNODE kb_pk, CDK_KBNODE kb_sig )
{
    cdkPKT_public_key *pk = NULL, *sig_pk = NULL;
    cdkPKT_signature *sig = NULL;
    CDK_KBNODE node;
    GCRY_MD_HD md;
    int digest_algo, is_expired = 0;
    int rc = 0;

    if ( !kb_pk || !kb_sig )
        return CDKERR_INV_VALUE;
  
    if ( kb_pk->pkt->pkttype != PKT_PUBLIC_KEY
         || kb_sig->pkt->pkttype != PKT_SIGNATURE )
        return CDKERR_INV_VALUE;
    pk = kb_pk->pkt->pkt.public_key;
    sig = kb_sig->pkt->pkt.signature;
    digest_algo = sig->digest_algo;

    if ( sig->sig_class == 0x20 ) /* key revocation */ {
        md = gcry_md_open( digest_algo, 0 );
        cdk_hash_public_key( pk, md );
        rc = signature_check( pk, sig, md, &is_expired );
        cdk_md_close( &md );
    }
    else if ( sig->sig_class == 0x28 ) /* subkey revocation */ {
        node = cdk_kbnode_find_prev( kb_pk, kb_sig, PKT_PUBLIC_SUBKEY );
        if ( !node )
            return CDKERR_NOKEY; /* no subkey for subkey revocation packet */
        md = gcry_md_open( digest_algo, 0 );
        cdk_hash_public_key( pk, md );
        cdk_hash_public_key( node->pkt->pkt.public_key, md );
        rc = signature_check( pk, sig, md, &is_expired );
        cdk_md_close( &md );
    }
    else if ( sig->sig_class == 0x18 ) /* key binding */ {
        node = cdk_kbnode_find_prev( kb_pk, kb_sig, PKT_PUBLIC_SUBKEY);
        if ( !node )
            return CDKERR_NOKEY; /* no subkey for subkey binding packet */
        md = gcry_md_open( digest_algo, 0 );
        cdk_hash_public_key( pk, md );
        cdk_hash_public_key( node->pkt->pkt.public_key, md );
        rc = signature_check( pk, sig, md, &is_expired );
        cdk_md_close( &md );
    }
    else if ( sig->sig_class == 0x1f ) /* direct key signature */ {
        md = gcry_md_open( digest_algo, 0 );
        cdk_hash_public_key( pk, md );
        rc = signature_check( pk, sig, md, &is_expired );
        cdk_md_close( &md );
    }
    else /* all other classes */ {
        node = cdk_kbnode_find_prev( kb_pk, kb_sig, PKT_USER_ID );
        if ( !node )
            return CDKERR_NOKEY; /* no user ID for key signature packet */     
        md = gcry_md_open( digest_algo, 0 );
        cdk_hash_public_key( pk, md );
        cdk_hash_user_id( node->pkt->pkt.user_id, sig->version, md );
        if ( pk->keyid[0] == sig->keyid[0]
             && pk->keyid[1] == sig->keyid[1] )
            rc = signature_check( pk, sig, md, &is_expired );
        else if ( khd ) {
            rc = cdk_keydb_get_pk( khd, sig->keyid, &sig_pk );
            if ( rc )
                return CDKERR_NOKEY;
            rc = signature_check( sig_pk, sig, md, &is_expired );
            cdk_pk_release( sig_pk );
        }
        cdk_md_close( &md );
    }

    return rc;
} /* signature_key_check */

int
cdk_key_check_sigs( CDK_KBNODE kb_pk, CDK_KEYDB_HD khd, int *r_status )
{
    cdkPKT_signature *sig = NULL;
    CDK_KBNODE k;
    u32 keyid = 0;
    int key_status = 0;
    int rc = 0;

    if ( !kb_pk || !r_status )
        return CDKERR_INV_VALUE;

    k = cdk_kbnode_find( kb_pk, PKT_PUBLIC_KEY );
    if ( !k )
        return CDKERR_NOKEY;
    if ( k->pkt->pkt.public_key->is_revoked )
        key_status |= CDK_KEY_REVOKED;
    if ( k->pkt->pkt.public_key->has_expired )
        key_status |= CDK_KEY_EXPIRED;
    if ( key_status ) {
        *r_status = key_status;
        return CDKERR_GENERAL;
    }
    keyid = cdk_pk_get_keyid( k->pkt->pkt.public_key, NULL );

    for ( k=kb_pk; k && k->pkt->pkttype; k=k->next ) {
        if ( k->pkt->pkttype == PKT_SIGNATURE ) {
            sig = k->pkt->pkt.signature;
            rc = signature_key_check( khd, kb_pk, k );
            if ( rc ) {
                if ( IS_UID_SIG(sig) && rc == CDKERR_NOKEY )
                    continue;
                if ( rc && rc != CDKERR_NOKEY ) {
                    *r_status = CDK_KEY_INVALID;
                    break; /* invalid self signature or key signature */
                }
                cdk_printf("KeyID %08X; Signer %08X: `%s'\n",
                           keyid, sig->keyid[1], cdk_strerror(rc));
            }
            else
                cdk_printf("Signature okay: signer %08X\n", sig->keyid[1]);
        }
    }
    if ( !rc || rc == CDKERR_NOKEY )
        *r_status = CDK_KEY_VALID;
  
    return rc;
} /* cdk_key_check_sigs */
