/* t-armor.c - Armor handling for data
 *  Copyright (C) 2001, 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <gcrypt.h>

#include "opencdk.h"
#include "main.h"

void
dump_data( const char *msg, byte *data, size_t len )
{
    int i;
        
    printf("%s\n", msg);
    for (i=0; i<len; i++)
        printf("%02d ", data[i]);
    printf("\n");
}                    

int
test_verify( const char *file, const char *sig )
{
    cdkPKT_public_key *pk;
    int rc;

    cdk_keydb_add_resource( "pub.gpg", 0 );
    pk = cdk_alloc( sizeof *pk );
    rc = cdk_verify_file( file, sig, &pk );
    if ( pk ) {
        printf("v%d %d%c\n",
               pk->version,
               pk->mpi[0]->bits,
               is_DSA(pk->pubkey_algo) ? 'D' : 'R' );
        printf( "Good signature from `%08X'\n", pk->keyid[1] );
	}
    else if ( rc ) {
        printf( "Error `%s'\n", cdk_strerror(rc) );
	}
  
    cdk_free( pk );
    return 0;
}

char
get_pref_letter(int type)
{
    switch (type) {
    case PREFTYPE_NONE: return '-';
    case PREFTYPE_SYM: return 's';
    case PREFTYPE_HASH: return 'h';
    case PREFTYPE_ZIP: return 'c';
    default: return '-';
    }
}
  
void
dump_selfsig( CDK_KBNODE key)
{
    cdkPKT_user_id *uid;
    CDK_KBNODE p;
    int i;

    for (p=key; p; p=p->next) {
        if (p->pkt->pkttype == PKT_USER_ID && p->pkt->pkt.user_id->prefs) {
            uid = p->pkt->pkt.user_id;
            fprintf(stderr, "`%s' [primary %d][mdc %d]\n",
                    uid->name, uid->is_primary, uid->mdc_feature);
            for (i=0; uid->prefs[i].type != PREFTYPE_NONE; i++)
                fprintf(stderr, "%c[%d] ",
                        get_pref_letter(uid->prefs[i].type),
                        uid->prefs[i].value);
        }
        fprintf(stderr, "\n");
    }
}


void
test_sign( const char *file )
{
    byte md[20] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10,
                   11, 12, 13, 14, 15, 16, 17, 18, 19, 20};
    byte buf[512], name[256];
    /*u32 keyid[2] = {0, 0x8B126B6E};*/
    /*u32 keyid[2] = {0, 0xCCC07C35};*/
    CDK_STRLIST locusr = NULL;
    int rc, i;
    /*size_t n;*/

    cdk_set_logging( 1 );
    /*gcry_control( GCRYCTL_SET_DEBUG_FLAGS, 1|2 );*/
    cdk_printf( "Sign `%s'\n", file );

    cdk_secmem_init( 16384 );
    cdk_keydb_add_resource( "sec.gpg", 1 );

    strcpy( name, file );
    strcat( name, ".sig" );

    cdk_strlist_add( &locusr, "opencdk@foo-bar.org" );
    rc = cdk_sign_file( NULL, locusr, file, name, 1 );
    if ( rc ) {
        cdk_printf( "Error `%s'\n", cdk_strerror(rc) );
    }

#if 0
    {
        struct subpkt_s *s;
        for (s=sig.hashed; s; s=s->next)
            printf("hashed: type `%d'\n", s->type);
        for (s=sig.unhashed; s; s=s->next)
            printf("unhashed: type `%d'\n", s->type);    
    }
#endif
    cdk_secmem_end( );
}

void
test_pwd(void)
{
    CDK_DEK dek = {0};
    CDK_STRING2KEY s2k = {0};
    int i;

    cdk_secmem_init( 16384 );
    s2k.hash_algo = GCRY_MD_SHA1;
    s2k.mode = 3;
    dek.algo = GCRY_CIPHER_CAST5;

    printf("keylen `%d'\n", gcry_cipher_get_algo_keylen(dek.algo));
    /*i = cdk_hash_passphrase(&dek, "test", &s2k, 0);
      if ( i )
      printf("hash_passphrase `%s'\n", cdk_strerror(i));*/

    for (i=0; i<8; i++)
        printf("%02X", s2k.salt[i]);
    printf("\n");
    for (i=0; i<dek.keylen; i++)
        printf("%02X", dek.key[i]);
    printf("\n");
    cdk_secmem_end( );
}

void
test_basic_cipher( void )
{
    CDK_KEYLIST pkl = NULL;
    CDK_STRLIST remusr = NULL, s;
    /*CDK_DEK dek;*/
    CDK_IOBUF outp = NULL;
    /*int i;*/
    int rc = 0;
    GCRY_SEXP opts = NULL;

    cdk_secmem_init( 16384 );
    /*cdk_keydb_add_resource( "/home/twoaday/.gnupg/pubring.gpg", 0 );*/
    cdk_keydb_add_resource( "pub.gpg", 0 );
    /*memset( &dek, 0, sizeof dek );
      _cdk_make_sesskey( &dek );
      printf( "SESSION KEY:\n" );
      printf( "algo=%d; keylen=%d mdc=%d\n", dek.algo, dek.keylen,
      dek.use_mdc );
      for ( i=0; i<dek.keylen; i++ )
      printf("%02X", dek.key[i]);
      printf("\n");*/
    
    cdk_strlist_add( &remusr, "<opencdk@foo-bar.org>" );
#if 0
    /*for ( s=remusr; s; s=s->next )
      printf("recipient `%s'\n", s->d);*/
    rc = cdk_pklist_build( remusr, &pkl, GCRY_PK_USAGE_ENCR );
    if ( rc ) { 
        printf( "pklist_build: `%s'\n", cdk_strerror( rc ) );
        goto leave;
    }
    outp = cdk_iobuf_temp( );
    rc = cdk_pklist_encrypt( pkl, &dek, outp );
    if ( rc )
        printf( "pklist_encrypt: `%s'\n", cdk_strerror( rc ) ); {
        FILE *fp = fopen("pubkey-enc.gpg", "w");
        if ( fp ) {
            size_t nbytes;
            byte *p = cdk_iobuf_getdata( outp, &nbytes );
            fwrite( p, 1, nbytes, fp );
            fclose( fp );
            cdk_free( p );
        }
    }
#endif
        /*cdk_set_logging( OPENCDK_LOG_DEBUG );*/
    gcry_sexp_build( &opts, NULL,
                     "(pubkey-enc((textmode%d)(compress%d)(armor%d)))",
                     0, 0, 0);
    rc = cdk_encrypt_file( opts, remusr, "foo", "foo.gpg" );
    if ( rc )
        printf("encrypt_file: %s\n", cdk_strerror( rc ) );

leave:
    gcry_sexp_release( opts );
    cdk_secmem_end( );
    cdk_iobuf_close( outp );
    cdk_pklist_release( pkl );
}

int
test_basic_sym_cipher( void )
{
    char *pw = "test";
    GCRY_SEXP opts;
    const char *fmt;
    int rc = 0;
  
    cdk_secmem_init( 16386 );
    fmt ="(symkey-enc((textmode%d)(compress%d)(armor%d)(digest%d)(cipher%d)))";
    gcry_sexp_build( &opts, NULL, fmt, 0, 0, 0, 0, 0 );
    rc = cdk_sym_encrypt_file( opts, pw, "foo", "foo.gpg" );
    if ( rc )
        printf("sym_encrypt_file: %s\n", cdk_strerror( rc ) );
    gcry_sexp_release( opts );

    return 0;
}

void
test_decrypt( const char *file )
{
    char *pw = "test";
    u32 keyid[2];
    int rc = 0;

    cdk_secmem_init( 16384 );
    cdk_keydb_add_resource( "sec.gpg", 1 );
#if 0
    rc = cdk_decrypt_get_keyid( file, keyid );
    if ( rc )
        printf("get_keyid `%s'\n", cdk_strerror( rc ) );
    if ( !rc )
        printf("%s: %08X\n", file, keyid[1]);
#endif
    cdk_set_logging( OPENCDK_LOG_DEBUG );
    if ( rc )
        printf("decrypt_file: `%s'\n", cdk_strerror( rc ) );
}

int
main(int argc, char **argv)
{
    /*test_pwd();*/
    test_basic_cipher( );
    /*test_basic_sym_cipher( );*/
    /*test_decrypt( "foo.gpg" );*/
    if (argc > 1)
        test_decrypt( argv[1] );
    /*if ( argc == 1 )
      test_verify("Makefile", "Makefile.sig");*/
    /*test_sign( "foo" );*/
    return 0;
}







