/* keydb.c - Key database routines
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"
#include "filters.h"
#include "stream.h"


#define KEYDB_ENTRIES 4
static struct cdk_keydb_handle_s keydb_table[KEYDB_ENTRIES];

int
cdk_keydb_add_resource (const char *name, int is_secret)
{
  int i;

  if (!name)
    return CDK_Inv_Value;
  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (!keydb_table[i].used)
	{
	  keydb_table[i].name = cdk_calloc (1, strlen (name) + 1);
	  if (!keydb_table[i].name)
	    return CDK_Out_Of_Core;
	  strcpy (keydb_table[i].name, name);
	  keydb_table[i].secret = is_secret;
	  keydb_table[i].used = 1;
	  keydb_table[i].type = CDK_DBTYPE_KEYRING;
	  keydb_table[i].offset = 0;
	  keydb_table[i].old_offset = 0;
	  break;
	}
    }

  return 0;
}


int
cdk_keydb_remove_resource (int id)
{
  int i;

  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (keydb_table[i].used && id == i)
	{
	  cdk_free (keydb_table[i].name);
	  keydb_table[i].name = NULL;
	  keydb_table[i].offset = 0;
	  keydb_table[i].old_offset = 0;
	  keydb_table[i].secret = 0;
	  keydb_table[i].used = 0;
	  break;
	}
    }
  return 0;
}


const char *
cdk_keydb_get_name (int id)
{
  int i;

  if (id == -1)
    {
      for (i = 0; i < KEYDB_ENTRIES; i++)
	{
	  if (keydb_table[i].used)
	    return keydb_table[i].name;
	}
      return NULL;
    }
  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (keydb_table[i].used && id == i)
	return keydb_table[i].name;
    }
  return NULL;
}


int
cdk_keydb_is_secret (int id)
{
  int i;

  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (keydb_table[i].used && id == i)
	return keydb_table[i].secret;
    }

  return 0;
}


int
cdk_keydb_find_idx (int is_secret, int *ret_pos)
{
  int i;

  if (!ret_pos)
    return CDK_Inv_Value;

  *ret_pos = -1;
  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (!keydb_table[i].used)
	continue;
      if (is_secret == cdk_keydb_is_secret (i))
	{
	  *ret_pos = i;
	  return 0;
	}
    }
  return CDK_General_Error;
}


CDK_KEYDB_HD
cdk_keydb_get_ctx (int is_secret, int id)
{
  int idx = 0;

  if (id == -1)
    {
      int rc = cdk_keydb_find_idx (is_secret, &idx);
      if (rc)
	return NULL;
      return &keydb_table[idx];
    }
  if (id >= 0 && id < KEYDB_ENTRIES)
    {
      if (keydb_table[id].used && keydb_table[id].secret == is_secret)
	return &keydb_table[id];
    }
  return NULL;
}


int
cdk_keydb_new (CDK_KEYDB_HD * r_hd, int type, byte * data, size_t count)
{
  CDK_KEYDB_HD hd;

  hd = cdk_calloc (1, sizeof *hd);
  if (!hd)
    return CDK_Out_Of_Core;
  
  hd->used = 1;
  switch (type)
    {
    case CDK_DBTYPE_ARMORED:
    case CDK_DBTYPE_KEYRING:
      data[count] = '\0';
      hd->name = strdup (data);
      if (!hd->name)
        {
          cdk_free (hd);
          return CDK_Out_Of_Core;
        }
      hd->type = type;
      break;
      
    case CDK_DBTYPE_DATA:
      hd->buf = cdk_stream_tmp ();
      if (!hd->buf)
        {
          cdk_free (hd);
          return CDK_Out_Of_Core;
        }
      hd->type = CDK_DBTYPE_DATA;
      cdk_stream_write (hd->buf, data, count);
      break;
    }
  if (r_hd)
    *r_hd = hd;
  return 0;
}


int
cdk_keydb_open (CDK_KEYDB_HD kdb, CDK_STREAM * ret_kr)
{
  int rc = 0;
  CDK_STREAM kr = NULL;

  if (!kdb || !ret_kr)
    return CDK_Inv_Value;

  if (kdb->type == CDK_DBTYPE_DATA && kdb->buf)
    {
      cdk_stream_seek (kdb->buf, 0);
      kr = kdb->buf;
    }
  else if (kdb->name && (kdb->type == CDK_DBTYPE_KEYRING
                         || kdb->type == CDK_DBTYPE_ARMORED))
    {
      rc = cdk_stream_open (kdb->name, &kr);
      if (rc)
	goto leave;
      if (kdb->type == CDK_DBTYPE_ARMORED)
        cdk_stream_set_armor_flag (kr, 0);
    }
  if (!rc)
    cdk_stream_seek (kr, kdb->offset);
 leave:
  if (rc)
    {
      cdk_stream_close (kr);
      kr = NULL;
    }
  *ret_kr = kr;
  return rc;
}


static int
find_by_keyid (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  u32 keyid[2];
  int key_found = 0;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || node->pkt->pkttype == CDK_PKT_SECRET_KEY
          || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
        {
          _cdk_keyid_get (node->pkt, keyid);
          switch (ks->type)
            {
            case CDK_DBSEARCH_SHORT_KEYID:
              if (keyid[1] == ks->u.keyid[1])
                {
                  key_found = 1;
                  break;
                }
              break;

            case CDK_DBSEARCH_KEYID:
              if (keyid[1] == ks->u.keyid[1] && keyid[0] == ks->u.keyid[0])
                {
                  key_found = 1;
                  break;
                }
              break;
            }
        }
    }
  return key_found;
}


static int
find_by_fpr (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  int key_found = 0;
  byte fpr[20];
  
  if (ks->type != CDK_DBSEARCH_FPR)
    return key_found;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || node->pkt->pkttype == CDK_PKT_SECRET_KEY
          || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
        {
          _cdk_fingerprint_get (node->pkt, fpr);
          if (!memcmp (ks->u.fpr, fpr, 20))
            {
              key_found = 1;
              break;
            }
        }
    }
  return key_found;
}          


static int
find_by_pattern (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  size_t uidlen;
  char *name;
  int key_found = 0;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype != CDK_PKT_USER_ID)
        continue;
      uidlen = node->pkt->pkt.user_id->len;
      name = node->pkt->pkt.user_id->name;
      switch (ks->type)
        {
        case CDK_DBSEARCH_EXACT:
          if (name
              && (strlen (ks->u.pattern) == uidlen
              || !strncmp (ks->u.pattern, name, uidlen)))
            {
              key_found = 1;
              break;
            }
          break;

        case CDK_DBSEARCH_SUBSTR:
          if (uidlen > 65536)
            break;
          if (name
              && strlen (ks->u.pattern) > uidlen)
            break;
          if (name
              && ascii_memistr (name, uidlen, ks->u.pattern))
            {
              key_found = 1;
              break;
            }
          break;
        }
    }
  return key_found;
}


void
cdk_keydb_search_reset (CDK_KEYDB_HD hd)
{
  if (hd)
    {
      hd->old_offset = 0;
      hd->offset = 0;
    }
}


int
cdk_keydb_search_new (CDK_DBSEARCH * r_dbs, int type, void * desc)
{
  CDK_DBSEARCH c;
  u32 * keyid;

  c = cdk_calloc (1, sizeof *c);
  if (!c)
    return CDK_Out_Of_Core;
  c->type = type;
  switch (type)
    {
    case CDK_DBSEARCH_EXACT:
    case CDK_DBSEARCH_SUBSTR:
      c->u.pattern = desc;
      break;

    case CDK_DBSEARCH_SHORT_KEYID:
      keyid = desc;
      c->u.keyid[1] = keyid[0];
      break;
      
    case CDK_DBSEARCH_KEYID:
      keyid = desc;
      c->u.keyid[0] = keyid[0];
      c->u.keyid[1] = keyid[1];
      break;

    case CDK_DBSEARCH_FPR:
      memcpy (c->u.fpr, desc, 20);
      break;

    default:
      cdk_free (c);
      return CDK_Inv_Mode;
    }
  if (r_dbs)
    *r_dbs = c;
  return 0;
}


void
cdk_keydb_search_free (CDK_DBSEARCH dbs)
{
  if (dbs)
    {
      dbs->type = 0;
      cdk_free (dbs);
    }
}


int
cdk_keydb_search (CDK_KEYDB_HD kdb, CDK_DBSEARCH ks, CDK_KBNODE * r_key)
{
  CDK_STREAM kr = NULL;
  CDK_KBNODE knode = NULL;
  int key_found = 0;
  int rc = 0;
  size_t pos = 0;

  if (!kdb || !r_key || !ks)
    return CDK_Inv_Value;

  if (r_key)
    *r_key = NULL;

  rc = cdk_keydb_open (kdb, &kr);
  if (rc)
    return rc;

  while (!key_found && !rc)
    {
      pos = cdk_stream_tell (kr);
      kdb->old_offset = pos;
      rc = cdk_keydb_get_keyblock (kr, &knode);
      if (rc)
        {
          if (rc == CDK_EOF && knode)
            rc = 0;
          if (rc)
            break;
        }    
      if (!knode && rc == CDK_EOF)
        rc = CDK_Error_No_Key;
      if (rc)
	break;
      pos = cdk_stream_tell (kr);
      kdb->offset = pos;

      switch (ks->type)
        {
        case CDK_DBSEARCH_SHORT_KEYID:
        case CDK_DBSEARCH_KEYID:
          key_found = find_by_keyid (knode, ks);
          break;

        case CDK_DBSEARCH_FPR:
          key_found = find_by_fpr (knode, ks);
          break;
          
	case CDK_DBSEARCH_EXACT:
	case CDK_DBSEARCH_SUBSTR:
          key_found = find_by_pattern (knode, ks);
	  break;
	}
      if (key_found)
        break;

      cdk_kbnode_release (knode);
      knode = NULL;
    }

  cdk_stream_close (kr);
  *r_key = key_found? knode : NULL;

  return rc;
}


int
cdk_keydb_get_bykeyid (CDK_KEYDB_HD kdb, u32 * keyid, CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!kdb || !keyid || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_SHORT_KEYID;
  ks.u.keyid[1] = keyid[1];

  return cdk_keydb_search (kdb, &ks, ret_pk);
}


int
cdk_keydb_get_byfpr (CDK_KEYDB_HD kdb, const byte * fpr, CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!kdb || !fpr || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_FPR;
  memcpy (ks.u.fpr, fpr, 20);

  return cdk_keydb_search (kdb, &ks, ret_pk);
}


int
cdk_keydb_get_bypattern (CDK_KEYDB_HD kdb, const char *patt,
			 CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!kdb || !patt || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_SUBSTR;
  ks.u.pattern = patt;

  return cdk_keydb_search (kdb, &ks, ret_pk);
}


static CDK_KBNODE
keydb_find_byusage (CDK_KBNODE root, int req_usage, int pk)
{
  CDK_KBNODE node;
  int pkttype = 0;

  if (!req_usage)
    return cdk_kbnode_find (root, CDK_PKT_PUBLIC_KEY);

  for (node = root; node; node = node->next)
    {
      pkttype = node->pkt->pkttype;
      if (pk && (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
                 || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
          && (node->pkt->pkt.public_key->pubkey_usage & req_usage))
	return node;
      if (!pk && (node->pkt->pkttype == CDK_PKT_SECRET_KEY
                  || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
          && (node->pkt->pkt.secret_key->req_usage & req_usage))
	return node;
    }
  return NULL;
}


CDK_KBNODE
_cdk_keydb_find_bykeyid (CDK_KBNODE root, u32 * keyid)
{
  CDK_KBNODE node;
  u32 kid[2];

  for (node = root; node; node = node->next)
    {
      _cdk_keyid_get (node->pkt, kid);
      if (kid[1] == keyid[1])
	return node;
    }

  return NULL;
}


int
cdk_keydb_get_sk_byname (const char *name, cdkPKT_secret_key ** ret_sk,
			 int alloced)
{
  CDK_KBNODE knode, node = NULL;
  cdkPKT_secret_key *sk = NULL;
  CDK_KEYDB_HD khd;
  int req_usage = 0;
  int rc = 0;

  if (!ret_sk)
    return CDK_Inv_Value;

  khd = cdk_keydb_get_ctx (0, -1);
  if (!khd)
    return CDK_Error_No_Keyring;
  rc = cdk_keydb_get_bypattern (khd, name, &knode);
  if (rc)
    goto leave;
  if (alloced)
    {
      req_usage = ret_sk[0]->req_usage;
      cdk_free (ret_sk[0]);
      node = keydb_find_byusage (knode, req_usage, 0);
    }
  if (!node)
    {
      node = cdk_kbnode_find (knode, CDK_PKT_SECRET_KEY);
      if (!node)
	{
	  rc = CDK_Error_No_Key;
	  goto leave;
	}
    }
  _cdk_copy_seckey (&sk, node->pkt->pkt.secret_key);
  cdk_kbnode_release (knode);

leave:
  *ret_sk = sk;
  return rc;
}


int
cdk_keydb_get_pk_byname (const char *name, cdkPKT_public_key ** ret_pk,
			 int alloced)
{
  CDK_KBNODE key, node = NULL;
  cdkPKT_public_key *pk = NULL;
  CDK_KEYDB_HD khd;
  const char *s;
  int req_usage = 0;
  int rc = 0;

  if (!name || !ret_pk)
    return CDK_Inv_Value;

  khd = cdk_keydb_get_ctx (0, -1);
  if (!khd)
    return CDK_Error_No_Keyring;
  rc = cdk_keydb_get_bypattern (khd, name, &key);
  if (rc)
    goto leave;
  if (alloced)
    {
      req_usage = ret_pk[0]->pubkey_usage;
      cdk_free (ret_pk[0]);
      node = keydb_find_byusage (key, req_usage, 1);
    }
  if (!node)
    {
      node = cdk_kbnode_find (key, CDK_PKT_PUBLIC_KEY);
      if (!node)
	{
	  rc = CDK_Error_No_Key;
	  goto leave;
	}
    }
  _cdk_copy_pubkey (&pk, node->pkt->pkt.public_key);
  for (node = key; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_USER_ID)
	{
	  s = node->pkt->pkt.user_id->name;
	  if (pk && !pk->uid && ascii_memistr (s, strlen (s), name))
	    {
	      _cdk_copy_userid (&pk->uid, node->pkt->pkt.user_id);
	      break;
	    }
	}
    }
  cdk_kbnode_release (key);

leave:
  *ret_pk = pk;

  return rc;
}


int
cdk_keydb_get_pk (CDK_KEYDB_HD khd, u32 * keyid, cdkPKT_public_key ** r_pk)
{
  struct cdk_keydb_search_s ks;
  CDK_KBNODE key = NULL, node = NULL;
  CDK_KEYDB_HD keydb_hd = NULL;
  cdkPKT_public_key *pk = NULL;
  int rc = 0;

  if (!keyid || !r_pk)
    return CDK_Inv_Value;

  if (khd == NULL)
    {
      keydb_hd = cdk_keydb_get_ctx (0, -1);
      if (!keydb_hd)
	return CDK_Error_No_Keyring;
    }
  else
    keydb_hd = khd;

  if (!keyid[0])
    {
      ks.type = CDK_DBSEARCH_SHORT_KEYID;
      ks.u.keyid[1] = keyid[1];
    }
  else
    {
      ks.type = CDK_DBSEARCH_KEYID;
      ks.u.keyid[0] = keyid[0];
      ks.u.keyid[1] = keyid[1];
    }
  rc = cdk_keydb_search (keydb_hd, &ks, &key);
  if (rc)
    goto leave;
  node = _cdk_keydb_find_bykeyid (key, keyid);
  if (!node)
    {
      rc = CDK_Error_No_Key;
      goto leave;
    }
  _cdk_copy_pubkey (&pk, node->pkt->pkt.public_key);
  cdk_kbnode_release (key);

leave:
  *r_pk = pk;

  return rc;
}


int
cdk_keydb_get_sk (CDK_KEYDB_HD khd, u32 * keyid, cdkPKT_secret_key ** ret_sk)
{
  CDK_KEYDB_HD hd;
  CDK_KBNODE snode, node;
  cdkPKT_secret_key *sk = NULL;
  int rc = 0;

  if (!khd)
    {
      hd = cdk_keydb_get_ctx (1, -1);
      if (!hd)
	return CDK_Error_No_Keyring;
    }
  else
    hd = khd;
  rc = cdk_keydb_get_bykeyid (hd, keyid, &snode);
  if (rc)
    goto leave;

  node = _cdk_keydb_find_bykeyid (snode, keyid);
  if (!node)
    {
      rc = CDK_Error_No_Key;
      goto leave;
    }

  _cdk_copy_seckey (&sk, node->pkt->pkt.secret_key);
  cdk_kbnode_release (snode);

leave:
  *ret_sk = sk;
  return rc;
}


static int
keydb_merge_selfsig (CDK_KBNODE key)
{
  CDK_KBNODE k, kbnode, unode;
  struct cdk_subpkt_s *s = NULL;
  cdkPKT_signature *sig = NULL;
  cdkPKT_user_id *uid = NULL;
  const byte *symalg = NULL, *hashalg = NULL, *compalg = NULL;
  size_t nsymalg = 0, nhashalg = 0, ncompalg = 0, n = 0;

  if (!key)
    return CDK_Inv_Value;

  for (k = key; k && k->pkt->pkttype; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE
	  && k->pkt->pkt.signature->sig_class == 0x13)
	{
	  unode = cdk_kbnode_find_prev (key, k, CDK_PKT_USER_ID);
	  if (!unode)
	    return CDK_Error_No_Key;
	  uid = unode->pkt->pkt.user_id;
	  sig = k->pkt->pkt.signature;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PRIMARY_UID);
	  if (s)
	    uid->is_primary = 1;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_FEATURES);
	  if (s && s->size == 1 && s->d[0] & 0x01)
	    uid->mdc_feature = 1;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_SYM);
	  if (s)
	    {
	      symalg = s->d;
	      nsymalg = s->size;
	      n += s->size + 1;
	    }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_HASH);
	  if (s)
	    {
	      hashalg = s->d;
	      nhashalg = s->size;
	      n += s->size + 1;
	    }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_ZIP);
	  if (s)
	    {
	      compalg = s->d;
	      ncompalg = s->size;
	      n += s->size + 1;
	    }
	  if (!n || !hashalg || !compalg || !symalg)
	    uid->prefs = NULL;
	  else
	    {
	      uid->prefs = cdk_calloc (1, sizeof (*uid->prefs) * (n + 1));
	      if (!uid->prefs)
		return CDK_Out_Of_Core;
	      n = 0;
	      for (; nsymalg; nsymalg--, n++)
		{
		  uid->prefs[n].type = PREFTYPE_SYM;
		  uid->prefs[n].value = *symalg++;
		}
	      for (; nhashalg; nhashalg--, n++)
		{
		  uid->prefs[n].type = PREFTYPE_HASH;
		  uid->prefs[n].value = *hashalg++;
		}
	      for (; ncompalg; ncompalg--, n++)
		{
		  uid->prefs[n].type = PREFTYPE_ZIP;
		  uid->prefs[n].value = *compalg++;
		}
	      /* end of list marker */
	      uid->prefs[n].type = PREFTYPE_NONE;
	      uid->prefs[n].value = 0;
	      uid->prefs_size = n;

	      kbnode = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_KEY);
	      if (kbnode && uid->prefs && n)
		{
		  cdkPKT_public_key *pk = kbnode->pkt->pkt.public_key;
		  pk->prefs = _cdk_copy_prefs (uid->prefs);
		  pk->prefs_size = n;
		}
	    }
	}
    }

  return 0;
}


static void
key_set_expire (cdkPKT_public_key * pk, u32 pk_expired)
{
  u32 key_expire = 0;
  u32 curtime = 0;

  if (pk)
    {
      key_expire = 0;
      curtime = make_timestamp ();
      pk->expiredate = pk->timestamp + pk_expired;
      key_expire = pk->expiredate;
      pk->has_expired = (key_expire >= curtime) ? 0 : 1;
    }
}


static int
keydb_parse_allsigs (CDK_KBNODE key, int check)
{
  CDK_KBNODE k, kb;
  cdkPKT_signature *sig;
  struct cdk_subpkt_s *s = NULL;
  int pk_expired = 0;
  int rc = 0;

  if (!key)
    return CDK_Inv_Value;

  if (cdk_kbnode_find (key, CDK_PKT_SECRET_KEY))
    return 0;

  /* reset */
  for (k = key; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_USER_ID)
	k->pkt->pkt.user_id->is_revoked = 0;
      else if (k->pkt->pkttype == CDK_PKT_PUBLIC_KEY
               || k->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
	k->pkt->pkt.public_key->is_revoked = 0;
    }

  for (k = key; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  sig = k->pkt->pkt.signature;
	  /* Revocation certificates for primary keys */
	  if (sig->sig_class == 0x20)
	    {
	      kb = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_KEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Revocation certificates for subkeys */
	  else if (sig->sig_class == 0x28)
	    {
	      kb = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_SUBKEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Revocation certifcates for user ID's */
	  else if (sig->sig_class == 0x30)
	    {
	      kb = cdk_kbnode_find_prev (key, k, CDK_PKT_USER_ID);
	      if (kb)
		{
		  kb->pkt->pkt.user_id->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Direct certificates for primary keys */
	  else if (sig->sig_class == 0x1F)
	    {
	      kb = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_KEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_invalid = 0;
		  s = cdk_subpkt_find (k->pkt->pkt.signature->hashed,
				       SIGSUBPKT_KEY_EXPIRE);
		  if (s)
		    {
		      pk_expired = buffer_to_u32 (s->d);
		      key_set_expire (kb->pkt->pkt.public_key, pk_expired);
		    }
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Direct certificates for subkeys */
	  else if (sig->sig_class == 0x18)
	    {
	      kb = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_SUBKEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_invalid = 0;
		  s = cdk_subpkt_find (k->pkt->pkt.signature->hashed,
				       SIGSUBPKT_KEY_EXPIRE);
		  if (s)
		    {
		      pk_expired = buffer_to_u32 (s->d);
		      key_set_expire (kb->pkt->pkt.public_key, pk_expired);
		    }
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	}
    }

  return rc;
}


int
cdk_keydb_get_keyblock (CDK_STREAM inp, CDK_KBNODE * r_knode)
{
  CDK_PACKET *pkt = NULL;
  CDK_KBNODE knode = NULL, node = NULL;
  u32 keyid[2], main_keyid[2];
  int rc = 0, rc2 = 0, old_off;
  int key_seen = 0, got_key = 0;

  if (!inp || !r_knode)
    return CDK_Inv_Value;

  memset (keyid, 0, sizeof keyid);
  memset (main_keyid, 0, sizeof main_keyid);
  
  while (1)
    {
      pkt = cdk_calloc (1, sizeof *pkt);
      if (!pkt)
	return CDK_Out_Of_Core;
      old_off = cdk_stream_tell (inp);
      rc = cdk_pkt_parse (inp, pkt);
      if (rc)
	break;
      if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || pkt->pkttype == CDK_PKT_SECRET_KEY
          || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
	{
          if (key_seen && (pkt->pkttype == CDK_PKT_PUBLIC_KEY
                            || pkt->pkttype == CDK_PKT_SECRET_KEY))
	    {
              cdk_stream_seek (inp, old_off);
	      break;
	    }
          if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
              || pkt->pkttype == CDK_PKT_SECRET_KEY)
            {
              _cdk_keyid_get (pkt, main_keyid);
              key_seen = 1;
            }
          else if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
                   || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
            {
              if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
		{
		  pkt->pkt.public_key->main_keyid[0] = main_keyid[0];
		  pkt->pkt.public_key->main_keyid[1] = main_keyid[1];
		}
	      else
		{
		  pkt->pkt.secret_key->main_keyid[0] = main_keyid[0];
		  pkt->pkt.secret_key->main_keyid[1] = main_keyid[1];
		}
	    }
	  /* we save this for the signature */
          _cdk_keyid_get (pkt, keyid);
          got_key = 1;
	}
      else if (pkt->pkttype == CDK_PKT_USER_ID)
        ;
      else if (pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  pkt->pkt.signature->key[0] = keyid[0];
	  pkt->pkt.signature->key[1] = keyid[1];
	}
      node = cdk_kbnode_new (pkt);
      if (!knode)
	knode = node;
      else
	cdk_kbnode_add (knode, node);
    }

  if (got_key)
    {
      keydb_merge_selfsig (knode);
      rc2 = keydb_parse_allsigs (knode, 0);
      if (!rc)
	rc = rc2;
    }

  *r_knode = got_key ? knode : NULL;
  return rc;
}


int
cdk_keydb_export (CDK_STREAM out, CDK_STRLIST remusr, int secret)
{
  CDK_KEYDB_HD hd;
  CDK_KBNODE knode, node;
  CDK_STRLIST r;
  int rc = 0;

  hd = cdk_keydb_get_ctx (secret, -1);
  if (!hd)
    return CDK_Error_No_Keyring;

  for (r = remusr; r; r = r->next)
    {
      rc = cdk_keydb_get_bypattern (hd, r->d, &knode);
      if (rc)
	break;
      for (node = knode; node; node = node->next)
	{
	  /* those packets are not intended for the real wolrd */
	  if (node->pkt->pkttype == CDK_PKT_RING_TRUST)
	    continue;
	  /* we never export local signed signatures */
	  if (node->pkt->pkttype == CDK_PKT_SIGNATURE &&
	      !node->pkt->pkt.signature->flags.exportable)
	    continue;
	  rc = cdk_pkt_build (out, node->pkt);
	  if (rc)
	    break;
	}
      cdk_kbnode_release (knode);
      knode = NULL;
    }

  return rc;
}


/* internal functions */
CDK_KBNODE
_cdk_keydb_get_pkblock (u32 * keyid)
{
  CDK_KEYDB_HD hd;
  CDK_KBNODE kbnode = NULL;
  int rc = 0;

  hd = cdk_keydb_get_ctx (0, -1);
  if (hd)
    {
      rc = cdk_keydb_get_bykeyid (hd, keyid, &kbnode);
      if (rc)
        kbnode = NULL;
    }
  return kbnode;
}


CDK_KBNODE
_cdk_keydb_get_skblock (u32 * keyid)
{
  CDK_KBNODE snode = NULL;
  CDK_KEYDB_HD hd;
  int rc = 0;

  hd = cdk_keydb_get_ctx (1, -1);
  if (hd)
    {
      rc = cdk_keydb_get_bykeyid (hd, keyid, &snode);
      if (rc)
        snode = NULL;
      cdk_keydb_search_reset (hd);
    }
  return snode;
}


