/* misc.c
 *        Copyright (C) 2002 Timo Schulz
 *        Copyright (C) 1998-2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "opencdk.h"
#include "main.h"


u32
make_timestamp (void)
{
    return (u32) time (NULL);
}


u32
buffer_to_u32 (const byte *buf)
{
  u32 u = 0;
  
  if (buf)
    {
      u  = buf[0] << 24;
      u |= buf[1] << 16;
      u |= buf[2] <<  8;
      u |= buf[3];
    }
  return u;
}


void
u32_to_buffer (u32 u, byte *buf)
{
  if (buf)
    {
      buf[0] = u >> 24;
      buf[1] = u >> 16;
      buf[2] = u >>  8;
      buf[3] = u      ;
    }
}


static const char *
parse_version_number (const char *s, int *number)
{
  int val = 0;

  if (*s == '0' && isdigit(s[1]))
    return NULL;
  /* leading zeros are not allowed */
  for (; isdigit(*s); s++)
    {
      val *= 10;
      val += *s - '0';     
    }
  *number = val;
  return val < 0? NULL : s;      
}


static const char *
parse_version_string (const char * s, int * major, int * minor, int * micro)
{
  s = parse_version_number (s, major);
  if (!s || *s != '.')
    return NULL;
  s++;
  s = parse_version_number (s, minor);
  if (!s || *s != '.')
    return NULL;
  s++;
  s = parse_version_number(s, micro);
  if (!s)
    return NULL;
  return s; /* patchlevel */
}


/**
 * cdk_check_version - Version control handling.
 * @req_version: The requested version
 *
 * Check that the the version of the library is at minimum the requested
 * one and return the version string; return NULL if the condition is
 * not satisfied.  If a NULL is passed to this function, no check is done,
 *but the version string is simply returned.
 **/
const char *
cdk_check_version (const char *req_version)
{
  const char *ver = VERSION;
  int my_major, my_minor, my_micro;
  int rq_major, rq_minor, rq_micro;
  const char *my_plvl, *rq_plvl;
  
  if (!req_version)
    return ver;
  my_plvl = parse_version_string (ver, &my_major, &my_minor, &my_micro);
  if (!my_plvl)
    return NULL;
  /* very strange our own version is bogus */
  rq_plvl = parse_version_string (req_version, &rq_major, &rq_minor,&rq_micro);
  if (!rq_plvl)
    return NULL;  /* req version string is invalid */
  if (my_major > rq_major
      || (my_major == rq_major && my_minor > rq_minor)
      || (my_major == rq_major && my_minor == rq_minor
          && my_micro > rq_micro)
      || (my_major == rq_major && my_minor == rq_minor
          && my_micro == rq_micro
          && strcmp (my_plvl, rq_plvl) >= 0))
    {
      return ver;    
    }
  return NULL;
}


void
cdk_strlist_free (CDK_STRLIST sl)
{
  CDK_STRLIST sl2;

  for(; sl; sl = sl2 )
    {
      sl2 = sl->next;
      cdk_free (sl);
    }
}


CDK_STRLIST
cdk_strlist_add (CDK_STRLIST *list, const char *string)
{
  CDK_STRLIST sl;

  if (!string)
    return NULL;
  
  sl = cdk_calloc (1, sizeof *sl + strlen (string) + 1);
  if (!sl)
    return NULL;
  strcpy (sl->d, string);
  sl->next = *list;
  *list = sl;
  return sl;
}


const char *
cdk_strlist_walk (CDK_STRLIST root, CDK_STRLIST * context)
{
  CDK_STRLIST n;
  
  if (! *context)
    {
      *context = root;
      n = root;
    }
  else
    {
      n = (*context)->next;
      *context = n;
    }

  return n? n->d : NULL;
}


static int
ascii_toupper (int c)
{
  if (c >= 'a' && c <= 'z')
    c &= ~0x20;
  
  return c;   
}

const char *
ascii_memistr (const char *buf, size_t buflen, const char *sub)
{
  const byte *t, *s;
  size_t n;

  for ( t=buf, n=buflen, s=sub ; n ; t++, n-- ) {
    if ( ascii_toupper(*t) == ascii_toupper(*s) ) {
      for( buf=t++, buflen = n--, s++;
           n && ascii_toupper(*t) == ascii_toupper(*s); t++, s++, n-- )
        ;
      if( !*s )
        return buf;
      t = buf;
      n = buflen;
      s = sub ;                        
    }
  }

  return NULL ;
}


int
ascii_strcasecmp (const char *a, const char *b)
{
  if (!a || !b)
    return 1;
  
  if (a == b)
    return 0;

  for (; *a && *b; a++, b++)
    {
      if (*a != *b && ascii_toupper (*a) != ascii_toupper (*b))
        break;         
    }
  
  return *a == *b? 0 : (ascii_toupper (*a) - ascii_toupper (*b));      
}

