/* cipher.c - Cipher filters
 *        Copyright (C) 2002 Timo Schulz
 *        Copyright (C) 1998-2001 Free Software Foundation
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "filters.h"


int
_cdk_cipher_test_algo (int algo)
{
  if (algo < 0 || algo > 110)
    return CDK_Inv_Algo;
  return gcry_cipher_test_algo (algo)? CDK_Inv_Algo : 0;
}


int
_cdk_md_test_algo (int algo)
{
  if (algo < 0 || algo > 110)
    return CDK_Inv_Algo;
  return gcry_md_test_algo (algo) ? CDK_Inv_Algo : 0;
}


int
hash_filter (void * opaque, FILE * in, FILE * out)
{
  md_filter_s * mfx = opaque;
  byte buf[4096];
  int nread;

  if (!mfx)
    return CDK_Inv_Value;

  if (!mfx->md)
    {    
      mfx->md = gcry_md_open (mfx->digest_algo, 0);
      if (!mfx->md)
        return CDK_Gcry_Error;
    }
  
  while (!feof (in))
    {
      nread = fread (buf, 1, sizeof buf-1, in);
      if (!nread)
        break;
      gcry_md_write (mfx->md, buf, nread);
    }  
  wipemem (buf, sizeof buf);
  return 0;
}


static int
write_header (cipher_filter_s * cfx, CDK_STREAM out)
{
  CDK_PACKET pkt;
  cdkPKT_encrypted ed;
  CDK_DEK dek = cfx->dek;
  byte temp[18];
  size_t blocksize = 0;
  int use_mdc = 0, cflags, nprefix;
  int rc = 0;

  blocksize = gcry_cipher_get_algo_blklen (dek->algo);
  if (blocksize < 8 || blocksize > 16)
    return CDK_Inv_Algo;

  use_mdc = dek->use_mdc;
  if (blocksize != 8)
    use_mdc = 1; /* enabled by default for all 128-bit block cipher */

  if (use_mdc && cfx->datalen)
    cfx->datalen += 22;
  
  init_packet (&ed);
  ed.len = cfx->datalen;
  ed.extralen = blocksize + 2;
  if (use_mdc)
    {
      ed.mdc_method = GCRY_MD_SHA1;
      cfx->mdc_hash = gcry_md_open (GCRY_MD_SHA1, 0);
      if (!cfx->mdc_hash)
	return CDK_Gcry_Error;
    }

  init_packet (&pkt);
  pkt.pkttype = use_mdc ? CDK_PKT_ENCRYPTED_MDC : CDK_PKT_ENCRYPTED;
  pkt.pkt.encrypted = &ed;
  rc = cdk_pkt_build (out, &pkt);
  if (rc)
    {
      gcry_md_close (cfx->mdc_hash);
      return rc;
    }
  nprefix = blocksize;
  gcry_randomize (temp, nprefix, GCRY_STRONG_RANDOM);
  temp[nprefix] = temp[nprefix - 2];
  temp[nprefix + 1] = temp[nprefix - 1];
  cflags = GCRY_CIPHER_SECURE;
  if (!use_mdc)
    cflags |= GCRY_CIPHER_ENABLE_SYNC;
  cfx->cipher_hd = gcry_cipher_open (dek->algo, GCRY_CIPHER_MODE_CFB, cflags);
  if (!cfx->cipher_hd)
    {
      gcry_md_close (cfx->mdc_hash);
      return CDK_Gcry_Error;
    }
  rc = gcry_cipher_setkey (cfx->cipher_hd, dek->key, dek->keylen);
  if (rc)
    {
      if (rc == GCRYERR_WEAK_KEY)
        rc = CDK_Weak_Key;
      else
        rc = CDK_Gcry_Error;
      gcry_md_close (cfx->mdc_hash);
      return rc;
    }
  gcry_cipher_setiv (cfx->cipher_hd, NULL, 0);
  if (cfx->mdc_hash)
    gcry_md_write (cfx->mdc_hash, temp, nprefix + 2);
  gcry_cipher_encrypt (cfx->cipher_hd, temp, nprefix + 2, NULL, 0);
  gcry_cipher_sync (cfx->cipher_hd);
  cdk_stream_write (out, temp, nprefix + 2);
  return rc;
}


static int
cipher_encode2 (void * opaque, FILE * in, FILE * out)
{
  cipher_filter_s * cfx = opaque;
  byte buf[4096], pktdata[22];
  int algo = GCRY_MD_SHA1, dlen = gcry_md_get_algo_dlen (algo);
  int rc = 0, nread;

  if (!cfx || !in || !out)
    return CDK_Inv_Value;

  while (!feof (in))
    {
      nread = fread (buf, 1, sizeof buf -1, in);
      if (!nread)
        break;
      if (cfx->mdc_hash)
        gcry_md_write (cfx->mdc_hash, buf, nread);
      rc = gcry_cipher_encrypt (cfx->cipher_hd, buf, nread, NULL, 0);
      if (rc)
        {
          rc = CDK_Gcry_Error;
          break;
        }
      fwrite (buf, 1, nread, out);
    }
  wipemem (buf, sizeof buf);
  if (!rc && cfx->mdc_hash)
    {
      if (dlen != 20)
        return CDK_Inv_Algo;
      /* we must hash the prefix of the MDC packet here */
      pktdata[0] = 0xd3;
      pktdata[1] = 0x14;
      gcry_md_putc (cfx->mdc_hash, pktdata[0]);
      gcry_md_putc (cfx->mdc_hash, pktdata[1]);
      gcry_md_final (cfx->mdc_hash);
      memcpy (pktdata + 2, gcry_md_read (cfx->mdc_hash, algo), dlen);
      gcry_cipher_encrypt (cfx->cipher_hd, pktdata, 22, NULL, 0);
      gcry_md_close (cfx->mdc_hash);
      cfx->mdc_hash = NULL;
      fwrite (pktdata, 1, 22, out);
    }
  return rc;
}


static int
read_header (cipher_filter_s * cfx, FILE * in)
{
  int rc = 0, blocksize, nprefix, cflags = 0, i = 0, c = 0;
  CDK_DEK dek = cfx->dek;
  byte temp[32];

  if (!cfx || !in)
    return CDK_Inv_Value;

  blocksize = gcry_cipher_get_algo_blklen (dek->algo);
  if (blocksize < 8 || blocksize > 16)
    return CDK_Inv_Algo;
  
  nprefix = blocksize;
  if (cfx->datalen && cfx->datalen < (nprefix + 2))
    return CDK_Inv_Value;
  if (cfx->mdc_method)
    cfx->mdc_hash = gcry_md_open (cfx->mdc_method, 0);
  if (!cfx->mdc_hash)
    cflags |= GCRY_CIPHER_ENABLE_SYNC;
  cfx->cipher_hd = gcry_cipher_open (dek->algo, GCRY_CIPHER_MODE_CFB, cflags);
  if (!cfx->cipher_hd)
    {
      gcry_md_close (cfx->mdc_hash);
      return CDK_Gcry_Error;
    }
  rc = gcry_cipher_setkey (cfx->cipher_hd, dek->key, dek->keylen);
  if (rc)
    {
      if (rc == GCRYERR_WEAK_KEY)
        rc = CDK_Weak_Key;
      else
        rc = CDK_Gcry_Error;
      return rc;
    }
  gcry_cipher_setiv (cfx->cipher_hd, NULL, 0);
  for (i = 0; i < (nprefix + 2); i++)
    {
      if ((c = fgetc (in)) == EOF)
        return CDK_File_Error;
      temp[i] = c;
    }
  gcry_cipher_decrypt (cfx->cipher_hd, temp, nprefix + 2, NULL, 0);
  gcry_cipher_sync (cfx->cipher_hd);
  if (temp[nprefix - 2] != temp[nprefix]
      || temp[nprefix - 1] != temp[nprefix + 1])
    return CDK_Chksum_Error;
  if (cfx->mdc_hash)
    gcry_md_write (cfx->mdc_hash, temp, nprefix + 2);
  return 0;
}


static int
cipher_decode2 (void * opaque, FILE * in, FILE * out)
{
  cipher_filter_s * cfx = opaque;
  byte buf[4096];
  int rc = 0, nread;

  if (!cfx || !in || !out)
    return CDK_Inv_Value;

  while (!feof (in))
    {
      nread = fread (buf, 1, sizeof buf-1, in);
      if (!nread)
        break;
      rc = gcry_cipher_decrypt (cfx->cipher_hd, buf, nread, NULL, 0);
      if (rc)
        {
          rc = CDK_Gcry_Error;
          break;
        }
      if (cfx->mdc_hash)
        gcry_md_write (cfx->mdc_hash, buf, nread);
      fwrite (buf, 1, nread, out);
    }
  wipemem (buf, sizeof buf);
  return rc;
}


int
cipher_decode (void *opaque, FILE * in, FILE * out)
{
  cipher_filter_s * cfx = opaque;
  int rc;
    
  if (!cfx || !in || !out)
    return CDK_Inv_Value;
    
  rc = read_header (cfx, in);
  if (!rc)
    rc = cipher_decode2 (cfx, in, out);

  return rc;
}
    

int
cipher_encode (void * opaque, FILE * in, FILE * out)
{
  cipher_filter_s * cfx = opaque;
  CDK_STREAM si, so;
  int rc;

  if (!cfx || !in || !out)
    return CDK_Inv_Value;

  so = cdk_stream_fpopen (out, 1);
  if (!so)
    return CDK_Out_Of_Core;
  si = cdk_stream_fpopen (in, 0);
  if (!si)
    {
      cdk_stream_close (si);
      return CDK_Out_Of_Core;
    }
  cfx->datalen = cdk_stream_get_length (si);
  rc = write_header (cfx, so);
  if (!rc)
    rc = cipher_encode2 (cfx, in, out);

  cdk_stream_close (si);
  cdk_stream_close (so);

  return rc;
}
