/* keydb.c - Key database routines
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"
#include "filters.h"
#include "stream.h"


#define KEYDB_ENTRIES 4
static struct cdk_keydb_handle_s keydb_table[KEYDB_ENTRIES];

int
cdk_keydb_add_resource (const char *name, int is_secret)
{
  int i;

  if (!name)
    return CDK_Inv_Value;
  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (!keydb_table[i].used)
	{
	  keydb_table[i].name = cdk_calloc (1, strlen (name) + 1);
	  if (!keydb_table[i].name)
	    return CDK_Out_Of_Core;
	  strcpy (keydb_table[i].name, name);
	  keydb_table[i].secret = is_secret;
	  keydb_table[i].used = 1;
	  keydb_table[i].type = CDK_DBTYPE_KEYRING;
	  keydb_table[i].offset = 0;
	  keydb_table[i].old_offset = 0;
	  break;
	}
    }

  return 0;
}


int
cdk_keydb_remove_resource (int id)
{
  int i;

  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (keydb_table[i].used && id == i)
	{
	  cdk_free (keydb_table[i].name);
	  keydb_table[i].name = NULL;
	  keydb_table[i].offset = 0;
	  keydb_table[i].old_offset = 0;
	  keydb_table[i].secret = 0;
	  keydb_table[i].used = 0;
	  break;
	}
    }
  return 0;
}


const char *
cdk_keydb_get_name (int id)
{
  int i;

  if (id == -1)
    {
      for (i = 0; i < KEYDB_ENTRIES; i++)
	{
	  if (keydb_table[i].used)
	    return keydb_table[i].name;
	}
      return NULL;
    }
  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (keydb_table[i].used && id == i)
	return keydb_table[i].name;
    }
  return NULL;
}


int
cdk_keydb_is_secret (int id)
{
  int i;

  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (keydb_table[i].used && id == i)
	return keydb_table[i].secret;
    }

  return 0;
}


int
cdk_keydb_find_idx (int is_secret, int *ret_pos)
{
  int i;

  if (!ret_pos)
    return CDK_Inv_Value;

  *ret_pos = -1;
  for (i = 0; i < KEYDB_ENTRIES; i++)
    {
      if (!keydb_table[i].used)
	continue;
      if (is_secret == cdk_keydb_is_secret (i))
	{
	  *ret_pos = i;
	  return 0;
	}
    }
  return CDK_General_Error;
}


CDK_KEYDB_HD
cdk_keydb_get_ctx (int is_secret, int id)
{
  int idx = 0;

  if (id == -1)
    {
      int rc = cdk_keydb_find_idx (is_secret, &idx);
      if (rc)
	return NULL;
      return &keydb_table[idx];
    }
  if (id >= 0 && id < KEYDB_ENTRIES)
    {
      if (keydb_table[id].used && keydb_table[id].secret == is_secret)
	return &keydb_table[id];
    }
  return NULL;
}


int
cdk_keydb_new (CDK_KEYDB_HD * r_hd, int type, void * data, size_t count)
{
  CDK_KEYDB_HD hd;
  char * name;

  hd = cdk_calloc (1, sizeof *hd);
  if (!hd)
    return CDK_Out_Of_Core;
  
  hd->used = 1;
  switch (type)
    {
    case CDK_DBTYPE_ARMORED:
    case CDK_DBTYPE_KEYRING:
      name = data;
      name[count] = '\0';
      hd->name = strdup (name);
      if (!hd->name)
        {
          cdk_free (hd);
          return CDK_Out_Of_Core;
        }
      hd->type = type;
      break;
      
    case CDK_DBTYPE_DATA:
      hd->buf = cdk_stream_tmp_from_mem (data, count);      
      if (!hd->buf)
        {
          cdk_free (hd);
          return CDK_Out_Of_Core;
        }
      hd->type = CDK_DBTYPE_DATA;
      break;
    }
  if (r_hd)
    *r_hd = hd;
  return 0;
}


int
cdk_keydb_open (CDK_KEYDB_HD kdb, CDK_STREAM * ret_kr)
{
  int rc = 0;
  CDK_STREAM kr = NULL;

  if (!kdb)
    return CDK_Inv_Value;

  if (kdb->type == CDK_DBTYPE_DATA && kdb->buf)
    {
      cdk_stream_seek (kdb->buf, 0);
      kr = kdb->buf;
    }
  else if (kdb->name && (kdb->type == CDK_DBTYPE_KEYRING
                         || kdb->type == CDK_DBTYPE_ARMORED))
    {
      rc = cdk_stream_open (kdb->name, &kr);
      if (rc)
	goto leave;
      if (kdb->type == CDK_DBTYPE_ARMORED)
        cdk_stream_set_armor_flag (kr, 0);
    }
  if (!rc)
    cdk_stream_seek (kr, kdb->offset);
 leave:
  if (rc)
    {
      cdk_stream_close (kr);
      kr = NULL;
    }
  if (ret_kr)
    *ret_kr = kr;
  return rc;
}


static int
find_by_keyid (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  u32 keyid[2];
  int key_found = 0;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || node->pkt->pkttype == CDK_PKT_SECRET_KEY
          || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
        {
          _cdk_keyid_get (node->pkt, keyid);
          switch (ks->type)
            {
            case CDK_DBSEARCH_SHORT_KEYID:
              if (keyid[1] == ks->u.keyid[1])
                {
                  key_found = 1;
                  break;
                }
              break;

            case CDK_DBSEARCH_KEYID:
              if (keyid[1] == ks->u.keyid[1] && keyid[0] == ks->u.keyid[0])
                {
                  key_found = 1;
                  break;
                }
              break;
            }
        }
    }
  return key_found;
}


static int
find_by_fpr (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  int key_found = 0;
  byte fpr[20];
  
  if (ks->type != CDK_DBSEARCH_FPR)
    return key_found;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || node->pkt->pkttype == CDK_PKT_SECRET_KEY
          || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
        {
          _cdk_fingerprint_get (node->pkt, fpr);
          if (!memcmp (ks->u.fpr, fpr, 20))
            {
              key_found = 1;
              break;
            }
        }
    }
  return key_found;
}


static int
find_by_pattern (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  size_t uidlen;
  char *name;
  int key_found = 0;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype != CDK_PKT_USER_ID)
        continue;
      uidlen = node->pkt->pkt.user_id->len;
      name = node->pkt->pkt.user_id->name;
      switch (ks->type)
        {
        case CDK_DBSEARCH_EXACT:
          if (name
              && (strlen (ks->u.pattern) == uidlen
              || !strncmp (ks->u.pattern, name, uidlen)))
            {
              key_found = 1;
              break;
            }
          break;

        case CDK_DBSEARCH_SUBSTR:
          if (uidlen > 65536)
            break;
          if (name && strlen (ks->u.pattern) > uidlen)
            break;
          if (name && _cdk_memistr (name, uidlen, ks->u.pattern))
            {
              key_found = 1;
              break;
            }
          break;
        }
    }
  return key_found;
}


void
cdk_keydb_search_reset (CDK_KEYDB_HD hd)
{
  if (hd)
    {
      hd->old_offset = 0;
      hd->offset = 0;
    }
}


int
cdk_keydb_search_new (CDK_DBSEARCH * r_dbs, int type, void * desc)
{
  CDK_DBSEARCH c;
  u32 * keyid;

  c = cdk_calloc (1, sizeof *c);
  if (!c)
    return CDK_Out_Of_Core;
  c->type = type;
  switch (type)
    {
    case CDK_DBSEARCH_EXACT:
    case CDK_DBSEARCH_SUBSTR:
      c->u.pattern = desc;
      break;

    case CDK_DBSEARCH_SHORT_KEYID:
      keyid = desc;
      c->u.keyid[1] = keyid[0];
      break;
      
    case CDK_DBSEARCH_KEYID:
      keyid = desc;
      c->u.keyid[0] = keyid[0];
      c->u.keyid[1] = keyid[1];
      break;

    case CDK_DBSEARCH_FPR:
      memcpy (c->u.fpr, desc, 20);
      break;

    default:
      cdk_free (c);
      return CDK_Inv_Mode;
    }
  if (r_dbs)
    *r_dbs = c;
  return 0;
}


void
cdk_keydb_search_free (CDK_DBSEARCH dbs)
{
  if (dbs)
    {
      dbs->type = 0;
      cdk_free (dbs);
    }
}


int
cdk_keydb_search (CDK_KEYDB_HD kdb, CDK_DBSEARCH ks, CDK_KBNODE * r_key)
{
  CDK_STREAM kr = NULL;
  CDK_KBNODE knode = NULL;
  int key_found = 0;
  int rc = 0;
  size_t pos = 0;

  if (!kdb || !ks)
    return CDK_Inv_Value;

  if (r_key)
    *r_key = NULL;

  rc = cdk_keydb_open (kdb, &kr);
  if (rc)
    return rc;

  while (!key_found && !rc)
    {
      pos = cdk_stream_tell (kr);
      kdb->old_offset = pos;
      rc = cdk_keydb_get_keyblock (kr, &knode);
      if (rc)
        {
          if (rc == CDK_EOF && knode)
            rc = 0;
          if (rc)
            break;
        }
      if (!knode && rc == CDK_EOF)
        rc = CDK_Error_No_Key;
      if (rc)
	break;
      pos = cdk_stream_tell (kr);
      kdb->offset = pos;

      switch (ks->type)
        {
        case CDK_DBSEARCH_SHORT_KEYID:
        case CDK_DBSEARCH_KEYID:
          key_found = find_by_keyid (knode, ks);
          break;

        case CDK_DBSEARCH_FPR:
          key_found = find_by_fpr (knode, ks);
          break;
          
	case CDK_DBSEARCH_EXACT:
	case CDK_DBSEARCH_SUBSTR:
          key_found = find_by_pattern (knode, ks);
	  break;
	}
      if (key_found)
        break;

      cdk_kbnode_release (knode);
      knode = NULL;
    }

  cdk_stream_close (kr);
  if (r_key)
    *r_key = key_found? knode : NULL;
  return rc;
}


int
cdk_keydb_get_bykeyid (CDK_KEYDB_HD kdb, u32 * keyid, CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!kdb || !keyid || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_SHORT_KEYID;
  ks.u.keyid[1] = keyid[1];

  return cdk_keydb_search (kdb, &ks, ret_pk);
}


int
cdk_keydb_get_byfpr (CDK_KEYDB_HD kdb, const byte * fpr, CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!kdb || !fpr || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_FPR;
  memcpy (ks.u.fpr, fpr, 20);

  return cdk_keydb_search (kdb, &ks, ret_pk);
}


int
cdk_keydb_get_bypattern (CDK_KEYDB_HD kdb, const char *patt,
			 CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!kdb || !patt || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_SUBSTR;
  ks.u.pattern = patt;

  return cdk_keydb_search (kdb, &ks, ret_pk);
}


static int
keydb_check_key (CDK_PACKET * pkt)
{
  cdkPKT_public_key * pk;
  
  if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
      || pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
    pk = pkt->pkt.public_key;
  else if (pkt->pkttype == CDK_PKT_SECRET_KEY
      || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
    pk = pkt->pkt.secret_key->pk;
  else
    return 0;
  return !pk->is_revoked && !pk->has_expired;
}


static CDK_KBNODE
keydb_find_byusage (CDK_KBNODE root, int req_usage, int pk)
{
  CDK_KBNODE node;
  int pkttype = 0;

  if (!req_usage)
    return cdk_kbnode_find (root, CDK_PKT_PUBLIC_KEY);

  for (node = root; node; node = node->next)
    {
      pkttype = node->pkt->pkttype;
      if (pk && (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
                 || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
          && keydb_check_key (node->pkt)
          && (node->pkt->pkt.public_key->pubkey_usage & req_usage))
	return node;
      if (!pk && (node->pkt->pkttype == CDK_PKT_SECRET_KEY
                  || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
          && keydb_check_key (node->pkt)
          && (node->pkt->pkt.secret_key->req_usage & req_usage))
	return node;
    }
  return NULL;
}


CDK_KBNODE
_cdk_keydb_find_bykeyid (CDK_KBNODE root, u32 * keyid)
{
  CDK_KBNODE node;
  u32 kid[2];

  for (node = root; node; node = node->next)
    {
      _cdk_keyid_get (node->pkt, kid);
      if (kid[1] == keyid[1])
	return node;
    }
  return NULL;
}


int
cdk_keydb_get_sk_byname (const char * name, cdkPKT_secret_key ** ret_sk,
			 int alloced)
{
  CDK_KBNODE knode = NULL, node = NULL;
  cdkPKT_secret_key * sk = NULL;
  CDK_KEYDB_HD hd;
  int req_usage = 0;
  int rc = 0;

  if (!ret_sk)
    return CDK_Inv_Value;

  hd = cdk_keydb_get_ctx (1, -1);
  if (!hd)
    return CDK_Error_No_Keyring;
  rc = cdk_keydb_search_auto (hd, name, &knode);
  if (rc)
    goto leave;
  if (alloced)
    {
      req_usage = ret_sk[0]->req_usage;
      cdk_free (ret_sk[0]);
      node = keydb_find_byusage (knode, req_usage, 0);
    }
  if (!node)
    {
      node = cdk_kbnode_find (knode, CDK_PKT_SECRET_KEY);
      if (!node)
	{
	  rc = CDK_Error_No_Key;
	  goto leave;
	}
    }
  sk = node->pkt->pkt.secret_key;
  _cdk_kbnode_clone (node);
  cdk_kbnode_release (knode);

leave:
  cdk_keydb_search_reset (hd);
  *ret_sk = sk;
  return rc;
}


int
cdk_keydb_get_pk_byname (const char * name, cdkPKT_public_key ** ret_pk,
			 int alloced)
{
  CDK_KBNODE knode, node = NULL;
  cdkPKT_public_key *pk = NULL;
  CDK_KEYDB_HD hd;
  const char *s;
  int req_usage = 0;
  int rc = 0;

  if (!name)
    return CDK_Inv_Value;

  hd = cdk_keydb_get_ctx (0, -1);
  if (!hd)
    return CDK_Error_No_Keyring;
  rc = cdk_keydb_search_auto (hd, name, &knode);
  if (rc)
    goto leave;
  if (alloced)
    {
      req_usage = ret_pk[0]->pubkey_usage;
      cdk_free (ret_pk[0]);
      node = keydb_find_byusage (knode, req_usage, 1);
    }
  if (!node)
    {
      node = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
      if (!node)
	{
	  rc = CDK_Error_No_Key;
	  goto leave;
	}
    }
  _cdk_copy_pubkey (&pk, node->pkt->pkt.public_key);
  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_USER_ID)
	{
	  s = node->pkt->pkt.user_id->name;
	  if (pk && !pk->uid && _cdk_memistr (s, strlen (s), name))
	    {
	      _cdk_copy_userid (&pk->uid, node->pkt->pkt.user_id);
	      break;
	    }
	}
    }
  cdk_kbnode_release (knode);

leave:
  cdk_keydb_search_reset (hd);
  if (ret_pk)
    *ret_pk = pk;
  return rc;
}


int
cdk_keydb_get_pk (CDK_KEYDB_HD hd, u32 * keyid, cdkPKT_public_key ** r_pk)
{
  struct cdk_keydb_search_s ks;
  CDK_KBNODE key = NULL, node = NULL;
  CDK_KEYDB_HD keydb_hd = NULL;
  cdkPKT_public_key *pk = NULL;
  int rc = 0;

  if (!keyid || !r_pk)
    return CDK_Inv_Value;

  if (hd == NULL)
    {
      keydb_hd = cdk_keydb_get_ctx (0, -1);
      if (!keydb_hd)
	return CDK_Error_No_Keyring;
    }
  else
    keydb_hd = hd;

  if (!keyid[0])
    {
      ks.type = CDK_DBSEARCH_SHORT_KEYID;
      ks.u.keyid[1] = keyid[1];
    }
  else
    {
      ks.type = CDK_DBSEARCH_KEYID;
      ks.u.keyid[0] = keyid[0];
      ks.u.keyid[1] = keyid[1];
    }
  rc = cdk_keydb_search (keydb_hd, &ks, &key);
  if (rc)
    goto leave;
  node = _cdk_keydb_find_bykeyid (key, keyid);
  if (!node)
    {
      rc = CDK_Error_No_Key;
      goto leave;
    }
  _cdk_copy_pubkey (&pk, node->pkt->pkt.public_key);
  cdk_kbnode_release (key);

leave:
  *r_pk = pk;
  return rc;
}


int
cdk_keydb_get_sk (CDK_KEYDB_HD hd, u32 * keyid, cdkPKT_secret_key ** ret_sk)
{
  CDK_KEYDB_HD khd;
  CDK_KBNODE snode, node;
  cdkPKT_secret_key *sk = NULL;
  int rc = 0;

  if (!hd)
    {
      khd = cdk_keydb_get_ctx (1, -1);
      if (!khd)
	return CDK_Error_No_Keyring;
    }
  else
    khd = hd;
  rc = cdk_keydb_get_bykeyid (hd, keyid, &snode);
  if (rc)
    goto leave;

  node = _cdk_keydb_find_bykeyid (snode, keyid);
  if (!node)
    {
      rc = CDK_Error_No_Key;
      goto leave;
    }

  sk = node->pkt->pkt.secret_key;
  _cdk_kbnode_clone (node);
  cdk_kbnode_release (snode);

leave:
  *ret_sk = sk;
  return rc;
}


static int
keydb_merge_selfsig (CDK_KBNODE key)
{
  CDK_KBNODE k, kbnode, unode;
  struct cdk_subpkt_s *s = NULL;
  cdkPKT_signature *sig = NULL;
  cdkPKT_user_id *uid = NULL;
  const byte *symalg = NULL, *hashalg = NULL, *compalg = NULL;
  size_t nsymalg = 0, nhashalg = 0, ncompalg = 0, n = 0;
  int key_usage = 0, key_expire = 0;

  if (!key)
    return CDK_Inv_Value;

  for (k = key; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE
	  && k->pkt->pkt.signature->sig_class == 0x13)
	{
	  unode = cdk_kbnode_find_prev (key, k, CDK_PKT_USER_ID);
	  if (!unode)
	    return CDK_Error_No_Key;
	  uid = unode->pkt->pkt.user_id;
	  sig = k->pkt->pkt.signature;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PRIMARY_UID);
	  if (s)
	    uid->is_primary = 1;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_FEATURES);
	  if (s && s->size == 1 && s->d[0] & 0x01)
	    uid->mdc_feature = 1;
          s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_KEY_EXPIRE);
          if (s && s->size == 4)
            key_expire = _cdk_buftou32 (s->d);
          s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_KEY_FLAGS);
          if (s)
            {
              if (s->d[0] & 3) /* cert + sign data */
                key_usage |= GCRY_PK_USAGE_SIGN;
              if (s->d[0] & 12) /* encrypt comm. + storage */
                key_usage |= GCRY_PK_USAGE_ENCR;
            }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_SYM);
	  if (s)
	    {
	      symalg = s->d;
	      nsymalg = s->size;
	      n += s->size + 1;
	    }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_HASH);
	  if (s)
	    {
	      hashalg = s->d;
	      nhashalg = s->size;
	      n += s->size + 1;
	    }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_ZIP);
	  if (s)
	    {
	      compalg = s->d;
	      ncompalg = s->size;
	      n += s->size + 1;
	    }
	  if (!n || !hashalg || !compalg || !symalg)
	    uid->prefs = NULL;
	  else
	    {
	      uid->prefs = cdk_calloc (1, sizeof (*uid->prefs) * (n + 1));
	      if (!uid->prefs)
		return CDK_Out_Of_Core;
	      n = 0;
	      for (; nsymalg; nsymalg--, n++)
		{
		  uid->prefs[n].type = CDK_PREFTYPE_SYM;
		  uid->prefs[n].value = *symalg++;
		}
	      for (; nhashalg; nhashalg--, n++)
		{
		  uid->prefs[n].type = CDK_PREFTYPE_HASH;
		  uid->prefs[n].value = *hashalg++;
		}
	      for (; ncompalg; ncompalg--, n++)
		{
		  uid->prefs[n].type = CDK_PREFTYPE_ZIP;
		  uid->prefs[n].value = *compalg++;
		}
	      /* end of list marker */
	      uid->prefs[n].type = CDK_PREFTYPE_NONE;
	      uid->prefs[n].value = 0;
	      uid->prefs_size = n;

	      kbnode = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_KEY);
	      if (kbnode)
		{
		  cdkPKT_public_key * pk = kbnode->pkt->pkt.public_key;
                  if (uid->prefs && n)
                    {
                      pk->prefs = _cdk_copy_prefs (uid->prefs);
                      pk->prefs_size = n;
                    }
                  if (key_expire)
                    {
                      pk->expiredate = pk->timestamp + key_expire;
                      pk->has_expired = pk->expiredate> _cdk_timestamp ()?0 :1;
                    } 
                  if (key_usage && !pk->pubkey_usage)
                    pk->pubkey_usage = key_usage;
                }
            }
        }
    }
  return 0;
}


static int
keydb_parse_allsigs (CDK_KBNODE knode, int check)
{
  CDK_KBNODE k, kb;
  cdkPKT_signature * sig;
  cdkPKT_public_key * pk;
  struct cdk_subpkt_s * s = NULL;
  u32 expiredate = 0, curtime = _cdk_timestamp ();
  int rc = 0;

  if (!knode)
    return CDK_Inv_Value;

  kb = cdk_kbnode_find (knode, CDK_PKT_SECRET_KEY);
  if (kb)
    return 0;

  /* reset */
  for (k = knode; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_USER_ID)
	k->pkt->pkt.user_id->is_revoked = 0;
      else if (k->pkt->pkttype == CDK_PKT_PUBLIC_KEY
               || k->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
	k->pkt->pkt.public_key->is_revoked = 0;
    }

  for (k = knode; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  sig = k->pkt->pkt.signature;
	  /* Revocation certificates for primary keys */
	  if (sig->sig_class == 0x20)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_KEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Revocation certificates for subkeys */
	  else if (sig->sig_class == 0x28)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_SUBKEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Revocation certifcates for user ID's */
	  else if (sig->sig_class == 0x30)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_USER_ID);
	      if (kb)
		{
		  kb->pkt->pkt.user_id->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Direct certificates for primary keys */
	  else if (sig->sig_class == 0x1F)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_KEY);
	      if (kb)
		{
                  pk = kb->pkt->pkt.public_key;
		  pk->is_invalid = 0;
		  s = cdk_subpkt_find (k->pkt->pkt.signature->hashed,
				       SIGSUBPKT_KEY_EXPIRE);
		  if (s)
		    {
		      expiredate = _cdk_buftou32 (s->d);
                      pk->expiredate = pk->timestamp + expiredate;
                      pk->has_expired = pk->expiredate > curtime? 0 : 1;
		    }
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Direct certificates for subkeys */
	  else if (sig->sig_class == 0x18)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_SUBKEY);
	      if (kb)
		{
                  pk = kb->pkt->pkt.public_key;
		  pk->is_invalid = 0;
		  s = cdk_subpkt_find (k->pkt->pkt.signature->hashed,
				       SIGSUBPKT_KEY_EXPIRE);
		  if (s)
		    {
		      expiredate = _cdk_buftou32 (s->d);
                      pk->expiredate = pk->timestamp + expiredate;
                      pk->has_expired = pk->expiredate > curtime? 0 : 1;
		    }
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	}
    }
  k = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
  if (k) /* v3 public keys have no additional signatures for the key */
    {
      pk = k->pkt->pkt.public_key;
      if (pk->version == 3)
        pk->is_invalid = 0;
    }
  return rc;
}


int
cdk_keydb_get_keyblock (CDK_STREAM inp, CDK_KBNODE * r_knode)
{
  CDK_PACKET *pkt = NULL;
  CDK_KBNODE knode = NULL, node = NULL;
  u32 keyid[2], main_keyid[2];
  int rc = 0, rc2 = 0, old_off;
  int key_seen = 0, got_key = 0;

  if (!inp)
    return CDK_Inv_Value;

  memset (keyid, 0, sizeof keyid);
  memset (main_keyid, 0, sizeof main_keyid);
  
  while (1)
    {
      pkt = cdk_calloc (1, sizeof *pkt);
      if (!pkt)
	return CDK_Out_Of_Core;
      old_off = cdk_stream_tell (inp);
      rc = cdk_pkt_parse (inp, pkt);
      if (rc)
	break;
      if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || pkt->pkttype == CDK_PKT_SECRET_KEY
          || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
	{
          if (key_seen && (pkt->pkttype == CDK_PKT_PUBLIC_KEY
                            || pkt->pkttype == CDK_PKT_SECRET_KEY))
	    {
              cdk_stream_seek (inp, old_off);
	      break;
	    }
          if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
              || pkt->pkttype == CDK_PKT_SECRET_KEY)
            {
              _cdk_keyid_get (pkt, main_keyid);
              key_seen = 1;
            }
          else if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
                   || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
            {
              if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
		{
		  pkt->pkt.public_key->main_keyid[0] = main_keyid[0];
		  pkt->pkt.public_key->main_keyid[1] = main_keyid[1];
		}
	      else
		{
		  pkt->pkt.secret_key->main_keyid[0] = main_keyid[0];
		  pkt->pkt.secret_key->main_keyid[1] = main_keyid[1];
		}
	    }
	  /* we save this for the signature */
          _cdk_keyid_get (pkt, keyid);
          got_key = 1;
	}
      else if (pkt->pkttype == CDK_PKT_USER_ID)
        ;
      else if (pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  pkt->pkt.signature->key[0] = keyid[0];
	  pkt->pkt.signature->key[1] = keyid[1];
	}
      node = cdk_kbnode_new (pkt);
      if (!knode)
	knode = node;
      else
	_cdk_kbnode_add (knode, node);
    }

  if (got_key)
    {
      keydb_merge_selfsig (knode);
      rc2 = keydb_parse_allsigs (knode, 0);
      if (!rc)
	rc = rc2;
    }
  if (r_knode)
    *r_knode = got_key ? knode : NULL;
  return rc;
}


static int
classify_data (const byte * buf, size_t len)
{
  int type = 0;
  int i;

  if (buf[0] == '0' && buf[1] == 'x')
    {
      buf += 2;
      len -= 2;
    }
  
  if (len == 8 || len == 16 || len == 40)
    {
      for (i = 0; i < len; i++)
        {
          if (!isxdigit (buf[i]))
            break;
        }
      if (i == len)
        {
          switch (len)
            {
            case 8: type = CDK_DBSEARCH_SHORT_KEYID; break;
            case 16: type = CDK_DBSEARCH_KEYID; break;
            case 40: type = CDK_DBSEARCH_FPR; break;
            }
        }
    }
  if (!type)
    type = CDK_DBSEARCH_SUBSTR;
  return type;
}


static void
keyid_from_str (const char * buf, u32 * keyid)
{
  if (!strncmp (buf, "0x", 2))
    buf += 2;

  if (strlen (buf) == 8)
    keyid[1] = strtoul (buf, NULL, 16);
  else if (strlen (buf) == 16)
    {
      keyid[0] = strtoul (buf, NULL, 16);
      keyid[1] = strtoul (buf + 8, NULL, 16);
    }
}


static void
fpr_from_str (const char * buf, byte * fpr)
{
  char tmp[3];
  int i;
  
  if (strlen (buf) != 40)
    memset (fpr, 0, 20);
  else
    {
      for (i = 0; i < 20; i++)
        {
          tmp[0] = buf[2*i];
          tmp[1] = buf[2*i+1];
          tmp[2] = '\0';
          fpr[i] = strtoul (tmp, NULL, 16);
        }
    }
}
      

int
cdk_keydb_search_auto (CDK_KEYDB_HD hd, const char *desc, CDK_KBNODE * r_knode)
{
  u32 keyid[2];
  byte fpr[20];
  int rc;

  switch (classify_data (desc, strlen (desc)))
    {
    case CDK_DBSEARCH_EXACT:
    case CDK_DBSEARCH_SUBSTR:
      rc = cdk_keydb_get_bypattern (hd, desc, r_knode);
      break;
      
    case CDK_DBSEARCH_KEYID:
    case CDK_DBSEARCH_SHORT_KEYID:
      keyid_from_str (desc, keyid);
      rc = cdk_keydb_get_bykeyid (hd, keyid, r_knode);
      break;

    case CDK_DBSEARCH_FPR:
      fpr_from_str (desc, fpr);
      rc = cdk_keydb_get_byfpr (hd, fpr, r_knode);
      break;

    default:
      rc = CDK_Inv_Mode;
    }
  return rc;
}


int
cdk_keydb_export (CDK_STREAM out, CDK_STRLIST remusr, int secret)
{
  CDK_KEYDB_HD hd;
  CDK_KBNODE knode, node;
  CDK_STRLIST r;
  int old_ctb = 0;
  int rc = 0;

  hd = cdk_keydb_get_ctx (secret, -1);
  if (!hd)
    return CDK_Error_No_Keyring;

  for (r = remusr; r; r = r->next)
    {
      rc = cdk_keydb_search_auto (hd, r->d, &knode);
      if (rc)
	break;
      for (node = knode; node; node = node->next)
	{
	  /* those packets are not intended for the real wolrd */
	  if (node->pkt->pkttype == CDK_PKT_RING_TRUST)
	    continue;
          if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
              && node->pkt->pkt.public_key->version == 3)
            old_ctb = 1;
	  /* we never export local signed signatures */
	  if (node->pkt->pkttype == CDK_PKT_SIGNATURE &&
	      !node->pkt->pkt.signature->flags.exportable)
            continue;
          node->pkt->old_ctb = old_ctb;
	  rc = cdk_pkt_build (out, node->pkt);
	  if (rc)
	    break;
	}
      cdk_kbnode_release (knode);
      knode = NULL;
    }

  return rc;
}


static CDK_PACKET *
find_key_packet (CDK_KBNODE knode, int * r_secret)
{
  CDK_PACKET * pkt;

  pkt = cdk_kbnode_find_packet (knode, CDK_PKT_PUBLIC_KEY);
  if (!pkt)
    {
      pkt = cdk_kbnode_find_packet (knode, CDK_PKT_SECRET_KEY);
      *r_secret = pkt? 1 : 0;
    }
  return pkt;
}


int
cdk_keydb_import (CDK_KEYDB_HD hd, CDK_KBNODE knode, int result[4])
{
  CDK_KBNODE node, chk = NULL;
  CDK_PACKET * pkt;
  CDK_STREAM out;
  u32 keyid[2];
  int rc = 0, is_sk = 0;

  memset (result, 0, 4 * sizeof (int));
  pkt = find_key_packet (knode, &is_sk);
  if (!pkt)
      return CDK_Inv_Packet;
  result[is_sk] = 1;
  cdk_pk_get_keyid (pkt->pkt.public_key, keyid);
  cdk_keydb_get_bykeyid (hd, keyid, &chk);
  if (chk)
    {
      cdk_kbnode_release (chk);
      return 0;
    }
  
  cdk_keydb_search_reset (hd);
  if (hd->buf)
    return CDK_Inv_Mode;
  rc = _cdk_stream_append (hd->name, &out);
  if (rc)
    return rc;
  
  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_RING_TRUST)
        continue; /* No uniformed syntax for this packet */
      rc = cdk_pkt_build (out, node->pkt);
      if (rc)
        break;
    }
  if (!rc)
    result[is_sk? 3 : 2] = 1;
  cdk_stream_close (out);
  return rc;
}  


/* internal functions */
CDK_KBNODE
_cdk_keydb_get_pkblock (u32 * keyid)
{
  CDK_KEYDB_HD hd;
  CDK_KBNODE kbnode = NULL;
  int rc = 0;

  hd = cdk_keydb_get_ctx (0, -1);
  if (hd)
    {
      rc = cdk_keydb_get_bykeyid (hd, keyid, &kbnode);
      if (rc)
        kbnode = NULL;
      cdk_keydb_search_reset (hd);
    }
  return kbnode;
}


CDK_KBNODE
_cdk_keydb_get_skblock (u32 * keyid)
{
  CDK_KBNODE snode = NULL;
  CDK_KEYDB_HD hd;
  int rc = 0;

  hd = cdk_keydb_get_ctx (1, -1);
  if (hd)
    {
      rc = cdk_keydb_get_bykeyid (hd, keyid, &snode);
      if (rc)
        snode = NULL;
      cdk_keydb_search_reset (hd);
    }
  return snode;
}


