/* keylist.c - Linked key lists
 *        Copyright (C) 2002 Timo Schulz
 *        Copyright (C) 1998-2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"


static int
pklist_key_is_present (CDK_KEYLIST pkl, cdkPKT_public_key * pk)
{
  for (; pkl; pkl = pkl->next)
    {
      if (_cdk_pubkey_compare (pkl->key.pk, pk) == 0)
	return 0;
    }
  return -1;
}


static int
pklist_get_pk (const char * remusr, cdkPKT_public_key ** ret_pk)
{
  if (!remusr || !ret_pk)
    return CDK_Inv_Value;
  return cdk_keydb_get_pk_byname (remusr, ret_pk, 1);
}


/* Here we check if *all* keys have the MDC feature. Even if one
   key doesn't support it, it is not used. */
static int
pklist_select_mdc (CDK_KEYLIST pkl)
{
  CDK_KEYLIST pkr;
  int mdc = 0;

  if (!pkl)
    return CDK_Inv_Value;

  for (pkr = pkl; pkr; pkr = pkr->next)
    {
      if (pkr->key.pk->uid) /* selected by user ID */
	mdc = pkr->key.pk->uid->mdc_feature;
      if (!mdc)
	return 0;
    }
  return 1;
}


static int
algo_available (int preftype, int algo)
{
  if (preftype == CDK_PREFTYPE_SYM)
    return algo && !_cdk_cipher_test_algo (algo);
  else if (preftype == CDK_PREFTYPE_HASH)
    return algo && !_cdk_md_test_algo (algo);
  else if (preftype == CDK_PREFTYPE_ZIP)
    return !algo || algo == 1 || algo == 2;
  else
    return 0;
}


static int
pklist_select_algo (CDK_KEYLIST pkl, int preftype)
{
  const struct cdk_prefitem_s * prefs;
  CDK_KEYLIST pkr;
  u32 bits[8];
  int compr_hack = 0, any = 0;
  int i = 0, j = 0;

  if (!pkl)
    return -1;

  memset (bits, ~0, 8 * sizeof *bits);
  for (pkr = pkl; pkr; pkr = pkr->next)
    {
      u32 mask[8];
      if (preftype == CDK_PREFTYPE_SYM)
	{
	  memset (mask, 0, 8 * sizeof *mask);
	  mask[0] |= (1 << 2); /* 3DES is implicitly there for everyone else */
	}
      if (pkr->key.pk->uid)
	prefs = pkr->key.pk->uid->prefs;
      else
	prefs = pkr->key.pk->prefs;
      any = 0;
      for (i = 0; prefs && prefs[i].type; i++)
	{
	  if (prefs[i].type == preftype)
	    {
	      mask[prefs[i].value / 32] |= 1 << (prefs[i].value % 32);
	      any = 1;
	    }
	}
      if ((!prefs || !any) && preftype == CDK_PREFTYPE_ZIP)
	{
	  mask[0] |= 3; /* asume no_compression and old pgp */
	  compr_hack = 1;
	}
      for (i = 0; i < 8; i++)
	bits[i] &= mask[i];
      /* Usable algorithms are now in bits:
         We now use the last key from pkl to select the algorithm we want
         to use. There are no preferences for the last key, we select the one
         corresponding to first set bit. */
      i = -1;
      any = 0;
      for (j = 0; prefs && prefs[j].type; j++)
	{
	  if (prefs[j].type == preftype)
	    {
	      if ((bits[prefs[j].value / 32] & (1 << (prefs[j].value % 32))))
		{
		  if (algo_available (preftype, prefs[j].value))
		    {
		      any = 1;
		      i = prefs[j].value;
		      break;
		    }
		}
	    }
	}
      if (!prefs || !any)
	{
	  for (j = 0; j < 256; j++)
	    if ((bits[j / 32] & (1 << (j % 32))))
	      {
		if (algo_available (preftype, j))
		  {
		    i = j;
		    break;
		  }
	      }
	}
      if (compr_hack && !i)
	{
	  /* selected no compression, but we should check whether
	     algorithm 1 is also available (the ordering is not relevant
	     in this case). */
	  if (bits[0] & (1 << 1))
	    i = 1; /* yep; we can use compression algo 1 */
	}
    }

  return i;
}


static int
is_duplicated_entry (CDK_STRLIST list, CDK_STRLIST item)
{
  for (; list && list != item; list = list->next)
    {
      if (!strcmp (list->d, item->d))
	return 1;
    }

  return 0;
}


int
cdk_pklist_select_algo (CDK_KEYLIST pkl)
{
  int algo = 0;

  if (!pkl)
    return GCRY_CIPHER_CAST5;

  algo = pklist_select_algo (pkl, CDK_PREFTYPE_SYM);
  return algo == -1? GCRY_CIPHER_CAST5 : algo;
}


int
cdk_pklist_use_mdc (CDK_KEYLIST pkl)
{
  return pkl? pklist_select_mdc (pkl) : 0;
}


void
cdk_pklist_release (CDK_KEYLIST pkl)
{
  CDK_KEYLIST pkr;

  for (; pkl; pkl = pkr)
    {
      pkr = pkl->next;
      _cdk_free_pubkey (pkl->key.pk);
      pkl->key.pk = NULL;
      cdk_free (pkl);
    }
}


int
cdk_pklist_build (CDK_STRLIST remusr, CDK_KEYLIST * ret_pkl, int use)
{
  CDK_KEYLIST pkl = NULL, r = NULL;
  cdkPKT_public_key *pk = NULL;
  int rc = 0;

  for (; remusr; remusr = remusr->next)
    {
      pk = cdk_calloc (1, sizeof *pk);
      if (!pk)
	{
	  return CDK_Out_Of_Core;
	  goto leave;
	}
      pk->pubkey_usage = use;
      rc = pklist_get_pk (remusr->d, &pk);
      if (rc)
	goto leave;
      else
	{
	  if (pklist_key_is_present (pkl, pk) == 0)
	    {
	      _cdk_free_pubkey (pk);
	      pk = NULL;
	      continue; /* key already in list so skip it */
	    }
	  r = cdk_calloc (1, sizeof *r);
	  if (!r)
	    {
	      rc = CDK_Out_Of_Core;
	      goto leave;
	    }
	  r->type = CDK_PKT_PUBLIC_KEY;
	  r->key.pk = pk;
	  r->next = pkl;
	  pkl = r;
	}
    }
leave:
  if (rc)
    {
      cdk_pklist_release (pkl);
      pkl = NULL;
    }
  *ret_pkl = pkl;
  return rc;
}


int
cdk_pklist_encrypt (CDK_KEYLIST pkl, CDK_DEK dek, CDK_STREAM outp)
{
  cdkPKT_public_key * pk = NULL;
  cdkPKT_pubkey_enc * enc = NULL;
  CDK_PACKET pkt;
  GCRY_MPI frame = NULL;
  int nbits = 0;
  int rc = 0;

  if (!pkl || !dek || !outp)
    return CDK_Inv_Value;

  if (pkl->type != CDK_PKT_PUBLIC_KEY)
    return CDK_Inv_Mode;

  for (; pkl; pkl = pkl->next)
    {
      pk = pkl->key.pk;
      cdk_free (enc);
      enc = cdk_calloc (1, sizeof *enc);
      if (!enc)
	return CDK_Out_Of_Core;
      enc->pubkey_algo = pk->pubkey_algo;
      cdk_pk_get_keyid (pk, enc->keyid);
      nbits = cdk_pk_get_nbits (pk);
      rc = _cdk_pkcs1_sesskey (&frame, dek, nbits);
      if (rc)
        goto fail;
      rc = cdk_pk_encrypt (pk, enc, frame);
      gcry_mpi_release (frame);
      if (rc)
        goto fail;
      else
	{
	  init_packet (&pkt);
          pkt.old_ctb = dek->rfc1991? 1 : 0;
	  pkt.pkttype = CDK_PKT_PUBKEY_ENC;
	  pkt.pkt.pubkey_enc = enc;
	  rc = cdk_pkt_build (outp, &pkt);
          if (rc)
            goto fail;
	}
    }

 fail:
  cdk_free (enc);
  return rc;
}


void
cdk_sklist_release (CDK_KEYLIST skl)
{
  CDK_KEYLIST sk_rover = NULL;

  for (; skl; skl = sk_rover)
    {
      sk_rover = skl->next;
      _cdk_free_seckey (skl->key.sk);
      skl->key.sk = NULL;
      cdk_free (skl);
    }
}


int
cdk_sklist_build (CDK_STRLIST locusr, CDK_KEYLIST * ret_skl, int unlock,
		  unsigned int use)
{
  CDK_KEYLIST r = NULL, skl = NULL;
  cdkPKT_secret_key * sk = NULL;
  int rc = 0;

  if (!ret_skl)
    return CDK_Inv_Value;

  if (!locusr)  /* use the default one */
    {
      sk = cdk_calloc (1, sizeof *sk);
      if (!sk)
	return CDK_Out_Of_Core;
      sk->req_usage = use;
      rc = cdk_keydb_get_sk_byname (NULL, &sk, 1);
      if (rc)
	{
	  _cdk_free_seckey (sk);
	  sk = NULL;
	}
      else
	{
          if (unlock)
            {
              rc = _cdk_seckey_unprotect2 (sk);
              if (rc)
                goto fail;
            }
	  r = cdk_calloc (1, sizeof *r);
	  if (!r)
	    return CDK_Out_Of_Core;
	  r->key.sk = sk;
	  r->next = skl;
	  r->mark = 0;
	  r->type = CDK_PKT_SECRET_KEY;
	  skl = r;
	}
    }
  else
    {
      CDK_STRLIST locusr_orig = locusr;
      for (; locusr; locusr = locusr->next)
	{
	  if (is_duplicated_entry (locusr_orig, locusr))
	    continue;
	  sk = cdk_calloc (1, sizeof *sk);
	  if (!sk)
	    return CDK_Out_Of_Core;
	  sk->req_usage = use;
	  rc = cdk_keydb_get_sk_byname (locusr->d, &sk, 1);
	  if (rc)
	    {
	      _cdk_free_seckey (sk);
	      sk = NULL;
	    }
	  else
	    {
              if (unlock)
                {
                  rc = _cdk_seckey_unprotect2 (sk);
                  if (rc)
                    goto fail; 
                }
	      r = cdk_calloc (1, sizeof *r);
	      if (!r)
		return CDK_Out_Of_Core;
	      r->key.sk = sk;
	      r->next = skl;
	      r->mark = 0;
              r->type = CDK_PKT_SECRET_KEY;
	      skl = r;
	    }
	}
    }

 fail:
  if (rc)
    {
      cdk_sklist_release (skl);
      skl = NULL;
    }
  *ret_skl = skl;
  return rc;
}


int
cdk_sklist_write_onepass (CDK_KEYLIST skl, CDK_STREAM outp, int sigclass)
{
  cdkPKT_onepass_sig *ops;
  CDK_KEYLIST r;
  CDK_PACKET pkt;
  int i, skcount = 0;
  int rc = 0;

  if (!skl || !outp)
    return CDK_Inv_Value;

  for (skcount = 0, r = skl; r; r = r->next)
    skcount++;

  for (; skcount; skcount--)
    {
      for (i = 0, r = skl; r; r = r->next)
	{
	  if (++i == skcount)
	    break;
	}
      ops = cdk_calloc (1, sizeof *ops);
      if (!ops)
	return CDK_Out_Of_Core;
      ops->version = 3;
      cdk_sk_get_keyid (r->key.sk, ops->keyid);
      ops->sig_class = sigclass;
      ops->digest_algo = _cdk_sig_hash_for (r->key.sk->pubkey_algo,
					    r->key.sk->version);
      ops->pubkey_algo = r->key.sk->pubkey_algo;
      ops->last = (skcount == 1);

      init_packet (&pkt);
      pkt.pkttype = CDK_PKT_ONEPASS_SIG;
      pkt.pkt.onepass_sig = ops;
      rc = cdk_pkt_build (outp, &pkt);
      cdk_pkt_free (&pkt);
      if (rc)
	return rc;
    }

  return rc;
}


int
cdk_sklist_write (CDK_KEYLIST skl, CDK_STREAM outp, GCRY_MD_HD hash,
		  int sigclass, int sigver)
{
  CDK_KEYLIST r = NULL;
  cdkPKT_signature *sig = NULL;
  CDK_PACKET pkt;
  GCRY_MD_HD md = NULL;
  byte digest[24]; /* maximal 192-bit for TIGER-192 */
  int rc = 0;

  if (!skl || !outp || !hash)
    return CDK_Inv_Value;

  for (r = skl; r; r = r->next)
    {
      sig = cdk_calloc (1, sizeof *sig);
      if (!sig)
	return CDK_Out_Of_Core;
      sig->version = sigver;
      _cdk_sig_create (r->key.sk->pk, sig);
      sig->sig_class = sigclass;
      md = gcry_md_copy (hash);
      _cdk_hash_sig_data (sig, md);
      gcry_md_final (md);

      memcpy (digest, gcry_md_read (md, sig->digest_algo),
	      gcry_md_get_algo_dlen (sig->digest_algo));
      rc = cdk_pk_sign (r->key.sk, sig, digest);
      if (rc)
        goto fail;
      init_packet (&pkt);
      pkt.old_ctb = sig->version == 3? 1 : 0;
      pkt.pkttype = CDK_PKT_SIGNATURE;
      pkt.pkt.signature = sig;
      rc = cdk_pkt_build (outp, &pkt);
      _cdk_free_signature (sig);
      if (rc)
        goto fail;
      gcry_md_close (md);
      md = NULL;
    }
  
 fail:
  gcry_md_close (md);
  return rc;
}
