/* stream.c - provides a STREAM object
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Cdk_Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "opencdk.h"
#include "filters.h"
#include "stream.h"
#include "types.h"
#include "main.h"


struct stream_filter_table_s filter_table[] = {
  {fDUMMY,     NULL,             NULL},
  {fARMOR,     armor_encode,     armor_decode},
  {fCIPHER,    cipher_encode,    cipher_decode},
  {fPLAINTEXT, plaintext_encode, plaintext_decode},
  {fCOMPRESS,  compress_encode,  compress_decode},
  {fHASH,      hash_filter,      hash_filter},
  {fTEXT,      text_encode,      text_decode},
  {0}  
};

static int stream_filter_walk_write (CDK_STREAM s);
static int stream_cache_flush (CDK_STREAM s, FILE * fp);


#ifndef HAVE_MKSTEMP
int *
mkstemp (char * template)
{
  static int pos = 0;
  char chars[] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

  pos = strlen (template) - 6;
  if (pos < 0)
    return NULL;

  gcry_randomize (template + pos, 6, GCRY_WEAK_RANDOM);
  for (; pos < strlen (template); pos++)
    template[pos] = chars[template[pos] % sizeof chars];
  return &pos;   
}
#endif


static char *
mk_tmpnam (void)
{
  static char template[11];
  
  memset (template, 0, sizeof template);
  strncpy (template, "cdk", 3);
  memset (template + 3, 'X', 6);
  mkstemp (template);
  return template;
}


/**
 * cdk_stream_open: create a new stream based on an existing file.
 * @file: The file to open
 * @ret_s: The new STREAM object
 **/
int
cdk_stream_open (const char * file, CDK_STREAM * ret_s)
{
  CDK_STREAM s;

  if (!file)
    return CDK_Inv_Value;

  _cdk_log_debug ("open stream `%s'\n", file);
  
  s = cdk_calloc (1, sizeof *s);
  if (!s)
    return CDK_Out_Of_Core;
  
  s->fp = fopen (file, "rb");
  if (!s->fp)
    {
      cdk_free (s);
      return CDK_File_Error;
    }
  s->fname = cdk_strdup (file);
  if (!s->fname)
    {
      fclose (s->fp);
      cdk_free (s);
      return CDK_Out_Of_Core;
    }
  s->flags.write = 0;
  if (ret_s)
    *ret_s = s;
  return 0;
}


/**
 * cdk_stream_new: Create a new stream into into the given file.
 * @file: The name of the new file
 * @ret_s: The new STREAM object
 **/
int
cdk_stream_new (const char * file, CDK_STREAM * ret_s)
{
  CDK_STREAM s;

  s = cdk_calloc (1, sizeof *s);
  if (!s)
    return CDK_Out_Of_Core;
  
  s->flags.write = 1;
  if (!file)
    {
      s->flags.temp = 1;
      s->fp = tmpfile ();
      if (!s->fp)
        {
          cdk_free (s);
          return CDK_File_Error;
        }
      if (ret_s)
        *ret_s = s;
      return 0;
    }
  s->fname = cdk_strdup (file);
  if (!s->fname)
    {
      cdk_free (s);
      return CDK_Out_Of_Core;
    }
  s->tname = cdk_strdup (mk_tmpnam ());
  if (!s->tname)
    {
      cdk_free (s->fname);
      cdk_free (s);
      return CDK_Out_Of_Core;
    }
  s->fp = fopen (s->tname, "w+b");
  if (!s->fp)
    {
      cdk_free (s->fname);
      cdk_free (s->tname);
      cdk_free (s);
      return CDK_File_Error;
    }
  if (ret_s)
    *ret_s = s;
  return 0;
}


CDK_STREAM
cdk_stream_tmp (void)
{
  CDK_STREAM s;
  int rc;
  
  rc = cdk_stream_new (NULL, &s);
  if (!rc)
    return s;
  return NULL;
}


CDK_STREAM
cdk_stream_tmp_from_mem (const void * buf, size_t count)
{
  CDK_STREAM s;
  int nwritten;

  s = cdk_stream_tmp ();
  if (!s)
    return NULL;
  
  nwritten = cdk_stream_write (s, buf, count);
  if (nwritten == EOF)
    {
      cdk_stream_close (s);
      return NULL;
    }
  cdk_stream_seek (s, 0);
  return s;
}

  

CDK_STREAM
cdk_stream_fpopen (FILE * fp, unsigned write_mode)
{
  CDK_STREAM s;

  s = cdk_calloc (1, sizeof *s);
  if (!s)
    return NULL;
  
  s->fp = fp;
  s->flags.filtrated = 1;
  s->flags.write = write_mode;

  return s;
}


int
_cdk_stream_append (const char * file, CDK_STREAM * ret_s)
{
  CDK_STREAM s;
  FILE * fp;
  int rc;

  if (ret_s)
    *ret_s = NULL;
  
  rc = cdk_stream_open (file, &s);
  if (rc)
    return rc;
  fp = fopen (file, "a+b");
  if (!fp)
    {
      cdk_stream_close (s);
      return CDK_File_Error;
    }
  fclose (s->fp);
  s->fp = fp;
  s->flags.write = 1;
  if (ret_s)
    *ret_s = s;
  return 0;
}

      

int
cdk_stream_control (CDK_STREAM s, int ctl, int val)
{
  if (!s)
    return CDK_Inv_Value;
  
  switch (ctl)
    {
    case CDK_STREAMCTL_RESET:
      s->flags.reset = val;
      break;

    default:
      return CDK_Inv_Value;
    }
  return 0;
}


int
cdk_stream_flush (CDK_STREAM s)
{
  int rc = 0;
  
  if (!s)
    return CDK_Inv_Value;

  if (!s->flags.filtrated)
    {
      rc = fseek (s->fp, 0, SEEK_SET);
      if (!rc)
        rc = fflush (s->fp);
      if (rc)
        rc = CDK_File_Error;
      if (!rc)
        rc = stream_filter_walk_write (s);
      s->flags.filtrated = 1;
    }
  return rc;
}


void
cdk_stream_tmp_set_mode (CDK_STREAM s, int val)
{
  if (s && s->flags.temp)
    s->fmode = val;
}


/**
 * cdk_stream_close: Close a stream and flush all buffers.
 * @s: The STREAM object.
 *
 * This function work different for read or write streams. When the
 * stream is for reading, the filtering is already done and we can
 * simply close the file and all buffers.
 * But for the case it's a write stream, we need to apply all registered
 * filters now. The file is closed in the filter function and not here.
 **/
int
cdk_stream_close (CDK_STREAM s)
{
  struct stream_filter_s * f, * f2;
  int rc = 0;

  if (s)
    {
      _cdk_log_debug ("close stream `%s'\n", s->fname? s->fname : "[temp]");
      if (!s->flags.filtrated)
        rc = cdk_stream_flush (s);
      else if (s->fname || s->flags.temp)
        {
          rc = fclose (s->fp);
          if (rc)
            rc = CDK_File_Error;
        }
      f = s->filters;
      while (f)
        {
          f2 = f->next;
          cdk_free (f);
          f = f2;
        }
      if (s->fname)
        cdk_free (s->fname);
      if (s->tname)
        {
          unlink (s->tname);
          cdk_free (s->tname);
        }
      cdk_free (s);
    }
  return rc;
}


/**
 * cdk_stream_eof: Return if the associated file handle was set to EOF.
 * @s: The STREAM object.
 *
 * This function will only work with read streams.
 **/
int
cdk_stream_eof (CDK_STREAM s)
{
  return s->flags.eof;
}


const char *
_cdk_stream_get_fname (CDK_STREAM s)
{
  return s->fname;
}


FILE *
_cdk_stream_get_fp (CDK_STREAM s)
{
  return s->fp;
}


int
_cdk_stream_get_errno (CDK_STREAM s)
{
  return s->error;
}


/**
 * cdk_stream_get_length: Return the length of the associated file handle.
 * @s: The STREAM object.
 *
 * This file only works for read stream because it's likely that the
 * write stream is not flushed or even no data was inserted.
 **/
unsigned
cdk_stream_get_length (CDK_STREAM s)
{
  struct stat statbuf;
  int rc;

  rc = fflush (s->fp);
  if (rc)
    {
      s->error = CDK_File_Error;
      return -1;
    }
  if (fstat (fileno (s->fp), &statbuf))
    {
      s->error = CDK_File_Error;
      return -1;
    }
  return statbuf.st_size;
}


static struct stream_filter_s*
filter_add (CDK_STREAM s)
{
  struct stream_filter_s * f;

  f = cdk_calloc (1, sizeof *f);
  if (!f)
    return NULL;
  
  f->next = s->filters;
  s->filters = f;
  return f;
}


static struct stream_filter_s *
filter_search (CDK_STREAM s, FILTER_FUNC fnc)
{
  struct stream_filter_s * f;

  for (f = s->filters; f; f = f->next)
    {
      if (f->fnc == fnc)
        return f;
    }
  return NULL;
}


struct stream_filter_s *
stream_filter_add (CDK_STREAM s, FILTER_FUNC fnc, int type)
{
  struct stream_filter_s * flt;

  s->flags.filtrated = 0;
  flt = filter_search (s, fnc);
  if (flt)
    return flt;
  
  flt = filter_add (s);
  if (!flt)
    return NULL;
  flt->fnc = fnc;
  flt->flags.enabled = 1;
  flt->tmp = NULL;
  flt->type = type;
  switch (type)
    {
    case fARMOR    : flt->opaque = &flt->u.afx; break;
    case fCIPHER   : flt->opaque = &flt->u.cfx; break;
    case fPLAINTEXT: flt->opaque = &flt->u.pfx; break;
    case fCOMPRESS : flt->opaque = &flt->u.zfx; break;
    case fHASH     : flt->opaque = &flt->u.mfx; break;
    case fTEXT     : flt->opaque = &flt->u.tfx; break;
    default        : flt->opaque = NULL;
    }
  return flt;
}


static int
stream_mode (CDK_STREAM s)
{
  if (s->flags.temp)
    return s->fmode;
  
  return s->flags.write;
}


/**
 * cdk_stream_filter_del: Delete a filter from the given STREAM object.
 * @s: The STREAM object
 * @type: The numberic filter ID.
 *
 **/
int
cdk_stream_filter_disable (CDK_STREAM s, int type)
{
  struct stream_filter_s * f;
  FILTER_FUNC fnc;
  int idx;

  if (!s || type > 6)
    return CDK_Inv_Value;
  idx = type;
  fnc = stream_mode (s)? filter_table[idx].encode : filter_table[idx].decode;
  f = filter_search (s, fnc);
  if (f)
      f->flags.enabled = 0;
  return 0;
}


static int
stream_fp_replace (CDK_STREAM s, FILE ** tmp)
{
  int rc;
  
  rc = fclose (s->fp);
  if (rc)
    return CDK_File_Error;
  s->fp = *tmp;
  *tmp = NULL;
  return 0;
}


/* This function is a little more complicated because we can't use tmpfile()
   because we have to rename the final file. So what are we doing here:
   In stream_new a tempfile was created. Now we create a second tempfile and
   the filter will be used. The old input handle will be replaced with the
   output handle. The pointer is set to the begin and we delete the old
   tempame in the context. It's replaced with the name of the current round.
*/
static int
stream_filter_walk_write (CDK_STREAM s)
{
  struct stream_filter_s * f;
  int rc = 0;
  
  if (s->flags.filtrated)
    return CDK_Inv_Value;

  for (f = s->filters; f; f = f->next)
    {
      if (!f->flags.enabled)
        continue;
      s->tmpn = mk_tmpnam ();
      f->tmp = fopen (s->tmpn, "w+b");
      if (!f->tmp)
        {
          rc = CDK_File_Error;
          break;
        }
      /* If there is no next filter, flush the cache. We also do this
         when the next filter is the armor filter because this filter
         is special and before it starts, all data should be written. */
      if ((!f->next || f->next->type == fARMOR) && s->cache.size)
        {
          rc = stream_cache_flush (s, f->tmp);
          if (rc)
            break;
        }
      rc = f->fnc (f->opaque, s->fp, f->tmp);
      _cdk_log_debug ("filter: walk_write type=%d rc=%d\n", f->type, rc);
      if (rc)
        {
          fclose (f->tmp);
          unlink (s->tmpn);
          break;
        }
      rc = stream_fp_replace (s, &f->tmp);
      if (rc)
        break;
      rc = fseek (s->fp, 0, SEEK_SET);
      if (rc)
        {
          rc = CDK_File_Error;
          break;
        }
      if (s->tname)
        {
          rc = unlink (s->tname);
          if (rc)
            {
              rc = CDK_File_Error;
              break;
            }
          cdk_free (s->tname);
        }
      s->tname = cdk_strdup (s->tmpn);
      if (!s->tname)
        {
          rc = CDK_Out_Of_Core;
          break;
        }
    }
  if (!rc)
    {
      if (!s->flags.temp)
        {
          fclose (s->fp);
          rc = rename (s->tname, s->fname);
          if (rc)
            rc = CDK_File_Error;
        }
    }
  return rc;
}


/* Here all data from the file handle is passed through all filters.
   The scheme works like this:
   Create a tempfile and use it for the output of the filter. Then the
   original file handle will be closed and replace with the temp handle.
   The file pointer will be set to the begin and the game starts again. */
static int
stream_filter_walk_read (CDK_STREAM s)
{
  struct stream_filter_s * f;
  int rc = 0;
  
  if (s->flags.filtrated)
    return 0;

  for (f = s->filters; f; f = f->next)
    {
      if (!f->flags.enabled)
        continue;
      f->tmp = tmpfile ();
      if (!f->tmp)
        {
          rc = CDK_File_Error;
          break;
        }
      rc = f->fnc (f->opaque, s->fp, f->tmp);
      _cdk_log_debug ("filter: walk_read type=%d rc=%d\n", f->type, rc);
      if (rc)
        break;
      
      /* if the filter is read-only, do not replace the FP because
         the contents were not altered in any way. */
      if (!f->flags.rdonly)
        {
          rc = stream_fp_replace (s, &f->tmp);
          if (rc)
            break;
        }
      else
        {
          fclose (f->tmp);
          f->tmp = NULL;
        }
      rc = fseek (s->fp, 0, SEEK_SET);
      if (rc)
        {
          rc = CDK_File_Error;
          break;
        }
      if (s->flags.reset)
        f->flags.enabled = 0;
    }
  return rc;
}


void *
_cdk_stream_get_opaque (CDK_STREAM s, int id)
{
  struct stream_filter_s * f;
  
  for (f = s->filters; f; f = f->next)
    {
      if (f->type == id)
        return f->opaque;
    }
  return NULL;
}


/**
 * cdk_stream_read: Try to read count bytes from the STREAM object.
 * @s: The STREAM object.
 * @buf: The buffer to insert the readed bytes.
 * @count: Request so much bytes.
 *
 * When this function is called the first time, it can take a while
 * because all filters need to be processed. Please remember that you
 * need to add the filters in reserved order.
 **/
int
cdk_stream_read (CDK_STREAM s, void * buf, size_t count)
{
  int nread;
  int rc;

  if (s->flags.write && !s->flags.temp)
    return EOF; /* this is a write stream */
  
  if (!s->cache.on && !s->flags.filtrated)
    {
      rc = stream_filter_walk_read (s);
      if (rc)
        {
          s->error = rc;
          return EOF;
        }
      s->flags.filtrated = 1;
    }
  if (!buf && !count)
    return 0;
  nread = fread (buf, 1, count, s->fp);  
  if (!nread)
      nread = EOF;
  if (feof (s->fp))
    s->flags.eof = 1;

  return nread;
}

      
int
cdk_stream_getc (CDK_STREAM s)
{
  unsigned char buf[2];
  int nread;
  
  nread = cdk_stream_read (s, buf, 1);
  if (nread == EOF)
    {
      s->error = CDK_File_Error;
      return -1;
    }
  return buf[0];
}


/**
 * cdk_stream_write: Try to write count bytes into the stream.
 * @s: The STREAM object
 * @buf: The buffer with the values to write.
 * @count: The size of the buffer.
 *
 * In this function we simply write the bytes to the stream. We can't
 * use the filters here because it would mean they have to support
 * partial flushing.
 **/
int
cdk_stream_write (CDK_STREAM s, const void * buf, size_t count)
{
  int nwritten;

  if (!s->flags.write)
    return CDK_Inv_Mode; /* this is a read stream */

  if (!buf && !count)
    return fflush (s->fp);

  if (s->cache.on)
    {
      if (s->cache.size + count > sizeof (s->cache.buf))
        return CDK_EOF;
      memcpy (s->cache.buf + s->cache.size, buf, count);
      s->cache.size += count;
      return 0;
    }
  
  nwritten = fwrite (buf, 1, count, s->fp);
  if (!nwritten)
    nwritten = EOF;  
  return nwritten;
}


int
cdk_stream_putc (CDK_STREAM s, int c)
{
  unsigned char buf[2];
  int nwritten;

  buf[0] = c;
  nwritten = cdk_stream_write (s, buf, 1);
  if (nwritten == EOF)
    {
      s->error = CDK_File_Error;
      return -1;
    }
  return 0;
}


int
cdk_stream_tell (CDK_STREAM s)
{
  return ftell (s->fp);
}


int
cdk_stream_seek (CDK_STREAM s, long offset)
{
  if (offset < cdk_stream_get_length (s))
    s->flags.eof = 0;
  return fseek (s->fp, offset, SEEK_SET);
}


int
cdk_stream_set_armor_flag (CDK_STREAM s, int type)
{
  struct stream_filter_s * flt;
  FILTER_FUNC fnc;
  int idx = fARMOR;

  fnc = stream_mode (s)? filter_table[idx].encode : filter_table[idx].decode;
  flt = stream_filter_add (s, fnc, idx);
  if (!flt)
    return CDK_Out_Of_Core;
  flt->u.afx.idx = flt->u.afx.idx2 = type;
  return 0;
}


int
cdk_stream_set_literal_flag (CDK_STREAM s, int mode, const char * fname)
{
  struct stream_filter_s * flt;
  FILTER_FUNC fnc;
  int idx = fPLAINTEXT;

  fnc = stream_mode (s)? filter_table[idx].encode : filter_table[idx].decode;
  flt = stream_filter_add (s, fnc, idx);
  if (!flt)
    return CDK_Out_Of_Core;
  flt->u.pfx.mode = mode;
  flt->u.pfx.filename = fname? cdk_strdup (fname) : NULL;
  return 0;
}


int
cdk_stream_set_cipher_flag (CDK_STREAM s, CDK_DEK dek, int use_mdc)
{
  struct stream_filter_s * flt;
  FILTER_FUNC fnc;
  int idx = fCIPHER;

  fnc = stream_mode (s)? filter_table[idx].encode : filter_table[idx].decode;
  flt = stream_filter_add (s, fnc, idx);
  if (!flt)
    return CDK_Out_Of_Core;
  dek->use_mdc = use_mdc;
  flt->u.cfx.dek = dek;
  flt->u.cfx.mdc_method = use_mdc? GCRY_MD_SHA1 : 0;
  return 0;
}


int
cdk_stream_set_compress_flag (CDK_STREAM s, int algo)
{
  struct stream_filter_s * flt;
  FILTER_FUNC fnc;
  int idx = fCOMPRESS;

  fnc = stream_mode (s)? filter_table[idx].encode : filter_table[idx].decode;
  flt = stream_filter_add (s, fnc, idx);
  if (!flt)
    return CDK_Out_Of_Core;
  flt->u.zfx.algo = algo;
  return 0;
}


int
cdk_stream_set_text_flag (CDK_STREAM s, const char * lf)
{
  struct stream_filter_s * flt;
  FILTER_FUNC fnc;
  int idx = fTEXT;

  fnc = stream_mode (s)?filter_table[idx].encode : filter_table[idx].decode;
  flt = stream_filter_add (s, fnc, idx);
  if (!flt)
    return CDK_Out_Of_Core;
  flt->u.tfx.lf = lf;
  return 0;
}


int
cdk_stream_set_hash_flag (CDK_STREAM s, int algo)
{
  struct stream_filter_s * flt;
  FILTER_FUNC fnc;
  int idx = fHASH;

  if (stream_mode (s))
    return CDK_Inv_Mode;
  
  fnc = filter_table[idx].encode;
  flt = stream_filter_add (s, fnc, idx);
  if (!flt)
    return CDK_Out_Of_Core;
  flt->u.mfx.digest_algo = algo;
  flt->flags.rdonly = 1;
  return 0;
}


int
cdk_stream_set_cache (CDK_STREAM s, int val)
{
  if (!s || !s->flags.write)
    return CDK_Inv_Value;
  s->cache.on = val;
  return 0;
}


static int
stream_cache_flush (CDK_STREAM s, FILE * fp)
{
  int nwritten;

  if (!s)
    return CDK_Inv_Value;

  nwritten = fwrite (s->cache.buf, 1, s->cache.size, fp);
  if (!nwritten)
    return CDK_File_Error;
  
  s->cache.size = 0;
  s->cache.on = 0;
  memset (s->cache.buf, 0, sizeof (s->cache.buf));
  
  return 0;
}


int
cdk_stream_kick_off (CDK_STREAM inp, CDK_STREAM out)
{
  byte buf[4096];
  int nread, nwritten;
  int rc = 0;

  while (!cdk_stream_eof (inp))
    {
      nread = cdk_stream_read (inp, buf, sizeof buf-1);
      if (nread == EOF)
        break;
      nwritten = cdk_stream_write (out, buf, nread);
      if (nwritten == EOF)
        rc = CDK_File_Error;
    }
  wipemem (buf, sizeof buf);
  return rc;
}


int
_cdk_stream_gets (CDK_STREAM s, char * buf, size_t count)
{
  int c, i;

  i = 0;
  while (!cdk_stream_eof (s) && count > 0)
    {
      if ((c = cdk_stream_getc (s)) == EOF || c == '\r' || c == '\n')
        {
          buf[i++] = '\0';
          break;   
        }
      buf[i++] = c;
      count--;   
    }
  return i;
}


