/* sig-check.c - Check signatures
 *        Copyright (C) 2001, 2002 Timo Schulz           
 *        Copyright (C) 1998-2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <time.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"


static void
hash_mpi_array (cdkPKT_public_key * pk, GCRY_MD_HD md, int v4)
{
  int i, npkey;
  CDK_MPI a;

  npkey = cdk_pk_get_npkey (pk->pubkey_algo);
  for (i = 0; i < npkey; i++)
    {
      a = pk->mpi[i];
      if (v4)
        {
          gcry_md_putc (md, a->bits >> 8);
          gcry_md_putc (md, a->bits);
        }
      gcry_md_write (md, a->data + 2, a->bytes);
    }
}
  

void
_cdk_hash_pubkey (cdkPKT_public_key * pk, GCRY_MD_HD md, int use_fpr)
{
  int i, npkey;
  u16 n;
  byte buf[4];

  if (!pk || !md)
    return;

  npkey = cdk_pk_get_npkey (pk->pubkey_algo);
  if (use_fpr && pk->version < 4 && is_RSA (pk->pubkey_algo))
    {
      hash_mpi_array (pk, md, 0);
      return;
    }
  n = pk->version < 4 ? 8 : 6;
  for (i = 0; i < npkey; i++)
    {
      n += pk->mpi[i]->bytes;
      n += 2;
    }
  
  gcry_md_putc (md, 0x99);
  gcry_md_putc (md, n >> 8);
  gcry_md_putc (md, n);
  gcry_md_putc (md, pk->version);
  
  buf[0] = pk->timestamp >> 24;
  buf[1] = pk->timestamp >> 16;
  buf[2] = pk->timestamp >> 8;
  buf[3] = pk->timestamp;
  gcry_md_write (md, buf, 4);

  if (pk->version < 4)
    {
      u16 a = pk->expiredate ? pk->expiredate : 0;
      gcry_md_putc (md, a >> 8);
      gcry_md_putc (md, a);
    }
  gcry_md_putc (md, pk->pubkey_algo);
  hash_mpi_array (pk, md, 1);
}


void
_cdk_hash_userid (cdkPKT_user_id * uid, int sig_version, GCRY_MD_HD md)
{
  const byte * data;
  byte buf[5];
  u32 dlen;

  if (!uid || !md)
    return;

  if (sig_version == 4)
    {
      if (uid->attrib_img)
	{
	  buf[0] = 0xd1;
	  buf[1] = uid->attrib_len >> 24;
	  buf[2] = uid->attrib_len >> 16;
	  buf[3] = uid->attrib_len >> 8;
	  buf[4] = uid->attrib_len;
	}
      else
	{
	  buf[0] = 0xb4;
	  buf[1] = uid->len >> 24;
	  buf[2] = uid->len >> 16;
	  buf[3] = uid->len >> 8;
	  buf[4] = uid->len;
	}
      gcry_md_write (md, buf, 5);
    }
  data = uid->attrib_img ? uid->attrib_img : (byte *) uid->name;
  dlen = uid->attrib_img ? uid->attrib_len : uid->len;
  gcry_md_write (md, data, dlen);
}


void
_cdk_hash_sig_data (cdkPKT_signature * sig, GCRY_MD_HD md)
{
  byte buf[4];
  size_t n = 0;

  if (!sig || !md)
    return;

  if (sig->version == 4)
    gcry_md_putc (md, sig->version);
  gcry_md_putc (md, sig->sig_class);
  if (sig->version < 4)
    {
      buf[0] = sig->timestamp >> 24;
      buf[1] = sig->timestamp >> 16;
      buf[2] = sig->timestamp >> 8;
      buf[3] = sig->timestamp;
      gcry_md_write (md, buf, 4);
    }
  else
    {
      gcry_md_putc (md, sig->pubkey_algo);
      gcry_md_putc (md, sig->digest_algo);
      if (sig->hashed)
	{
	  _cdk_subpkt_hash (sig->hashed, &n, md);
          sig->hashed_size = n;
          n = sig->hashed_size + 6;
	}
      else
	{
	  gcry_md_putc (md, 0);
	  gcry_md_putc (md, 0);
	  n = 6;
	}
      gcry_md_putc (md, sig->version);
      gcry_md_putc (md, 0xff);
      buf[0] = n >> 24;
      buf[1] = n >> 16;
      buf[2] = n >> 8;
      buf[3] = n;
      gcry_md_write (md, buf, 4);
    }
}


static void
cache_sig_result (cdkPKT_signature * sig, int res)
{
  if (!res)
    {
      sig->flags.checked = 1;
      sig->flags.valid = 1;
    }
  else if (res == CDK_Bad_Sig)
    {
      sig->flags.checked = 1;
      sig->flags.valid = 0;
    }
  else
    {
      sig->flags.checked = 0;
      sig->flags.valid = 0;
    }
}


int
_cdk_signature_check (cdkPKT_public_key * pk, cdkPKT_signature * sig,
                      GCRY_MD_HD digest, int * r_expired)
{
  byte md[24];
  time_t cur_time = _cdk_timestamp ();
  int digest_algo;
  int rc;

  if (!pk || !sig || !digest)
    return CDK_Inv_Value;

  if (sig->flags.checked)
    return sig->flags.valid ? 0 : CDK_Bad_Sig;

  if (pk->version == 4 && pk->pubkey_algo == GCRY_PK_ELG_E)
    return CDK_Inv_Algo;
  if (pk->timestamp > sig->timestamp || pk->timestamp > cur_time)
    return CDK_Time_Conflict;

  digest_algo = sig->digest_algo;
  if (pk->expiredate && (pk->expiredate + pk->timestamp) > cur_time)
    {
      if (r_expired)
	*r_expired = 1;
    }

  _cdk_hash_sig_data (sig, digest);
  gcry_md_final (digest);
  memcpy (md, gcry_md_read (digest, sig->digest_algo),
	  gcry_md_get_algo_dlen (sig->digest_algo));
  if (md[0] != sig->digest_start[0] || md[1] != sig->digest_start[1])
    return CDK_Bad_Sig;

  rc = cdk_pk_verify (pk, sig, md);
  cache_sig_result (sig, rc);

  return rc;
}


static int
check_key_signature (CDK_KEYDB_HD hd, CDK_KBNODE knode, CDK_KBNODE snode)
{
  cdkPKT_public_key * pk = NULL, * sig_pk = NULL;
  cdkPKT_signature * sig = NULL;
  CDK_KBNODE node;
  GCRY_MD_HD md;
  int digest_algo, is_expired = 0;
  int rc = 0;

  if (!knode || !snode)
    return CDK_Inv_Value;

  if (knode->pkt->pkttype != CDK_PKT_PUBLIC_KEY
      || snode->pkt->pkttype != CDK_PKT_SIGNATURE)
    return CDK_Inv_Value;
  pk = knode->pkt->pkt.public_key;
  sig = snode->pkt->pkt.signature;
  digest_algo = sig->digest_algo;

  md = gcry_md_open (digest_algo, 0);
  if (!md)
    return CDK_Gcry_Error;

  if (sig->sig_class == 0x20) /* key revocation */
    {
      _cdk_hash_pubkey (pk, md, 0);
      rc = _cdk_signature_check (pk, sig, md, &is_expired);
    }
  else if (sig->sig_class == 0x28) /* subkey revocation */
    {
      node = cdk_kbnode_find_prev (knode, snode, CDK_PKT_PUBLIC_SUBKEY);
      if (!node)
	return CDK_Error_No_Key; /* no subkey for subkey revocation packet */
      _cdk_hash_pubkey (pk, md, 0);
      _cdk_hash_pubkey (node->pkt->pkt.public_key, md, 0);
      rc = _cdk_signature_check (pk, sig, md, &is_expired);
    }
  else if (sig->sig_class == 0x18) /* key binding */
    {
      node = cdk_kbnode_find_prev (knode, snode, CDK_PKT_PUBLIC_SUBKEY);
      if (!node)
	return CDK_Error_No_Key; /* no subkey for subkey binding packet */
      _cdk_hash_pubkey (pk, md, 0);
      _cdk_hash_pubkey (node->pkt->pkt.public_key, md, 0);
      rc = _cdk_signature_check (pk, sig, md, &is_expired);
    }
  else if (sig->sig_class == 0x1f) /* direct key signature */
    {
      _cdk_hash_pubkey (pk, md, 0);
      rc = _cdk_signature_check (pk, sig, md, &is_expired);
    }
  else /* all other classes */
    {
      node = cdk_kbnode_find_prev (knode, snode, CDK_PKT_USER_ID);
      if (!node)
	return CDK_Error_No_Key; /* no user ID for key signature packet */
      _cdk_hash_pubkey (pk, md, 0);
      _cdk_hash_userid (node->pkt->pkt.user_id, sig->version, md);
      if (pk->keyid[0] == sig->keyid[0] && pk->keyid[1] == sig->keyid[1])
	rc = _cdk_signature_check (pk, sig, md, &is_expired);
      else if (hd)
	{
	  rc = cdk_keydb_get_pk (hd, sig->keyid, &sig_pk);
	  if (rc)
	    return CDK_Error_No_Key;
	  rc = _cdk_signature_check (sig_pk, sig, md, &is_expired);
	  _cdk_free_pubkey (sig_pk);
	}
    }
  gcry_md_close (md);
  return rc;
}


int
cdk_key_check_sigs (CDK_KBNODE knode, CDK_KEYDB_HD hd, int * r_status)
{
  cdkPKT_signature * sig = NULL;
  CDK_KBNODE k;
  u32 keyid = 0;
  int key_status = 0;
  int rc = 0;

  if (!knode || !r_status)
    return CDK_Inv_Value;

  k = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
  if (!k)
    return CDK_Error_No_Key;
  if (k->pkt->pkt.public_key->is_revoked)
    key_status |= CDK_KEY_REVOKED;
  if (k->pkt->pkt.public_key->has_expired)
    key_status |= CDK_KEY_EXPIRED;
  if (key_status)
    {
      *r_status = key_status;
      return CDK_General_Error;
    }
  keyid = cdk_pk_get_keyid (k->pkt->pkt.public_key, NULL);

  for (k = knode; k && k->pkt->pkttype; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  sig = k->pkt->pkt.signature;
	  rc = check_key_signature (hd, knode, k);
	  if (rc)
	    {
	      if (IS_UID_SIG (sig) && rc == CDK_Error_No_Key)
		{
		  sig->flags.missing_key = 1;
		  continue;
		}
	      if (rc && rc != CDK_Error_No_Key)
		{
		  *r_status = CDK_KEY_INVALID;
		  break; /* invalid self signature or key signature */
		}
              _cdk_log_debug ("keyid %08X; signer %08X: `%s'\n", keyid,
                              cdk_sig_get_keyid (sig, 0), cdk_strerror (rc));
	    }
	  else
            _cdk_log_debug ("signature okay: signer %08X\n", sig->keyid[1]);
	}
    }
  if (!rc || rc == CDK_Error_No_Key)
    *r_status = CDK_KEY_VALID;
  return rc;
}
