/* sign.c - Signing routines
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <time.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"
#include "filters.h"
#include "stream.h"


static void
sig_calc_subpkt_size (cdkPKT_signature * sig)
{
  size_t nbytes;
  
  if (sig->hashed)
    {
      _cdk_subpkt_get_array (sig->hashed, 1, &nbytes);
      sig->hashed_size = nbytes;
    }
  if (sig->unhashed)
    {
      _cdk_subpkt_get_array (sig->unhashed, 1, &nbytes);
      sig->unhashed_size = nbytes;
    }  
}

  
int
_cdk_sig_hash_for (int pubkey_algo, int pktver)
{
  if (is_DSA (pubkey_algo))
    return GCRY_MD_SHA1;
  else if (is_RSA (pubkey_algo) && pktver < 4)
    return GCRY_MD_MD5;
  return GCRY_MD_SHA1; /* default message digest */
}


int
_cdk_sig_create (cdkPKT_public_key * pk, cdkPKT_signature * sig)
{
  CDK_SUBPKT node;
  byte buf[8];

  if (!sig)
    return CDK_Inv_Value;

  if (pk)
    {
      if (!sig->version)
        sig->version = pk->version;
      sig->pubkey_algo = pk->pubkey_algo;
      sig->digest_algo = _cdk_sig_hash_for (pk->pubkey_algo, pk->version);
      cdk_pk_get_keyid (pk, sig->keyid);
    }
  sig->timestamp = _cdk_timestamp ();
  if (sig->version == 3)
    return 0;
  
  sig->hashed = sig->unhashed = NULL;

  _cdk_u32tobuf (sig->keyid[0], buf);
  _cdk_u32tobuf (sig->keyid[1], buf + 4);
  node = cdk_subpkt_new (8);
  if (node)
    {
      node->type = SIGSUBPKT_ISSUER;
      node->size = 8;
      memcpy (node->d, buf, 8);
    }
  sig->unhashed = node;
  
  _cdk_u32tobuf (sig->timestamp, buf);
  node = cdk_subpkt_new (4);
  if (node)
    {
      node->type = SIGSUBPKT_SIG_CREATED;
      node->size = 4;
      memcpy (node->d, buf, 4);
    }
  sig->hashed = node;
  
  if (sig->expiredate)
    {
      u32 u = sig->expiredate - sig->timestamp;
      _cdk_u32tobuf (u, buf);
      node = cdk_subpkt_new (4);
      if (node)
        {
          node->type = SIGSUBPKT_SIG_EXPIRE;
          memcpy (node->d, buf, 4);
          node->size = 4;
        }
      _cdk_subpkt_add (sig->hashed, node);
    }
  sig_calc_subpkt_size (sig);

  return 0;
}


int
_cdk_sig_complete (cdkPKT_signature * sig, cdkPKT_secret_key * sk,
                   GCRY_MD_HD md)
{
  byte digest[24];

  if (!sig || !sk)
    return CDK_Inv_Value;
  
  sig_calc_subpkt_size (sig);
  _cdk_hash_sig_data (sig, md);
  gcry_md_final (md);

  memcpy (digest, gcry_md_read (md, sig->digest_algo),
          gcry_md_get_algo_dlen (sig->digest_algo));
  return cdk_pk_sign (sk, sig, digest);
}


static int
write_plaintext (CDK_STREAM inp, CDK_STREAM out)
{
  CDK_PACKET pkt;
  cdkPKT_plaintext * pt;
  const char * s = _cdk_stream_get_fname (inp);
  int rc;

  if (!inp || !out)
    return CDK_Inv_Value;
  
  cdk_stream_seek (inp, 0);
  pt = cdk_calloc (1, sizeof *pt + strlen (s));
  if (!pt)
    return CDK_Out_Of_Core;
  pt->len = cdk_stream_get_length (inp);
  pt->mode = 'b';
  pt->timestamp = _cdk_timestamp ();
  pt->namelen = strlen (s);
  pt->buf = inp;
  strcpy (pt->name, s);
  pkt.pkttype = CDK_PKT_PLAINTEXT;
  pkt.pkt.plaintext = pt;
  rc = cdk_pkt_build (out, &pkt);
  cdk_free (pt);

  return rc;
}


static int
write_pubkey_enc_list (CDK_HD hd, CDK_STREAM out, CDK_STRLIST remusr)
{
  CDK_KEYLIST pkl;
  int rc;

  if (!out)
    return CDK_Inv_Value;
  
  rc = cdk_pklist_build (hd->db.pub, remusr, &pkl, GCRY_PK_USAGE_ENCR);
  if (rc)
    return rc;

  cdk_free (hd->dek);
  rc = cdk_dek_new (&hd->dek, 0);
  if (!rc)
    {
      hd->dek->algo = cdk_pklist_select_algo (pkl);
      hd->dek->use_mdc = cdk_pklist_use_mdc (pkl);
      rc = cdk_pklist_encrypt (pkl, hd->dek, out);
    }
  cdk_pklist_release (pkl);
  return rc;
}


static int
get_sig_version (CDK_HD hd, CDK_KEYLIST kl)
{
  CDK_KEYLIST l;

  if (hd->opt.compat)
    return 3;
  
  for (l = kl; l; l = l->next)
    {
      if ((l->type == CDK_PKT_PUBLIC_KEY && l->key.pk->version == 3)
          || (l->type == CDK_PKT_SECRET_KEY && l->key.sk->version == 3))
        return 3;
    }
  return 4;
}


/**
 * cdk_file_sign: Sign a file.
 * @locusr: List of userid which should be used for signing
 * @remusr: If encrypt is valid, the list of recipients
 * @file: Name of the input file
 * @output: Name of the output file
 * @detached: Create a detached signature (1=yes, 0=no)
 * @encrypt: enable sign and encrypt
 *
 **/
int
cdk_file_sign (CDK_HD hd, CDK_STRLIST locusr, CDK_STRLIST remusr,
               const char * file, const char * output,
               int detached, int encryptflag)
{
  CDK_STREAM inp = NULL, out = NULL;
  CDK_KEYLIST list;
  cdkPKT_secret_key * sk;
  md_filter_s * mfx;
  int sig_version, digest_algo;
  int rc = 0;

  if (!file || !output)
    return CDK_Inv_Value;
  if (encryptflag && !remusr)
    return CDK_Inv_Value;
  if (detached && encryptflag)
    return CDK_Inv_Mode;

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;

  rc = cdk_stream_new (output, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  rc = cdk_sklist_build (hd->db.sec, locusr, &list, 1, GCRY_PK_USAGE_SIGN);
  if (rc)
    {
      cdk_stream_close (inp);
      cdk_stream_close (out);
      return rc;
    }

  sk = list->key.sk;
  digest_algo = _cdk_sig_hash_for (sk->pubkey_algo, sk->version);

  if (hd->opt.armor)
    cdk_stream_set_armor_flag (out, 0);
  
  if (encryptflag)
    {
      cdk_stream_set_cache (out, 1);
      rc = write_pubkey_enc_list (hd, out, remusr);
      if (rc)
        goto leave;
      cdk_stream_set_cipher_flag (out, hd->dek, hd->dek->use_mdc);
      cdk_stream_set_cache (out, 0);
    }
  
  cdk_stream_set_hash_flag (inp, digest_algo);
  /* kick off the filter */
  sig_version = get_sig_version (hd, list);
  cdk_stream_read (inp, NULL, 0);
  mfx = _cdk_stream_get_opaque (inp, fHASH);
  if (mfx && mfx->md)
    {
      if (!detached)
        {
          rc = cdk_sklist_write_onepass (list, out, 0x00);
          if (!rc)
            rc = write_plaintext (inp, out);
        }
      if (!rc)
        rc = cdk_sklist_write (list, out, mfx->md, 0x00, sig_version);
      gcry_md_close (mfx->md);
    }

leave:
  cdk_sklist_release (list);
  cdk_stream_close (inp);
  cdk_stream_close (out);
  return rc;
}


static void
put_hash_line (CDK_STREAM out, int digest_algo, int key_ver)
{
  const char * s = NULL;

  if (key_ver == 3)
    {
      cdk_stream_putc (out, '\n');
      return;
    }

  switch (digest_algo)
    {
    case GCRY_MD_MD5   : s = "Hash: MD5\n\n"; break;
    case GCRY_MD_SHA1  : s = "Hash: SHA1\n\n"; break;
    case GCRY_MD_RMD160: s = "Hash: RIPEMD160\n\n"; break;
    default            : s = "Hash: SHA1\n\n"; break;
    }
  cdk_stream_write (out, s, strlen (s));
}


void
_cdk_trim_string (char * s, int canon)
{
  while (s && *s && (   s[strlen (s)-1] == '\t'
                     || s[strlen (s)-1] == '\r'
                     || s[strlen (s)-1] == '\n'
                     || s[strlen (s)-1] == ' '))
    s[strlen (s) -1] = '\0';
  if (canon)
    strcat (s, "\r\n");
}


/**
 * cdk_file_clearsign: Make a cleartext signature of a file.
 * @hd: Handle
 * @locusr: List of userIDs
 * @file: Name of the intput file
 * @output: Name of the output file
 *
 **/
int
cdk_file_clearsign (CDK_HD hd, CDK_STRLIST locusr,
                    const char * file, const char * output)
{
  GCRY_MD_HD md = NULL;
  CDK_STREAM inp = NULL, out = NULL, tmp;
  CDK_KEYLIST list;
  cdkPKT_secret_key * sk;
  const char * s;
  int digest_algo, sig_version;
  char buf[1024+2];
  int rc, nread = 0;
  
  if (!locusr || !file || !output)
    return CDK_Inv_Value;

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;
  
  rc = cdk_stream_new (output, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  rc = cdk_sklist_build (hd->db.sec, locusr, &list, 1, GCRY_PK_USAGE_SIGN);
  if (rc)
    {
      cdk_stream_close (inp);
      cdk_stream_close (out);
      return rc;
    }

  sk = list->key.sk;
  digest_algo = _cdk_sig_hash_for (sk->pubkey_algo, sk->version);
  md = gcry_md_open (digest_algo, 0);
  if (!md)
    {
      rc = CDK_Gcry_Error;
      goto leave;
    }

  s = _cdk_armor_get_lineend ();
  strcpy (buf, "-----BEGIN PGP SIGNED MESSAGE-----");
  strcat (buf, s);
  cdk_stream_write (out, buf, strlen (buf));
  put_hash_line (out, digest_algo, sk->version);
  
  while (!cdk_stream_eof (inp))
    {
      nread = _cdk_stream_gets (inp, buf, sizeof buf-1);
      if (!nread)
        break;
      _cdk_trim_string (buf, 1);
      gcry_md_write (md, buf, strlen (buf));
      if (buf[0] == '-')
        {
          memmove (&buf[2], buf, nread + 1);
          buf[1] = ' ';
        }
      if (strlen (s) == 1)
        {
          buf[strlen (buf) - 1] = '\0';
          buf[strlen (buf) - 1] = '\n';
        }
      cdk_stream_write (out, buf, strlen (buf));
    }
  cdk_stream_write (out, s, strlen (s));
  tmp = cdk_stream_tmp ();
  if (!tmp)
    {
      rc = CDK_Out_Of_Core;
      goto leave;
    }
  cdk_stream_tmp_set_mode (tmp, 1);
  cdk_stream_set_armor_flag (tmp, CDK_ARMOR_SIGNATURE);

  sig_version = get_sig_version (hd, list);
  rc = cdk_sklist_write (list, tmp, md, 0x01, sig_version);
  if (rc)
    {
      cdk_stream_close (tmp);
      goto leave;
    }
  
  rc = cdk_stream_flush (tmp);
  if (rc)
    goto leave;
  
  while (!cdk_stream_eof (tmp))
    {
      nread = cdk_stream_read (tmp, buf, sizeof buf-1);
      if (!nread)
        break;
      cdk_stream_write (out, buf, nread);
    }
  cdk_stream_close (tmp);
  
 leave:
  gcry_md_close (md);
  cdk_sklist_release (list);
  cdk_stream_close (inp);
  cdk_stream_close (out);
  return rc;
}

