/* sadb.c */

/* This file contains all of the routines necessary to manage the SADB including
   the netlink call to allow SADM messages to be conveyed between kernel
   space and user space */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/skbuff.h>
#include <linux/in.h>
#include <linux/in6.h>
#include <asm/uaccess.h>
#include <linux/netdevice.h>
#include <linux/netlink.h>
#include <net/sock.h>
#include "sadb.h"
#include "transform.h"
#include "ipsec.h"

#include <asm/param.h>
#include <linux/timer.h>
/* random.h - def's for get_random_bytes         */
#include <linux/random.h>
#include "keymgmt.h"

struct sadb sadb;
__u16 registered_km_port;
struct ipsec_system_policy ipsec_sys_policy;
struct socket *ipsecsk[3];

struct timer_list sa_expire_check_timer;

static void sadb_lock(void);
static void sadb_unlock(void);
unsigned short addr_exist( struct net_addr addr);
unsigned short addr_equal(struct net_addr addr1, struct net_addr addr2);
unsigned short sadb_max_plen(unsigned short family);
unsigned short sadb_hash_plen(unsigned short family);
short get_sadb_hash(struct net_addr dst);
unsigned short sadb_hash_match(struct net_addr addr1, struct net_addr addr2);
unsigned short sadb_prefix_match( struct net_addr addr1, struct net_addr addr2, unsigned short plen);
struct sadb_dst_node *get_sadb_node(struct net_addr dst,unsigned short protocol, unsigned long spi, unsigned short outbound, unsigned short check_expired);
struct sadb_dst_node *get_sadb_outbound_node( struct sadb_dst_node *sadbp, struct net_addr dst, unsigned short protocol, unsigned long spi);
struct sadb_dst_node *get_sadb_inbound_node( struct sadb_dst_node *sadbp, struct net_addr dst, unsigned short protocol,unsigned long spi);
struct sadb_dst_node *get_sadb_inbound_dst_node(struct net_addr dst,unsigned short protocol, unsigned long spi);
void sadb_add(struct sadb_dst_node *d);
void sadb_outbound_add(short h, struct sadb_dst_node *d);
void sadb_inbound_add(short h, struct sadb_dst_node *d);
void sadb_delete(struct sadb_dst_node *d);
void sadb_outbound_delete(short h, struct sadb_dst_node *d);
void sadb_inbound_delete(short h, struct sadb_dst_node *d);
int null_inbound_sa_found(struct net_addr dst);
static int sadb_netlink_call(int minor,  struct sk_buff *skb);
void sadb_init(void);
void printk_key(key_type k);
void printk_net_addr(struct net_addr a);
void printk_sadb_dst_node(struct sadb_dst_node *d);
void sadb_flush(unsigned short type);
void sadb_mss_adjust(void);
int sadb_netlink_attach(int unit, int (*function)(int, struct sk_buff *skb));
void sadb_netlink_detach(int unit);
int sadb_netlink_post(int unit, struct sk_buff *skb);

struct sadb_dst_node *sadb_add_partial(struct sadb_dst_node *d);
unsigned long get_spi (unsigned long lowval, unsigned long highval, struct sadb_dst_node d);
void sadb_update(struct sadb_dst_node *d);
void sadb_outbound_update(short h, struct sadb_dst_node *d);
void sadb_inbound_update(short h, struct sadb_dst_node *d);
unsigned short sadb_check_sa(struct sadb_dst_node *d);
int sa_expired(struct sadb_dst_node *d);
int sa_expires_soon(struct sadb_dst_node *d);
void sadb_expire_check(void);
int sadb_kick_keymgmt(__u32 daddr, unsigned long spi, unsigned long peer_spi);
void sadb_keymgmt_delete_sa(struct sadb_dst_node *d);

/*
 *      SADB locking. One of these days we'll get this working
 */

#if 0
static struct wait_queue *sadb_wait;
static atomic_t sadb_users = ATOMIC_INIT(0);
#endif

static void sadb_lock(void)
{
#if 0
/* this isn't working properly */
        while (atomic_read(&sadb_users))
                sleep_on(&sadb_wait);
        atomic_inc(&sadb_users);
#endif
#ifdef MODULE
    ipsec_inc_mod_count();
#endif

}

static void sadb_unlock(void)
{
#if 0
/* this isn't working properly */
        if (atomic_dec_and_test(&sadb_users)) 
                wake_up(&sadb_wait);
#endif
#ifdef MODULE
    ipsec_dec_mod_count();
#endif
}

unsigned short __inline__ addr_exist(addr)
	struct net_addr addr;
{

switch (addr.addr_family)
{
	case AF_INET:
		return (addr.addr_union.ip_addr.s_addr);
	break;
	default:
		return 0;
	break;
}
}

unsigned short __inline__ addr_equal(addr1,addr2)
	struct net_addr addr1,addr2;
{
    if (addr1.addr_family != addr2.addr_family) return 0;

    switch (addr1.addr_family)
    {
	case AF_INET:
		return (addr1.addr_union.ip_addr.s_addr == addr2.addr_union.ip_addr.s_addr);
	break;
	default:
		return 0;
	break;
    }
}

/* maximum size in bits of address prefix */
unsigned short __inline__ sadb_max_plen(unsigned short family)
{
	switch(family){
	case AF_INET:
		return 32;
	break;
	case AF_INET6:
		return 128;
	break;
	default:	return 0;
	}
}

/* size of prefix used by hash function */
unsigned short __inline__ sadb_hash_plen(unsigned short family)
{
	switch(family){
	case AF_INET:
		return SADB_HASH_PLEN;
	break;
	case AF_INET6:
		return SADB_HASH6_PLEN;
	break;
	default:	return 0;
	}
}

short __inline__ get_sadb_hash(struct net_addr dst)
{
	switch(dst.addr_family){
	case AF_INET:
		return ((dst.addr_union.ip_addr.s_addr&(0xffffffff>>(sadb_max_plen(dst.addr_family) - SADB_HASH_PLEN)))%MAX_SADB_HASH_SIZE);
	break;
	default: break;
	}
	return -1;
}

void printk_key(key_type k)
{
	printk(KERN_INFO "key length\t%d\n",k.key_len);
	bprintk(k.key_data,k.key_len,"key data");
}

void printk_net_addr(struct net_addr a)
{
	switch(a.addr_family)
	{
	case AF_INET:
		printk("AF_INET ");
		printk("%s",ntoa(a.addr_union.ip_addr.s_addr));
	break;
	case AF_INET6:
		printk("<family %d>  ",a.addr_family);
	break;
	}
}

void printk_sadb_dst_node(struct sadb_dst_node *d)
{
	printk(KERN_INFO "SADB Entry:\n");
	printk(KERN_INFO "destination\t");
	printk_net_addr(d->dst);
	printk("/%d\n",d->prefix_len);
	printk(KERN_INFO "spi\t\t%lx\n",d->sa_info.spi);
	printk(KERN_INFO "uid\t\t%d\n",d->sa_info.uid);
	printk(KERN_INFO "peer\t\t");
	printk_net_addr(d->sa_info.peer_addr);
	printk("\n");
	printk(KERN_INFO "peer spi\t%lx\n",d->sa_info.peer_spi);
	printk(KERN_INFO "flags\t%x\n",d->sa_info.flags);
	printk(KERN_INFO "sequence number\t%lx\n",d->sa_info.sn);
	printk(KERN_INFO "TCP MSS Delta\t%d\n",d->sa_info.mss_delta);
	printk(KERN_INFO "protocol\t%d\n",d->sa_info.protocol);
	switch(d->sa_info.protocol)
	{
	case IPPROTO_ESP:
		printk(KERN_INFO "ESP Cipher Algorithm Parameters\n");
		printk(KERN_INFO "crypto alg id\t%d\n",d->sa_info.alg.esp.crypto_alg_id);
		printk(KERN_INFO "crypto ivec length\t%d\n",d->sa_info.alg.esp.crypto_ivec_length);
		printk(KERN_INFO "outbound key\n");
		printk_key(d->sa_info.alg.esp.outbound_crypto_key);
		printk(KERN_INFO "\n");
		printk(KERN_INFO "inbound key\n");
		printk_key(d->sa_info.alg.esp.inbound_crypto_key);
		printk(KERN_INFO "ESP Authenticator Algorithm Parameters\n");
		printk(KERN_INFO "alg id\t%d\n",d->sa_info.alg.esp.auth_alg_id);
		printk(KERN_INFO "ivec length\t%d\n",d->sa_info.alg.esp.auth_ivec_length);
		printk(KERN_INFO "data length\t%d\n",d->sa_info.alg.esp.auth_data_length);
		printk(KERN_INFO "outbound key\n");
		printk_key(d->sa_info.alg.esp.outbound_auth_key);
		printk(KERN_INFO "\n");
		printk(KERN_INFO "inbound key\n");
		printk_key(d->sa_info.alg.esp.inbound_auth_key);
	break;
	case IPPROTO_AH:
		printk(KERN_INFO "AH Algorithm Parameters\n");
		printk(KERN_INFO "alg id\t%d\n",d->sa_info.alg.esp.auth_alg_id);
		printk(KERN_INFO "ivec length\t%d\n",d->sa_info.alg.ah.auth_ivec_length);
		printk(KERN_INFO "data length\t%d\n",d->sa_info.alg.ah.auth_data_length);
		printk(KERN_INFO "outbound key\n");
		printk_key(d->sa_info.alg.ah.outbound_auth_key);
		printk(KERN_INFO "\n");
		printk(KERN_INFO "inbound key\n");
		printk_key(d->sa_info.alg.ah.inbound_auth_key);
	break;
	default:
	break;
	}
}

#if 0 /* For Debugging Purposes */
void dump_sadb()
{
	int i;

	printk(KERN_INFO "************************DUMP_SADB BEGIN*****************************\n");
	for (i=0;i<MAX_SADB_HASH_SIZE;i++){
		struct sadb_dst_node **s2, *tmps, *d;

		s2=&sadb.outbound[i];

		while (*s2)
		{
			printk(KERN_INFO "BEGIN_CHAIN s2=%x, *s2=%x, s2->noh = %x\n",s2, *s2, &((*s2)->next_out_hash));
			d = *s2;
			while(d)
			{
				printk(KERN_INFO "d = %x, spi = %x, next_out = %x, next_out_hash = %x\n", d, d->sa_info.spi, d->next_out,d->next_out_hash);
				d = d->next_out;
			}
			printk(KERN_INFO "END_CHAIN\n");
			s2 = &((*s2)->next_out_hash);
		}

		s2=&sadb.inbound[i];
		while (*s2)
		{
			d = *s2;
			printk(KERN_INFO "BEGIN_CHAIN s2=%x, *s2=%x, s2->nih = %x\n",s2,*s2,&((*s2)->next_in_hash));
			while(d)
			{
				printk(KERN_INFO "d = %x, spi = %x, next_in = %x, next_in_hash = %x\n", d,d->sa_info.spi, d->next_in,d->next_in_hash);
				d = d->next_in;
			}
			printk(KERN_INFO "END_CHAIN\n");
			s2 = &((*s2)->next_in_hash);
		}
	}
	printk(KERN_INFO "************************DUMP_SADB END*****************************\n");
}
#endif

unsigned short __inline__ sadb_prefix_match(addr1,addr2,plen)
	struct net_addr addr1;
	struct net_addr addr2;
	unsigned short plen; /* in bits */
{

    if (addr1.addr_family != addr2.addr_family) return 0;

    switch (addr1.addr_family)
    {
	case AF_INET:
		if ((plen/8) > sizeof (addr1.addr_union.ip_addr.s_addr)) return 0;

		return((addr1.addr_union.ip_addr.s_addr&(htonl(0xffffffff<<(32-plen))))==(addr2.addr_union.ip_addr.s_addr&(htonl(0xffffffff<<(32-plen)))));
	break;
	default:
		return 0;
	break;
    }
}

/* This function is used to determine if a collision has occured within
   the hash table */
/* If the v6 hash function hashes on the prefix of the v6 address then
   this function should be made into a macro */
unsigned short sadb_hash_match(struct net_addr addr1, struct net_addr addr2)
{
    switch (addr1.addr_family)
    {
	case AF_INET:
		return sadb_prefix_match(addr1,addr2,SADB_HASH_PLEN);
	break;
	default:
		return 0;
	break;
    }

}

int sa_expired(struct sadb_dst_node *d)
{
	struct timeval tv;

	if (!(d->sa_info.flags & IPSEC_LIFETIME_PERMANENT_FLAG))
	{
		do_gettimeofday (&tv); 

		return(((d->sa_info.lifetime.bytes_remaining<=SA_BYTES_EXPIRED) && 
                	(!(d->sa_info.lifetime.bytes_remaining==-1 ))) || 
               		((tv.tv_sec>=(d->sa_info.lifetime.time_expired-SA_TIME_EXPIRED))&&
                	(d->sa_info.lifetime.time_expired>0)));
	}
	return 0;
}

int sa_expires_soon(struct sadb_dst_node *d)
{
	struct timeval tv;

	if (!(d->sa_info.flags & IPSEC_LIFETIME_PERMANENT_FLAG))
	{
		do_gettimeofday (&tv); 

		return(((d->sa_info.lifetime.bytes_remaining<=SA_BYTES_EXPIRED_WARNING) && 
                	(!(d->sa_info.lifetime.bytes_remaining==-1 ))) || 
               		((tv.tv_sec>=(d->sa_info.lifetime.time_expired-SA_TIME_EXPIRED_WARNING))&&
                	(d->sa_info.lifetime.time_expired>0)));
	}	
	return 0;
}

void sadb_expire_check()
{
	int i;
	struct timeval tv;
	struct ipsec_transform *transform;

	do_gettimeofday(&tv);
	ipsec_mss_delta = 0;

	sadb_lock();
	for (i=0;i<MAX_SADB_HASH_SIZE;i++){
		struct sadb_dst_node **s2, *tmps, *d;

		s2=&sadb.outbound[i];

		while (*s2)
		{
			d = *s2;
			while(d)
			{
				if (d->sa_info.flags & IPSEC_INBOUND_FLAG)
				{
 					d = d->next_out;
					continue;
				}
				if (sa_expired(d))
				{
					if (!(d->sa_info.flags & IPSEC_PARTIAL_FLAG))
					{
						switch (d->sa_info.protocol)
						{
						case IPPROTO_ESP:
							if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
								if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
							if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
								if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
						break;
						case IPPROTO_AH:
							if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
								if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
						break;
						default:
						break;
						}
					}
					tmps = d;
					d = d->next_out;
					sadb_delete(tmps);
				}
				else 
				{
					ipsec_mss_delta = max(ipsec_mss_delta, d->sa_info.mss_delta);
					d = d->next_out;
				}
			}
			if (*s2)
				s2 = &((*s2)->next_out_hash);
		}
		s2=&sadb.inbound[i];
		while (*s2)
		{
			d = *s2;
			while(d)
			{
				if (sa_expired(d))
				{
					if (!(d->sa_info.flags & IPSEC_PARTIAL_FLAG))
					{
						switch (d->sa_info.protocol)
						{
						case IPPROTO_ESP:
							if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
								if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
							if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
								if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
						break;
						case IPPROTO_AH:
							if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
								if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
						break;
						default:
						break;
						}
					}
					tmps = d;
					d = d->next_in;
					sadb_delete(tmps);
				}
				else
					d = d->next_in;
			}
			if (*s2)
				s2 = &((*s2)->next_in_hash);
		}
	}
	sadb_unlock();
}
/* get a SA entry based on dest address, spi, protocol, and direction */
struct sadb_dst_node *get_sadb_node(struct net_addr dst,unsigned short protocol, unsigned long spi, unsigned short outbound, unsigned short check_expired)
{
	short hash;
	struct sadb_dst_node *sadbp, *result;
	if ((hash = get_sadb_hash(dst))<0) return 0;

	switch(dst.addr_family)
	{
	case AF_INET:
		if (outbound)
		{
			sadbp = sadb.outbound[hash];
			result = get_sadb_outbound_node(sadbp,dst,protocol,spi);
		}
		else
		{
			sadbp = sadb.inbound[hash];
			result =  get_sadb_inbound_node(sadbp,dst,protocol,spi);
		}
		break;
	case AF_INET6:
		if (outbound)
		{
			sadbp = sadb.outbound6[hash];
			result = get_sadb_outbound_node(sadbp,dst,protocol,spi);
		}
		else
		{
			sadbp = sadb.inbound6[hash];
			result = get_sadb_inbound_node(sadbp,dst,protocol,spi);
		}
		break;
	default: return NULL;
	}
	if (check_expired)
	{
	    if (result)
	    {
		if (!(result->sa_info.flags & IPSEC_LIFETIME_PERMANENT_FLAG))
		{
			if (sa_expires_soon(result))
			{
				switch(dst.addr_family)
				{
				case AF_INET:
					if (result->sa_info.flags & IPSEC_NEG_KM_FLAG){
						if (!(result->sa_info.flags & IPSEC_KM_KICKED_FLAG)) {
							if (outbound){
#ifdef SI_TESTER_VERBOSE
                    						printk(KERN_INFO "SA EXPIRING SOON (KICKING outbound spi=%lx)\n",
		      							result->sa_info.spi);
#endif
								sadb_kick_keymgmt(dst.addr_union.ip_addr.s_addr, result->sa_info.spi, result->sa_info.peer_spi);
								result->sa_info.flags |= IPSEC_KM_KICKED_FLAG;
							}
#ifdef SI_TESTER_VERBOSE
		   					else
                    						printk(KERN_INFO "SA EXPIRING SOON (NOT KICKING inbound spi=%lx)\n",
		      							result->sa_info.spi);
#endif
						}
					}
					return result;
				case AF_INET6:
					return NULL;
				}
			}
		}
	    }
	}
	return result;	
}

struct sadb_dst_node *get_sadb_outbound_node( struct sadb_dst_node *sadbp, struct net_addr dst, unsigned short protocol, unsigned long spi)
{

	if (!sadbp) return NULL;

	sadb_lock();
	while (sadbp)
	{
		if (sadb_hash_match(sadbp->dst,dst))
			break;
		sadbp = sadbp->next_out_hash;
	}

	while (sadbp)
	{
		if (addr_equal(sadbp->dst,dst)) 
			break;


		if (!sadb_hash_match(sadbp->dst,dst)){
			sadb_unlock();
			return NULL;
		}

		if ((sadbp->prefix_len < sadb_max_plen(dst.addr_family)) && sadb_prefix_match(sadbp->dst,dst,sadbp->prefix_len))
			break;

		sadbp = sadbp->next_out_hash;
	}


	while (sadbp)
	{
/* Make sure the SA is neither a PARTIAL SA nor an expired SA */
/* non-0 ret'd ==> OK */
	if (sadb_check_sa(sadbp)) {
	    if (!spi){
		sadb_unlock();
		return sadbp;
	    }
	    else if ((sadbp->sa_info.spi == spi) &&
	        (protocol==sadbp->sa_info.protocol))
            {
		  sadb_unlock();
		  return sadbp;
            }
        } /* end if sadb_check_sa */
	  sadbp = sadbp->next_out;
	} /* end while */

	sadb_unlock();
        return NULL;

}

struct sadb_dst_node *get_sadb_inbound_node( struct sadb_dst_node *sadbp, struct net_addr dst,unsigned short protocol, unsigned long spi)
{

	if (!sadbp) return NULL;

	sadb_lock();
	while (sadbp)
	{
		if (sadb_hash_match(sadbp->sa_info.peer_addr,dst))
			break;
		sadbp = sadbp->next_in_hash;
	}

	while (sadbp)
	{
		if (addr_equal(sadbp->sa_info.peer_addr,dst))
			break;

		if (!sadb_hash_match(sadbp->sa_info.peer_addr,dst)){
			sadb_unlock();
			return NULL;
		}

		sadbp = sadbp->next_in_hash;
	}

	while (sadbp)
	{
	    if ((sadbp->sa_info.spi == spi) &&
	        (protocol==sadbp->sa_info.protocol) &&
/* Make sure the SA is neither a PARTIAL SA nor an expired SA */
/*                 non-0 ret'd ==> OK */
		   sadb_check_sa(sadbp))
            {
		  sadb_unlock();
		  return sadbp;
            }
	    sadbp = sadbp->next_in;
	}

	sadb_unlock();
        return NULL;
}

/* This is a brute force search of the entire inbound SADB.  This is 
specifically for KM gets & deletes.  It should NEVER be used by the IPsec 
per-packet code or anywhere else where performance is critical.  This 
routine searches the inbound database based on the actuall destination 
address instead of the peer address in get_sadb_inbound_node */

struct sadb_dst_node *get_sadb_inbound_dst_node(struct net_addr dst,unsigned short protocol, unsigned long spi)
{
	int i;

	sadb_lock();
	for (i=0;i<MAX_SADB_HASH_SIZE;i++){
		struct sadb_dst_node **s2, *d;

		s2=&sadb.inbound[i];
		while (*s2)
		{
			d = *s2;
			while(d)
			{
				if ((d->sa_info.protocol == protocol) &&
                                    (d->sa_info.spi == spi) &&
				    (addr_equal(d->dst,dst)))
				{
					sadb_unlock();
					return d;
				}
				d = d->next_in;
			}
			s2 = &((*s2)->next_in_hash);
		}
	}
	sadb_unlock();
	return NULL;
}

/* Make sure the SA is neither a PARTIAL SA nor an expired SA 
   						non-0 ret'd ==> OK  */

unsigned short sadb_check_sa(struct sadb_dst_node *d)
{
	return(!(d->sa_info.flags & IPSEC_PARTIAL_FLAG) &&
               !(sa_expired(d)));
}

/* check for an existing NULL_SA */
int null_inbound_sa_found(struct net_addr dst)
{
	short hash;

	struct sadb_dst_node *sadbp;
	if ((hash = get_sadb_hash(dst))<0) return 0;
	sadbp = sadb.inbound[hash];

	sadb_lock();
	while (sadbp)
	{
		if (sadb_hash_match(sadbp->sa_info.peer_addr,dst))
			break;
		sadbp = sadbp->next_in_hash;
	}

	while (sadbp)
	{
		if (addr_equal(sadbp->sa_info.peer_addr,dst))
			break;

		if (!sadb_hash_match(sadbp->sa_info.peer_addr,dst)){
			sadb_unlock();
			return 0;
		}

		sadbp = sadbp->next_in_hash;
	}

	while (sadbp)
	{
	    if (sadbp->sa_info.protocol == PROTO_NULL_SA)
            {
		  sadb_unlock();
		  return 1;
            }
	    sadbp = sadbp->next_in;
	}

	sadb_unlock();
        return 0;
}

/* SADB Garbage collection timer.  When the timer expires, scan the
   SADB for expired SAs and physically delete them, then restart the
   timer */

static void sa_timer_expire(unsigned long data)
{
	sadb_expire_check();
	init_timer(&sa_expire_check_timer);
	sa_expire_check_timer.expires = jiffies + (HZ*SA_EXPIRE_CHECK);
	sa_expire_check_timer.data = 0;
	sa_expire_check_timer.function = sa_timer_expire;
	add_timer(&sa_expire_check_timer);
}

void sadb_add(struct sadb_dst_node *d)
{

	struct sadb_dst_node *newd;
	short in_hash,out_hash;
	unsigned short flags;

	if (d->sa_info.spi < 256) 
	{
		printk(KERN_INFO "sadb_add: spi %lx reserved\n",d->sa_info.spi);
		return;
	}

	if (d->sa_info.peer_spi < 256) 
	{
		printk(KERN_INFO "sadb_add: peer spi %lx reserved\n",d->sa_info.peer_spi);
		return;
	}

	if ((d->sa_info.protocol != IPPROTO_ESP)  && (d->sa_info.protocol != IPPROTO_AH) && (d->sa_info.protocol != PROTO_NULL_SA))
	{
		printk(KERN_INFO "sadb_add: unknown protocol %x\n",d->sa_info.protocol);
		return;
	}

	if ((d->sa_info.next_encap.protocol != IPPROTO_ESP)  && (d->sa_info.next_encap.protocol != IPPROTO_AH) && (d->sa_info.next_encap.protocol != 0))
	{
		printk(KERN_INFO "sadb_add: unknown protocol for Linked SA %x\n",d->sa_info.next_encap.protocol);
		return;
	}

	if ((d->sa_info.next_encap.protocol)  && (d->sa_info.next_encap.spi < 256))
	{
		printk(KERN_INFO "sadb_add: illegal spi for Linked SA %lx\n",d->sa_info.next_encap.spi);
		return;
	}

	if ((!addr_equal(d->dst, d->sa_info.peer_addr)) && (!(d->sa_info.flags & IPSEC_TUNNEL_FLAG))) 
	{
		printk(KERN_INFO "sadb_add: transport mode, dest and peer must be equal\n");
		return;
	}

	if ((d->prefix_len < sadb_max_plen(d->dst.addr_family)) && (!(d->sa_info.flags & IPSEC_TUNNEL_FLAG))) 
	{
		printk(KERN_INFO "sadb_add: transport mode, prefix length must be equal to %d\n",
			sadb_max_plen(d->dst.addr_family));
		return;
	}

	if (d->prefix_len > sadb_max_plen(d->dst.addr_family))
	{
		printk(KERN_INFO "sadb_add: prefix length must be <= %d\n", sadb_max_plen(d->dst.addr_family));
		return;
	}

	if (d->dst.addr_family != d->sa_info.peer_addr.addr_family) 
	{
		printk(KERN_INFO "sadb_add: address family mismatch between dest and peer\n");
		return;
	}

	if (d->prefix_len < sadb_hash_plen(d->dst.addr_family))
	{
		printk(KERN_INFO "sadb_add: prefix length must be >= %d\n", sadb_hash_plen(d->dst.addr_family));
		return;
	}

	if ((d->sa_info.next_encap.protocol == PROTO_NULL_SA) ||
	    ((d->sa_info.protocol == PROTO_NULL_SA) && (d->sa_info.next_encap.spi)))
	{
		printk(KERN_INFO "sadb_add: NULL_SA not permitted in multiple encapsulation chain\n");
		return;
	}

	if ((d->sa_info.protocol == PROTO_NULL_SA) && (!(d->sa_info.flags & IPSEC_LIFETIME_PERMANENT_FLAG)))
	{
		printk(KERN_INFO "sadb_add: NULL_SA must have LIFETIME_PERMANENT set\n");
		return;
	}

	if (((d->sa_info.next_encap.protocol == 0) && (d->sa_info.next_encap.spi)) ||
	    ((d->sa_info.next_encap.protocol) && (d->sa_info.next_encap.spi == 0)))
	{
		printk(KERN_INFO "sadb_add: next encapsulation protocol must both be zero or both be no-zero\n");
		return;
	}
        
	if ((in_hash = get_sadb_hash(d->sa_info.peer_addr))<0)
	{
 		printk(KERN_INFO "sadb_add: hash failure for in_hash\n");
		return;
	}

	if ((out_hash = get_sadb_hash(d->dst))<0) 
	{
 		printk(KERN_INFO "sadb_add: hash failure for out_hash\n");
		return;
	}

	if ((newd = (struct sadb_dst_node *)kmalloc(sizeof(struct sadb_dst_node),GFP_ATOMIC)) == NULL)
	{
		printk(KERN_INFO "sadb_add: kmalloc failure\n");
		return;
	}

	memcpy(newd, d, sizeof(struct sadb_dst_node));
	newd->sa_info.rpwin_bitmap=(__u64)0;
	newd->sa_info.lastsn = 0;
	newd->next_out_hash = 0;
	newd->next_out = 0;
	newd->next_in_hash = 0;
	newd->next_in = 0;
	flags = newd->sa_info.flags;

	switch(d->dst.addr_family)
	{
	case AF_INET:
/* Add inbound only SA or bi-directional SA to the inbound chain  */
		if ((flags & IPSEC_INBOUND_FLAG))
		sadb_inbound_add(in_hash,newd);
/* Add outbound only SA or bi-directional SA to the outbound chain  */
		if ((flags & IPSEC_OUTBOUND_FLAG))
		sadb_outbound_add(out_hash,newd);
		break;
	case AF_INET6:
/* Add inbound only SA or bi-directional SA to the inbound chain  */
		if ((flags & IPSEC_INBOUND_FLAG))
		sadb_inbound_add(in_hash,newd);
/* Add outbound only SA or bi-directional SA to the outbound chain  */
		if ((flags & IPSEC_OUTBOUND_FLAG))
		sadb_outbound_add(out_hash,newd);
		break;
	default: break;
	}

}

/* Partial SAs are used by IKE as a reserved place holder which gets filled
in once the IKE negotiation is complete */

struct sadb_dst_node *sadb_add_partial(struct sadb_dst_node *d)
{

	struct sadb_dst_node *newd;
	short in_hash,out_hash;
	unsigned short flags;

	if (d->prefix_len < sadb_hash_plen(d->dst.addr_family))
	{
		printk(KERN_INFO "sadb_add_partial: prefix length must be >= %d\n", sadb_hash_plen(d->dst.addr_family));
		return NULL;
	}

	if ((in_hash = get_sadb_hash(d->sa_info.peer_addr))<0) 
	{
		printk(KERN_INFO "sadb_add_partial: hash failure for in_hash\n");
		return NULL;
	}

	if ((out_hash = get_sadb_hash(d->dst))<0) 
	{
		printk(KERN_INFO "sadb_add_partial: hash failure for out_hash\n");
		return NULL;
	}

	if ((newd = (struct sadb_dst_node *)kmalloc(sizeof(struct sadb_dst_node),GFP_ATOMIC)) == NULL)
	{
		printk(KERN_INFO "sadb_add_partial: kmalloc failure\n");
		return NULL;
	}

	memcpy(newd, d, sizeof(struct sadb_dst_node));
	newd->sa_info.rpwin_bitmap=(__u64)0;
	newd->sa_info.lastsn = 0;
	newd->next_out_hash = 0;
	newd->next_out = 0;
	newd->next_in_hash = 0;
	newd->next_in = 0;
	newd->sa_info.flags |= IPSEC_PARTIAL_FLAG;
	flags = newd->sa_info.flags;
	switch(d->dst.addr_family)
	{
	case AF_INET:
/* Add inbound only SA or bi-directional SA to the inbound chain  */
		if (flags & IPSEC_INBOUND_FLAG)
			sadb_inbound_add(in_hash,newd);
/* Add outbound only SA or bi-directional SA to the inbound chain  */
		if (flags & IPSEC_OUTBOUND_FLAG)
			sadb_outbound_add(out_hash,newd);
		break;
	case AF_INET6:
		if (flags & IPSEC_INBOUND_FLAG)
			sadb_inbound_add(in_hash,newd);
		if (flags & IPSEC_OUTBOUND_FLAG)
			sadb_outbound_add(out_hash,newd);
		break;
	default: break;
	}
	return (newd);
}

/*----------------------------------------------------------------------
 * get_spi():
 *      Get a unique spi value for a key management daemon/program.  
 *      Most of this code is taken from the NRL/Cisco ISAKMP
 ----------------------------------------------------------------------*/

/* get_spi picks a legal random spi value and insures that it is not already 
in use for a given spi,destination,protocol */

unsigned long get_spi (unsigned long lowval, unsigned long highval, struct sadb_dst_node d)
{
  unsigned long val;
  int count;
  int maxcount = 1000;
  
  if ((lowval == 0) || (highval == 0))
	{
		printk(KERN_INFO "get_spi: invalid range %lu to %lu\n",lowval, highval);
		return(-1);
	}

  if (lowval > highval) {
    unsigned long temp;
    temp = lowval;
    lowval = highval;
    highval = lowval;
  }

  count = 0;
  do {
     count++;
    /* 
     *  This may not be "random enough".
     */
    get_random_bytes(&val,sizeof(val));
    val = lowval + (val % (highval - lowval + 1));

    if (lowval == highval)
       count = maxcount;

    if (val > 255)
    {
       if ((get_sadb_node(d.dst, d.sa_info.protocol, val,1,1)==NULL) && 
           (get_sadb_inbound_dst_node(d.dst, d.sa_info.protocol, val) == NULL))
       return (val);
    }

    }  while ((count < maxcount));

    return(-1);
}

/* used to replace a partial SA with a full SA after the IKE negotiation */
void sadb_update(struct sadb_dst_node *d)
{

	short out_hash, in_hash;

	if (d->sa_info.spi < 256) 
	{
		printk(KERN_INFO "sadb_update: spi %lx reserved\n",d->sa_info.spi);
		return;
	}

	if (d->sa_info.peer_spi < 256) 
	{
		printk(KERN_INFO "sadb_update: peer spi %lx reserved\n",d->sa_info.peer_spi);
		return;
	}


	if ((d->sa_info.protocol != IPPROTO_ESP)  && (d->sa_info.protocol != IPPROTO_AH))
	{
		printk(KERN_INFO "sadb_update: unknown protocol %x\n",d->sa_info.protocol);
		return;
	}

	if ((!addr_equal(d->dst, d->sa_info.peer_addr)) && (!d->sa_info.flags & IPSEC_TUNNEL_FLAG)) 
	{
		printk(KERN_INFO "sadb_update: transport mode, dest and peer must be equal\n");
		return;
	}

	if ((d->prefix_len < sadb_max_plen(d->dst.addr_family)) && (!d->sa_info.flags & IPSEC_TUNNEL_FLAG)) 
	{
		printk(KERN_INFO "sadb_update: transport mode, prefix length must be equal to %d\n",
			sadb_max_plen(d->dst.addr_family));
		return;
	}

	if (d->prefix_len > sadb_max_plen(d->dst.addr_family))
	{
		printk(KERN_INFO "sadb_update: prefix length must be <= %d\n", sadb_max_plen(d->dst.addr_family));
		return;
	}

	if (d->dst.addr_family != d->sa_info.peer_addr.addr_family) 
	{
		printk(KERN_INFO "sadb_update: address family mismatch between dest and peer\n");
		return;
	}

	if (d->prefix_len < sadb_hash_plen(d->dst.addr_family))
	{
		printk(KERN_INFO "sadb_update: prefix length must be >= %d\n", sadb_hash_plen(d->dst.addr_family));
		return;
	}

	if ((in_hash = get_sadb_hash(d->sa_info.peer_addr))<0) 
	{
		printk(KERN_INFO "sadb_update: hash failure for in_hash\n");
		return;
	}

	if ((out_hash = get_sadb_hash(d->dst))<0) 
	{
		printk(KERN_INFO "sadb_update: hash failure for out_hash\n");
		return;
	}

		if (d->sa_info.flags & IPSEC_OUTBOUND_FLAG)
/* Update outbound only SA or bi-directional SA                      */
			sadb_outbound_update(out_hash,d);
		else
/* Update inbound only SA or bi-directional SA                      */
			sadb_inbound_update(in_hash,d);
	return;
}

/* used to replace a partial SA with a full SA after the IKE negotiation */
void sadb_outbound_update(short h, struct sadb_dst_node *d)
{
	struct sadb_dst_node **s, *tmps, *no, *noh, *ni, *nih;

	sadb_lock();
	s = &sadb.outbound[h];
	tmps = NULL;
	while (*s)
	{
		if (addr_equal((*s)->dst,d->dst))
			break;
		s = &((*s)->next_out_hash);
	}

	if (!*s) {
		printk(KERN_INFO "sadb_outbound_update: entry not found for destination %s\n", ntoa(d->dst.addr_union.ip_addr.s_addr));
		sadb_unlock();
		return;
	}

	while (*s)
	{
		if (((*s)->sa_info.spi == d->sa_info.spi) &&
	            ((*s)->sa_info.flags & IPSEC_PARTIAL_FLAG)) 
		{
			tmps = *s;
/* Save inbound/outbound chain pointers before copying info into partial sadb entry */
			ni = tmps->next_in;
			nih = tmps->next_in_hash;
			no = tmps->next_out;
			noh = tmps->next_out_hash;

			memcpy(tmps, d, sizeof(struct sadb_dst_node));

/* Restore inbound/outbound chain pointers                               */
			tmps->next_in = ni;
			tmps->next_in_hash = nih;
			tmps->next_out = no;
			tmps->next_out_hash = noh;
			sadb_unlock();
			return;
		}
		else
		{
			s = &((*s)->next_out);
		}
	}
	printk(KERN_INFO "sadb_outbound_update: partial entry not found for destination %s spi %lx\n", ntoa(d->dst.addr_union.ip_addr.s_addr) ,d->sa_info.spi);
	sadb_unlock();
}

/* used to replace a partial SA with a full SA after the IKE negotiation */
void sadb_inbound_update(short h, struct sadb_dst_node *d)
{
	struct sadb_dst_node **s, *tmps, *no, *noh, *ni, *nih;

	sadb_lock();
	s = &sadb.inbound[h];
	tmps = NULL;
	while (*s)
	{
		if (addr_equal((*s)->sa_info.peer_addr,d->sa_info.peer_addr))
			break;
		s = &((*s)->next_in_hash);
	}

	if (!*s) {
		printk(KERN_INFO "sadb_inbound_update: entry not found for destination %s\n", ntoa(d->sa_info.peer_addr.addr_union.ip_addr.s_addr));
		sadb_unlock();
		return;
	}

	while (*s)
	{
		if (((*s)->sa_info.peer_spi == d->sa_info.peer_spi) &&
	            ((*s)->sa_info.flags & IPSEC_PARTIAL_FLAG)) 
		{
			tmps = *s;
/* Save inbound/outbound chain pointers before copying info into partial sadb entry */
			ni = tmps->next_in;
			nih = tmps->next_in_hash;
			no = tmps->next_out;
			noh = tmps->next_out_hash;

			memcpy(tmps, d, sizeof(struct sadb_dst_node));

/* Restore inbound/outbound chain pointers                               */
			tmps->next_in = ni;
			tmps->next_in_hash = nih;
			tmps->next_out = no;
			tmps->next_out_hash = noh;
			sadb_unlock();
			return;
		}
		else
			s = &((*s)->next_in);
	}
	printk(KERN_INFO "sadb_inbound_update: partial entry not found for destination %s spi %lx\n", ntoa(d->sa_info.peer_addr.addr_union.ip_addr.s_addr) ,d->sa_info.peer_spi);
	sadb_unlock();
}

void sadb_outbound_add(short h, struct sadb_dst_node *d)
{
	struct sadb_dst_node **prevp, **s;

	sadb_lock();
	prevp = NULL;
	s = &sadb.outbound[h];

	while (*s)
	{
		if (sadb_hash_match((*s)->dst,d->dst))
			break;
		prevp = s;
		s = &((*s)->next_out_hash);
	}

	while (*s)
	{
		if (((d->prefix_len > (*s)->prefix_len) && (d->prefix_len < sadb_max_plen(d->dst.addr_family))) ||
		    (((*s)->prefix_len < sadb_max_plen(d->dst.addr_family)) && (d->prefix_len == sadb_max_plen(d->dst.addr_family))) ||
		     (!sadb_hash_match((*s)->dst,d->dst)))
		{
			d->next_out_hash = (*s);
                        if (prevp){
				if (*prevp)
					(*prevp)->next_out_hash = d;
			}else{
				sadb.outbound[h] = d;
			}
			sadb_unlock();
			return;
		}
		if (addr_equal((*s)->dst,d->dst))
			break;	
		prevp = s;
		s = &((*s)->next_out_hash);
	}

	while (*s)
	{
		if (((*s)->sa_info.spi == d->sa_info.spi) &&
		   ((*s)->sa_info.protocol == d->sa_info.protocol))
	        {
			 sadb_unlock();
			 return;
		}
		s = &((*s)->next_out);
	}
	
	*s = d;
	sadb_unlock();
}

void sadb_inbound_add(short h, struct sadb_dst_node *d)
{
	struct sadb_dst_node **prevp, **s;

	sadb_lock();
	s = &sadb.inbound[h];
	prevp = NULL;
	while (*s)
	{
		if (sadb_hash_match((*s)->sa_info.peer_addr,d->sa_info.peer_addr))
			break;
		prevp = s;
		s = &((*s)->next_in_hash);
	}

	while (*s)
	{
		if (addr_equal((*s)->sa_info.peer_addr,d->sa_info.peer_addr))
			break;

		if (!sadb_hash_match((*s)->sa_info.peer_addr, d->sa_info.peer_addr))
		{
			d->next_in_hash = *s;
			if (*prevp)
				(*prevp)->next_in_hash = d;
			sadb_unlock();
			return;
		}

		prevp = s;
		s = &((*s)->next_in_hash);
	}

	while (*s)
	{
		if (((*s)->sa_info.spi == d->sa_info.spi) &&
		     (addr_equal((*s)->dst, d->dst)) &&
		     ((*s)->sa_info.protocol == d->sa_info.protocol))
		{
			 sadb_unlock();
			 return;
		}
		s = &((*s)->next_in);
	}
	
	*s = d;
	sadb_unlock();
}

void sadb_delete(struct sadb_dst_node *d)
{

	short in_hash,out_hash;

	if (d->sa_info.spi < 256) 
	{
		printk(KERN_INFO "sadb_delete: spi %lx reserved\n",d->sa_info.spi);
		return;
	}

	if ((in_hash = get_sadb_hash(d->sa_info.peer_addr))<0) 
	{
		printk(KERN_INFO "sadb_delete: hash failure for in_hash\n");
		return;
	}
	if ((out_hash = get_sadb_hash(d->dst))<0)
	{
		printk(KERN_INFO "sadb_delete: hash failure for out_hash\n");
		return;
	}
/* Delete negotiated SA from keymgmt daemon's SA structure list */
        if ((d->sa_info.flags & IPSEC_NEG_KM_FLAG))
		sadb_keymgmt_delete_sa(d);
/* delete, order (inbound vs. outbound) is important since the last delete actually frees the memory */
	switch(d->dst.addr_family)
	{
	case AF_INET:
		if (d->sa_info.flags & IPSEC_INBOUND_FLAG)
			sadb_inbound_delete(in_hash,d);
		if (d->sa_info.flags & IPSEC_OUTBOUND_FLAG)
			sadb_outbound_delete(out_hash,d);
		break;
	case AF_INET6:
		if (d->sa_info.flags & IPSEC_INBOUND_FLAG)
			sadb_inbound_delete(in_hash,d);
		if (d->sa_info.flags & IPSEC_OUTBOUND_FLAG)
			sadb_outbound_delete(out_hash,d);
		break;
	default: break;
	};
}

void sadb_outbound_delete(short h, struct sadb_dst_node *d)
{
	struct sadb_dst_node **prev_hashp, **prevp, **s, *tmps;

	sadb_lock();
	s = &sadb.outbound[h];
	prev_hashp = NULL;
	prevp = NULL;
	tmps = NULL;
	while (*s)
	{
		if (addr_equal((*s)->dst,d->dst))
			break;
		prev_hashp = s;
		s = &((*s)->next_out_hash);
	}

	if (!*s) {
		sadb_unlock();
		return;
	}

	while (*s)
	{
		if (((*s)->sa_info.spi == d->sa_info.spi) &&
		    ((*s)->sa_info.protocol == d->sa_info.protocol))
		{
			tmps = *s;
			if (prevp)
				(*prevp)->next_out = tmps->next_out;
			else
			{
				*s = tmps->next_out;
				if (*s)	
					(*s)->next_out_hash = tmps->next_out_hash;
				else
					*s = tmps->next_out_hash;

				if (prev_hashp)
					(*prev_hashp)->next_out_hash = *s;
			}

			kfree(tmps);
			sadb_unlock();
			return;
		}
		else
		{
			prevp = s;
			s = &((*s)->next_out);
		}
	}
	sadb_unlock();
}

void sadb_inbound_delete(short h, struct sadb_dst_node *d)
{
	struct sadb_dst_node **prev_hashp, **prevp, **s, *tmps;

	sadb_lock();
	s = &sadb.inbound[h];
	prev_hashp = NULL;
	prevp = NULL;
	tmps = NULL;
	while (*s)
	{
		if (addr_equal((*s)->sa_info.peer_addr,d->sa_info.peer_addr))
			break;
		prev_hashp = s;
		s = &((*s)->next_in_hash);
	}

	if (!*s) {
		sadb_unlock();
		return;
	}

	while (*s)
	{
		if (((*s)->sa_info.spi == d->sa_info.spi) &&
		    (addr_equal((*s)->dst, d->dst))&&
		    ((*s)->sa_info.protocol == d->sa_info.protocol))
		{
			tmps = *s;
			if (prevp)
				(*prevp)->next_in = tmps->next_in;
			else
			{
				*s = tmps->next_in;
				if (*s)	
					(*s)->next_in_hash = tmps->next_in_hash;
				else
					*s = tmps->next_in_hash;

				if (prev_hashp)
					(*prev_hashp)->next_in_hash = *s;
			}
			sadb_unlock();
			return;
		}
		else
		{
			prevp = s;
			s = &((*s)->next_in);
		}
	}
	sadb_unlock();
}

void sadb_flush(unsigned short type)
{
	int i;
	struct ipsec_transform *transform;

	sadb_lock();
	for (i=0;i<MAX_SADB_HASH_SIZE;i++){
		struct sadb_dst_node **s2, *tmps, *d;

		s2=&sadb.outbound[i];

		while (*s2)
		{
			d = *s2;
			while(d)
			{
				if (d->sa_info.flags & IPSEC_INBOUND_FLAG)
				{
 					d = d->next_out;
					continue;
				}
				switch (d->sa_info.protocol)
				{
				case IPPROTO_ESP:
					if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
					if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
				break;
				case IPPROTO_AH:
					if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
				break;
				default:
				break;
				}
				tmps = d;
				d = d->next_out;
				kfree(tmps);
			}
			s2 = &((*s2)->next_out_hash);
		}
		sadb.outbound[i] = NULL;

		s2=&sadb.inbound[i];
		while (*s2)
		{
			d = *s2;
			while(d)
			{
				switch (d->sa_info.protocol)
				{
				case IPPROTO_ESP:
					if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
					if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
				break;
				case IPPROTO_AH:
					if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
				break;
				default:
				break;
				}
				tmps = d;
				d = d->next_in;
				kfree(tmps);
			}
			s2 = &((*s2)->next_in_hash);
		}
		sadb.inbound[i] = NULL;
	}
	ipsec_mss_delta = 0;
	sadb_unlock();
}

void sadb_mss_adjust(void)
{
	int i;

	sadb_lock();
	ipsec_mss_delta = 0;
	for (i=0;i<MAX_SADB_HASH_SIZE;i++){
		struct sadb_dst_node **s2, *d;

		s2=&sadb.outbound[i];

		while (*s2)
		{
			d = *s2;
			while(d)
			{
				ipsec_mss_delta = max(ipsec_mss_delta,d->sa_info.mss_delta);
				d = d->next_out;
			}
			s2 = &((*s2)->next_out_hash);
		}
	}
	sadb_unlock();
}

/* These functions were pretty much taken out of the af_netlink.c file
   I got tired of re-writing all the sadb netlink calls everytime the
   netlink interface changed, and I was a bit concerned that
   "someday soon" the attach, detach, and post functions would disappear
   that I just included a version of them here.  */

int sadb_netlink_attach(int unit, int (*function)(int, struct sk_buff *skb))
{
        struct sock *sk = netlink_kernel_create(unit, NULL);
        if (sk == NULL)
                return -ENOBUFS;
        sk->protinfo.af_netlink.handler = function;
	switch (unit)
	{
	case NETLINK_IPSEC:
        	ipsecsk[0] = sk->socket;
	break;
	case NETLINK_IKE:
        	ipsecsk[1] = sk->socket;
	break;
	case NETLINK_GETSPI:
        	ipsecsk[2] = sk->socket;
	break;
	default:
		printk(KERN_INFO "sadb_netlink_attach: unknown NETLINK socket identifier\n");
	break;
	}
        return 0;

}
void sadb_netlink_detach(int unit)
{
        struct socket *sock;

	switch (unit)
	{
	case NETLINK_IPSEC:
		sock = ipsecsk[0];
        	ipsecsk[0] = NULL;
	break;
	case NETLINK_IKE:
		sock = ipsecsk[1];
        	ipsecsk[1] = NULL;
	break;
	case NETLINK_GETSPI:
		sock = ipsecsk[2];
        	ipsecsk[2] = NULL;
	break;
	default:
		printk(KERN_INFO "sadb_netlink_detach: unknown NETLINK socket identifier\n");
	return;
	}
        sock_release(sock);
}

int sadb_netlink_post(int unit, struct sk_buff *skb)
{
        struct socket *sock;

	switch (unit)
	{
	case NETLINK_IPSEC:
		sock = ipsecsk[0];
	break;
	case NETLINK_IKE:
		sock = ipsecsk[1];
	break;
	case NETLINK_GETSPI:
		sock = ipsecsk[2];
	break;
	default:
		printk(KERN_INFO "sadb_netlink_post: unknown NETLINK socket identifier\n");
        	return -EUNATCH;;
	}
        if (sock) {
		netlink_broadcast(sock->sk, skb, 0, ~0, GFP_ATOMIC);
               	return 0;
        }
        return -EUNATCH;;
}

static int sadb_netlink_call(minor,skb)
int minor;
struct sk_buff *skb;
{
	struct sadb_msg *sadbm, *s;
 	struct sadb_dst_node *d,*sadbp=NULL;
	struct ipsec_transform *transform;
	struct sk_buff *tmpskb;
	int i=0,nlidx;
	int k;
	int found;

/* variables added for get_spi - SADBM_GETSPI: compute spi  */
 	unsigned long lowval, highval;


	sadbm = (struct sadb_msg *)skb->data;
	if (sadbm->m_sadbm.sadbm_flags & SADBM_MANUAL)
		nlidx = NETLINK_IPSEC;
	else if (sadbm->m_sadbm.sadbm_flags & SADBM_IKE_GETSPI)
		nlidx = NETLINK_GETSPI;
	else
		nlidx = NETLINK_IKE;

	switch (sadbm->m_sadbm.sadbm_type)
	{
	case SADBM_GET:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_GET spi=%lx peer_spi=%lx\n", 
			sadbm->sadb_info.dst.sa_info.spi, 
			sadbm->sadb_info.dst.sa_info.peer_spi);
#endif
		switch (sadbm->request_type){
		case SADB_SYSPOL:
			tmpskb = skb_copy(skb,GFP_ATOMIC);
			s = (struct sadb_msg *)tmpskb->data;
			s->sadb_info.sys_pol.inbound.prot_flag = ipsec_sys_policy.inbound.prot_flag;
			s->sadb_info.sys_pol.outbound.prot_flag = ipsec_sys_policy.outbound.prot_flag;
			s->m_sadbm.sadbm_flags = SADBM_DONE;
			if (sadb_netlink_post(nlidx,tmpskb))
				kfree_skb(tmpskb);
			
		break;
		case SADB_DST:
			if (addr_exist(sadbm->sadb_info.dst.dst))
			{
				tmpskb = skb_copy(skb,GFP_ATOMIC);
				s = (struct sadb_msg *)tmpskb->data;
				s->m_sadbm.sadbm_flags = SADBM_DONE;
				d = NULL;
                                if (sadbm->sadb_info.dst.sa_info.flags & IPSEC_OUTBOUND_FLAG)
					d = get_sadb_node(sadbm->sadb_info.dst.dst, sadbm->sadb_info.dst.sa_info.protocol, sadbm->sadb_info.dst.sa_info.spi, 1,0);
                                if ((d==NULL) && (sadbm->sadb_info.dst.sa_info.flags & IPSEC_INBOUND_FLAG))
					d = get_sadb_inbound_dst_node(sadbm->sadb_info.dst.dst, sadbm->sadb_info.dst.sa_info.protocol, sadbm->sadb_info.dst.sa_info.spi);

 			        if (d != NULL)
				{
					memcpy(&(s->sadb_info.dst),d, sizeof(struct sadb_dst_node));
				}
				else{
					memset(&(s->sadb_info.dst),0,sizeof(struct sadb_dst_node));
				}
				if (sadb_netlink_post(nlidx,tmpskb))
					kfree_skb(tmpskb);
			} else
			{
				sadb_lock();
				if (sadbm->sadb_info.dst.sa_info.flags & IPSEC_OUTBOUND_FLAG)
				{
				for (i=0;i<MAX_SADB_HASH_SIZE;i++){
					sadbp=sadb.outbound[i];
					while (sadbp)
					{
						d = sadbp;
						while(d)
						{
							tmpskb = skb_copy(skb,GFP_ATOMIC);
							s = (struct sadb_msg *)tmpskb->data;
							s->m_sadbm.sadbm_flags = SADBM_MORE;
							memcpy(&(s->sadb_info.dst),d, sizeof(struct sadb_dst_node));
							if (sadb_netlink_post(nlidx,tmpskb))
								kfree_skb(tmpskb);
 							d = d->next_out;
						}
    					           sadbp = sadbp->next_out_hash;
					}
				}
				}

				if (sadbm->sadb_info.dst.sa_info.flags & IPSEC_INBOUND_FLAG)
				{
				for (i=0;i<MAX_SADB_HASH_SIZE;i++){
					sadbp=sadb.inbound[i];
					while (sadbp)
					{
						d = sadbp;
						while(d)
						{
							if ((sadbm->sadb_info.dst.sa_info.flags & IPSEC_OUTBOUND_FLAG) && 
                                                            (d->sa_info.flags & IPSEC_OUTBOUND_FLAG)){
 								d = d->next_in;
								continue;
							}
							tmpskb = skb_copy(skb,GFP_ATOMIC);
							s = (struct sadb_msg *)tmpskb->data;
							s->m_sadbm.sadbm_flags = SADBM_MORE;
							memcpy(&(s->sadb_info.dst),d, sizeof(struct sadb_dst_node));
							if (sadb_netlink_post(nlidx,tmpskb))
								kfree_skb(tmpskb);
 							d = d->next_in;
						}
 						sadbp = sadbp->next_in_hash;
					}
				}
				}
/* one last bogus message to say that we are done */
				tmpskb = skb_copy(skb,GFP_ATOMIC);
				s = (struct sadb_msg *)tmpskb->data;
				s->m_sadbm.sadbm_flags = SADBM_DONE;
				memset(&(s->sadb_info.dst),0,sizeof(struct sadb_dst_node));
				if (sadb_netlink_post(nlidx,tmpskb))
					kfree_skb(tmpskb);
				sadb_unlock();
			}
		break;
		default:
			tmpskb = skb_copy(skb,GFP_ATOMIC);
			s = (struct sadb_msg *)tmpskb->data;
			s->m_sadbm.sadbm_flags = SADBM_DONE;
			memset(&(s->sadb_info.dst),0,sizeof(struct sadb_dst_node));
			if (sadb_netlink_post(nlidx,tmpskb))
				kfree_skb(tmpskb);
		break;
		}
	break;
	case SADBM_SET:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_ADD spi=%lx peer_spi=%lx\n", 
			sadbm->sadb_info.dst.sa_info.spi, 
			sadbm->sadb_info.dst.sa_info.peer_spi);
#endif
		switch (sadbm->request_type){
		case SADB_SYSPOL:
			ipsec_sys_policy.inbound.prot_flag = sadbm->sadb_info.sys_pol.inbound.prot_flag;
			ipsec_sys_policy.outbound.prot_flag = sadbm->sadb_info.sys_pol.outbound.prot_flag;
		break;
		case SADB_DST:
			switch (sadbm->sadb_info.dst.sa_info.protocol)
			{
			case IPPROTO_ESP:
				if ((transform = ipsec_get_transform(sadbm->sadb_info.dst.sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
				{
					if (transform->sadb_add_handler(&sadbm->sadb_info.dst)){
						if (sadbm->sadb_info.dst.sa_info.alg.esp.auth_alg_id != NO_ALG){
							if ((transform = ipsec_get_transform(sadbm->sadb_info.dst.sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
								if (transform->sadb_add_handler(&sadbm->sadb_info.dst))
									sadb_add(&sadbm->sadb_info.dst);
						}
						else
							sadb_add(&sadbm->sadb_info.dst);
					}
				}
			break;
			case IPPROTO_AH:
				if ((transform = ipsec_get_transform(sadbm->sadb_info.dst.sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
				{
					if (transform->sadb_add_handler(&sadbm->sadb_info.dst))
						sadb_add(&sadbm->sadb_info.dst);
				}
			break;
			case PROTO_NULL_SA:
				sadb_add(&sadbm->sadb_info.dst);
			break;
			default:
			break;
			}
		break;
		default:
		break;
		}
	break;
	case SADBM_DELETE:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_DELETE spi=%lx peer_spi=%lx\n", 
			sadbm->sadb_info.dst.sa_info.spi, 
			sadbm->sadb_info.dst.sa_info.peer_spi);
#endif
		switch (sadbm->request_type){
		case SADB_DST:
			d = NULL;
                        if (sadbm->sadb_info.dst.sa_info.flags & IPSEC_OUTBOUND_FLAG)
				d = get_sadb_node(sadbm->sadb_info.dst.dst, sadbm->sadb_info.dst.sa_info.protocol, sadbm->sadb_info.dst.sa_info.spi, 1,0);
                        if ((d==NULL) && (sadbm->sadb_info.dst.sa_info.flags & IPSEC_INBOUND_FLAG))
				d = get_sadb_inbound_dst_node(sadbm->sadb_info.dst.dst, sadbm->sadb_info.dst.sa_info.protocol, sadbm->sadb_info.dst.sa_info.spi);

 			if (d != NULL)
			{
				switch (sadbm->sadb_info.dst.sa_info.protocol)
				{
				case IPPROTO_ESP:
					if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
					if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
				break;
				case IPPROTO_AH:
					if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
						if (transform->sadb_delete_handler) transform->sadb_delete_handler(d);
				break;
				case PROTO_NULL_SA:
				break;
				default:
				break;
				}
			sadb_delete(d);
			sadb_mss_adjust();
			}
		break;
		default:
		break;
		}
	break;
	case SADBM_CHANGE: /* TBD */
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_CHANGE\n");
#endif
	break;
	case SADBM_FLUSH: 
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_FLUSH\n");
#endif
		sadb_flush(sadbm->request_type);
	break;
	case SADBM_REGISTER:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_REGISTER\n");
#endif
		registered_km_port = sadbm->sadb_info.udp_port;
/* Zero out ISAKMP addr array */
	memset(&isakmp_addr_array,0, ISAKMP_ADDR_ARRAY_SIZE*sizeof(__u32));
/* code to kick key mgmt for ISAKMP initiator (send SADBM_ACQUIRE)  */
	if (sadbm->request_type == KICK_KEY_MGMT) {
#ifdef SI_TESTER_VERBOSE
			printk(KERN_INFO "SADBM_REGISTER with KICK_KEY_MGMT\n");
#endif
			tmpskb = skb_copy(skb,GFP_ATOMIC);
			s = (struct sadb_msg *)tmpskb->data;
			s->m_sadbm.sadbm_version = SADBM_VERSION;
			s->m_sadbm.sadbm_type = SADBM_ACQUIRE;
			s->sadb_info.dst.dst.addr_family = AF_INET;
			s->sadb_info.dst.sa_info.flags |= IPSEC_NEG_KM_FLAG;
                	s->sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;
/* Make sure that spi and peer_spi are set to 0
     to let keymgmt daemon know that this is a new SA,
     not an SA that is expiring and needs to be replaced */
	                s->sadb_info.dst.sa_info.spi = 0;
       		        s->sadb_info.dst.sa_info.peer_spi = 0;
#ifdef SI_TESTER_VERBOSE
			printk(KERN_INFO "SADBM_REGISTER (POST=ACQUIRE spi=%lx peer-spi=%lx)\n",
				s->sadb_info.dst.sa_info.spi, 
				s->sadb_info.dst.sa_info.peer_spi);
#endif

			if (sadb_netlink_post(nlidx,tmpskb))
				kfree_skb(tmpskb);
	}
#ifdef SI_TESTER_VERBOSE
	else
	  printk(KERN_INFO "SADBM_REGISTER without KICK_KEY_MGMT\n");
#endif
	break;
	case SADBM_UNREGISTER:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_UNREGISTER\n");
#endif
		registered_km_port = 0;
/* Zero out ISAKMP addr array */
	memset(&isakmp_addr_array,0, ISAKMP_ADDR_ARRAY_SIZE*sizeof(__u32));
	break;
	case SADBM_GETSPI:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_GETSPI\n");
#endif
		lowval = 300;
		highval = 2500001;
		if ((sadbm->sadb_info.dst.sa_info.spi = get_spi(lowval,highval,sadbm->sadb_info.dst))<256)
			{
			printk(KERN_INFO "get_spi failure\n");
			break;
			}
/* TTT Set peer_spi = to spi to enable sadb_add_partial for a 1-way inbound SA */
		sadbm->sadb_info.dst.sa_info.peer_spi = sadbm->sadb_info.dst.sa_info.spi;
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_GETSPI spi=%lx peer_spi=%lx\n", 
			sadbm->sadb_info.dst.sa_info.spi, 
			sadbm->sadb_info.dst.sa_info.peer_spi);
#endif
		tmpskb = skb_copy(skb,GFP_ATOMIC);
		s = (struct sadb_msg *)tmpskb->data;
			if (sadbm->request_type == SADB_ADD_PARTIAL_SA) {
#ifdef SI_TESTER_VERBOSE
			    printk(KERN_INFO "SADBM_GETSPI - adding partial SA\n");
#endif
		  	    if ((sadbp = sadb_add_partial(&sadbm->sadb_info.dst))) {
				memcpy(&(s->sadb_info.dst),sadbp, sizeof(struct sadb_dst_node));
			    }
		  	    else {
				printk(KERN_INFO "sadb_add_partial failure\n");
			    }
                       }
#ifdef SI_TESTER_VERBOSE
		       else
			    printk(KERN_INFO "SADBM_GETSPI - NOT adding partial SA\n");
	  	       printk(KERN_INFO "SADBM_GETSPI (POST=GETSPI)\n");
#endif
		if (sadb_netlink_post(nlidx,tmpskb))
		kfree_skb(tmpskb);
	break;
	case SADBM_UPDATE:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_UPDATE spi=%lx peer_spi=%lx\n", 
		sadbm->sadb_info.dst.sa_info.spi, 
		sadbm->sadb_info.dst.sa_info.peer_spi);
#endif
		found = 0;
		for (k=0;k<ISAKMP_ADDR_ARRAY_SIZE;k++)
		{
			if (isakmp_addr_array[k] == sadbm->sadb_info.dst.dst.addr_union.ip_addr.s_addr) 
			{
			isakmp_addr_array[k] = 0;
			found = 1;
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_UPDATE: ISAKMP address array entry found for %s \n", 
			ntoa(sadbm->sadb_info.dst.dst.addr_union.ip_addr.s_addr));
#endif
			break;
			}
		}
		if (!found)
			printk(KERN_INFO "SADBM_UPDATE: ISAKMP address array entry NOT found for %s \n", 
			ntoa(sadbm->sadb_info.dst.dst.addr_union.ip_addr.s_addr));
		switch (sadbm->sadb_info.dst.sa_info.protocol)
		{
		case IPPROTO_ESP:
			if ((transform = ipsec_get_transform(sadbm->sadb_info.dst.sa_info.alg.esp.crypto_alg_id,CIPHER_TRANSFORM))!=NULL)
			{
				if (transform->sadb_add_handler(&sadbm->sadb_info.dst)){
					if (sadbm->sadb_info.dst.sa_info.alg.esp.auth_alg_id != NO_ALG){
						if ((transform = ipsec_get_transform(sadbm->sadb_info.dst.sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))!=NULL)
							if (transform->sadb_add_handler(&sadbm->sadb_info.dst))
								sadb_update(&sadbm->sadb_info.dst);
					}
					else
						sadb_update(&sadbm->sadb_info.dst);
				}
			}
		break;
		case IPPROTO_AH:
			if ((transform = ipsec_get_transform(sadbm->sadb_info.dst.sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))!=NULL)
			{
				if (transform->sadb_add_handler(&sadbm->sadb_info.dst))
					sadb_update(&sadbm->sadb_info.dst);
			}
		break;
		case IPPROTO_NG:
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "SADBM_UPDATE: no update performed for %s \n", 
			ntoa(sadbm->sadb_info.dst.dst.addr_union.ip_addr.s_addr));
#endif
		break;
		default:
		break;
		}
	break;
	default:
	break;
	}
	return(0);
}

void sadb_init()
{

	if (sadb_netlink_attach(NETLINK_IPSEC, sadb_netlink_call))
	{
		printk(KERN_ERR "NIST SADB Initialization Failure - cannot attach to manual IPSEC netlink socket\n");
		return;
	}
	if (sadb_netlink_attach(NETLINK_IKE, sadb_netlink_call))
	{
		printk(KERN_ERR "NIST SADB Initialization Failure - cannot attach to IKE netlink socket\n");
		return;
	}
	if (sadb_netlink_attach(NETLINK_GETSPI, sadb_netlink_call))
	{
		printk(KERN_ERR "NIST SADB Initialization Failure - cannot attach to IKE GETSPI netlink socket\n");
		return;
	}
	printk(KERN_INFO "NIST Security Association Data Base Initialization\n");
	memset(&sadb,0,sizeof(sadb));
/* Initialize ISAKMP addr array - used to save addresses for which
 * 	ISAKMP is currently negotiating 
 */
	memset(&isakmp_addr_array,0, ISAKMP_ADDR_ARRAY_SIZE*sizeof(__u32));
	ipsec_sys_policy.inbound.prot_flag = PROTECTED|NON_PROTECTED|NULL_ASSOCIATION;
	ipsec_sys_policy.outbound.prot_flag = PROTECTED|NON_PROTECTED|NULL_ASSOCIATION;
	transform_init(); 

	init_timer(&sa_expire_check_timer);
	sa_expire_check_timer.expires = jiffies + (HZ*SA_EXPIRE_CHECK);
	sa_expire_check_timer.data = 0;
	sa_expire_check_timer.function = sa_timer_expire;
	add_timer(&sa_expire_check_timer);

	registered_km_port = 0;

}

void sadb_close()
{

	del_timer(&sa_expire_check_timer);
	sadb_flush(0);
	sadb_netlink_detach(NETLINK_IPSEC);
	sadb_netlink_detach(NETLINK_IKE);
	sadb_netlink_detach(NETLINK_GETSPI);
	registered_km_port = 0;
	printk(KERN_INFO "NIST Security Association Data Base Flushed and Removed\n");
}

int sadb_kick_keymgmt(__u32 daddr, unsigned long spi, unsigned long peer_spi)
{	
    struct sadb_msg *s;
    struct sk_buff *tmpskb;
    int first_zero, k;

/* For a new negotiation (spi == 0), see if kernel already kicked -
 *   if so, don't kick again
 * For a re-negotiation (spi != 0), let PlutoPlus decide whether to
 *   negotiate
 */
    if (!spi)
    {
        first_zero = -1;
	for (k=0;k<ISAKMP_ADDR_ARRAY_SIZE;k++){
		if ((isakmp_addr_array[k] == 0) && (first_zero == -1))
			first_zero=k;
		if (isakmp_addr_array[k] == daddr) 
			return k+1;
	}
	if (first_zero >= 0)
		isakmp_addr_array[first_zero] = daddr;
	else 
		return -1;
    }

	tmpskb=alloc_skb((sizeof(struct sadb_msg)), GFP_ATOMIC);
	if(tmpskb)
	{
		skb_put(tmpskb,sizeof(struct sadb_msg));
		s = (struct sadb_msg *)tmpskb->data;
		s->m_sadbm.sadbm_version = SADBM_VERSION;
		s->m_sadbm.sadbm_type = SADBM_ACQUIRE;
		s->m_sadbm.sadbm_msglen = sizeof(struct sadb_msg);

		s->sadb_info.dst.sa_info.flags |= IPSEC_NEG_KM_FLAG;
		s->sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;

/* Send spi & peer_spi of SA that is expiring and needs to be replaced */
		s->sadb_info.dst.sa_info.spi = spi;
		s->sadb_info.dst.sa_info.peer_spi = peer_spi;
#ifdef SI_TESTER_VERBOSE
	printk(KERN_INFO "sadb_kick_keymgmt: kernel sending SADBM_ACQUIRE spi=%lx peer_spi=%lx\n", spi, peer_spi);
#endif

		s->sadb_info.dst.dst.addr_family = AF_INET;
		s->sadb_info.dst.dst.addr_union.ip_addr.s_addr = daddr;
		if(sadb_netlink_post(NETLINK_IKE, tmpskb))
			kfree_skb(tmpskb);
	}
	else
		printk(KERN_INFO "sadb_kick_keymgmt: alloc_skb failure\n");
		return 0;
}

void sadb_keymgmt_delete_sa(struct sadb_dst_node *d)
{	
    struct sadb_msg *s;
    struct sk_buff *tmpskb;


	tmpskb=alloc_skb((sizeof(struct sadb_msg)), GFP_ATOMIC);
	if(tmpskb)
	{
		skb_put(tmpskb,sizeof(struct sadb_msg));
		s = (struct sadb_msg *)tmpskb->data;
/*		memset(&s,0,sizeof(struct sadb_msg));  */
		s->m_sadbm.sadbm_version = SADBM_VERSION;
		s->m_sadbm.sadbm_type = SADBM_DELETE;
#ifdef SI_TESTER_VERBOSE
		printk(KERN_INFO "sadb_keymgmt_delete_sa: kernel sending SADBM_DELETE addr=%s spi= %lx peerspi = %lx flags = %x\n", 
		ntoa(d->dst.addr_union.ip_addr.s_addr),
		d->sa_info.spi, d->sa_info.peer_spi, d->sa_info.flags);
#endif
		s->m_sadbm.sadbm_msglen = sizeof(struct sadb_msg);

	s->sadb_info.dst.sa_info.flags = d->sa_info.flags;
	s->sadb_info.dst.sa_info.spi = d->sa_info.spi;
	s->sadb_info.dst.sa_info.peer_spi = d->sa_info.peer_spi;

		s->sadb_info.dst.dst.addr_family = AF_INET;
		s->sadb_info.dst.dst.addr_union.ip_addr.s_addr =
				d->dst.addr_union.ip_addr.s_addr;
		if(sadb_netlink_post(NETLINK_IKE, tmpskb))
			kfree_skb(tmpskb);
	}
	else
		printk(KERN_INFO "sadb_keymgmt_delete_sa: alloc_skb failure\n");

}
