/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/*
 * ipsec_doi.c - This file contains various IPsec DOI and Oakley 
 * 		resolution routines.
 */

#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include "constants.h"
#include "state.h"
#include "packet.h"
#include "defs.h"
#include DES_H
#include SHA_H
#include MD5_H
#include <sadb.h>
#include "argdefs.h"

/* Weak and semi week keys as taken from
 * %A D.W. Davies
 * %A W.L. Price
 * %T Security for Computer Networks
 * %I John Wiley & Sons
 * %D 1984
 * Many thanks to smb@ulysses.att.com (Steven Bellovin) for the reference
 * (and actual cblock values).
 */
/* added possible weak keys as of Schneier's book */
#define NUM_WEAK_KEY	64
static unsigned char weak_keys[NUM_WEAK_KEY][DES_KEY_SZ]={
  /* weak keys */
  {0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01},
  {0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE},
  {0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F},
  {0xE0,0xE0,0xE0,0xE0,0xE0,0xE0,0xE0,0xE0},
  /* semi-weak keys */
  {0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE},
  {0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01},
  {0x1F,0xE0,0x1F,0xE0,0x0E,0xF1,0x0E,0xF1},
  {0xE0,0x1F,0xE0,0x1F,0xF1,0x0E,0xF1,0x0E},
  {0x01,0xE0,0x01,0xE0,0x01,0xF1,0x01,0xF1},
  {0xE0,0x01,0xE0,0x01,0xF1,0x01,0xF1,0x01},
  {0x1F,0xFE,0x1F,0xFE,0x0E,0xFE,0x0E,0xFE},
  {0xFE,0x1F,0xFE,0x1F,0xFE,0x0E,0xFE,0x0E},
  {0x01,0x1F,0x01,0x1F,0x01,0x0E,0x01,0x0E},
  {0x1F,0x01,0x1F,0x01,0x0E,0x01,0x0E,0x01},
  {0xE0,0xFE,0xE0,0xFE,0xF1,0xFE,0xF1,0xFE},
  {0xFE,0xE0,0xFE,0xE0,0xFE,0xF1,0xFE,0xF1},
  /* possible weak keys */
  {0x1F,0x1F,0x01,0x01,0x0E,0x0E,0x01,0x01},
  {0xE0,0x01,0x01,0xE0,0xF1,0x01,0x01,0xF1},
  {0x01,0x1F,0x1F,0x01,0x01,0x0E,0x0E,0x01},
  {0xFE,0x1F,0x01,0xE0,0xFE,0x0E,0x01,0xF1},
  {0x1F,0x01,0x01,0x1F,0x0E,0x01,0x01,0x0E},
  {0xFE,0x01,0x1F,0xE0,0xFE,0x01,0x0E,0xF1},
  {0x01,0x01,0x1F,0x1F,0x01,0x01,0x0E,0x0E},
  {0xE0,0x1F,0x1F,0xE0,0xF1,0x0E,0x0E,0xF1},
  {0xFE,0x01,0x01,0xFE,0xFE,0x01,0x01,0xFE},
  {0xE0,0xE0,0x01,0x01,0xF1,0xF1,0x01,0x01},
  {0xE0,0x1F,0x01,0xFE,0xF1,0x0E,0x01,0xFE},
  {0xFE,0xFE,0x01,0x01,0xFE,0xFE,0x01,0x01},
  {0xE0,0x01,0x1F,0xFE,0xF1,0x01,0x0E,0xFE},
  {0xFE,0xE0,0x1F,0x01,0xFE,0xF1,0x0E,0x01},
  {0xFE,0x1F,0x1F,0xFE,0xFE,0x0E,0x0E,0xFE},
  {0xE0,0xFE,0x1F,0x01,0xF1,0xFE,0x0E,0x01},
  {0xFE,0xE0,0x01,0x1F,0xFE,0xF1,0x01,0x0E},
  {0x1F,0xFE,0x01,0xE0,0x0E,0xFE,0x01,0xF1},
  {0xE0,0xFE,0x01,0x1F,0xF1,0xFE,0x01,0x0E},
  {0x01,0xFE,0x1F,0xE0,0x01,0xFE,0x0E,0xF1},
  {0xE0,0xE0,0x1F,0x1F,0xF1,0xF1,0x0E,0x0E},
  {0x1F,0xE0,0x01,0xFE,0x0E,0xF1,0x01,0xFE},
  {0xFE,0xFE,0x1F,0x1F,0xFE,0xFE,0x0E,0x0E}, 
  {0x01,0xE0,0x1F,0xFE,0x01,0xF1,0x0E,0xFE},
  {0xFE,0x1F,0xE0,0x01,0xFE,0x0E,0xF1,0x01},
  {0x01,0x01,0xE0,0xE0,0x01,0x01,0xF1,0xF1},
  {0xE0,0x1F,0xFE,0x01,0xF1,0x0E,0xFE,0x01},
  {0x1F,0x1F,0xE0,0xE0,0x0E,0x0E,0xF1,0xF1},
  {0xFE,0x01,0xE0,0x1F,0xFE,0x01,0xF1,0x0E},
  {0x1F,0x01,0xFE,0xE0,0x0E,0x0E,0xF1,0xF1},
  {0xE0,0x01,0xFE,0x1F,0xF1,0x01,0xFE,0x0E},
  {0x01,0x1F,0xFE,0xE0,0x01,0x0E,0xFE,0xF1},
  {0x1F,0x01,0xE0,0xFE,0x0E,0x01,0xF1,0xFE},
  {0x01,0xE0,0xE0,0x01,0x01,0xE1,0xE1,0x01},
  {0x01,0x1F,0xE0,0xFE,0x01,0x0E,0xF1,0xFE},
  {0x1F,0xFE,0xE0,0x01,0x0E,0xFE,0xF0,0x01},
  {0x01,0x01,0xFE,0xFE,0x01,0x01,0xFE,0xFE},
  {0x1F,0xFE,0xE0,0x01,0x0E,0xF1,0xFE,0x01},
  {0x1F,0x1F,0xFE,0xFE,0x0E,0x0E,0xFE,0xFE},
  {0x01,0xFE,0xFE,0x01,0x01,0xFE,0xFE,0x01},
  {0x1F,0xE0,0xE0,0xF1,0x0E,0xF1,0xF1,0x0E},
  {0xFE,0xFE,0xE0,0xE0,0xFE,0xFE,0xF1,0xF1},
  {0x01,0xFE,0xE0,0x1F,0x01,0xFE,0xF1,0x0E},
  {0xE0,0xFE,0xFE,0xE0,0xF1,0xFE,0xFE,0xF1},
  {0x01,0xE0,0xFE,0x1F,0x01,0xF1,0xFE,0x0E},
  {0xFE,0xE0,0xE0,0xFE,0xFE,0xF1,0xF1,0xFE},
  {0x1F,0xFE,0xFE,0x1F,0x0E,0xFE,0xFE,0x0E},
  {0xE0,0xE0,0xFE,0xFE,0xF1,0xF1,0xFE,0xFE}};

#define PRINTADDRESS(x) (get_address(x) != (char *) NULL ? get_address(x) : "(unknown address family)")

extern char *get_address();

/************************************************************************/
u_char *get_preshared_key();
void initiate_quick();

/************************************************************************/

/*
 * Send a notification to the peer. We could make a decision on
 * whether to send the notification, based on the type and the
 * destination, if we care to.
 * XXX It doesn't handle DELETE notifications (which are also  informational exchanges).
 * Called by : ipsecdoi_handle_rfirst (ipsec_doi_mainxchg.c)
 */
void send_notification(int              sock, 
		       u_int16_t        type, 
		       u_char          *spi,
		       u_char           spilen, 
		       u_char           protoid, 
		       u_char          *icookie,
		       u_char          *rcookie, 
		       u_int32_t        msgid, 
		       struct sockaddr  sa)
{
  u_char                        buffer[sizeof(struct isakmp_hdr) + 
				      sizeof(struct isakmp_notification)];
  struct isakmp_hdr            *isa;
  struct isakmp_notification   *isan;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E send_notification  | in ipsecdoi.c");
#endif
  isa =  (struct isakmp_hdr *) buffer;
  isan = (struct isakmp_notification *) (buffer + sizeof(struct isakmp_hdr));
  bzero(buffer, sizeof(struct isakmp_hdr) +
	sizeof(struct isakmp_notification));
  
  if (icookie != (u_char *) NULL)
    bcopy(icookie, isa->isa_icookie, COOKIE_SIZE);
  
  if (rcookie != (u_char *) NULL)
    bcopy(rcookie, isa->isa_rcookie, COOKIE_SIZE);
  
  /* Standard header */
  isa->isa_np = ISAKMP_NEXT_N;
  isa->isa_maj = ISAKMP_MAJOR_VERSION;
  isa->isa_min = ISAKMP_MINOR_VERSION;
  isa->isa_xchg = ISAKMP_XCHG_INFO;
  isa->isa_msgid = msgid;
  isa->isa_length = htonl(sizeof(struct isakmp_hdr) + 
			  sizeof(struct isakmp_notification) +
			  spilen);
  
  /* Notification header */
  isan->isan_type = htons(type);
  isan->isan_doi = htonl(ISAKMP_DOI_IPSEC);
  isan->isan_length = htons(sizeof(struct isakmp_notification) + spilen);
  isan->isan_spisize = spilen;
  bcopy(spi, (u_char *)isan + sizeof(struct isakmp_notification), spilen);
  isan->isan_protoid = protoid;
  
#ifdef DEBUG
  fprintf(stdout, "sending Notify Message type %d to %s, port %d\n", type, 
      PRINTADDRESS(sa), get_port(sa));
#endif
  
  if (sendto(sock, buffer, ntohl(isa->isa_length), 0, &sa,
	     sizeof(sa)) != ntohl(isa->isa_length))
    log(0, "sendto() failed in send_notification() to %s, port %d", 
	PRINTADDRESS(sa), get_port(sa), 0);
#ifdef DEBUG
  packet_dump("\n\tSend a Notification\n", DUMP_MSG, 0);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(buffer, DUMP_SENT_NO_ENC, 0);
  show_payloads(buffer, 0);
  if (arg_verbose & DEBUG_VERBOSE)
  	fprintf(stdout, "transmitted %d bytes\n", ntohl(isa->isa_length));
#endif DEBUG
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q  send_notification  | in ipsecdoi.c");
#endif
}

/************************************************************************/
/*
 * Get an ESP Proposal. The second argument could be used to decide on
 * a per-peer basis what the proposal(s) should be. 
 * If the fourth argumentis true, then we include the encapsulation 
 * attribute.
 * XXX For now, we offer one very basic proposal.
 * Called by : oakley_get_sa (ipsec_doi.c)
 */

u_char * oakley_get_esp_prop(int               *length, 
			     struct sockaddr_in sin, 
			     struct state      *st,
			     int                tunnel,
			     int	        kernelfd,
			     int		initiator)
{
  u_char                   p[8192];
  u_char                  *q;
  u_char                  *pp;
  int                      i;
  int                      proposal = 1;
  int                      transform = 1;
  int                      k;
  u_int32_t                spi, peerspi;
  struct isakmp_proposal  *isap;
  struct isakmp_transform *isat;
  struct isakmp_attribute *isaa;
  struct isakmp_attribute_var_val    *isaatv;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E oakley_get_esp_prop  | in ipsecdoi.c");
#endif
  bzero(p, 8192);
  
  /* Proposal header -- ESP */
  isap = (struct isakmp_proposal *) p;
  i = sizeof(struct isakmp_proposal);  
  isap->isap_np = ISAKMP_NEXT_NONE;
  isap->isap_proposal = proposal++;
  isap->isap_protoid = PROTO_IPSEC_ESP;
  isap->isap_spisize = DOI_SPI_SIZE_IPSEC;
  isap->isap_notrans = 1;
  st->st_protoid = isap->isap_protoid;  
  
  /* If we are the responder, send peer_spi to kernel_getspi, to be placed in
     the partial SA (convert peerspi to host byte order - 
     ipsec transform routines do this for spi, but not for peer_spi)  */
  if (initiator)
    peerspi = 0;
  else
    peerspi = ntohl(*(unsigned long *)st->st_peer_spi);

/* Make sure encapsulation mode is set correctly */
  if (tunnel)
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TUNNEL;
  else
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TRANSPORT;
  /* Get an SPI -- it is assumed it's already in network byte order */
  spi = kernel_getspi(kernelfd, sin, initiator, 
	peerspi, st->st_att_encapsulation_mode);
  bcopy(&spi, p + i, sizeof(spi));
  st->st_spi = (u_char *) calloc(DOI_SPI_SIZE_IPSEC, sizeof(u_char));
  if (st->st_spi == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_esp_prop()", 0, 0, 0);
  bcopy(&spi, st->st_spi, sizeof(spi));
  st->st_spi_len = DOI_SPI_SIZE_IPSEC;
  i += DOI_SPI_SIZE_IPSEC;
   
  /* Transform header */
  isat = (struct isakmp_transform *) (p + i);  
  isat->isat_np = ISAKMP_NEXT_NONE;
  isat->isat_transnum = transform++;
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set enc alg as specified by -e option (default = IPSEC_DES3_ALG) */
  isat->isat_transid = arg_enc_alg;       
  st->st_esp_keymat_len = arg_enc_key_len;
  isat->isat_length = i;		/* Use as a temporary variable */
  st->st_transid = isat->isat_transid;
  i += sizeof(struct isakmp_transform);  
  
st->st_att_sa_duration_seconds = arg_ipsec_life_duration_seconds;
st->st_expire = arg_ipsec_life_duration_seconds;
if (arg_ipsec_life_duration_seconds)
  {
  /* SA lifetime type - seconds */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_TYPE;  
  isaa->isaat_lv = htons(SA_LIFE_TYPE_SECONDS);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);  
  i += sizeof(struct isakmp_attribute);  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "SA lifetime type: %d\n", ntohs(isaa->isaat_lv));
#endif
    
  /* SA lifetime duration - seconds */
#ifndef TEST_VAR_ATTR
/* Make SA Lifetime for ESP a basic attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_DURATION;
  isaa->isaat_lv = htons(arg_ipsec_life_duration_seconds);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);  
  i += sizeof(struct isakmp_attribute);
#else
/* Make SA Lifetime for ESP a variable length attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = htons(SA_LIFE_DURATION);
  isaa->isaat_lv = htons(4);
  i += sizeof(struct isakmp_attribute);

  isaatv = (struct isakmp_attribute_var_val *) (p + i);
  isaatv->isaatv_val = htonl(arg_ipsec_life_duration_seconds);
  i += sizeof(struct isakmp_attribute_var_val);
#endif TEST_VAR_ATTR
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "IPSEC SA lifetime: %u seconds\n", arg_ipsec_life_duration_seconds);
#endif
  }
  
st->st_att_sa_duration_kbytes = arg_ipsec_life_duration_kbytes;
if (arg_ipsec_life_duration_kbytes)
  {
  /* SA lifetime type - kilobytes */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_TYPE;  
  isaa->isaat_lv = htons(SA_LIFE_TYPE_KBYTES);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "SA lifetime type: %d\n", ntohs(isaa->isaat_lv));
#endif
    
  /* SA lifetime duration - kilobytes */
#ifndef TEST_VAR_ATTR
/* Make SA Lifetime for ESP a basic attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_DURATION;
  isaa->isaat_lv = htons(arg_ipsec_life_duration_kbytes);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
#else
/* Make SA Lifetime for ESP a variable length attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = htons(SA_LIFE_DURATION);
  isaa->isaat_lv = htons(4);
  i += sizeof(struct isakmp_attribute);

  isaatv = (struct isakmp_attribute_var_val *) (p + i);
  isaatv->isaatv_val = htonl(arg_ipsec_life_duration_kbytes);
  i += sizeof(struct isakmp_attribute_var_val);
#endif TEST_VAR_ATTR
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "IPSEC SA lifetime: %u kilobytes\n", arg_ipsec_life_duration_kbytes);
#endif
  }

  /* Replay protection */
  /* TODO Should now be in a notification payload */
  
  /* Encapsulation mode */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_ENCAPSULATION_MODE;
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set encapsulation mode as specified by -t option 
     (default = transport mode) */
  if (tunnel)
    {
      isaa->isaat_lv = htons(ENCAPSULATION_MODE_TUNNEL);
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TUNNEL;
    }
  else
    {
      isaa->isaat_lv = htons(ENCAPSULATION_MODE_TRANSPORT);
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TRANSPORT;
    }
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  st->st_att_encapsulation_mode = ntohs(isaa->isaat_lv);
  i += sizeof(struct isakmp_attribute);  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Encapsulation mode: %d\n", ntohs(isaa->isaat_lv));
#endif
  
  /* group attribute - only send this for PFS */
  if(arg_pfs_requested)
	{
  	isaa = (struct isakmp_attribute *) (p + i);
  	isaa->isaat_type = SA_GROUP_DESCRIPTION;
  	isaa->isaat_lv = htons(OAKLEY_DEFAULT_GROUP);
  	st->st_att_group = OAKLEY_DEFAULT_GROUP;
  	isaa->isaat_af = 1;
  	isaa->isaat_type = htons(isaa->isaat_type);
  	i += sizeof(struct isakmp_attribute);
	}

  /* HMAC algorithm */
  /* Create an HMAC entry only if authentication is also requested      */
  if (arg_key_mgmt_goal & GOAL_AUTHENTICATE)
    {
      isaa = (struct isakmp_attribute *) (p + i);
      isaa->isaat_type = SA_AUTHENTICATION_ALGORITHM;
      /* XXX Temporary code: should be handled by IPSEC policy mechanism */
      /* XXX set auth alg as specified by -a option (default = IPSEC_HMACMD596) */
      switch (arg_auth_alg)
	{
	case AH_MD5:
	  isaa->isaat_lv = htons(AUTH_ALGO_HMAC_MD5);
	  st->st_ah_keymat_len = 16;
	  st->st_att_authentication_alg = AUTH_ALGO_HMAC_MD5;
	  break;
	case AH_SHA:
          isaa->isaat_lv = htons(AUTH_ALGO_HMAC_SHA_1);
	  st->st_ah_keymat_len = 20;
          st->st_att_authentication_alg = AUTH_ALGO_HMAC_SHA_1;
	  break;
	default:
	  log(0, "The authentication algorithm attribute is not recognized\n", 0, 0, 0);
	}
      isaa->isaat_af = 1;
      isaa->isaat_type = htons(isaa->isaat_type);
      st->st_att_authentication_alg = ntohs(isaa->isaat_lv);
      i += sizeof(struct isakmp_attribute);      
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "HMAC algorithm: %d\n", ntohs(isaa->isaat_lv));
#endif
    }
  
  /* SA key length */
/* Create a key length entry only for crypto algorithms with variable key size */
if ((arg_enc_alg == ESP_RC5 ) || (arg_enc_alg == ESP_BLOWFISH)) {
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_KEY_LENGTH;  
/* Send key length in bits */
  isaa->isaat_lv = htons(8 * arg_enc_key_len);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "SA key length: %d bytes (%d bits)\n", arg_enc_key_len, 
		ntohs(isaa->isaat_lv));
#endif
}
  
  /* A couple of default values */
  st->st_ca_name = CA_DISTINGUISHED_NAME_DNSSEC;  
  isat->isat_length = htons(i - isat->isat_length);
  isap->isap_length = htons(i);
  
  q = (u_char *)calloc(i, sizeof(u_char));
  if (q == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_esp_prop()", 0, 0, 0);
  
  bcopy(p, q, i);
  *length = i;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q  oakley_get_esp_prop   | in ipsecdoi.c");
#endif
  return q;
}

/************************************************************************/
/*
 * Get an AH Proposal.
 * XXX For now, we offer one very basic proposal.
 * Called by : oakley_get_sa (ipsec_doi.c)
 */
u_char * oakley_get_ah_prop(int               *length, 
			    struct sockaddr_in sin, 
			    struct state      *st,
			    int                tunnel,
			    int		       kernelfd,
			    int		       initiator)
{
  u_char                   p[8192];
  u_char                  *q;
  u_char                  *pp;
  int                      i;
  int                      proposal = 1;
  int                      transform = 1;
  int                      k;
  u_int32_t                spi;
  u_int32_t                peerspi;
  struct isakmp_proposal  *isap;
  struct isakmp_transform *isat;
  struct isakmp_attribute *isaa;
  struct isakmp_attribute_var_val    *isaatv;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E oakley_get_ah_prop  | in ipsecdoi.c");
#endif
  bzero(p, 8192);
  
  /* Proposal header -- AH */
  isap = (struct isakmp_proposal *) p;
  i = sizeof(struct isakmp_proposal);
	
  isap->isap_np = ISAKMP_NEXT_NONE;
  isap->isap_proposal = proposal++;
  isap->isap_protoid = PROTO_IPSEC_AH;
  isap->isap_spisize = DOI_SPI_SIZE_IPSEC;
  isap->isap_notrans = 1;
  st->st_protoid = isap->isap_protoid;
    
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
  fprintf(stdout, "Protocol: %d\n", isap->isap_protoid);
#endif
  
  /* If we are the responder, send peer_spi to kernel_getspi, to be placed in
     the partial SA (convert peerspi to host byte order - 
     ipsec transform routines do this for spi, but not for peer_spi)  */
  if (initiator)
    peerspi = 0;
  else
    peerspi = ntohl(*(unsigned long *)st->st_peer_spi);
  /* Get an SPI -- it is assumed it's already in network byte order */
/* Make sure encapsulation mode is set correctly */
  if (tunnel)
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TUNNEL;
  else
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TRANSPORT;
  spi = kernel_getspi(kernelfd, sin, initiator, 
	peerspi, st->st_att_encapsulation_mode);

  bcopy(&spi, p + i, sizeof(spi));
  
  st->st_spi = (u_char *) calloc(DOI_SPI_SIZE_IPSEC, sizeof(u_char));
  if (st->st_spi == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_ah_prop()", 0, 0, 0);
  bcopy(&spi, st->st_spi, sizeof(spi));
  st->st_spi_len = DOI_SPI_SIZE_IPSEC;
  
  i += DOI_SPI_SIZE_IPSEC;
  
  /* Transform header -- AH-MD5 */
  isat = (struct isakmp_transform *) (p + i);
  
  isat->isat_np = ISAKMP_NEXT_NONE;
  isat->isat_transnum = transform++;
  /*10/28*/
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set auth alg as specified by -a option (default = IPSEC_HMACMD596) */
  isat->isat_transid = arg_auth_alg; 
  st->st_ah_keymat_len = arg_auth_key_len;
  st->st_esp_keymat_len = 0;
  isat->isat_length = i;		/* Use as a temporary variable */
  st->st_transid = isat->isat_transid;
  i += sizeof(struct isakmp_transform);
  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Transform: %d\n", isat->isat_transid);
#endif
  
st->st_att_sa_duration_seconds = arg_ipsec_life_duration_seconds;
st->st_expire = arg_ipsec_life_duration_seconds;
if (arg_ipsec_life_duration_seconds)
  {
  /* SA lifetime type - seconds */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_TYPE;
  isaa->isaat_lv = htons(SA_LIFE_TYPE_SECONDS);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "SA lifetime type: %d\n", ntohs(isaa->isaat_lv));
#endif
  
  /* SA lifetime duration - seconds */
#ifndef TEST_VAR_ATTR
/* Make SA Lifetime for AH a basic attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_DURATION;
  isaa->isaat_lv = htons(arg_ipsec_life_duration_seconds);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);  
  i += sizeof(struct isakmp_attribute);
#else
/* Make SA Lifetime for AH a variable length attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = htons(SA_LIFE_DURATION);
  isaa->isaat_lv = htons(4);
  i += sizeof(struct isakmp_attribute);

  isaatv = (struct isakmp_attribute_var_val *) (p + i);
  isaatv->isaatv_val = htonl(arg_ipsec_life_duration_seconds);
  i += sizeof(struct isakmp_attribute_var_val);
#endif TEST_VAR_ATTR
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "IPSEC SA lifetime: %u seconds\n", arg_ipsec_life_duration_seconds);
#endif
  }
  
st->st_att_sa_duration_kbytes = arg_ipsec_life_duration_kbytes;
if (arg_ipsec_life_duration_kbytes)
  {
  /* SA lifetime type - kilobytes */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_TYPE;
  isaa->isaat_lv = htons(SA_LIFE_TYPE_KBYTES);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "SA lifetime type: %d\n", ntohs(isaa->isaat_lv));
#endif
  
  /* SA lifetime duration - kilobytes */
#ifndef TEST_VAR_ATTR
/* Make SA Lifetime for AH a basic attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_LIFE_DURATION;
  isaa->isaat_lv = htons(arg_ipsec_life_duration_kbytes);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
#else
/* Make SA Lifetime for AH a variable length attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = htons(SA_LIFE_DURATION);
  isaa->isaat_lv = htons(4);
  i += sizeof(struct isakmp_attribute);

  isaatv = (struct isakmp_attribute_var_val *) (p + i);
  isaatv->isaatv_val = htonl(arg_ipsec_life_duration_kbytes);
  i += sizeof(struct isakmp_attribute_var_val);
#endif TEST_VAR_ATTR
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "IPSEC SA lifetime: %u kilobytes\n", arg_ipsec_life_duration_kbytes);
#endif
  }

  /* Replay protection */
  /* TODO :  Should now be in a notification payload */
  
  /* group attribute - only send this for PFS */
  if(arg_pfs_requested)
	{
  	isaa = (struct isakmp_attribute *) (p + i);
  	isaa->isaat_type = SA_GROUP_DESCRIPTION;
  	isaa->isaat_lv = htons(OAKLEY_DEFAULT_GROUP);
  	st->st_att_group = OAKLEY_DEFAULT_GROUP;
  	isaa->isaat_af = 1;
  	isaa->isaat_type = htons(isaa->isaat_type);
  	i += sizeof(struct isakmp_attribute);
	}

  /* Encapsulation mode */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_ENCAPSULATION_MODE;
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set encapsulation mode as specified by -t option 
     (default = transport mode) */
  if (tunnel)
    {
      isaa->isaat_lv = htons(ENCAPSULATION_MODE_TUNNEL);
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TUNNEL;
    }
  else
    {
      isaa->isaat_lv = htons(ENCAPSULATION_MODE_TRANSPORT);
      st->st_att_encapsulation_mode = ENCAPSULATION_MODE_TRANSPORT;
    }
  isaa->isaat_af = 1;
   isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);

  /* Authentication algorithm */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = SA_AUTHENTICATION_ALGORITHM;
  switch (arg_auth_alg)
    {
    case AH_MD5:
      isaa->isaat_lv = htons(AUTH_ALGO_HMAC_MD5);
      st->st_att_authentication_alg = AUTH_ALGO_HMAC_MD5;
      break;
    case AH_SHA:
      isaa->isaat_lv = htons(AUTH_ALGO_HMAC_SHA_1);
      st->st_att_authentication_alg = AUTH_ALGO_HMAC_SHA_1;
	break;
/*    case AH_DES:
      isaa->isaat_lv = htons(AUTH_ALGO_DES_MAC);
      st->st_att_authentication_alg = AUTH_ALGO_DES_MAC;
      break; */
    default:
      log(0, "The authentication algorithm attribute is not recognized\n", 0, 0, 0);
    }
  /*  isaa->isaat_lv = htons(auth_alg);*/
  isaa->isaat_af = 1;
  isaa->isaat_type = htons (isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
  
  isat->isat_length = htons(i - isat->isat_length);
  isap->isap_length = htons(i);
  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Encapsulation mode: %d\n", ntohs(isaa->isaat_lv));
#endif
  
  /* A few default values */
  st->st_ca_name = CA_DISTINGUISHED_NAME_DNSSEC;
  
  q = (u_char *) calloc(i, sizeof(u_char));
  if (q == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_ah_prop()", 0, 0, 0);
  
  bcopy(p, q, i);
  *length = i;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q oakley_get_ah_prop   | in ipsecdoi.c");
#endif
  return q;
}

/************************************************************************/
/*
 * Get an Oakley Proposal.
 * XXX For now, we offer one very basic proposal.
 * XXX Since we only do this one proposal, we also copy the value in
 * XXX the state object at this point. This will have to change if/when
 * XXX we start offering multiple proposals (see ipsec_prop_to_state()).
 * Called by : oakley_get_sa (ipsec_doi.c)
 */
u_char * oakley_get_oakley_prop(int               *length, 
				struct sockaddr_in sin, 
				struct state      *st,
				int	      kernelfd,
				int		initiator)
{
  u_char                   p[8192];
  u_char                  *q;
  u_char                  *pp;
  int                      i;
  int                      k;
  int                      proposal = 1;
  int                      transform = 1;
  u_int32_t                spi;
  struct isakmp_proposal  *isap;
  struct isakmp_transform *isat;
  struct isakmp_attribute *isaa;
  struct isakmp_attribute_var_val    *isaatv;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E oakley_get_oakley_prop  | in ipsecdoi.c");
#endif
  bzero(p, 8192);
  
  /* Proposal header -- ISAKMP */
  isap = (struct isakmp_proposal *) p;
  i = sizeof(struct isakmp_proposal);
  
  isap->isap_np = ISAKMP_NEXT_NONE;
  isap->isap_proposal = proposal++;
  isap->isap_protoid = PROTO_ISAKMP;
  isap->isap_spisize = DOI_SPI_SIZE_ISAKMP; 
  isap->isap_notrans = 1;
  st->st_protoid = PROTO_ISAKMP;
  
  /* Use 8 bytes of the initiator cookie to be the ISAKMP SPI. 
     We also could not set any SPI at all... */
  bcopy(st->st_icookie, p + i, DOI_SPI_SIZE_ISAKMP); 
  
  st->st_spi = (u_char *) calloc(DOI_SPI_SIZE_ISAKMP, sizeof(u_char));
  if (st->st_spi == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_oakley_prop()", 0, 0, 0);
  bcopy(st->st_icookie, st->st_spi, DOI_SPI_SIZE_ISAKMP);
  st->st_spi_len = DOI_SPI_SIZE_ISAKMP;

  i += DOI_SPI_SIZE_ISAKMP; 
  
  /* Transform header -- Oakley */
  isat = (struct isakmp_transform *) (p + i);
  
  isat->isat_np = ISAKMP_NEXT_NONE;
  isat->isat_transnum = transform++;
  isat->isat_transid = KEY_OAKLEY;
  st->st_transid = KEY_OAKLEY;
  isat->isat_length = i;		/* Use of as a temporary variable */
  i += sizeof(struct isakmp_transform);
  
  /* Encryption algorithm attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = OAKLEY_ENCRYPTION_ALGORITHM;
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set isakmp enc alg (default = OAKLEY_DES_CBC) */
  /* DES or 3DES ; should be managed by policy mechanism*/
  isaa->isaat_lv = htons(arg_isakmp_enc);
  isaa->isaat_af = 1;
  /* DES or 3DES ; should be managed by policy mechanism*/
  st->st_enc = arg_isakmp_enc;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
  /* Hash algorithm attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = OAKLEY_HASH_ALGORITHM;
  /* XXX Acceptable algorithm = SHA or MD5                          */
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set isakmp auth alg as specified by -s option (default = OAKLEY_MD5) */
  isaa->isaat_lv = htons(arg_isakmp_auth); 
  isaa->isaat_af = 1;
  /* XXX Temporary code: should be handled by IPSEC policy mechanism */
  /* XXX set isakmp auth alg as specified by -s option (default = OAKLEY_MD5) */
  st->st_hash = arg_isakmp_auth; 
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
  /* Authentication method attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = OAKLEY_AUTHENTICATION_METHOD;
  
  pp = (u_char *) get_preshared_key(st, &k);
  if (pp != (u_char *) NULL)
      free(pp);
  else
    {
      /* XXX Find another way to authenticate */
#ifndef WIT
	  log(1,"Pre-shared secret key not found for %s - using generic key",
#else
/* WIT exits on errors */
	  log(1,"Pre-shared secret key not found for %s",
#endif WIT
			  PRINTADDRESS(sin), 0, 0);
    }
  st->st_auth = OAKLEY_PRESHARED_KEY;
  isaa->isaat_lv = htons(OAKLEY_PRESHARED_KEY);
  
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
  /* Group Description attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = OAKLEY_GROUP_DESCRIPTION;
  isaa->isaat_lv = htons(OAKLEY_DEFAULT_GROUP);
  isaa->isaat_af = 1;
  st->st_group = OAKLEY_DEFAULT_GROUP;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
  /* Lifetime Type attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = OAKLEY_LIFE_TYPE;
  isaa->isaat_lv = htons(OAKLEY_LIFE_SECONDS);
  isaa->isaat_af = 1;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
  
  /* Lifetime Duration attribute */
/* Make SA Lifetime a basic attribute */
#ifndef TEST_VAR_ATTR
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = OAKLEY_LIFE_DURATION;
  isaa->isaat_lv = htons(arg_oakley_life_duration_seconds);   
  isaa->isaat_af = 1;
  st->st_expire = arg_oakley_life_duration_seconds;
  isaa->isaat_type = htons(isaa->isaat_type);
  i += sizeof(struct isakmp_attribute);
#else
/* Make SA Lifetime a variable length attribute */
  isaa = (struct isakmp_attribute *) (p + i);
  isaa->isaat_type = htons(OAKLEY_LIFE_DURATION);
  isaa->isaat_lv = htons(4);
  st->st_expire = arg_oakley_life_duration_seconds;
  i += sizeof(struct isakmp_attribute);

  isaatv = (struct isakmp_attribute_var_val *) (p + i);
  isaatv->isaatv_val = htonl(arg_oakley_life_duration_seconds);
  i += sizeof(struct isakmp_attribute_var_val);
#endif TEST_VAR_ATTR
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Oakley SA lifetime: %u seconds\n", arg_oakley_life_duration_seconds);
#endif
  
  isat->isat_length = htons(i - isat->isat_length);
  isap->isap_length = htons(i);
  
  q = (u_char *) calloc(i, sizeof(u_char));
  if (q == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_oakley_prop()", 0, 0, 0);
  
  bcopy(p, q, i);
  *length = i;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q oakley_get_oakley_prop   | in ipsecdoi.c");
#endif
  
  return q;
}


/************************************************************************/
/*
 * Construct an SA payload for Oakley, depending on the goal. Return
 * a valid SA payload, length of the buffer on the first argument.
 * Called by : oakley_initiate (ipsec_doi.c)
 * Called by : initiate_quick (ipsec_doi_quickxchg.c)
 * Called by : ipsecdoi_handle_quick_r1 (ipsec_doi_quickxchg.c)
 */
u_char * oakley_get_sa(int               *length, 
		       struct sockaddr_in sin, 
		       int                goal, 
		       struct             state *st, 
		       int                kernelfd, 
		       int                initiator)
{
  struct isakmp_sa  *isa;
  u_char             p[8192];
  u_char            *q;
  int                i;
  int                j;
  int                k;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E  oakley_get_sa  | in ipsecdoi.c");
#endif
  bzero(p, 8192);
    
  isa = (struct isakmp_sa *) p;
  
  isa->isasa_np = ISAKMP_NEXT_NONE;
  isa->isasa_doi = htonl(ISAKMP_DOI_IPSEC);
  
  i = sizeof(struct isakmp_sa);
  
  /* We don't do labeled integrity/secrecy, so forget the LDI */
  p[i + IPSEC_DOI_SITUATION_LENGTH - 1] = SIT_IDENTITY_ONLY;
  i += IPSEC_DOI_SITUATION_LENGTH;
  
  if (goal & GOAL_KEYMANAGEMENT)
    {
      q = oakley_get_oakley_prop(&j, sin, st, kernelfd, initiator);
      if (q == (u_char *) NULL)
	{
#ifdef DEBUG_IN_OUT
	  in_out_functions(" Q  oakley_get_sa -er1- | in ipsecdoi.c");	    
#endif
	  return q;
	}
      bcopy(q, p + i, j);
      free(q);
      i += j;
    }
  else
    {
      if (goal & GOAL_ENCRYPT)	/* Does authentication too */
	{
	  q = oakley_get_esp_prop(&j, sin, st, goal & GOAL_TUNNEL, kernelfd,
				  initiator);
	  if (q == (u_char *) NULL)
	    {
#ifdef DEBUG_IN_OUT
	      in_out_functions(" Q  oakley_get_sa -2- | in ipsecdoi.c");	    
#endif
	      return q;
	    }
	  bcopy(q, p + i, j);
	  free(q);
	  i += j;
	}
      else
	if (goal & GOAL_AUTHENTICATE)
	  {
	    q = oakley_get_ah_prop(&j, sin, st, goal & GOAL_TUNNEL, kernelfd,
				   initiator);
	    if (q == (u_char *) NULL)
	      {
#ifdef DEBUG_IN_OUT
		in_out_functions(" Q  oakley_get_sa -3- | in ipsecdoi.c");	    
#endif
		return q;
	      }
	    bcopy(q, p + i, j);
	    free(q);
	    i += j;
	  }
	else
	  exit_log("invalid goal (%d) in oakley_get_sa", goal, 0, 0); /* XXX */
    }
  
  isa->isasa_length = htons(i);
  q = (u_char *) calloc(i, sizeof(u_char));
  if (q == (u_char *) NULL)
    exit_log("calloc() failed in oakley_get_sa()", 0, 0, 0);
  
  bcopy(p, q, i);
  *length = i;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q oakley_get_sa   | in ipsecdoi.c");
#endif
  
  return q;
}
 
/************************************************************************/
/*
 * Initiate an Oakley exchange.
 * Called by ipsecdoi_initiate (ipsec_doi.c)
 *	oldspi == 0 ==> no existing IPsec SA
 *	oldspi != 0 ==> spi of existing IPsec SA that is expiring
 */
void oakley_initiate(int                sock, 
		     struct sockaddr_in sin, 
		     int                goal,
		     int		kernelfd,
		     u_int32_t	  	oldspi)
{
  struct isakmp_hdr isa;
  struct isakmp_sa *isasa;
  struct event     *ev;
  struct state     *st, *st2;
  u_char           *buffer;
  u_char           *p;
  int               j, k, repl_found = 0;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
  /* 
   * If there's already an ISAKMP SA established, use that and
   * go directly to Quick Mode.
   * XXX If an ISAKMP SA is *being* established, we foolishly
   * XXX try to establish another one, in parallel. We could
   * XXX issue an event to wait for it to finish and then try
   * XXX to establish it.
   */
#ifdef DEBUG_IN_OUT
  in_out_functions(" E oakley_initiate   | in ipsecdoi.c");
#endif
#ifdef DEBUG
  fprintf(stdout, "Initiate an Oakley exchange: send HDR + SA to %s \n", inet_ntoa(sin.sin_addr));
#endif
  st = (struct state *) find_phase1_state(sin);
  if (st != (struct state *) NULL)
    {
#ifdef DEBUG
      fprintf(stdout, "Begin Quick Mode Exchange with %s, port %d, goal : ",
	  PRINTADDRESS(sin), get_port(sin));
      if (goal & GOAL_KEYMANAGEMENT)
	fprintf(stdout, "KEYMANAGEMENT\n");
      else 
	{
	  if (goal & GOAL_ENCRYPT)
	    fprintf(stdout,"ENCRYPT\n");
	  else 
	    if (goal & GOAL_AUTHENTICATE)
	      fprintf(stdout, "AUTHENTICATE\n");
	}            
#endif
      st->st_goal = goal;

/*    Before initiating a Phase 2 negotiation,
 *     		check to see if another Phase 2 SA already exists
 */
#ifdef TEST_PR_STATE
	pr_state_table();
#endif TEST_PR_STATE
  	if (oldspi) 
    	  {
    	    st2 = (struct state *) find_phase2_state_to_expire(sin, oldspi);
  	    if (st2 == (struct state *) NULL)
		/* Neither expiring SA nor replacement SA found 
		 * Initiate Quick Mode Negotiation */
      		log(0, "oakley_initiate: expiring IPsec SA NOT found addr=%s spi=%lx",
				inet_ntoa(sin.sin_addr), ntohl(oldspi), 0);
  	    else if (st2 == (struct state *) -1)
		{
	       /* Replacement SA found, but expiring SA not found 
		* No Need to Initiate Quick Mode Negotiation */
      		log(0, "oakley_initiate: expiring IPsec SA NOT found addr=%s spi=%lx",
				inet_ntoa(sin.sin_addr), ntohl(oldspi), 0);
#ifdef TEST_KERN
          		fprintf(stdout, "oakley_initiate: replacement phase2 SA found addr=%s spi=%lx\n",
				inet_ntoa(sin.sin_addr), ntohl(oldspi));
#endif TEST_KERN
		return;
		}
  	    else
		{
	       /* Expiring SA found */
	         st2->st_flags |= ST_FLAG_OUTBOUND_EXPIRED;
        	 if (st2->st_flags & ST_FLAG_REPL_FOUND) 
			{
		/* Both expiring SA and replacement SA found 
		 * No Need to Initiate Quick Mode Negotiation */
    			/* Turn OFF replacement_found flag */
	                st2->st_flags ^= ST_FLAG_REPL_FOUND;
#ifdef TEST_KERN
          		fprintf(stdout, "oakley_initiate: replacement phase2 SA found addr=%s spi=%lx\n",
				inet_ntoa(sin.sin_addr), ntohl(oldspi));
#endif TEST_KERN
			return;
			}
		}
#ifdef TEST_KERN
      			fprintf(stdout, "oakley_initiate: replacement phase2 SA NOT found addr=%s spi=%lx\n",
				inet_ntoa(sin.sin_addr), ntohl(oldspi));
#endif TEST_KERN
    	  }
     /* Replacement SA not found 
      * Initiate Quick Mode Negotiation */
  	    initiate_quick(sock, st, kernelfd);
            return;
    }
  
  /* No phase1 state has been found*/
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout,"\t No phase1 state has been found\n");
#endif
  bzero(&isa, sizeof(isa));
  
  /* Standard header initialization */
  isa.isa_maj = ISAKMP_MAJOR_VERSION;
  isa.isa_min = ISAKMP_MINOR_VERSION;
  isa.isa_np = ISAKMP_NEXT_SA;
  isa.isa_xchg = ISAKMP_XCHG_IDPROT;
  isa.isa_length = sizeof(isa);
  /* R-cookie, flags and MessageID is left zero */
    
  /* Initiator cookie */
  get_cookie(ISAKMP_INITIATOR, &(isa.isa_icookie), COOKIE_SIZE, sin);
  
  st = (struct state *) get_state();
  
  bcopy(&sin, &(st->st_peer), sizeof(sin));
  
  /* Get an ISAKMP proposal (Oakley) */
  /* Arg#6 = 1 ==> tells oakley_get_sa that we are the initiator */
  bcopy(isa.isa_icookie, st->st_icookie, COOKIE_SIZE);
  p = oakley_get_sa(&j, sin, GOAL_KEYMANAGEMENT, st, kernelfd, 1);
  if (p == (u_char *) NULL)
    {
      free_state(st);
      return;
    }
  
  isa.isa_length += j;
  isa.isa_length = htonl(isa.isa_length);
  
  buffer = (u_char *) calloc(sizeof(isa) + j, sizeof(u_char));
  if (buffer == (u_char *) NULL)
    exit_log("calloc() failed in oakley_initiate()", 0, 0, 0);
  
  bcopy(&isa, buffer, sizeof(isa));
  bcopy(p, buffer + sizeof(isa), j);
  
  isasa = (struct isakmp_sa *) (buffer + sizeof(isa));

  /* fill up some attributes of the new state */
  st->st_myidentity_protoid = TRANSPORT_PROT_UDP; 
  st->st_myidentity_port = ntohs(ISAKMP_PORT); 
  st->st_peeridentity_protoid = TRANSPORT_PROT_UDP;
  st->st_peeridentity_port = ntohs(ISAKMP_PORT);
  st->st_situation = SIT_IDENTITY_ONLY;
  st->st_goal = goal;
  st->st_state = OAKLEY_MAIN_I_1;
  st->st_packet = buffer;
  st->st_packet_len = sizeof(isa) + j;
  st->st_doi = ISAKMP_DOI_IPSEC;
    
  /* Copy sent SA */
  st->st_sa = (u_char *) calloc(j, sizeof(u_char));
  if (st->st_sa == (u_char *) NULL)
    exit_log("calloc() failed in oakley_initiate()", 0, 0, 0);
  bcopy(p, st->st_sa, j);

  free(p);

  insert_state(st);
  
  /* Transmit */
  if (sendto(sock, buffer, sizeof(isa) + j, 0, (struct sockaddr *)&sin,
	     sizeof(sin)) != sizeof(isa) + j)
    log(0, "sendto() failed in oakley_initiate()", 0, 0, 0);
#ifdef DEBUG
  packet_dump("\n\tINITIATOR->RESPONDER ***** MAIN MODE MESSAGE 1 : HDR + SA", DUMP_MSG, 0);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(buffer, DUMP_SENT_NO_ENC, 0);
  show_payloads(buffer, 0);
  if (arg_verbose & DEBUG_PACKETS)
  {
        fprintf(stdout, "-----\n");
    	show_header((struct isakmp_hdr *) buffer);
  }
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "w");
  fprintf(f_demo, "MAIN MODE MESSAGE 1 SENT BY INITIATOR\n");
  fclose(f_demo);
#endif DEMO
  
  /* Set up a retransmission event, half a minute henceforth */
  event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
#ifdef DEBUG
  fprintf(stdout, "\n=======================================================================\n");
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q oakley_initiate  | in ipsecdoi.c");
#endif
}



/************************************************************************/
/*
 * Demux for IPsec DOI initiator.
 * Called by Kernel_handle (kernel_comm.c)
 */
/* Added kernelfd as arg to ipsecdoi_initiate - 
   needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     
void ipsecdoi_initiate(int                sock, 
		       struct sockaddr_in sin, 
		       int                goal, 
		       int                protocol,
		       int		  kernelfd,
		       u_int32_t	  oldspi)
{
#ifdef DEBUG_IN_OUT
  in_out_functions(" E  ipsecdoi_initiate | in ipsecdoi.c");
#endif
  switch (protocol)
    {
    case KEY_OAKLEY:
      /* Added kernelfd as arg to oakley_initiate - 
	 needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     
      oakley_initiate(sock, sin, goal, kernelfd, oldspi);
      break;

    default:
      log(0, "unsupported protocol %d specified in ipsecdoi_initiate()",
	  protocol, 0, 0);
      return;
    }
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_initiate   | in ipsecdoi.c");
#endif
}

/************************************************************************/
/* 
 * Examine a list of IPsec DOI policies for IPsec and copy 
 * them into the state.
 * XXX The policy is hardwired now; we only accept a few select
 * XXX algorithms. 
 * XXX It also doesn't handle multiprotocol policies. 
 * XXX This should change so that 0 is the value returned if all
 * XXX went fine; otherwise, return a notification error type.
 * Called by : ipsecdoi_handle_quick_r1 (ipsec_doi_quickxchg.c)
 * Called by : ipsecdoi_handle_quick_i1 (ipsec_doi_quickxchg.c)
 */
int ipsecdoi_ipsec_sa_copy(u_char           *from, 
			   int               length,
			   struct sockaddr   sa, 
			   struct state     *st,
		  	   int              initiator)
{
  struct isakmp_proposal     *isapf;
  struct isakmp_attribute    *isaat;
  struct isakmp_attribute_var_val    *isaatv;
  struct isakmp_transform    *isat;
  struct isakmp_sa           *isasa;
  struct isakmp_generic      *isag;
  int                         d;
  int                         i;
  int                         j;
  int                         k;
  int                         l;
  int			      l1;
  int                         acceptable;
  int                         len;
  u_char                     *pp;
  int                         saduration_seconds;
  int                         saduration_kbytes;
  int                         salifetype;
  int                         groupdescription;
  int                         caname;
  int                         encapsulate;
  int                         authentication_alg;
  int                         protoid;
  int                         transid;
  int                         key_length = 0;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_ipsec_sa_copy   | in ipsecdoi.c");
#endif
  /* i points the proposal payload, ie after HDR + HASH + SA payload. */
  isag = (struct isakmp_generic *) (from + sizeof(struct isakmp_hdr));
  i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
    IPSEC_DOI_SITUATION_LENGTH + ntohs(isag->isag_length);
    
  isasa = (struct isakmp_sa *) (from + sizeof(struct isakmp_hdr) +
				ntohs(isag->isag_length));
  len = ntohs(isasa->isasa_length) - sizeof(struct isakmp_sa) -
	  IPSEC_DOI_SITUATION_LENGTH;
  
  /* Verify that DOI is IPsec */
  if (ntohl(isasa->isasa_doi) != ISAKMP_DOI_IPSEC)
    {
      log(0, "unsupported DOI %d from %s, port %d", ntohl(isasa->isasa_doi),
	  PRINTADDRESS(st->st_peer), get_port(st->st_peer));
      return 0;
    }

  /* Verify that this is a supported situation */
  bcopy(from + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
	ntohs(isag->isag_length), &d, sizeof(d));
  if (ntohl(d) != SIT_IDENTITY_ONLY)
    {
#ifdef DEBUG
      fprintf(stdout, "Unsupported situation %d from %s, port %d\n", d,
	  PRINTADDRESS(st->st_peer), get_port(st->st_peer));
#endif
      return 0;
    }
  
  while (1)
    { /* begin while (1) */
      /* Set up some defaults */
      saduration_seconds = arg_ipsec_life_duration_seconds;
      saduration_kbytes = arg_ipsec_life_duration_kbytes;
      caname = CA_DISTINGUISHED_NAME_DNSSEC;
      encapsulate = ENCAPSULATION_MODE_TUNNEL;
      salifetype = 0;
      groupdescription = 0;
      authentication_alg = 0;
      
      isapf = (struct isakmp_proposal *) (from + i);
      acceptable = 1;
      
      /* Length mismatch */
      if (ntohs(isapf->isap_length) > len)
	return 0;
      
      j = i + sizeof(struct isakmp_proposal) + isapf->isap_spisize;
      isat = (struct isakmp_transform *) (from + j);
      
      if (ntohs(isat->isat_length) > len + sizeof(struct isakmp_proposal) +
	  isapf->isap_spisize)
	return 0;
      
      protoid = isapf->isap_protoid;
#ifndef WIT
/* Check that responder's proposal matches proposal sent by initiator */
/* WIT - Also check that initiator's proposal matches responder's expectations */
      if (initiator)
#endif WIT
          if (arg_key_mgmt_goal & GOAL_ENCRYPT)
		 {
	         if (protoid != PROTO_IPSEC_ESP)
		    {
		    log(0, "Incorrect protocol ID %d in proposal (expected ESP)", protoid, 0, 0);
	            return 0;
		    }
		 }
  	  else if (protoid != PROTO_IPSEC_AH)
		{
		log(0, "Incorrect protocol ID %d in proposal (expected AH)", protoid, 0, 0);
	        return 0;
		}
      transid = isat->isat_transid;
      
      /* Check that the transform IDs/Protocols are valid */
      switch (protoid)
	{ /* begin switch protoid */
	case PROTO_IPSEC_AH:
	  switch (transid)
	    { /* begin switch transid */	    
	    case AH_MD5:
	      st->st_ah_keymat_len = 16;
	      break;
	    case AH_SHA:
	      st->st_ah_keymat_len = 20;
	      break;
	    default:
	      log(0, "unknown/unsupported transform %d in AH proposal", transid, 0, 0, 0);
	      return 0;
	      break;
	    } /* end switch transid */	      
#ifndef WIT
/* Check that responder's proposal matches proposal sent by initiator */
/* WIT - Also check that initiator's proposal matches responder's expectations */
      if (initiator)
#endif WIT
          if (transid != arg_auth_alg)
		{
		log(0, "Incorrect AH auth alg ID %d in proposal (expected %d)", 
			transid, arg_auth_alg, 0);
	        return 0;
		}
	  break;
	  
	case PROTO_IPSEC_ESP:
/* Intialize key_length to an invalid value to ensure it's set properly */
          key_length = -1;
	  switch (transid)
	    { /* begin switch transid */	    
	    case ESP_DES:
	      key_length = 8;
	      break;
	    case ESP_3DES:
	      key_length = 24;
	      break;
	    case ESP_BLOWFISH:
/* BLOWFISH key len must be specified in a KEY_LENGTH attribute entry */
	      break;
	    case ESP_DES_IV64:
	      key_length = 8;	      
	      break;
	    case ESP_RC5:
/* RC5 key len must be specified in a KEY_LENGTH attribute entry */
	      break;
	    case ESP_IDEA:
	      key_length = 16;
	      break;
	    case ESP_NULL:
	      key_length = 0;
	      break;
	    default:
	      log(0, "unknown/unsupported transform %d in ESP proposal",
		  transid, 0, 0, 0);
	      return 0;
	      break;
	    } /* end switch transid */	      
#ifndef WIT
/* Check that responder's proposal matches proposal sent by initiator */
/* WIT - Also check that initiator's proposal matches responder's expectations */
      if (initiator)
#endif WIT
          if (transid != arg_enc_alg)
		{
		log(0, "Incorrect ESP enc alg ID %d in proposal (expected %d)", 
			transid, arg_enc_alg, 0);
	        return 0;
		}
	  break;
	  
	default:
	  log(0, "unknown/unsupported protocol id %d", protoid, 0, 0);
	  return 0;
	} /* end switch protoid */
      
      if ((isapf->isap_spisize != DOI_SPI_SIZE_IPSEC) ||
	  (isapf->isap_notrans != 1) ||
	  (isat->isat_np != ISAKMP_NEXT_NONE))
	{
	  /* Next proposal, if there */
	  if (isapf->isap_np == ISAKMP_NEXT_P)
	    {
	      i += ntohs(isapf->isap_length);
	      if (length - i <= sizeof(struct isakmp_proposal))
		return 0;
	      
	      len += ntohs(isapf->isap_length) + isapf->isap_spisize;
	      isapf = (struct isakmp_proposal *) (from + i);
	      if (ntohs(isapf->isap_length) + i > length)
		return 0;
	      
	      continue;
	    }
	  else
	    return 0;
	}
      
      j += sizeof(struct isakmp_transform);
      
      k = ntohs(isat->isat_length) - sizeof(struct isakmp_transform);
      while (k != 0)
	{ /* begin while k != 0 */
	  isaat = (struct isakmp_attribute *) (from + j);
	  isaat->isaat_type = ntohs(isaat->isaat_type);
	  l = ntohs(isaat->isaat_lv);
	  
	  if (isaat->isaat_af)
	    {
	      j += sizeof(struct isakmp_attribute);
	      k -= sizeof(struct isakmp_attribute);
	    }
	  else
	    {
/* Receiver - accept variable length attribute */
	      j += sizeof(struct isakmp_attribute);
	      k -= sizeof(struct isakmp_attribute);
	      l1 = l;
	      isaatv = (struct isakmp_attribute_var_val *) (from + j);
	      l = ntohl(isaatv->isaatv_val);
	      j += l1;
	      k -= l1;
	    }
	  
	  /* 
	   * XXX We should also be checking whether the attribute
	   * is/should be TLV or TV 
	   */
	  switch (isaat->isaat_rtype)
	    { /* begin switch isaat->isaat_rtype */
	    case SA_LIFE_TYPE:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "SA life type %d\n", l);
#endif
	      if ((l != SA_LIFE_TYPE_SECONDS) &&
		  (l != SA_LIFE_TYPE_KBYTES))
		acceptable = 0;
	      else
		  salifetype = l;
	      break;
	      
	    case SA_LIFE_DURATION:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "SA life duration %u\n", l);
#endif
	      switch (salifetype)
		{ /* begin switch salifetype */
		case SA_LIFE_TYPE_SECONDS:
#ifdef DEBUG
		  if (l==0)
		  {
		    fprintf(stdout,"The life duration is 0 seconds; we accept it in DEBUG mode\n");
		    saduration_seconds = l;
		  }
		  else
		    {
#endif
		      if ((l < SA_LIFE_DURATION_MINIMUM) ||
			  (l > SA_LIFE_DURATION_MAXIMUM))			
			acceptable = 0;
		      else
			{
		    saduration_seconds = l;
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "IPSEC SA lifetime: %u seconds\n", l);
#endif
			}
#ifdef DEBUG
		    }
#endif
		  break;
		  
		case SA_LIFE_TYPE_KBYTES:
#ifdef DEBUG
		  if (l==0)
		    {
		      fprintf(stdout,"The life duration is 0 kbytes; we accept it in DEBUG mode\n");
		      saduration_kbytes = l;
		    }
		  else
		    {
#endif
		      if (l < SA_LIFE_DURATION_K_MINIMUM)
			acceptable = 0;
		      else
			{
		          saduration_kbytes = l;
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "IPSEC SA lifetime: %u kilobytes\n", l);
#endif
		         }
#ifdef DEBUG
	            }
#endif
		  break;
		  
		default:
#ifdef DEBUG
		    fprintf(stdout, "Life duration given without lifetype\n");
#endif
		  acceptable = 0;
		} /* end switch salifetype */
	      
	      break;
		    
	    case SA_GROUP_DESCRIPTION:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "group description %d\n", l);
#endif
	      if (l != OAKLEY_DEFAULT_GROUP)
		acceptable = 0;
	      else
		 groupdescription = l;
	      break;
	      
	    case SA_ENCAPSULATION_MODE:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "encapsulation mode %d\n", l);
#endif
	      if ((l != ENCAPSULATION_MODE_TUNNEL) &&
		  (l != ENCAPSULATION_MODE_TRANSPORT))
		acceptable = 0;
	      else
		  encapsulate = l;
#ifndef WIT
/* Check that responder's proposal matches proposal sent by initiator */
/* WIT - Also check that initiator's proposal matches responder's expectations */
      if (initiator)
#endif WIT
          if ((arg_key_mgmt_goal & GOAL_TUNNEL) &&
	     (l == ENCAPSULATION_MODE_TRANSPORT))
		{
		log(0, "Incorrect encapsulation mode %d in proposal (expected TUNNEL)", l, 0, 0);
	        return 0;
		}
          else if (!(arg_key_mgmt_goal & GOAL_TUNNEL) &&
	     (l == ENCAPSULATION_MODE_TUNNEL))
		{
		log(0, "Incorrect encapsulation mode %d in proposal (expected TRANSPORT)", l, 0, 0);
	        return 0;
		}
	      break;
	      
	    case SA_AUTHENTICATION_ALGORITHM:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "authentication algorithm %d\n", l);
#endif
	      switch (l)
		{ /* begin switch l */	    
		case AUTH_ALGO_HMAC_MD5:
		  st->st_ah_keymat_len = 16;
		  authentication_alg = l;
		  break;
		case AUTH_ALGO_HMAC_SHA_1:
		  st->st_ah_keymat_len = 20;
		  authentication_alg = l;
		  break;
		case AUTH_ALGO_DES_MAC:
		case AUTH_ALGO_KPDK:
		  log(0, "Unsupported attribute (authentication algo # %d) in ESP proposal", l, 0, 0, 0);
		  acceptable = 0;
		  break;
		default:
		  log(0, "Unknown attribute (authentication # %d) in ESP proposal", l, 0, 0, 0);
		  acceptable = 0;
		  break;
		} /* end switch l */	    
#ifndef WIT
/* Check that responder's proposal matches proposal sent by initiator */
/* WIT - Also check that initiator's proposal matches responder's expectations */
      if (initiator)
#endif WIT
          if (((arg_auth_alg == AH_MD5) && (l != AUTH_ALGO_HMAC_MD5)) ||
              ((arg_auth_alg == AH_SHA) && (l != AUTH_ALGO_HMAC_SHA_1)))
		{
		log(0, "Incorrect auth alg ID %d in proposal", 
			l, 0, 0);
	        return 0;
		} 
	      break;
	      
	    case SA_KEY_LENGTH:
/* Convert key length from bits to bytes */
	      l = l/8;
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
 		fprintf(stdout, "key length %d bytes (%d bits)\n", l, l*8);
#endif
	if (transid == ESP_RC5)
	  if ((l != 5) &&
	     (l != 16) &&
	     (l != 20))
		acceptable = 0;
	else if (transid == ESP_BLOWFISH)
	  if ((l < 5) ||
	     (l > 56))
		acceptable = 0;
	else 
	{
          log(0, "key length should not be specified for enc alg %d ", transid, 0, 0);
	  acceptable = 0;
	}
     if (acceptable) 
		  key_length = l;
     else
        log(0, "Invalid key length %d bytes (%d bits) for enc alg %d ", 
        		l, l*8, transid);
#ifndef WIT
/* Check that responder's proposal matches proposal sent by initiator */
/* WIT - Also check that initiator's proposal matches responder's expectations */
      if (initiator)
#endif WIT
          if (l != arg_enc_key_len)
		{
 		log(0, "Incorrect key length %d bytes (%d bits) in proposal (expected %d bytes)", 
 			l, l*8, arg_enc_key_len);
	        return 0;
		}
	      break;

	    default:
	      acceptable = 0;
	      log(0, "unsupported attribute %d from %s, port %d",
		  isaat->isaat_rtype, PRINTADDRESS(sa), get_port(sa));
	      break;
	    } /* end switch isaat->isaat_rtype */

	  isaat->isaat_type = htons(isaat->isaat_type);
	  
	  /* Something unacceptable with this policy */
	  if (acceptable == 0)
	    break;
	} /* end while k != 0 */
	  
/* If key_length was not set, policy is unacceptable */
	  if (key_length == -1)
		acceptable = 0;
      
      /* Something went wrong.*/
      if (acceptable == 0)
	{
	  /* Next proposal, if there */
	  if (isapf->isap_np == ISAKMP_NEXT_P)
	    {
#ifdef DEBUG
	      fprintf(stdout, "There is another proposal:\n");
#endif
	      i += ntohs(isapf->isap_length);
	      
	      if (length - i <= sizeof(struct isakmp_proposal))
		return 0;
	      
	      len += ntohs(isapf->isap_length) + isapf->isap_spisize;
	      isapf = (struct isakmp_proposal *) (from + i);
	      
	      if (ntohs(isapf->isap_length) + i > length)
		return 0;
	      
	      continue;
	    }
	  else
	      log(0,"Something wrong with the attributes of a proposed transform\n", 0, 0, 0);
	      return 0;
	}
#ifdef DEBUG
      else
	/* Should check if there is not another proposal with the same proposal number,
	   meaning there is an "AND". Something like ESP-DES ^ AH-MD5. 
	   We should be able to handle such proposals in the future. */
	{
	  /* if ((isapf->isap_np == ISAKMP_NEXT_P)&&
	      (isapf->isap_proposal == proposal number of accepted proposal
	      ...*/
      	   if (arg_verbose & DEBUG_PACKETS)
		   {
      			fprintf(stdout, "Accepted proposal(%d) :\n", ntohs(isapf->isap_length));
      			format_dump(isapf, ntohs(isapf->isap_length));
		   }
	}
#endif
	    
      st->st_proposal = (u_char *) calloc(ntohs(isapf->isap_length), 
					  sizeof(u_char));
      if (st->st_proposal == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_ipsec_sa_copy()", 0, 0, 0);
      
      bcopy(isapf, st->st_proposal, ntohs(isapf->isap_length));

      /* Copy SPI */
      st->st_peer_spi = (u_char *) calloc(DOI_SPI_SIZE_IPSEC, sizeof(u_char));
      if (st->st_peer_spi == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_ipsec_sa_copy()", 0, 0, 0);
      
      bcopy((u_char *)isapf + sizeof(struct isakmp_proposal), st->st_peer_spi, 
		DOI_SPI_SIZE_IPSEC);
      st->st_peer_spi_len = DOI_SPI_SIZE_IPSEC;
      
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "peer spi copied in ipsecdoi_ipsec_sa_copy \n");
#endif
      
      /* Change the order to suit the numbers in the draft */      
      /* Copy all accepted attributes */
      st->st_att_sa_duration_seconds = saduration_seconds;
      st->st_att_sa_duration_kbytes = saduration_kbytes;  
      st->st_att_group = groupdescription;
      st->st_att_encapsulation_mode = encapsulate;
      st->st_att_authentication_alg = authentication_alg;
      st->st_esp_keymat_len  = key_length;
      /* st->st_att_compress_dictionnary_size */
      /* st->st_att_compress_private_algorithm */
      /*      st->st_peer_ca_name = caname; */ /* not used anymore */
      st->st_protoid = protoid;
      st->st_transid = transid;
      st->st_peer_protoid = protoid;
      st->st_peer_transid = transid;
      
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
	  fprintf(stdout, "Peer SPI accepted (%d):\t", st->st_peer_spi_len);
	  format_dump(st->st_peer_spi, st->st_peer_spi_len);
	}
      if (arg_verbose & DEBUG_VALUES)
      {
          fprintf(stdout, "-----\nProperties of the accepted proposal:\nProtocol: %s(%d)\nTransform: %d\n", 
	      ((protoid == PROTO_IPSEC_AH) ? "AH" : "ESP"), protoid, 
	      transid);
          fprintf(stdout, "Expiration: %u seconds OR %u kilobytes\n", 
	      saduration_seconds,
	      saduration_kbytes);
          fprintf(stdout, "Encapsulation mode: %s(%d)\nAuthentication algorithm: %d\n",
	      ((encapsulate == ENCAPSULATION_MODE_TUNNEL) ? "tunnel" : "transport"),
	      encapsulate, authentication_alg);
          fprintf(stdout, "Group description: %d\n-----\n", 
	      groupdescription);      
      }
#endif
#ifdef DEBUG_IN_OUT
      in_out_functions(" Q  ipsecdoi_ipsec_sa_copy  | in ipsecdoi.c");
#endif
      return 1;
    } /* end while (1) */
}


/************************************************************************/
/*
 * Examine a list of IPsec DOI policies and copy the acceptable one. If
 * none is acceptable, return 0. Also copy the accepted proposal in the
 * state object.
 * XXX The policy is hardwired now; we only accept a policy
 * XXX with the Oakley transform.
 * XXX It also doesn't handle policies that span multiple protocols.
 * Called by : ipsecdoi_handle_rfirst (ipsec_doi_mainxchg.c)
 */
int ipsecdoi_oakley_sa_copy(u_char          *from, 
			    u_char          *to, 
			    int              length,
			    struct sockaddr  sa, 
			    struct state    *st)
{
  struct isakmp_proposal  *isapf;
  struct isakmp_proposal  *isapt;
  struct isakmp_attribute *isaat;
  struct isakmp_transform *isat;
  struct isakmp_sa        *isasa;
  struct isakmp_attribute_var_val    *isaatv;
  int                      l;
  int                      k;
  int                      j;
  int                      i;
  int                      enc;
  int                      hash;
  int                      group;
  int                      auth;
  int                      acceptable;
  int                      len;
  int                      d;
  int			      l1;
  u_char                  *pp;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E  ipsecdoi_oakley_sa_copy  | in ipsecdoi.c");
#endif
  i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +    IPSEC_DOI_SITUATION_LENGTH;
  isapt = (struct isakmp_proposal *) (to + i);
  
  isasa = (struct isakmp_sa *) (from + sizeof(struct isakmp_hdr));
  len = ntohs(isasa->isasa_length) - sizeof(struct isakmp_sa) -
    IPSEC_DOI_SITUATION_LENGTH;
  
  isasa = (struct isakmp_sa *) (to + sizeof(struct isakmp_hdr));
  
  while (1)
    {
      enc = 0;
      hash = 0;
      group = 0;
      auth = 0;
      isapf = (struct isakmp_proposal *) (from + i);
      acceptable = 1;
      
      /* Length mismatch */
      if (ntohs(isapf->isap_length) > len)
	return 0;
      
      j = i + sizeof(struct isakmp_proposal) + isapf->isap_spisize;
      isat = (struct isakmp_transform *) (from + j);
      
      if (ntohs(isat->isat_length) > len + sizeof(struct isakmp_proposal) +
	  isapf->isap_spisize)
	return 0;

      if ((isapf->isap_protoid != PROTO_ISAKMP) ||
	  /* cf isakmp-09.txt $3.5 */
	  (isapf->isap_spisize >= DOI_SPI_SIZE_ISAKMP_MAX) ||
	  (isapf->isap_notrans != 1) ||
	  (isat->isat_np != ISAKMP_NEXT_NONE) ||
	  (isat->isat_transid != KEY_OAKLEY))
	{
	  /* Next proposal, if there */
	  if (isapf->isap_np == ISAKMP_NEXT_P)
	    {
	      i += ntohs(isapf->isap_length);
	      
	      if (length - i <= sizeof(struct isakmp_proposal))
		return 0;
	      
	      len += ntohs(isapf->isap_length) + isapf->isap_spisize;
	      isapf = (struct isakmp_proposal *) (from + i);
	      
	      if (ntohs(isapf->isap_length) + i > length)
		return 0;
	      
	      continue;
	    }
	  else
	    return 0;
	}
      
      st->st_transid = isat->isat_transid;
      st->st_protoid = isapf->isap_protoid;
      
      j += sizeof(struct isakmp_transform);
      
      k = ntohs(isat->isat_length) - sizeof(struct isakmp_transform);
      while (k != 0)
	{
	  isaat = (struct isakmp_attribute *) (from + j);
	  isaat->isaat_type = ntohs(isaat->isaat_type);
	  
	  l = ntohs(isaat->isaat_lv);
	  
	  if (isaat->isaat_af)
	    {
	      j += sizeof(struct isakmp_attribute);
	      k -= sizeof(struct isakmp_attribute);
	    }
	  else
	    {
/* Receiver - accept variable length attribute */
	      j += sizeof(struct isakmp_attribute);
	      k -= sizeof(struct isakmp_attribute);
	      l1 = l;
	      isaatv = (struct isakmp_attribute_var_val *) (from + j);
	      l = ntohl(isaatv->isaatv_val);
	      j += l1;
	      k -= l1;
	    }
	  
	  /* 
	   * XXX We should also be checking whether the attribute
	   * is/should be TLV or TV 
	   */
	  switch (isaat->isaat_rtype)
	    {
	    case OAKLEY_ENCRYPTION_ALGORITHM:
	      enc = 1;
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "encryption algorithm %d\n", l);
#endif
if  (l == OAKLEY_DES_CBC) 
		{
		  st->st_enc = l;
		  arg_isakmp_enc_length = 8;
		}
	      else 
		if 
		  (l == OAKLEY_3DES_CBC)
		  {
		    st->st_enc = l;
		    arg_isakmp_enc_length = 24;
		  }
	      else
		acceptable = 0;
#ifdef WIT
/* WIT - Check that initiator's proposal matches responder's expectations */
          if (l != arg_isakmp_enc)
		{
		log(0, "Incorrect ISAKMP encryption alg %d in proposal (expected %d)", 
			l, arg_isakmp_enc, 0);
	        return 0;
		}
#endif WIT
	  break;
	      break;
	      
	    case OAKLEY_HASH_ALGORITHM:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "hash algorithm %d\n", l);
#endif
	      if ((l != OAKLEY_MD5) &&
		  (l != OAKLEY_SHA))
		acceptable = 0;
	      st->st_hash = l;
	      hash = 1;
#ifdef WIT
/* WIT - Check that initiator's proposal matches responder's expectations */
          if (l != arg_isakmp_auth)
		{
		log(0, "Incorrect ISAKMP hash alg %d in proposal (expected %d)", 
			l, arg_isakmp_auth, 0);
	        return 0;
		}
#endif WIT
	      break;
	      
	    case OAKLEY_AUTHENTICATION_METHOD:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "authentication method %d\n", l);
#endif
      /*	      if ((l != OAKLEY_PRESHARED_KEY) &&*/
	      /*(l != OAKLEY_RSA_SIG))*/
	      if (l != OAKLEY_PRESHARED_KEY)
		  acceptable = 0;
	      else
		{
		  /*	      if (l == OAKLEY_PRESHARED_KEY)*/
		  /*		{*/
		  /* check if there is a key */
		  pp = (u_char *) get_preshared_key(st, &d);
		  if (pp == (u_char *) NULL)
		    {
#ifndef WIT
	  		log(1,"Pre-shared secret key not found for %s - using generic key",
#else
		    acceptable = 0;
/* WIT exits on errors */
	  		log(1,"Pre-shared secret key not found for %s",
#endif WIT
			      PRINTADDRESS(sa), 0, 0);
		    }
		  else
		    free(pp);		  
		}
	      
	      st->st_auth = l;
	      auth = 1;
	      break;
	      
	    case OAKLEY_GROUP_DESCRIPTION:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "group description %d\n", l);
#endif
	      if (l != OAKLEY_DEFAULT_GROUP)
		acceptable = 0;
	      st->st_group = l;
	      group = 1;
	      break;
	      
	    case OAKLEY_LIFE_TYPE:
#ifdef DEBUG
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "life type %d\n", l);
#endif
	      if (l != OAKLEY_LIFE_SECONDS)
		acceptable = 0;
	      break;
	      
	    case OAKLEY_LIFE_DURATION:
#ifdef DEBUG	   
	      if(arg_verbose & DEBUG_VERBOSE)
		fprintf(stdout, "life duration %d\n", l);
#endif
	      if ((l < OAKLEY_ISAKMP_SA_LIFETIME_MINIMUM) ||
		  (l > OAKLEY_ISAKMP_SA_LIFETIME_MAXIMUM))
		acceptable = 0;
	      else
		st->st_expire = l;
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Oakley SA lifetime: %u \n", l);
#endif
	      break;
	      
	      /* XXX Handle more */
	    default:
	      acceptable = 0;
	      log(0, "unsupported attribute %d from %s, port %d",
		  isaat->isaat_rtype, PRINTADDRESS(sa), get_port(sa));
	      break;
	    }
	  
	  isaat->isaat_type = htons(isaat->isaat_type);
	  
	  /* Something unacceptable with this policy */
	  if (acceptable == 0)
	    break;
	}
      
      /* 
       * One of the default attributes not negotiated or unacceptable
       * policy 
       */
      if ((enc == 0) || (auth == 0) || (hash == 0) || (group == 0) ||
	  (acceptable == 0))
	{
	  /* Next proposal, if there */
	  if (isapf->isap_np == ISAKMP_NEXT_P)
	    {
	      i += ntohs(isapf->isap_length);
	      
	      if (length - i <= sizeof(struct isakmp_proposal))
		return 0;
	      
	      len += ntohs(isapf->isap_length) + isapf->isap_spisize;
	      isapf = (struct isakmp_proposal *) (from + i);
	      
	      if (ntohs(isapf->isap_length) + i > length)
		return 0;
	      
	      continue;
	    }
	  else
	    return 0;
	}
      
	/* Copy proposal */
      bcopy(isapf, isapt, ntohs(isapf->isap_length));
      isapt->isap_np = ISAKMP_NEXT_NONE;
      
      isasa->isasa_length = htons(ntohs(isapf->isap_length) +
				  sizeof(struct isakmp_sa) +
				  IPSEC_DOI_SITUATION_LENGTH);
      
      st->st_proposal = (u_char *) calloc(ntohs(isapt->isap_length), 
					  sizeof(u_char));
      if (st->st_proposal == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_oakley_sa_copy()", 0, 0, 0);
      
      bcopy(isapt, st->st_proposal, ntohs(isapt->isap_length));
      
      /* Copy SPI */
      st->st_spi = (u_char *) calloc(DOI_SPI_SIZE_ISAKMP, sizeof(u_char));
      if (st->st_spi == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_oakley_sa_copy()", 0, 0, 0);      
      bcopy((u_char *)isapf + sizeof(struct isakmp_proposal), st->st_spi,
	    isapf->isap_spisize);
      st->st_spi_len = isapf->isap_spisize;
      
#ifdef DEBUG_IN_OUT
	in_out_functions(" Q ipsecdoi_oakley_sa_copy   | in ipsecdoi.c");
#endif
	return 1;
    }
}


/************************************************************************/
/*
 * Get preshared key for remote IP. The file should be human readable,
 * and the format should be:
 * IP address
 * secret
 * <repeat>
 * with the secret being at most 254 characters.
 * Called by : oakley_get_oakley_prop (ipsec_doi.c)
 * Called by : ipsecdoi_oakley_sa_copy (ipsec_doi.c)
 * Called by : generate_skeyids (ipsec_doi.c)
 */
u_char * get_preshared_key(struct state  *st, 
			   int           *length)
{
  u_char              buffer[256];
  u_char             *p = (u_char *) NULL;
  int                 i;
  int                 match;
  FILE               *fp;
  struct sockaddr_in  sin;
    
#ifdef DEBUG_IN_OUT
  in_out_functions(" E get_preshared_key   | in ipsecdoi.c");
#endif
  bcopy(&(st->st_peer), &sin, sizeof(sin));

#ifdef WIT
/* WIT - Each dest has its own shared secrets file in /tmp/wit */
    fp = fopen(secret_file_name, "r");
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "open the pre-shared-secret-key-file: %s\n", secret_file_name);
#else
    fp = fopen(SHARED_SECRETS_FILE, "r");
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "open the pre-shared-secret-key-file: %s\n", SHARED_SECRETS_FILE);
#endif DEBUG
#endif WIT
  
  if (fp == (FILE *) NULL)
#ifndef WIT
    exit_log("fopen() of file %s failed in get_preshared_key()", 
		SHARED_SECRETS_FILE, 0, 0);
#else
    exit_log("fopen() of file %s failed in get_preshared_key()", 
		secret_file_name, 0, 0);
#endif WIT
  
  while (fgets(buffer, 254, fp) != (char *) NULL)
    {
      i = strlen(inet_ntoa(sin.sin_addr));
      if (!strncmp(inet_ntoa(sin.sin_addr), buffer, i))
	match = 1; 
      
      if (fgets(buffer, 254, fp) == (char *) NULL)
	exit_log("bad secrets file", 0, 0, 0);
      
      if (match == 1)
	{
	  p = (u_char *) strdup(buffer);
	  if (p == (u_char *) NULL)
	    exit_log("strdup() failed in get_preshared_key()", 0, 0, 0);
	  
	  /* Get rid of trailing newline */
	 if (p[strlen(p) - 1] == '\n')
	    p[strlen(p) - 1] = '\0';
	 else
	    p[strlen(p)] = '\0';
	  break;
	}
    }

/* BEGIN No pre-shared secret key for peer; use generic secret key if not WIT */
#ifndef WIT
  if (p == (u_char *) NULL)
     {
	  p = (u_char *) strdup("mekmitasdigoat");
	  if (p == (u_char *) NULL)
	    exit_log("strdup() failed in get_preshared_key()", 0, 0, 0);
	 }
#endif WIT
/* END No pre-shared secret key for peer; use generic secret key if not WIT */
  
  if (p != (u_char *) NULL)
    {
    *length = strlen(p);
#ifdef DEBUG	  
   if (arg_verbose & DEBUG_VALUES)
	  fprintf(stdout, "\n-----\nPre-shared secret key used with %s is \"%s\"(len = %d)\n-----\n", inet_ntoa(sin.sin_addr), p, *length);
#endif
    }
  fclose(fp);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q get_preshared_key   | in ipsecdoi.c");
#endif
  return p;
}

/************************************************************************/
/*
 * SKEYID for preshared keys.
 * Called by generate_skeyids (ipsec_doi.c)
 */
void skeyid_preshared(struct state *st, 
		      u_char       *id, 
		      int           length)
{
  int                     k;
  struct MD5Context       md5ctx;
  SHA_CTX                 shactx;
  char                    buffer[HMAC_BUFSIZE];
  char                    buffer2[HMAC_BUFSIZE];
  struct isakmp_generic  *isag;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E skeyid_preshared  | in ipsecdoi.c");
#endif
  bzero(buffer, HMAC_BUFSIZE);
  bzero(buffer2, HMAC_BUFSIZE);
  
  if (st->st_prf == 0)
    {
      /* Prepare the two pads for the HMAC */
      if (length > HMAC_BUFSIZE)
	{
	  switch (st->st_hash)
	    {
	    case OAKLEY_MD5:
	      MD5Init(&md5ctx);
	      MD5Update(&md5ctx, id, length);
	      MD5Final(buffer, &md5ctx);
	      bcopy(buffer, buffer2, 16);
	      break;
	      
	    case OAKLEY_SHA:
	      SHAInit(&shactx);
	      SHAUpdate(&shactx, id, length);
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, buffer, 20);
	      bcopy(buffer, buffer2, 20);
	      break;
	      
	    default:
	      exit_log("unknown/unsupported hash algorithm %d in skeyid_preshared()", st->st_hash, 0, 0);
	    }
	}
      else
	{
	  bcopy(id, buffer, length);
	  bcopy(id, buffer2, length);
	}
      
      for (k = 0; k < HMAC_BUFSIZE; k++)
	{
	  buffer[k] ^= HMAC_IPAD;
	  buffer2[k] ^= HMAC_OPAD;
	}

      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  st->st_skeyid = (u_char *) calloc(16, sizeof(u_char));
	  if (st->st_skeyid == (u_char *) NULL)
	    exit_log("calloc() failed in skeyid_preshared()", 0, 0, 0);
	  st->st_skeyid_len = 16;
	  
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buffer, HMAC_BUFSIZE);
	  
	  k = sizeof(struct isakmp_generic);
	  
	  isag = (struct isakmp_generic *) st->st_ni;
	  MD5Update(&md5ctx, st->st_ni + k, 
		    ntohs(isag->isag_length) - k);
	  
	  isag = (struct isakmp_generic *) st->st_nr;
	  MD5Update(&md5ctx, st->st_nr + k, 
		    ntohs(isag->isag_length) - k);
	  
	  MD5Final(st->st_skeyid, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buffer2, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, st->st_skeyid, 16);
	  MD5Final(st->st_skeyid, &md5ctx);
	  return;
	  
	case OAKLEY_SHA:
	  st->st_skeyid = (u_char *) calloc(20, sizeof(u_char));
	  if (st->st_skeyid == (u_char *) NULL)
	    exit_log("calloc() failed in skeyid_preshared()", 0, 0, 0);
	  st->st_skeyid_len = 20;
	  
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buffer, HMAC_BUFSIZE);
	  
	  k = sizeof(struct isakmp_generic);
	  
	  isag = (struct isakmp_generic *) st->st_ni;
	  SHAUpdate(&shactx, st->st_ni + k, 
		    ntohs(isag->isag_length) - k);
	  
	  isag = (struct isakmp_generic *) st->st_nr;
	  SHAUpdate(&shactx, st->st_nr + k, 
		    ntohs(isag->isag_length) - k);
	  
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buffer2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, st->st_skeyid, 20);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid, 20);
#ifdef DEBUG_IN_OUT
	  in_out_functions(" Q skeyid_preshared   | in ipsecdoi.c");
#endif
	  return;
	  
	default:
	  exit_log("unknown hash algorithm %d in skeyid_preshared", st->st_hash, 0, 0);
	}
    }
  else
    {
      /* XXX Handle 3DES */
    }
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q skeyid_preshared   | in ipsecdoi.c");
#endif
}


/************************************************************************/
/* modified 04/21/98 */
/*
 * Generate the SKEYID_*
 * Called by : ipsecdoi_handle_r1 (ipsec_doi_mainxchg.c)
 * Called by : ipsecdoi_handle_i2 (ipsec_doi_mainxchg.c)
 */
void generate_skeyids(struct state *st)
{
  u_char             *id;
  int                 k;
  int                 k1;
  int                 l;
  int                 len;

  int                 i;
  int                 j;
  int                 keylen;
  int                 numbytes;
  struct MD5Context   md5ctx;
  SHA_CTX             shactx;
  u_char              buf1[HMAC_BUFSIZE];
  u_char              buf2[HMAC_BUFSIZE];
  u_char             *gxy;
  u_char              c;
  MP_INT              temp;
  MP_INT              temp2;
  u_char             *gi;
  u_char             *gr;
  int                 gilen;
  int                 grlen;
  int                 gxylen;
  int                 gibegin;      /* to skip possible leading zero in gi */
  int                 grbegin;      /* to skip possible leading zero in gr */
  int                 gxybegin;     /* to skip possible leading zero in gxy */

#ifdef DEBUG
  struct isakmp_generic *isag; 
  int                 nonce_size;
#endif
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E generate_skeyids  | in ipsecdoi.c");
#endif

  mpz_init(&temp);
  mpz_init(&temp2);
  
  bzero(buf1, HMAC_BUFSIZE);
  bzero(buf2, HMAC_BUFSIZE);

  gilen = mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1 ;
    if (gilen < 96)
	gilen = 96;

  gi = (u_char *) calloc(gilen, sizeof(u_char));
  if (gi == (u_char *) NULL)
    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
    
  mpz_set(&temp, &(st->st_gi)); 
  
  for (k = gilen - 1 ; k >= 0; k--)
    {
      gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }

  /* Strip leading byte of zeroes, if necessary*/
  gibegin = 0;
  if ((gi[0] == 0) && (gilen > 96)) {
    gibegin++;
    gilen--;
  }
  grlen = mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1;
    if (grlen < 96)
	grlen = 96;
  gr = (u_char *) calloc(grlen, sizeof(u_char));
  if (gr == (u_char *) NULL)
    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
  
  mpz_set(&temp, &(st->st_gr));
  
  for (k = grlen - 1; k >= 0; k--)
    {
      gr[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }
  
  /* Strip leading byte of zeroes, if necessary*/
  grbegin = 0;
  if ((gr[0] == 0) && (grlen > 96)) {
    grbegin++;
    grlen--;
  }
  
  mpz_clear(&temp);
  mpz_clear(&temp2);
  
  switch (st->st_auth)
    {
    case OAKLEY_PRESHARED_KEY:
      /* Generate the SKEYID */
      id = get_preshared_key(st, &len);
      skeyid_preshared(st, id, len);
      free(id);
      break;
      
    case OAKLEY_DSS_SIG:
    case OAKLEY_RSA_SIG:
      /* XXX */
      break;
      
     case OAKLEY_RSA_ENC:
       /* XXX */
       break;
       
    default:
      exit_log("unknown authentication method %d in generate_skeyids",
	       st->st_auth, 0, 0);
     }
  
  /* Get the g^xy for HMAC'ing */
  gxylen = mpz_sizeinbase(&(st->st_shared), 16) / 2 + 1;
    if (gxylen < 96)
	gxylen = 96;
  gxy = (u_char *) calloc(gxylen, sizeof(u_char));
  if (gxy == (u_char *) NULL)
    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
  
  mpz_init(&temp);
  mpz_init(&temp2);
  mpz_set(&temp, &(st->st_shared));
  
  for (k = gxylen - 1; k >= 0; k--)
    {
      gxy[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }
  
  /* Strip leading byte of zeroes, if necessary*/
  gxybegin = 0;
  if ((gxy[0] == 0) && (gxylen > 96)) {
    gxybegin++;
    gxylen--;
  }

#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
  	fprintf(stdout, "Generate the SKEYID SKEYID_d SKEYID_a SKEYID_e\n");
   if (arg_verbose & DEBUG_VALUES)
   {
  	fprintf(stdout, "***** PHASE 1 Computed Values: *****\n");
      fprintf(stdout, "Initiator Cookie(%d): ", COOKIE_SIZE);
  	format_dump(st->st_icookie, COOKIE_SIZE); 
      fprintf(stdout, "Responder Cookie(%d): ", COOKIE_SIZE);
  	format_dump(st->st_rcookie, COOKIE_SIZE); 
  	isag = (struct isakmp_generic *) st->st_ni;
  	nonce_size = ntohs(isag->isag_length)-sizeof(struct isakmp_generic);
  	fprintf(stdout, "Initiator Nonce(%d) : ", nonce_size);
  	format_dump((st->st_ni) + sizeof(struct isakmp_generic), nonce_size); 
  	isag = (struct isakmp_generic *) st->st_nr;
  	nonce_size = ntohs(isag->isag_length)-sizeof(struct isakmp_generic);
  	fprintf(stdout, "\nResponder Nonce(%d) : ", nonce_size);
  	format_dump((st->st_nr) + sizeof(struct isakmp_generic), nonce_size); 
  	fprintf(stdout, "-----\n***** Diffie-Hellman Exchange Values: *****\n");
  	fprintf(stdout, "\nInitiator's Public Value - g^x(%d) : \n", gilen);
  	format_dump(gi + gibegin, gilen);
  	fprintf(stdout, "\nResponder's Public Value - g^y(%d) : \n", gilen);
  	format_dump(gr + grbegin, grlen);
  	fprintf(stdout, "\nShared Secret - g^xy(%d):\n", gxylen);
  	format_dump(gxy+gxybegin, gxylen);
  	fprintf(stdout, "-----\n");
   }
#endif
  
  mpz_clear(&temp);
  mpz_clear(&temp2);

  if (st->st_prf == 0)
    {
      /* Prepare the two pads for the HMAC */
      if (st->st_skeyid_len <= HMAC_BUFSIZE)
	{
	  bcopy(st->st_skeyid, buf1, st->st_skeyid_len);
	  bcopy(st->st_skeyid, buf2, st->st_skeyid_len);
	}
      else
	{
	  switch (st->st_hash)
	    {
	    case OAKLEY_MD5:
	      MD5Init(&md5ctx);
	      MD5Update(&md5ctx, st->st_skeyid, st->st_skeyid_len);
	      MD5Final(buf1, &md5ctx);
	      bcopy(buf1, buf2, 16);
	      break;
	      
	    case OAKLEY_SHA:
	      SHAInit(&shactx);
	      SHAUpdate(&shactx, st->st_skeyid, st->st_skeyid_len);
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, buf1, 20);
	      bcopy(buf1, buf2, 20);
	      break;
	      
	    default:
	      exit_log("unknown/unsupported hash algorithm %d in generate_skeyids()", st->st_hash, 0, 0);
	    }
	}
	
      for (k = 0; k < HMAC_BUFSIZE; k++)
	{
	  buf1[k] ^= HMAC_IPAD;
	  buf2[k] ^= HMAC_OPAD;
	}
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  st->st_skeyid_d_len = 16;
	  st->st_skeyid_d = (u_char *) calloc(16, sizeof(u_char));
	  if (st->st_skeyid_d == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  st->st_skeyid_a_len = 16;
	  st->st_skeyid_a = (u_char *) calloc(16, sizeof(u_char));
	  if (st->st_skeyid_a == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  /* If the encryption algorithm requires more than 16 bytes of
	     keying material, SKEYID_e is expanded to be able to store that
	     keying material and not recompute it. */
	  /* Allocate a multiple of 16 bytes for the key, since MD5 operates on 
	     16-byte blocks */
	  numbytes = ((arg_isakmp_enc_length + 15) / 16) * 16;

	  st->st_skeyid_e_len = (arg_isakmp_enc_length > 16) ? arg_isakmp_enc_length : 16;
	  st->st_skeyid_e = (u_char *) calloc(numbytes, sizeof(u_char));
	  if (st->st_skeyid_e == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);

	  /* SKEYID_D */
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);

	  MD5Update(&md5ctx, &gxy[gxybegin], gxylen);
	  MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
	  MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
	  c = 0;
	  MD5Update(&md5ctx, &c, 1);
	  MD5Final(st->st_skeyid_d, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, st->st_skeyid_d, st->st_skeyid_d_len);
	  MD5Final(st->st_skeyid_d, &md5ctx);
	  
	  /* SKEYID_A */
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, st->st_skeyid_d, st->st_skeyid_d_len);
	  MD5Update(&md5ctx, &gxy[gxybegin], gxylen);
	  MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
	  MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
	  c = 1;
	  MD5Update(&md5ctx, &c, 1);
	  MD5Final(st->st_skeyid_a, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, st->st_skeyid_a, st->st_skeyid_a_len);
	  MD5Final(st->st_skeyid_a, &md5ctx);
		
	  /* SKEYID_E */
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, st->st_skeyid_a, st->st_skeyid_a_len);
	  MD5Update(&md5ctx, &gxy[gxybegin], gxylen);
	  MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
	  MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
	  c = 2;
	  MD5Update(&md5ctx, &c, 1);
	  MD5Final(st->st_skeyid_e, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);

	  /*	  MD5Update(&md5ctx, st->st_skeyid_e, st->st_skeyid_e_len);*/
	  MD5Update(&md5ctx, st->st_skeyid_e, 16);
	  MD5Final(st->st_skeyid_e, &md5ctx);

	  if (numbytes > 16)
	    {
#ifdef DEBUG
   if (arg_verbose & DEBUG_VALUES)
   {
	      fprintf(stdout, "\nExtending skeyid_e from 16 bytes to %d bytes\n",
	      	st->st_skeyid_e_len);
	      fprintf(stdout, "Initial skeyid_e (%d):\t", 16);
	      format_dump(st->st_skeyid_e, 16);
   }
#endif	      	    	      
	      extend_skeyid_e(st, numbytes);
	    }
	  
	  /* IV */
	  st->st_iv_len = 16;
	  st->st_iv = (u_char *) calloc(st->st_iv_len, sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, &gi[gibegin], gilen);
	  MD5Update(&md5ctx, &gr[grbegin], grlen);
	  MD5Final(st->st_iv, &md5ctx);
		
	  break;
	  
	case OAKLEY_SHA:
	  st->st_skeyid_d_len = 20;
	  st->st_skeyid_d = (u_char *) calloc(20, sizeof(u_char));
	  if (st->st_skeyid_d == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  st->st_skeyid_a_len = 20;
	  st->st_skeyid_a = (u_char *) calloc(20, sizeof(u_char));
	  if (st->st_skeyid_a == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  /* If the encryption algorithm requires more than 16 bytes of
	     keying material, SKEYID_e is expended to be able to store that
	     keying material and not recompute it. */
	  /* Allocate a multiple of 20 bytes for the key, since SHA operates on 
	     20-byte blocks */
	  numbytes = ((arg_isakmp_enc_length + 19) / 20) * 20;

	  st->st_skeyid_e_len = (arg_isakmp_enc_length > 20) ? arg_isakmp_enc_length : 20;
	  st->st_skeyid_e = (u_char *) calloc(numbytes, sizeof(u_char));
	  if (st->st_skeyid_e == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  /* SKEYID_D */
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, &gxy[gxybegin], gxylen);
	  SHAUpdate(&shactx, st->st_icookie, COOKIE_SIZE);
	  SHAUpdate(&shactx, st->st_rcookie, COOKIE_SIZE);
	  c = 0;
	  SHAUpdate(&shactx, &c, 1);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid_d, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, st->st_skeyid_d, st->st_skeyid_d_len);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid_d, 20);
	  
	  /* SKEYID_A */
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, st->st_skeyid_d, st->st_skeyid_d_len);
	  SHAUpdate(&shactx, &gxy[gxybegin], gxylen);
	  SHAUpdate(&shactx, st->st_icookie, COOKIE_SIZE);
	  SHAUpdate(&shactx, st->st_rcookie, COOKIE_SIZE);
	  c = 1;
	  SHAUpdate(&shactx, &c, 1);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid_a, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, st->st_skeyid_a, st->st_skeyid_a_len);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid_a, 20);
	  
	  /* SKEYID_E */
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, st->st_skeyid_a, st->st_skeyid_a_len);
	  SHAUpdate(&shactx, &gxy[gxybegin], gxylen);
	  SHAUpdate(&shactx, st->st_icookie, COOKIE_SIZE);
	  SHAUpdate(&shactx, st->st_rcookie, COOKIE_SIZE);
	  c = 2;
	  SHAUpdate(&shactx, &c, 1);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid_e, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, st->st_skeyid_e, 20);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_skeyid_e, 20);
	  if (numbytes > 20)
	    {
#ifdef DEBUG
   if (arg_verbose & DEBUG_VALUES)
   {
	      fprintf(stdout, "\nExtending skeyid_e from 20 bytes to %d bytes\n",
	      	st->st_skeyid_e_len);
	      fprintf(stdout, "Initial skeyid_e (%d):\t", 20);
	      format_dump(st->st_skeyid_e, 20);
   }
#endif	      	    	      
	      extend_skeyid_e(st, numbytes);
	    }
	  
	  /* IV */
	  st->st_iv_len = 20;
	  st->st_iv = (u_char *) calloc(st->st_iv_len, sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
	  
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, &gi[gibegin], gilen);
	  SHAUpdate(&shactx, &gr[grbegin], grlen);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, st->st_iv, 20);
				
	  break;
	  
	default:
	  exit_log("unknown hash function %d in generate_skeyids",
		   st->st_hash, 0, 0);
	}
    }
  else
    {
      /* TODO : Handle a negotiated prf */
      /* for example 3DES_CBC_HMAC */
    }

    if ((st->st_enc == ESP_DES) ||
	(st->st_enc == ESP_DES_IV64)) 
      {
	/* Fix parity of DES key(s), check for weak key(s) 
               If weak key is found, fix if possible */
 	if (check_des_key(st->st_skeyid_e, numbytes, 1)) 
	  exit_log("check_des_key() failed for skeyid_e in generate_skeyids()", 0, 0, 0);
      }

    if (st->st_enc == ESP_3DES)
      {
	/* Fix parity of 3DES key(s), check for weak key(s) 
               If weak key is found, DO NOT FIX */
	if (check_3des_key(st->st_skeyid_e, numbytes)) 
	  exit_log("check_3des_key() failed for skeyid_e in generate_skeyids()", 0, 0, 0);
      }

#ifdef DEBUG
   if (arg_verbose & DEBUG_VALUES)
   {
  	fprintf(stdout, "\nSkeyid   (%d):\t", st->st_skeyid_len);
  	format_dump(st->st_skeyid, st->st_skeyid_len);
  	fprintf(stdout, "Skeyid_d (%d):\t", st->st_skeyid_d_len);
  	format_dump(st->st_skeyid_d, st->st_skeyid_d_len);  
  	fprintf(stdout, "Skeyid_a (%d):\t", st->st_skeyid_a_len);
  	format_dump(st->st_skeyid_a, st->st_skeyid_a_len);
  	fprintf(stdout, "Skeyid_e (%d):\t", st->st_skeyid_e_len);
  	format_dump(st->st_skeyid_e, st->st_skeyid_e_len);
  	fprintf(stdout, "IV       (%d):\t", st->st_iv_len);
  	format_dump(st->st_iv, st->st_iv_len);
   }
#endif
  
  free(gi);
  free(gr);
  free(gxy);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q generate_skeyids  | in ipsecdoi.c");
#endif
}


/************************************************************************/
/*
 * If the third argument is 1, generate HASH_I, if 0 generate HASH_R.
 * If fourth argument is 1, I'm the Initiator; if 0, I'm the responder..
 * Called by : ipsecdoi_handle_i2 (ipsec_doi_mainxchg.c)
 * Called by : ipsecdoi_handle_r2 (ipsec_doi_mainxchg.c)
 * Called by : ipsecdoi_handle_i3 (ipsec_doi_mainxchg.c)
 */
u_char * get_hash(struct state *st, 
		  int          *length, 
		  int           hashi, 
		  int           initiator)
{
  u_char                 *hash;
  u_char                 *gi;
  u_char                 *gr;
  struct MD5Context       md5ctx;
  SHA_CTX                 shactx;
  u_char                  buf1[HMAC_BUFSIZE];
  u_char                  buf2[HMAC_BUFSIZE];
  int                     k;
  int                     k1;
  int                     l;
  int                     gilen;
  int                     grlen;
  int                     gibegin;       /* to skip possible leading zero in gi */
  int                     grbegin;       /* to skip possible leading zero in gi */
  MP_INT                  temp;
  MP_INT                  temp2;
  struct identity         id;
  struct isakmp_sa       *isasa;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E get_hash   | in ipsecdoi.c");
#endif
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
  	fprintf(stdout, "\n Generate the HASH\n");
#endif
  mpz_init(&temp);
  mpz_init(&temp2);
  
  bzero(buf1, HMAC_BUFSIZE);
  bzero(buf2, HMAC_BUFSIZE);
  
  gilen = mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1;
    if (gilen < 96)
	gilen = 96;
  gi = (u_char *) calloc(gilen, sizeof(u_char));
  if (gi == (u_char *) NULL)
    exit_log("calloc() failed in get_hash()", 0, 0, 0);
    
  mpz_set(&temp, &(st->st_gi));

  for (k = gilen - 1 ; k >= 0; k--)
    {
      gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }

  /* Strip leading byte of zeroes, if necessary*/
  gibegin = 0;
  if ((gi[0] == 0) && (gilen > 96)) {
    gibegin++;
    gilen--;
  }

  grlen = mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1;
    if (grlen < 96)
	grlen = 96;
  gr = (u_char *) calloc(grlen, sizeof(u_char));
  if (gr == (u_char *) NULL)
    exit_log("calloc() failed in get_hash()", 0, 0, 0);
  
  mpz_set(&temp, &(st->st_gr));
  
  for (k = grlen - 1; k >= 0; k--)
    {
      gr[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }

  /* Strip leading byte of zeroes, if necessary */
  grbegin = 0;
  if ((gr[0] == 0) && (grlen > 96)) {
    grbegin++;
    grlen--;
  }

  mpz_clear(&temp);
  mpz_clear(&temp2);
  
  isasa = (struct isakmp_sa *) st->st_sa;
  
  if (st->st_prf == 0)
    {
      /* Prepare the two pads for the HMAC */
      if (st->st_skeyid_len <= HMAC_BUFSIZE)
	{
	  bcopy(st->st_skeyid, buf1, st->st_skeyid_len);
	  bcopy(st->st_skeyid, buf2, st->st_skeyid_len);
 	}
      else
	{
	  switch (st->st_hash)
	    {
	    case OAKLEY_MD5:
	      MD5Init(&md5ctx);
	      MD5Update(&md5ctx, st->st_skeyid, st->st_skeyid_len);
	      MD5Final(buf1, &md5ctx);
	      bcopy(buf1, buf2, 16);
	      break;
	      
	    case OAKLEY_SHA:
	      SHAInit(&shactx);
	      SHAUpdate(&shactx, st->st_skeyid, st->st_skeyid_len);
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, buf1, 20);
	      bcopy(buf1, buf2, 20);
	      break;
	      
	    default:
	      exit_log("unknown/unsupported hash algorithm %d in get_hash()", st->st_hash, 0, 0);
	    }
	}
      
      for (k = 0; k < HMAC_BUFSIZE; k++)
	{
	  buf1[k] ^= HMAC_IPAD;
	  buf2[k] ^= HMAC_OPAD;
	}
      
     switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  *length = 16;
	  hash = (u_char *) calloc(16, sizeof(u_char));
	  if (hash == (u_char *) NULL)
	    exit_log("calloc() failed in get_hash()", 0, 0, 0);
	  
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	  
	  if (hashi)
	    {
	      MD5Update(&md5ctx, &gi[gibegin], gilen);
	      MD5Update(&md5ctx, &gr[grbegin], grlen);
	      MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
	      MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
	    }
	  else
	    {
	      /* modif 03/20/98 */
	      /* if we generate HASH_R, g^xr comes first */
	      /*	      MD5Update(&md5ctx, &gi[gibegin], gilen);*/
	      /*	      MD5Update(&md5ctx, &gr[grbegin], grlen);*/
	      MD5Update(&md5ctx, &gr[grbegin], grlen);
	      MD5Update(&md5ctx, &gi[gibegin], gilen);
	      /* end modif */
	      MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
	      MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
	    }
	  
	  /* SAp */
	  MD5Update(&md5ctx, st->st_sa + sizeof(struct isakmp_generic), 
		    ntohs(isasa->isasa_length) - 
		    sizeof(struct isakmp_generic));
	  bzero(&id, sizeof(id));
	  
	  if (hashi)
	    {
	      if (initiator)
		{ 
		  id.id_type = st->st_myidentity_type;
		  id.id_protoid = st->st_myidentity_protoid;
		  id.id_port = st->st_myidentity_port;
		}
	      else
		{
		  id.id_type = st->st_peeridentity_type;
		  id.id_protoid = st->st_peeridentity_protoid;
		  id.id_port = st->st_peeridentity_port;
		}
	    }
	  else
	    {
	      if (initiator)
		{
		  id.id_type = st->st_peeridentity_type;
		  id.id_protoid = st->st_peeridentity_protoid;
		  id.id_port = st->st_peeridentity_port;
		}
	      else
		{
		  id.id_type = st->st_myidentity_type;
		  id.id_protoid = st->st_myidentity_protoid;
		  id.id_port = st->st_myidentity_port;
		}
	    }
	  
	  MD5Update(&md5ctx, (u_char *) &id, sizeof(id));
	  
	  if (hashi)
	    {
	      if (initiator)
		MD5Update(&md5ctx, st->st_myidentity, 
			  st->st_myidentity_len);
	      else
		MD5Update(&md5ctx, st->st_peeridentity,
			  st->st_peeridentity_len);
	    }
	  else
	    {
	      if (initiator)
		MD5Update(&md5ctx, st->st_peeridentity,
			      st->st_peeridentity_len);
	      else
		    MD5Update(&md5ctx, st->st_myidentity,
			      st->st_myidentity_len);
	    }
	  MD5Final(hash, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, hash, 16);
	  MD5Final(hash, &md5ctx);
	  break;
	  
	case OAKLEY_SHA:
	  *length = 20;
	  hash = (u_char *) calloc(20, sizeof(u_char));
	  if (hash == (u_char *) NULL)
	    exit_log("calloc() failed in get_hash()", 0, 0, 0);
	  
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	  
	  if (hashi)
	    {
	      SHAUpdate(&shactx, &gi[gibegin], gilen);
	      SHAUpdate(&shactx, &gr[grbegin], grlen);
	      SHAUpdate(&shactx, st->st_icookie, COOKIE_SIZE);
	      SHAUpdate(&shactx, st->st_rcookie, COOKIE_SIZE);
	    }
	  else
	    {

	      /*	      SHAUpdate(&shactx, &gi[gibegin], gilen);
			      SHAUpdate(&shactx, &gr[grbegin], grlen);*/
	      SHAUpdate(&shactx, &gr[grbegin], grlen);
	      SHAUpdate(&shactx, &gi[gibegin], gilen);
	      SHAUpdate(&shactx, st->st_rcookie, COOKIE_SIZE);
	      SHAUpdate(&shactx, st->st_icookie, COOKIE_SIZE);
	    }
	  
	  /* SAp */
	  SHAUpdate(&shactx, st->st_sa + 
		    sizeof(struct isakmp_generic), 
		    ntohs(isasa->isasa_length) - 
		    sizeof(struct isakmp_generic));
	  
	  bzero(&id, sizeof(id));
	  
	  if (hashi)
	    {
	      if (initiator)
		{ 
		  id.id_type = st->st_myidentity_type;
		  id.id_protoid = st->st_myidentity_protoid;
		  id.id_port = st->st_myidentity_port;
		}
	      else
		{
		  id.id_type = st->st_peeridentity_type;
		  id.id_protoid = st->st_peeridentity_protoid;
		  id.id_port = st->st_peeridentity_port;
		    }
	    }
	  else
	    {
	      if (initiator)
		{
		  id.id_type = st->st_peeridentity_type;
		  id.id_protoid = st->st_peeridentity_protoid;
		  id.id_port = st->st_peeridentity_port;
		    }
	      else
		{
		  id.id_type = st->st_myidentity_type;
		  id.id_protoid = st->st_myidentity_protoid;
		  id.id_port = st->st_myidentity_port;
		}
	    }

	  SHAUpdate(&shactx, (u_char *) &id, sizeof(id));
	  
	  if (hashi)
	    {
	      if (initiator)
		SHAUpdate(&shactx, st->st_myidentity, 
			  st->st_myidentity_len);
	      else
		SHAUpdate(&shactx, st->st_peeridentity, 
			  st->st_peeridentity_len);
	      }
	  else
	    {
	      if (initiator)
		SHAUpdate(&shactx, st->st_peeridentity, 
			  st->st_peeridentity_len);
	      else
		SHAUpdate(&shactx, st->st_myidentity, 
			  st->st_myidentity_len);
	    }
	    
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, hash, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, hash, 20);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, hash, 20);
	  
	  break;
	  
	default:
	  exit_log("unknown hash algorithm %d in get_hash", st->st_hash,
		   0, 0);
	}
    }
  else
    {
      /* XXX Handle 3DES */
    }
  
  free(gi);
  free(gr);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q get_hash  | in ipsecdoi.c");
#endif
  return hash;
}


/************************************************************************/
/* 
 * Compute HASH(1), HASH(2) of Quick Mode.
 * Called by : initiate_quick (ipsec_doi_quickxchg.c)
 * Called by : ipsecdoi_handle_quick_r1 (ipsec_doi_quickxchg.c)
 * Called by : ipsecdoi_handle_quick_i1 (ipsec_doi_quickxchg.c)
 */
void compute_hash(struct state     *st, 
		  u_char           *packet, 
		  int               length, 
		  int               hash2)
{
  struct MD5Context       md5ctx;
  SHA_CTX                 shactx;
  u_char                 *p;
  u_char                  buf1[HMAC_BUFSIZE];
  u_char                  buf2[HMAC_BUFSIZE];
  int                     k;
  int                     ignore;
  struct isakmp_generic  *isag;
  struct isakmp_generic  *isag2;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E compute_hash  | in ipsecdoi.c");
#endif
  p = packet + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
  
  /* Ignore the ISAKMP header and the HASH(?) space */
  isag = (struct isakmp_generic *) (packet + sizeof(struct isakmp_hdr));
  ignore = ntohs(isag->isag_length) + sizeof(struct isakmp_hdr);
  isag2 = (struct isakmp_generic *) st->st_ni;
  
  bzero(buf1, HMAC_BUFSIZE);
  bzero(buf2, HMAC_BUFSIZE);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
  	fprintf(stdout, "computing hash: skipping %d bytes at beginning of packet\n", ignore);
#endif
  
  /* Prepare the two pads for the HMAC */
  if (st->st_skeyid_a_len <= HMAC_BUFSIZE)
    {
      bcopy(st->st_skeyid_a, buf1, st->st_skeyid_a_len);
      bcopy(st->st_skeyid_a, buf2, st->st_skeyid_a_len);
    }
  else
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, st->st_skeyid_a, st->st_skeyid_a_len);
	  MD5Final(buf1, &md5ctx);
	  bcopy(buf1, buf2, 16);
	  break;
	  
	case OAKLEY_SHA:
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, st->st_skeyid_a, st->st_skeyid_a_len);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, buf1, 20);
	  bcopy(buf1, buf2, 20);
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_hash()", st->st_hash, 0, 0);
	}
    }
  
  for (k = 0; k < HMAC_BUFSIZE; k++)
    {
      buf1[k] ^= HMAC_IPAD;
      buf2[k] ^= HMAC_OPAD;
    }   
  
  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, (u_char *) &(st->st_msgid),
		    sizeof(st->st_msgid));
	  if (hash2)
	    MD5Update(&md5ctx, st->st_ni + sizeof(struct isakmp_generic),
		      ntohs(isag2->isag_length) -
		      sizeof(struct isakmp_generic));
	  MD5Update(&md5ctx, packet + ignore, length - ignore);
	  MD5Final(p, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, p, 16);
	  MD5Final(p, &md5ctx);
	  
	  break;
		
	case OAKLEY_SHA:
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, (u_char *) &(st->st_msgid),
		    sizeof(st->st_msgid));
	  if (hash2)
	    SHAUpdate(&shactx, st->st_ni + 
		      sizeof(struct isakmp_generic),
		      ntohs(isag2->isag_length) -
		      sizeof(struct isakmp_generic));
	  SHAUpdate(&shactx, packet + ignore, length - ignore);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, p, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, p, 20);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, p, 20);
	  
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_hash()", st->st_hash, 0, 0);
	}
    }
  else
    {
      /* XXX Handle 3DES */
    }

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q compute_hash  | in ipsecdoi.c");
#endif
}


/************************************************************************/
/*
 * Produce the new key material of Quick Mode.
 * 
 * Called by : ipsecdoi_handle_quick_i1 (ipsec_doi_quickxchg.c)
 * Called by : ipsecdoi_handle_quick_r2 (ipsec_doi_quickxchg.c)
 */
void compute_keymat(struct state      *st, 
		    int                keylength,  /* not used yet */
		    int                kernelfd,
		    int                initiator)
{
  struct MD5Context      md5ctx;
  struct MD5Context      md5ctx2;
  SHA_CTX                shactx;
  SHA_CTX                shactx2;
  u_char                 buf1[HMAC_BUFSIZE];
  u_char                 buf2[HMAC_BUFSIZE];
  struct isakmp_generic *isag;
  int                    i;
  int                    k;
  int                    nilen;
  int                    nrlen;
  int                    keylen;
  int                    numbytes;
  MP_INT                 temp;
  MP_INT                 temp2;
  u_char                *gxy;
  int                    gxylen;
  int                    gxybegin;     /* to skip possible leading zero in gxy */
#ifdef DEBUG
  int                    nonce_size;
#endif

#ifdef DEBUG_IN_OUT
  in_out_functions(" E compute_keymat  | in ipsecdoi.c");
#endif
  bzero(buf1, HMAC_BUFSIZE);
  bzero(buf2, HMAC_BUFSIZE);
    
  isag = (struct isakmp_generic *) st->st_ni;
  nilen = ntohs(isag->isag_length) - sizeof(struct isakmp_generic);
  
  isag = (struct isakmp_generic *) st->st_nr;
  nrlen = ntohs(isag->isag_length) - sizeof(struct isakmp_generic);
  
  /* Compute key(s) with length determined by ESP alg keylen and/or 
     AH alg keylen     */
  /* Compute keymaterial and not keys */
  /* Should be managed by policy mechanism */
  keylen = st->st_esp_keymat_len + st->st_ah_keymat_len;
  st->st_keymat_len = keylen;
  st->st_peer_keymat_len = keylen;
  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  /* Allocate a multiple of 16 bytes for the key, since MD5 operates on 
	     16-byte blocks */
	  numbytes = ((keylen + 15) / 16) * 16;
	  break;
	case OAKLEY_SHA:
	  /* Allocate a multiple of 20 bytes for the key, since SHA operates on 
	     20-byte blocks */
	  numbytes = ((keylen + 19) / 20) * 20;
	  break;
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_keymat()", st->st_hash, 0, 0);
	}
    }
  else
    {
      /* XXX Handle 3DES-CBC MAC */
      numbytes = keylen;
    }

  st->st_keymat = (u_char *) calloc(numbytes, sizeof(u_char));
  if (st->st_keymat == (u_char *) NULL)
    exit_log("calloc() failed in compute_keymat()", 0, 0, 0);
  
  st->st_peer_keymat = (u_char *) calloc(numbytes, sizeof(u_char));
  if (st->st_peer_keymat == (u_char *) NULL)
    exit_log("calloc() failed in compute_keymat()", 0, 0, 0);
  
  /* Prepare the two pads for the HMAC */
  if (st->st_skeyid_d_len <= HMAC_BUFSIZE)
    {
      bcopy(st->st_skeyid_d, buf1, st->st_skeyid_d_len);
      bcopy(st->st_skeyid_d, buf2, st->st_skeyid_d_len);
    }
  else
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, st->st_skeyid_d, st->st_skeyid_d_len);
	  MD5Final(buf1, &md5ctx);
	  bcopy(buf1, buf2, 16); 
	  break;
	  
	case OAKLEY_SHA:
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, st->st_skeyid_d, st->st_skeyid_d_len);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, buf1, 20);
	  bcopy(buf1, buf2, 20); 
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_keymat()", st->st_hash, 0, 0);
	}
    }
  
  for (k = 0; k < HMAC_BUFSIZE; k++)
    {
      buf1[k] ^= HMAC_IPAD;
      buf2[k] ^= HMAC_OPAD;
    }   
  
  if(st->st_pfs_handle)
    {
      gxylen = mpz_sizeinbase(&(st->st_shared), 16) / 2 + 1;
    	if (gxylen < 96)
		gxylen = 96;
      gxy = (u_char *) calloc(gxylen, sizeof(u_char));
      if (gxy == (u_char *) NULL)
	exit_log("calloc() failed in compute_keymat()", 0, 0, 0);
      mpz_init(&temp);
      mpz_init(&temp2);
      mpz_set(&temp, &(st->st_shared));      
      for (k = gxylen - 1; k >= 0; k--)
	{
	  gxy[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	  mpz_set(&temp, &temp2);
	}      
      /* Strip leading byte of zeroes, if necessary*/
      gxybegin = 0;
  if ((gxy[0] == 0) && (gxylen > 96)) {
	gxybegin++;
	gxylen--;
      }
      mpz_clear(&temp);
      mpz_clear(&temp2);
    }
#ifdef DEBUG
  if ((st->st_pfs_handle) && (arg_verbose & DEBUG_VERBOSE))
    {  
      fprintf(stdout, "PFS: shared secret -- gxy(%d):\n", gxylen);
      format_dump(gxy+gxybegin, gxylen);
    }
   if (arg_verbose & DEBUG_VALUES)
   {
  fprintf(stdout, "***** PHASE 2 Computed Values: *****\n");
  fprintf(stdout, "SPI    (%d) :\t", st->st_spi_len);
  format_dump(st->st_spi, st->st_spi_len);
  fprintf(stdout, "peer SPI(%d):\t", st->st_peer_spi_len);
  format_dump(st->st_peer_spi, st->st_peer_spi_len);
  isag = (struct isakmp_generic *) st->st_ni;
  nonce_size = ntohs(isag->isag_length)-sizeof(struct isakmp_generic);
  fprintf(stdout, "\nInitiator Nonce(%d) : ", nonce_size);
  format_dump((st->st_ni) + sizeof(struct isakmp_generic), nonce_size); 
  isag = (struct isakmp_generic *) st->st_nr;
  nonce_size = ntohs(isag->isag_length)-sizeof(struct isakmp_generic);
  fprintf(stdout, "\nResponder Nonce(%d) : ", nonce_size);
  format_dump((st->st_nr) + sizeof(struct isakmp_generic), nonce_size); 
   }
#endif

  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	  /* how to compute the keying material is described in draft-ietf-ipsec-isakmp-oakley-06.txt &5.5*/
	  /* KEYMAT = K1 | k2 | k3...
	     K1 = prf(SKEYID-D, [g(qm)^xy | ] protocol_id | SPI | Ni_b Nr_b)
	     K2 = prf(SKEYID-D, K1 | [g(qm)^xy | ] protocol_id | SPI | Ni_b Nr_b)
	     K3 = prf(SKEYID-D, K2 | [g(qm)^xy | ] protocol_id | SPI | Ni_b Nr_b)
	     ... */
	case OAKLEY_MD5:
	  for (i = 0; i < st->st_keymat_len; i+=16) 
	    {
	      /* inner MD5 */
	      MD5Init(&md5ctx);
	      MD5Init(&md5ctx2);
	      MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	      MD5Update(&md5ctx2, buf1, HMAC_BUFSIZE);
	      if (i!=0)
		{ 
		  MD5Update(&md5ctx, st->st_keymat + i - 16, 16);
 		  MD5Update(&md5ctx2, st->st_peer_keymat + i - 16, 16);
		}
	      if(st->st_pfs_handle)		    
		{
 		  MD5Update(&md5ctx, &gxy[gxybegin], gxylen);
		  MD5Update(&md5ctx2, &gxy[gxybegin], gxylen);
		}	   
	      MD5Update(&md5ctx, &(st->st_protoid), sizeof(st->st_protoid));
	      MD5Update(&md5ctx2, &(st->st_peer_protoid), sizeof(st->st_protoid));
	      MD5Update(&md5ctx, st->st_spi, st->st_spi_len);
	      MD5Update(&md5ctx2, st->st_peer_spi, st->st_peer_spi_len);
	      MD5Update(&md5ctx, st->st_ni + sizeof(struct isakmp_generic), nilen);
	      MD5Update(&md5ctx2, st->st_ni + sizeof(struct isakmp_generic), nilen);
	      MD5Update(&md5ctx, st->st_nr + sizeof(struct isakmp_generic), nrlen);
	      MD5Update(&md5ctx2, st->st_nr + sizeof(struct isakmp_generic), nrlen);
	      MD5Final(st->st_keymat+i, &md5ctx);
	      MD5Final(st->st_peer_keymat+i, &md5ctx2);

	      /* outer MD5 */
	      MD5Init(&md5ctx);
	      MD5Init(&md5ctx2);
	      MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	      MD5Update(&md5ctx2, buf2, HMAC_BUFSIZE);
	      MD5Update(&md5ctx, st->st_keymat+i, 16);       
	      MD5Update(&md5ctx2, st->st_peer_keymat+i, 16); 
	      MD5Final(st->st_keymat+i, &md5ctx);
	      MD5Final(st->st_peer_keymat+i, &md5ctx2); 
	    }
	  break;
	  
	case OAKLEY_SHA:
	  /* inner SHA */
	  for (i = 0 ; i < st->st_keymat_len ; i += 20) 
	    {
	      /* inner  SHA*/
	      SHAInit(&shactx);
	      SHAInit(&shactx2);
	      SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	      SHAUpdate(&shactx2, buf1, HMAC_BUFSIZE);
	      if (i!=0)
		{ 
		  SHAUpdate(&shactx, st->st_keymat + i - 20, 20);       
		  SHAUpdate(&shactx2, st->st_peer_keymat + i - 20, 20); 
		}
	      if(st->st_pfs_handle)		    
		{
		  SHAUpdate(&shactx, &gxy[gxybegin], gxylen);
		  SHAUpdate(&shactx2, &gxy[gxybegin], gxylen);
		}	  
	      SHAUpdate(&shactx, &(st->st_protoid), sizeof(st->st_protoid));
	      SHAUpdate(&shactx2, &(st->st_peer_protoid), sizeof(st->st_protoid));
	      SHAUpdate(&shactx, st->st_spi, st->st_spi_len);
	      SHAUpdate(&shactx2, st->st_peer_spi, st->st_peer_spi_len);
	      SHAUpdate(&shactx, st->st_ni + sizeof(struct isakmp_generic), nilen);
	      SHAUpdate(&shactx2, st->st_ni + sizeof(struct isakmp_generic), nilen);
	      SHAUpdate(&shactx, st->st_nr + sizeof(struct isakmp_generic), nrlen);
	      SHAUpdate(&shactx2, st->st_nr + sizeof(struct isakmp_generic), nrlen);
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, st->st_keymat + i, 20);
	      SHAFinal(&shactx2);
	      bcopy((&shactx2)->buffer, st->st_peer_keymat + i, 20);

	      /* outer SHA */
	      SHAInit(&shactx);
	      SHAInit(&shactx2);
	      SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	      SHAUpdate(&shactx2, buf2, HMAC_BUFSIZE);
	      SHAUpdate(&shactx, st->st_keymat + i, 20);       
	      SHAUpdate(&shactx2, st->st_peer_keymat + i, 20); 
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, st->st_keymat + i, 20);
	      SHAFinal(&shactx2);
	      bcopy((&shactx2)->buffer, st->st_peer_keymat + i, 20);
 	    }
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_keymat()", st->st_hash, 0, 0);
	}
    }
  else
    {
      /* CHANGEHERE */
      /* here we only produce keymaterial, it is already taken care of 
	 upper */
      /* XXX Handle 3DES */
    }
  
  /* If enc_alg is one of the DES-type alg's, check the key for parity and weakness
   * numbytes = number of bytes of keymat that were computed, but are not needed 
   *		for auth_key 
   * Repeat the procedure for the peer's enc key
   */
     numbytes -= st->st_ah_keymat_len; 
    
    if ((st->st_protoid == PROTO_IPSEC_ESP) &&
	( (st->st_transid == ESP_DES) || (st->st_transid == ESP_DES_IV64))) 
      {
      /* Fix parity of DES key(s), check for weak key(s) 
	 If weak key is found, DO NOT FIX - renegotiate */
      if (check_des_key(st->st_keymat, numbytes, 0)) 
	exit_log("check_des_key() failed for KEYMAT in compute_keymat()", 0, 0, 0);
      if (check_des_key(st->st_peer_keymat, numbytes, 0)) 
	exit_log("check_des_key() failed for Peer KEYMAT in compute_keymat()", 0, 0, 0);
    }
  
  if ((st->st_protoid == PROTO_IPSEC_ESP) &&
      (st->st_transid == ESP_3DES)) 
    {
      /* Fix parity of 3DES key(s), check for weak key(s) 
	 If weak key is found, DO NOT FIX */
      if (check_3des_key(st->st_keymat, numbytes))
	exit_log("check_3des_key() failed for KEYMAT in compute_keymat()", 0, 0, 0);
      if (check_3des_key(st->st_peer_keymat, numbytes))
	exit_log("check_3des_key() failed for Peer KEYMAT in compute_keymat()", 0, 0, 0);
    }
    
#ifdef DEBUG
  if (arg_verbose & DEBUG_VALUES)
  {
  fprintf(stdout, "\nFINAL KEYMAT   (%d):\n", st->st_keymat_len);
  format_dump(st->st_keymat, st->st_keymat_len);
  fprintf(stdout, "Peer FINAL KEYMAT(%d):\n", st->st_peer_keymat_len);
  format_dump(st->st_peer_keymat, st->st_peer_keymat_len);
  }
#endif    
  
  /* Call kernel with UPDATE message to complete the partial SA */
   kernel_update(kernelfd, st, initiator);
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q compute_keymat  | in ipsecdoi.c");
#endif
}

/**********************************************************************/
/* Check ESP DES key to ensure its parity is correct and it's not a known
   weak key
   *
   *      The DES key-checking code is mostly taken from the NRL/Cisco ISAKMP 
   */

/* Correct a DES key's parity */
static void make_odd (unsigned char *component)
{
  if((*component)%2)
    *component &= 0xfe;
  else
    *component |= 0x01;
}


/**********************************************************************/
/* Check if a DES key has odd parity */
static int is_odd (unsigned char component)
{
  int i=0;
  
  while(component){
    if(component%2)
      i++;
    component = component>>1;
  }
  return(i%2);
}

/**********************************************************************/
/* Fix DES key's parity
 * Phase 1: take 1st DES_KEY_SZ bytes of keymat which aren't 
 *                                       in the weak key list
 * Phase 2: reject key if 1st DES_KEY_SZ bytes of keymat are 
 *                                       in the weak key list
 */
int check_des_key(unsigned char  *keymat, 
		  int             keylen,
		  int 		  fix_key)
{
  unsigned char  des_key[DES_KEY_SZ];
  int            i;
  int            j;
  int            status = 1;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E check_des_key  | in ipsecdoi.c");
#endif
  if((keymat == NULL) || (keylen < DES_KEY_SZ))
    return(status);
  
/*  Phase 1 (fix_key == 1) ==> Find 1st DES_KEY_SZ non-weak bytes
    Phase 2 (fix_key == 0) ==> Reject if 1st DES_KEY_SZ bytes are weak */
  for(i=0; i<=(fix_key?(keylen - DES_KEY_SZ):0); i++){ 
    status = 0;
    bcopy((char *)(keymat + i), des_key, DES_KEY_SZ);
    for(j=0; j<DES_KEY_SZ; j++)
      if(!is_odd(des_key[j]))
	make_odd(&des_key[j]);
    for(j=0; j<NUM_WEAK_KEY; j++)
      if(bcmp(des_key, weak_keys[j], DES_KEY_SZ) == 0){
	status = 1;
	break;
      }
    if(status == 0){
      bcopy(des_key, keymat, DES_KEY_SZ);
      break;
    }
  }
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q check_des_key  | in ipsecdoi.c");
#endif
  return(status);
}

/**********************************************************************/
/* Fix 3DES key's parity
 * reject key if 1st DES_KEY_SZ bytes of keymat =
 *               2nd DES_KEY_SZ bytes of keymat
 *         OR if 2nd DES_KEY_SZ bytes of keymat =
 *               3rd DES_KEY_SZ bytes of keymat
 */
int check_3des_key(unsigned char  *keymat, 
		  int             keylen)
{
  int            i;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E check_3des_key  | in ipsecdoi.c");
#endif
  if((keymat == NULL) || (keylen < 3*DES_KEY_SZ) ||
        (bcmp(keymat, keymat + DES_KEY_SZ, DES_KEY_SZ) == 0) ||
        (bcmp(keymat + DES_KEY_SZ, keymat + 2*DES_KEY_SZ, DES_KEY_SZ) == 0))
    return(1);
  
  for(i=0; i<3*DES_KEY_SZ; i++){ 
      if(!is_odd(*(keymat + i)))
	make_odd(keymat + i);
  }
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q check_3des_key  | in ipsecdoi.c");
#endif
  return(0);
}


/************************************************************************/
/*
 * Compute HASH(3) in Quick Mode.
 * Called by ipsecdoi_handle_quick_i1 (ipsec_doi_quickxchg.c) 
 * Called by ipsecdoi_handle_quick_r2 (ipsec_doi_quickxchg.c) 
 */
void compute_hash_3(struct state  *st, 
		    u_char        *buffer)
{
  struct MD5Context       md5ctx;
  SHA_CTX                 shactx;
  u_char                 *p;
  u_char                  buf1[HMAC_BUFSIZE];
  u_char                  buf2[HMAC_BUFSIZE];
  struct isakmp_generic  *isag;
  struct isakmp_generic  *isag2;
  int                     k;
  int                     nilen;
  int                     nrlen;
  u_char                  c = 0;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E compute_hash_3  | in ipsecdoi.c");
#endif
  bzero(buf1, HMAC_BUFSIZE);
  bzero(buf2, HMAC_BUFSIZE);
  
  p = buffer + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
  isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
  
  isag2 = (struct isakmp_generic *) st->st_ni;
  nilen = ntohs(isag2->isag_length) - sizeof(struct isakmp_generic);
  
  isag2 = (struct isakmp_generic *) st->st_nr;
  nrlen = ntohs(isag2->isag_length) - sizeof(struct isakmp_generic);
  
  /* Prepare the two pads for the HMAC */
  if (st->st_skeyid_a_len <= HMAC_BUFSIZE)
    {
      bcopy(st->st_skeyid_a, buf1, st->st_skeyid_a_len);
      bcopy(st->st_skeyid_a, buf2, st->st_skeyid_a_len);
    }
  else
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, st->st_skeyid_a, st->st_skeyid_a_len);
	  MD5Final(buf1, &md5ctx);
	  bcopy(buf1, buf2, 16);
	  break;
	  
	case OAKLEY_SHA:
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, st->st_skeyid_a, st->st_skeyid_a_len);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, buf1, 20);
	  bcopy(buf1, buf2, 20);
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_hash_3()", st->st_hash, 0, 0);
	}
    }
  
  for (k = 0; k < HMAC_BUFSIZE; k++)
    {
      buf1[k] ^= HMAC_IPAD;
      buf2[k] ^= HMAC_OPAD;
    }   
  
  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, &c, sizeof(c));
	  MD5Update(&md5ctx, (u_char *) &(st->st_msgid),
		    sizeof(st->st_msgid));
	  MD5Update(&md5ctx, st->st_ni + sizeof(struct isakmp_generic), 
		    nilen);
	  MD5Update(&md5ctx, st->st_nr + sizeof(struct isakmp_generic), 
		    nrlen);
	  MD5Final(p, &md5ctx);
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	  MD5Update(&md5ctx, p, 16);
	  MD5Final(p, &md5ctx);
	  
	  break;
	  
	case OAKLEY_SHA:
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, &c, sizeof(c));
	  SHAUpdate(&shactx, (u_char *) &(st->st_msgid),
		    sizeof(st->st_msgid));
	  SHAUpdate(&shactx, st->st_ni + sizeof(struct isakmp_generic),
		    nilen);
	  SHAUpdate(&shactx, st->st_nr + sizeof(struct isakmp_generic),
			   nrlen);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, p, 20);
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	  SHAUpdate(&shactx, p, 20);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, p, 20);
	  
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in compute_hash_3()", st->st_hash, 0, 0);
	}
    }
  else
    {
      /* XXX Handle 3DES */
    }

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q compute_hash_3   | in ipsecdoi.c");
#endif
}


/**********************************************************************/
/* the length of the keying material (skeyid_e) is (for now) either 16<->MD5 
   or 20<->SHA which is not enougth.
   we need 24 bytes of keying material for 3DES, to do this, use the technique
   described for phase 1 in draft-ietf-ipsec-isakmp-oakley-06 in Appendix B :
   Keymat = k1 | k2 | k3 ... with
   k1 = prf(SKEYID_e, 0);
   k2 = prf(SKEYID_e, k1);
   k3 = prf(SKEYID_e, K3);... 
   "prf" being the negotiated prf or the HMAC version of the negotiated hash function */
int extend_skeyid_e(struct state *st, int numbytes)
{

  struct MD5Context      md5ctx;
  struct MD5Context      md5ctx2;
  SHA_CTX                shactx;
  SHA_CTX                shactx2;
  u_char                 buf1[HMAC_BUFSIZE];
  u_char                 buf2[HMAC_BUFSIZE];
  u_char                 c;
  int                    i;
  int                    k;
  int                    keylen;
  u_char                *key_material;
  int                    key_material_len;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E extend_skeyid_e  | in ipsecdoi.c");
#endif
  bzero(buf1, HMAC_BUFSIZE);
  bzero(buf2, HMAC_BUFSIZE);

  keylen =  arg_isakmp_enc_length;  
  key_material_len = keylen;

  key_material = (u_char *) calloc(numbytes, sizeof(u_char));
  if (key_material == (u_char *) NULL)
    exit_log("calloc() failed in ipsec_3DES_get_key ()", 0, 0, 0);
  
  /* Prepare the two pads for the HMAC */
  if (key_material_len <= HMAC_BUFSIZE)
    {
      bcopy(st->st_skeyid_e, buf1, st->st_skeyid_e_len);
      bcopy(st->st_skeyid_e, buf2, st->st_skeyid_e_len);
    }
  else
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, st->st_skeyid_e, st->st_skeyid_e_len);
	  MD5Final(buf1, &md5ctx);
	  bcopy(buf1, buf2, 16); 
	  break;
	  
 	case OAKLEY_SHA:
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, st->st_skeyid_e, st->st_skeyid_e_len);
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, buf1, 20);
	  bcopy(buf1, buf2, 20); 
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in ipsec_3DES_get_key()", st->st_hash, 0, 0);
	}
    }
  
  for (k = 0; k < HMAC_BUFSIZE; k++)
    {
      buf1[k] ^= HMAC_IPAD;
      buf2[k] ^= HMAC_OPAD;
    }   
  
  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  for (i = 0; i < key_material_len; i+=16) 
	    {
	      /* inner MD5 */
	      MD5Init(&md5ctx);
	      MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
	      if (i!=0)
		{ 
		  MD5Update(&md5ctx, key_material + i - 16, 16);
		}
	      else 
		{
		  c = 0;
		  MD5Update(&md5ctx, &c, 1);
		}
	      MD5Final(key_material+i, &md5ctx);

	      /* outer MD5 */
	      MD5Init(&md5ctx);
	      MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
	      MD5Update(&md5ctx, key_material + i, 16);       
	      MD5Final(key_material + i, &md5ctx);
	    }
	  break;
	  
	case OAKLEY_SHA:
	  for (i = 0 ; i < key_material_len ; i += 20) 
	    {
	      /* inner  SHA*/
	      SHAInit(&shactx);
	      SHAUpdate(&shactx, buf1, HMAC_BUFSIZE);
	      if (i!=0)
		{ 
		  SHAUpdate(&shactx, key_material + i - 20, 20);       
		}
	      else
		{
		  c = 0;
		  SHAUpdate(&shactx, &c, 1);
		}
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, key_material + i, 20);

	      /* outer SHA */
	      SHAInit(&shactx);
	      SHAUpdate(&shactx, buf2, HMAC_BUFSIZE);
	      SHAUpdate(&shactx, key_material + i, 20);       
	      SHAFinal(&shactx);
	      bcopy((&shactx)->buffer, key_material + i, 20);
 	    }
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in extend_skeyid_e()", st->st_hash, 0, 0);
	}
    }
  else
    {
      /* XXX Handle 3DES_CBC_HMAC */
    }
 
#ifdef DEBUG
 if (arg_verbose & DEBUG_VERBOSE)
 {
  	fprintf(stdout, "\nExtended skeyid_e(%d):\n", key_material_len);
  	format_dump(key_material, key_material_len);
 }
#endif    
  bcopy(key_material, st->st_skeyid_e, numbytes);

}
