/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/* ipsec_doi_mainxchg.c - Constructs and handles Main Mode packets */

#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include "constants.h"
#include "state.h"
#include "packet.h"
#include "defs.h"
#include DES_H

/* for arg_verbose */
#include "argdefs.h"

#define PRINTADDRESS(x) (get_address(x) != (char *) NULL ? get_address(x) : "(unknown address family)")

extern char *get_address();

extern u_char * get_hash();

/************************************************************************/
/*
 * Handle a Main Mode Oakley first packet (responder side).
 * Called by : comm_handle (demux.c)
 */
void ipsecdoi_handle_rfirst(int             sock, 
			    u_char         *buffer, 
			    int             length, 
			    struct sockaddr sa,
			    int             kernelfd)
{
  struct isakmp_hdr   *isa;
  struct isakmp_sa    *isasa; 
  struct isakmp_sa    *isas;
  struct state        *st;
  u_char              *packet;
  int                  i;
  int                  j;
  int                  k;
  struct sockaddr_in   sin;
  struct sockaddr_in   sin1;
  struct sockaddr_in   sin2;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_rfirst   | in ipsecdoi_mainxchg.c");
#endif
#ifdef DEBUG
  if (arg_verbose & DEBUG_PACKETS)
  {
        fprintf(stdout, "-----\n");
    	show_header((struct isakmp_hdr *) buffer);
  }
  fprintf(stdout,"=======================================================================\n");
  fprintf(stdout,"Handle a main mode Oakley first packet\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "w");
  fprintf(f_demo, "MAIN MODE MESSAGE 1 RECEIVED FROM INITIATOR\n");
  fclose(f_demo);
#endif DEMO

bcopy(&sa, &sin1, sizeof(sin1));
#ifdef WIT
/* WIT - Check that initiator's address matches responder's expectations 
		(otherwise, shared secrets won't match */
    if (arg_key_mgmt_goal & GOAL_TUNNEL) 
		sin2.sin_addr.s_addr = arg_gateway_addr;
    else
		sin2.sin_addr.s_addr = arg_dest_addr;
    if (sin1.sin_addr.s_addr != sin2.sin_addr.s_addr)
	{
/* These 2 statements do not work if combined */
		fprintf(stdout, "Unexpected responder address %s ", 
			(char *)inet_ntoa(sin1.sin_addr));
		fprintf(stdout, "(expected %s)\n", 
			(char *)inet_ntoa(sin2.sin_addr));
		
      		return;    
	}
#endif WIT
  
  isa = (struct isakmp_hdr *) buffer;
  isasa = (struct isakmp_sa *) (buffer + sizeof(struct isakmp_hdr));
  
  if (ntohs(isasa->isasa_length) + sizeof(struct isakmp_hdr) >
      length)
    {
      free(buffer);	
      log(0, "malformed packet m1 from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
      send_notification(sock, PAYLOAD_MALFORMED, NULL, 0,
			PROTO_ISAKMP, isa->isa_icookie, isa->isa_rcookie, 0,
			sa);
      return;    
    }
  
  /* XXX Check various fields ? */
  
  /* Set up state */
  st = (struct state *) get_state();
  
  bcopy(isa->isa_icookie, st->st_icookie, COOKIE_SIZE);
  
  get_cookie(ISAKMP_RESPONDER, st->st_rcookie, COOKIE_SIZE, sa);
  
  /* fill the attribute of the state */
  st->st_peer = sa;
  st->st_doi = ISAKMP_DOI_IPSEC;
  st->st_situation = SIT_IDENTITY_ONLY; /* We only support this */
  /* change 04/98 : do it later, when we are sure we sent the message */
  /*  st->st_state = OAKLEY_MAIN_R_1;*/
  /* set port and protoid of responder, but these can change 
     depending on how the peer fills the ID payload */
  st->st_myidentity_protoid = TRANSPORT_PROT_UDP;
  st->st_peeridentity_protoid = TRANSPORT_PROT_UDP;
  st->st_myidentity_port = ntohs(ISAKMP_PORT);
  st->st_peeridentity_port = ntohs (ISAKMP_PORT);
  
  /* The reply packet can't be larger than the original request */
  packet = (u_char *) calloc(length, sizeof(u_char));
  if (packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
  
  isa = (struct isakmp_hdr *) packet;
  
  /* Copy header as received, and responder cookie */
  bcopy(buffer, packet, sizeof(struct isakmp_hdr));
  bcopy(st->st_rcookie, isa->isa_rcookie, COOKIE_SIZE);
  isa->isa_length = 0;
  
  isas = (struct isakmp_sa *) (packet + sizeof(struct isakmp_hdr));
  
  isas->isasa_doi = ntohl(ISAKMP_DOI_IPSEC);
  i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
    IPSEC_DOI_SITUATION_LENGTH;
  packet[i - 1] = SIT_IDENTITY_ONLY;
  
  if (ipsecdoi_oakley_sa_copy(buffer, packet, length, sa, st) == 0)
    {
      free(buffer);
      free(packet);
      free_state(st);
#ifdef WIT
      exit_log("no acceptable proposal from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
#else
      log(0, "no acceptable proposal from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
#endif
      /* XXX Could send notification back */
      return;
    }
  
  st->st_sa = (u_char *) calloc(ntohs(isasa->isasa_length), sizeof(u_char));
  if (st->st_sa == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
  
  bcopy(isas, st->st_sa, ntohs(isasa->isasa_length));
  
  /* XXX Handle Aggressive Mode here ? */
  
  free(buffer);
  
  st->st_packet = packet;
  
  /* Get my identity */
  k = sizeof(sin);
  if (getsockname(sock, (struct sockaddr *)&sin, &k) < 0)
    exit_log("getsockname() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
  
  if (k <= 0)
    exit_log("getsockname() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
  
  /* Save my identity */
  st->st_myidentity = (u_char *) calloc(1, sizeof(struct in_addr));
  if (st->st_myidentity == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
  bcopy(&(sin.sin_addr), st->st_myidentity, sizeof(struct in_addr));
  st->st_myidentity_len = sizeof(struct in_addr);
  st->st_myidentity_type = ID_IPV4_ADDR;
  
  length = ntohs(isas->isasa_length) + sizeof(struct isakmp_hdr);
#ifdef DEBUG
/* These 2 statements do not work if combined */
  fprintf(stdout, "%s sending %d bytes ", inet_ntoa(sin.sin_addr), length);
  fprintf(stdout, "to %s, port %d", PRINTADDRESS(sa), get_port(sa));
#endif
  isa->isa_length = ntohl(length);
  st->st_packet_len = length;
    
  /* Save peer identity */
  st->st_peeridentity = (u_char *) calloc(1, sizeof(struct in_addr));
  if (st->st_peeridentity == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
  bcopy(&(sin1.sin_addr), st->st_peeridentity, sizeof(struct in_addr));
  st->st_peeridentity_len = sizeof(struct in_addr);
  st->st_peeridentity_type = ID_IPV4_ADDR;
  
  if (sendto(sock, packet, length, 0, &sa, sizeof(sa)) != length)
    log(0, "sendto() failed in ipsecdoi_handle_rfirst() for %s, port %d",
	PRINTADDRESS(sa), get_port(sa), 0);
  
#ifdef DEBUG
  packet_dump("\tRESPONDER->INITIATOR ***** MAIN MODE MESSAGE 2 : HDR + SA", DUMP_MSG);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(packet, DUMP_SENT_NO_ENC, 0);
  show_payloads(packet, 0);
  if (arg_verbose & DEBUG_PACKETS)
  {
        fprintf(stdout, "-----\n");
    	show_header((struct isakmp_hdr *) buffer);
  }
#endif DEBUG
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 2 SENT BY RESPONDER\n");
  fclose(f_demo);
#endif DEMO

  /* advance state */
  st->st_state = OAKLEY_MAIN_R_1;

  insert_state(st);
  event_schedule(EVENT_CLEANUP, EVENT_CLEANUP_DELAY, st, 0);

#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q  ipsecdoi_handle_rfirst  | in ipsecdoi_mainxchg.c");
#endif
}




/************************************************************************/
/*
 * Handle HDR;SA from responder (verify proposal), and send back
 * a HDR;KE;Ni.
 * Called by : comm_handle (demux.c)
 */
void ipsecdoi_handle_i1(int             sock, 
			u_char         *buffer, 
			int             length, 
			struct sockaddr sa,
			struct state   *st)
{
  struct sockaddr_in         sa2;
  struct isakmp_hdr         *isa;
  struct isakmp_hdr         *isa2;
  struct isakmp_sa          *isasa;
  struct isakmp_proposal    *isap;
  struct isakmp_transform   *isat;
  struct isakmp_attribute   *isaat;
  struct isakmp_generic     *isag;
  u_int32_t                  situation;
  u_char                     tmp[LOCALSECRETSIZE];
  u_char                    *packet;
  MP_INT                     temp;
  MP_INT                     temp2;
  u_char                    *gi;
  int                        gilen;
  int                        gibegin;
  int                        gioffset;
  int                        i;
  int                        k;
  int                        k1;
  struct sockaddr_in         sin;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_i1  | in ipsecdoi_mainxchg.c");
#endif
#ifdef DEBUG
  if (arg_verbose & DEBUG_PACKETS)
  {
    fprintf(stdout, "-----\n");
  	show_header((struct isakmp_hdr *) buffer);
  }
  fprintf(stdout,"=======================================================================\n");
  fprintf(stdout,"Handle Message 2 (HDR + SA) from responder\nSend back HDR + KE + Ni\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 2 RECEIVED FROM RESPONDER\n");
  fclose(f_demo);
#endif DEMO

  isa = (struct isakmp_hdr *) buffer;
  
  if (length <= sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa))
    {
      free(buffer);
      log(0, "too short (%d bytes) packet received from %s, port %d", length,
	  PRINTADDRESS(sa), get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  isasa = (struct isakmp_sa *) (buffer + sizeof(struct isakmp_hdr));
  
  if (htons(isasa->isasa_length) + sizeof(struct isakmp_hdr) !=
      length)
    {
      free(buffer);
      log(0, "malformed HDR;SA packet m1a from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }
  
  i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa);
  
  /* Not sure if this should ever happen...i'm just paranoid */
  bcopy(buffer + i, &situation, IPSEC_DOI_SITUATION_LENGTH);
  situation = ntohl(situation);
  if (situation != SIT_IDENTITY_ONLY)
    {
      free(buffer);
      log(0, "bad situation %d from %s, port %d", situation,
	  PRINTADDRESS(sa), get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  i += IPSEC_DOI_SITUATION_LENGTH;
  
  isap = (struct isakmp_proposal *) (buffer + i);
  if (ntohs(isap->isap_length) + i != length)
    {
      free(buffer);
      log(0, "bad proposal length from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_proposal) + isap->isap_spisize;
  
  /* XXX Verify proposal we received back */
  
  mpz_init(&(st->st_sec));
  mpz_init(&(st->st_gi));
  st->st_sec_in_use = 1;
  st->st_gi_in_use = 1;
  
  mpz_init(&temp);
  mpz_init(&temp2);
  
  get_rnd_bytes(tmp, LOCALSECRETSIZE);
#ifdef DEBUG
   if (arg_verbose & DEBUG_VALUES)
   {
  	fprintf(stdout, "\nInitiator's Secret - x(%d) : \n", LOCALSECRETSIZE);
  	format_dump(tmp, LOCALSECRETSIZE);
   }
#endif
  
  for (k = 0; k < LOCALSECRETSIZE; k++)
    {
      mpz_mul_ui(&(st->st_gi), &(st->st_sec), 256);
      mpz_add_ui(&(st->st_sec), &(st->st_gi), tmp[k]);
    }
  
  mpz_powm(&(st->st_gi), &groupgenerator, &(st->st_sec), &groupmodulo);
  mpz_init(&temp);
  mpz_init(&temp2);
  
  gilen = mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1;
    if (gilen < 96)
	gilen = 96;
  gi = (u_char *) calloc(gilen, sizeof(u_char));
  if (gi == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  
  mpz_set(&temp, &(st->st_gi));
  
  for (k = gilen - 1 ; k >= 0; k--)
    {
      gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }
  
/* Strip leading byte of zeroes, if necessary */
  gibegin = 0;
  if ((gi[0] == 0) && (gilen > 96)) {
    gibegin++;
    gilen--;
  }

  mpz_clear(&temp);
  mpz_clear(&temp2);
  
  i = sizeof(struct isakmp_hdr) + 2 * sizeof(struct isakmp_generic) +
    gilen + DEFAULTNONCESIZE;

  packet = (u_char *) calloc(i, sizeof(u_char));
  if (packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  
  bcopy(isa, packet, sizeof(struct isakmp_hdr));

  isa2 = (struct isakmp_hdr *) packet;
  isa2->isa_np = ISAKMP_NEXT_KE;
  isag = (struct isakmp_generic *) (packet + sizeof(struct isakmp_hdr));
  isag->isag_np = ISAKMP_NEXT_NONCE;
  isa2->isa_length = htonl(i);

  isag->isag_length = htons(gilen + sizeof(struct isakmp_generic));
  i -= (sizeof(struct isakmp_generic) + DEFAULTNONCESIZE);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "Group modulo (groupmodulo) in ipsecdoi_handle_i1: ");
      mpz_out_str(stdout, 16, &(groupmodulo));
      fprintf(stdout, "\n");
      fprintf(stdout, "Group generator (groupgenerator) in ipsecdoi_handle_i1: ");
      mpz_out_str(stdout, 16, &(groupgenerator));
      fprintf(stdout, "\n");
      fprintf(stdout, "Local secret (st_sec) in ipsecdoi_handle_i1: ");
      mpz_out_str(stdout, 16, &(st->st_sec));
      fprintf(stdout, "\n");
      fprintf(stdout, "Public value sent (st_gi) in ipsecdoi_handle_i1: ");
      mpz_out_str(stdout, 16, &(st->st_gi));
      fprintf(stdout, "\n");
      fprintf(stdout, "Public value is %d bytes long.\n", gilen);
    }
#endif 

  gioffset = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic) - gibegin;
  for (k = gibegin; k < gibegin + gilen; k++) 
    packet[k + gioffset] = gi[k];
  
  /* Generate Ni */
  isag = (struct isakmp_generic *) (packet + i);
  isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    DEFAULTNONCESIZE);
  get_rnd_bytes(packet + i + sizeof(struct isakmp_generic),
		DEFAULTNONCESIZE);
  
  /* Copy the Ni nonce in the state object */
  st->st_ni = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (st->st_ni == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  
  bcopy(isag, st->st_ni, ntohs(isag->isag_length));
  
  /* Copy accepted/received proposal in state object */
  st->st_proposal = (u_char *) calloc(ntohs(isap->isap_length),
				      sizeof(u_char));
  if (st->st_proposal == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "copying %d bytes of proposal into state object\n",
	    ntohs(isap->isap_length));
#endif
  bcopy(isap, st->st_proposal, ntohs(isap->isap_length));
  
  /* Reinsert the state, using the responder cookie we just received */
  delete_state(st); /* this just deletes the entry of the state from the hash table */
  bcopy(isa->isa_rcookie, st->st_rcookie, COOKIE_SIZE);
  insert_state(st);
  
  /* Delete previous retransmission event */
  delete_event(st);
  
  /* Free previous packet and place the new one in the state object */
  free(st->st_packet);
  st->st_packet_len = ntohl(isa2->isa_length);
  st->st_packet = packet;
  /* change 04/98 : do it later, when we are sure we sent the message */
  /*  st->st_state = OAKLEY_MAIN_I_2;*/
  
  /* Get my identity */
  k = sizeof(sa2);
  if (getsockname(sock, (struct sockaddr *)&sa2, &k) < 0)
    exit_log("getsockname() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  
  if (k <= 0)
    exit_log("getsockname() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  
  /* Save my identity */
  k = sizeof(sa2);
  st->st_myidentity = (u_char *) calloc(1, sizeof(struct in_addr));
  if (st->st_myidentity == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  bcopy(&(sa2.sin_addr), st->st_myidentity, sizeof(struct in_addr));
  st->st_myidentity_len = sizeof(struct in_addr);
  st->st_myidentity_type = ID_IPV4_ADDR;
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "my identity is %s\n", inet_ntoa(sa2.sin_addr));
#endif
  
  free(buffer);

  /* Save peer identity */
    bcopy(&sa, &sin, sizeof(sin));
  st->st_peeridentity = (u_char *) calloc(1, sizeof(struct in_addr));
  if (st->st_peeridentity == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
  bcopy(&(sin.sin_addr), st->st_peeridentity, sizeof(struct in_addr));
  st->st_peeridentity_len = sizeof(struct in_addr);
  st->st_peeridentity_type = ID_IPV4_ADDR;

  if (sendto(sock, packet, st->st_packet_len, 0, &sa, sizeof(sa)) !=
      st->st_packet_len)
    log(0, "sendto() failed in ipsecdoi_handle_i1()", 0, 0, 0);

#ifdef DEBUG
  packet_dump("\tINITIATOR->RESPONDER ***** MAIN MODE MESSAGE 3 : HDR + KE + Ni", DUMP_MSG);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(packet, DUMP_SENT_NO_ENC, 0);
  show_payloads(packet, 0);
#endif DEBUG
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 3 SENT BY INITIATOR\n");
  fclose(f_demo);
#endif DEMO
  /* advance state */
  st->st_state = OAKLEY_MAIN_I_2;
  
  /* Schedule for retransmission */
  event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);    
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_i1   | in ipsecdoi_mainxchg.c");
#endif
}



/************************************************************************/
/*
 * Handle HDR;KE;Nr from responder. Send a HDR*;IDii;HASH_I back.
 * Called by : comm_handle (demux.c)
 */
void ipsecdoi_handle_i2(int               sock, 
			u_char           *buffer, 
			int               length, 
			struct sockaddr   sa,
			struct state     *st)
{
  struct isakmp_hdr        *isa;
  struct isakmp_generic    *isag;
  MP_INT                    temp;
  MP_INT                    temp2;
  int                       i;
  int                       k;
  int                       k1;
  int                       blocksize;
  int                       padding;
  u_char                   *hash;
  struct identity          *ide;
  struct sockaddr_in        sin;
  key_schedule              des_cbc_keys;

  char              *three_des_cbc_key;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_i2  | in ipsecdoi_mainxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"=======================================================================\n");
  fprintf(stdout, "Handle Message 4 (HDR + KE + Nr) from responder\nSend back HDR* + IDii + HASH_I\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 4 RECEIVED FROM RESPONDER\n");
  fclose(f_demo);
#endif DEMO
  isa = (struct isakmp_hdr *) buffer;
  if (length <= sizeof(struct isakmp_hdr) + 
      2 * sizeof(struct isakmp_generic))
    {
      free(buffer);
      log(0, "too short HDR;KE;Nr packet (%d bytes) from %s, port %d", 
	  length, PRINTADDRESS(sa), get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
  i = htons(isag->isag_length);
  if (i + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic) > 
      length)
    {
      free(buffer);
      log(0, "malformed packet m2 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }
  
  if (i - sizeof(struct isakmp_generic) < MINIMUM_PUBLIC_VALUE_SIZE)
    {
      free(buffer);
      log(0, "too short public size (%d) sent from %s, port %d",
	  i - sizeof(struct isakmp_generic), PRINTADDRESS(sa), 
	  get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  if (isag->isag_np != ISAKMP_NEXT_NONCE)
    {
      free(buffer);
      log(0, "unexpected header %d in HDR;KE;Nr packet from %s, port %d",
	  isag->isag_np, PRINTADDRESS(sa), get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  mpz_init(&temp);
  mpz_init(&temp2);
  
  /* Get gr from buffer & convert to MP_INT (stored in temp) */
  i += sizeof(struct isakmp_hdr);
  for (k = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
       k < i; k++)
    {
      mpz_mul_ui(&temp2, &temp, 256);
      mpz_add_ui(&temp, &temp2, buffer[k]);
    }
  
  /* Check the Nr header */
  isag = (struct isakmp_generic *) (buffer + i);
  if (ntohs(isag->isag_length) + i != length)
    {
      free(buffer);
      log(0, "malformed packet m3 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }    
  
  mpz_init(&(st->st_shared));
  mpz_init(&(st->st_gr));
  
  st->st_shared_in_use = 1;
  st->st_gr_in_use = 1;
  
  /* Move temp to st_gr */
  mpz_set(&(st->st_gr), &temp);
  
  /* Compute shared secret */
  mpz_powm(&(st->st_shared), &(st->st_gr), &(st->st_sec), &groupmodulo);
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "Public value received (st_gr) in ipsecdoi_handle_i2: ");
      mpz_out_str(stdout, 16, &(st->st_gr));
      fprintf(stdout, "\n");
      fprintf(stdout, "Shared secret (st_shared) in ipsecdoi_handle_i2: ");
      mpz_out_str(stdout, 16, &(st->st_shared));
      fprintf(stdout, "\n");
    }
#endif 
  
  /* Save Nr */
  st->st_nr = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (st->st_nr == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i2()", 0, 0, 0);
  
  bcopy(isag, st->st_nr, ntohs(isag->isag_length));
  
  /* Delete old retransmit event */
  delete_event(st);
  
  /* Free old packet */
  free(st->st_packet);
  
  /* Advance state */
  /* change 04/98 : do it later, when we are sure we sent the message */
  /*  st->st_state = OAKLEY_MAIN_I_3;*/
  
  /* Generate SKEYID, SKEYID_A, SKEYID_D, SKEYID_E */
  generate_skeyids(st);
  
  /* Generate HASH_I (arg3=1) for initiator (arg4=1) */
  hash = get_hash(st, &i, 1, 1);
  
  st->st_packet_len = sizeof(struct isakmp_hdr) + 
    2 * sizeof(struct isakmp_generic) + i +
    st->st_myidentity_len + sizeof(struct identity);
  /* Padding */
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;      
    case OAKLEY_3DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;      
    default:
      exit_log("unknown/unsupported ISAKMP encryption algorithm %d",
	       st->st_enc, 0, 0);
    }
  
  k = (blocksize - 
       (st->st_packet_len - sizeof(struct isakmp_hdr)) 
       % blocksize) % blocksize;
  st->st_packet_len += k;
  
  st->st_packet = (u_char *) calloc(st->st_packet_len, sizeof(u_char));
  if (st->st_packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i2()", 0, 0, 0);
  
  padding = k;
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VALUES)
    fprintf(stdout, "padding is %d bytes\n", k);
#endif
  
  bcopy(isa, st->st_packet, sizeof(struct isakmp_hdr));
  isa = (struct isakmp_hdr *) st->st_packet;
  isa->isa_length = htonl(st->st_packet_len);
  isa->isa_np = ISAKMP_NEXT_ID;
  
  /* IDii */
  isag = (struct isakmp_generic *) (st->st_packet + 
				    sizeof(struct isakmp_hdr));
  isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    st->st_myidentity_len + sizeof(struct identity));
  isag->isag_np = ISAKMP_NEXT_HASH;
  
  ide = (struct identity *) (st->st_packet + sizeof(struct isakmp_hdr) +
			     sizeof(struct isakmp_generic));
  
  ide->id_type = st->st_myidentity_type;
  ide->id_protoid = st->st_myidentity_protoid; 
  ide->id_port = st->st_myidentity_port; 
  bcopy(st->st_myidentity,
	st->st_packet + sizeof(struct isakmp_hdr) +
	sizeof(struct isakmp_generic) + sizeof(struct identity), 
	st->st_myidentity_len);
  
  /* HASH_I */
  isag = (struct isakmp_generic *) (st->st_packet +
				    sizeof(struct isakmp_hdr) +
				    ntohs(isag->isag_length));
  isag->isag_length = htons(i + sizeof(struct isakmp_generic));
  
  bcopy(hash, st->st_packet + st->st_packet_len - (i + padding), i);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VALUES)
  {
  	fprintf(stdout, "-----\nInitiator Hash(%d): \t", i);
  	format_dump(hash, i);
  	fprintf(stdout, "-----\n");
  }
#endif
  
  free(hash);
  
  isa->isa_flags |= ISAKMP_FLAG_ENCRYPTION;
  
#ifdef DEBUG
      packet_dump("\tINITIATOR->RESPONDER ***** MAIN MODE MESSAGE 5 : HDR* + IDii + HASH_I", DUMP_MSG);
  if (arg_verbose & DEBUG_PACKETS)
  {
      	packet_dump(st->st_packet, DUMP_SENT_BEFORE_ENC, 0);
        fprintf(stdout, "Encrypting %d bytes; algorithm: %s(%d) \n-----IV: ",
	      st->st_packet_len - sizeof(struct isakmp_hdr),
	      ((st->st_enc == OAKLEY_DES_CBC) ? "DES_CBC" : "3DES_CBC"),
	      st->st_enc);
	    format_dump(st->st_iv, st->st_iv_len);
  }
  show_payloads(st->st_packet, 0);
#endif DEBUG
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 5 (ENCRYPTED) SENT BY INITIATOR\n");
  fclose(f_demo);
#endif DEMO

  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      /* Weak key detection already done */
      deskey(des_cbc_keys, st->st_skeyid_e, DES_KEY_ENCRYPT); 
      des_cbc_encrypt(st->st_packet + sizeof(struct isakmp_hdr),
		      st->st_packet_len - sizeof(struct isakmp_hdr),
		      des_cbc_keys, 
		      st->st_iv, 
		      DES_ENCRYPT);
      
      /* Update IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_i2()",
		     0, 0, 0);
	}
      
      st->st_iv_len = DES_CBC_BLOCK_SIZE;
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE,
	    st->st_iv, DES_CBC_BLOCK_SIZE);      
      break;
      
    case OAKLEY_3DES_CBC:
      /* Weak key detection already done */
      /* Encrypt */
      /* Choice to recompute the key and keymaterial everytime we receive a message encrypted with 3DES 
	 or to store the key or the key material in the state 
	 The second solution being much faster but is it crypto-correct? */
      three_des_cbc_key = (char *) malloc(sizeof(key_schedule)*3);
      if (three_des_cbc_key == (unsigned char) NULL)
	exit_log("malloc failed for three_des_cbc_key\n", 0,0,0);
      
      /* construct the key from st->st_skeyid_e, encrypt, erase the computed key */
      deskey(* (key_schedule *) (three_des_cbc_key), 
	     st->st_skeyid_e, 
	     0);
      deskey(* (key_schedule *) (three_des_cbc_key+sizeof(key_schedule)),
	     st->st_skeyid_e + 8, 
	     1);
      deskey(* (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
	     st->st_skeyid_e+16, 
	     0);
      des3_cbc_encrypt(st->st_packet + sizeof( struct isakmp_hdr),
		       st->st_packet_len - sizeof( struct isakmp_hdr),
		       * (key_schedule*) (three_des_cbc_key),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
		       st->st_iv,
		       DES_ENCRYPT);
      free(three_des_cbc_key);

      /* To update the IV, do the same as OAKLEY_DES_CBC */
      /* Update IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE, sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_i2()", 0, 0, 0);
	}      
      st->st_iv_len = DES_CBC_BLOCK_SIZE;
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE,
	    st->st_iv, DES_CBC_BLOCK_SIZE);      
      break;
      /* XXX Support more */
    default:
      exit_log("unsupported ISAKMP encryption algorithm %d", st->st_enc,
	       0, 0);
    }

  if (sendto(sock, st->st_packet, st->st_packet_len, 0, &sa, sizeof(sa)) !=
      st->st_packet_len)
    log(0, "sendto() failed in ipsecdoi_handle_i2()", 0, 0, 0);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_PACKETS)
    	packet_dump(st->st_packet, DUMP_SENT_AFTER_ENC, st->st_packet_len);
#endif

  /* advance state */
  st->st_state = OAKLEY_MAIN_I_3;  

  event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_i2   | in ipsecdoi_mainxchg.c");
#endif
}


/************************************************************************/
/*
 * Handle HDR;IDir;HASH_R from responder.
 * Called by : comm_handle (demux.c)
 */
 /* Added kernelfd as arg to ipsecdoi_handle_i3 - 
          needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     
void ipsecdoi_handle_i3(int                sock, 
			u_char            *buffer, 
			int                length, 
			struct sockaddr    sa,
			struct state      *st, 
			int                kernelfd)
{
  struct isakmp_hdr         *isa;
  struct isakmp_generic     *isag;
  int                        i;
  int                        k;
  struct identity           *id;
  u_char                    *hash;
  struct sockaddr_in         sin;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_i3  | in ipsecdoi_mainxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"=======================================================================\n");
  fprintf(stdout,"Handle Message 6 (HDR* + IDir + HASH_R) from responder\nSend nothing back\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 6 (ENCRYPTED) RECEIVED FROM RESPONDER\n");
  fclose(f_demo);
#endif DEMO
  isa = (struct isakmp_hdr *) buffer;
  if (length <= sizeof(struct isakmp_hdr) + 
      2 * sizeof(struct isakmp_generic) + sizeof(struct identity))
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  if (isa->isa_np != ISAKMP_NEXT_ID)
    {
      log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i = sizeof(struct isakmp_hdr);
  
  /* IDir */
  isag = (struct isakmp_generic *) (buffer + i);
  if (ntohs(isag->isag_length) + i >= length)
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa), 
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_generic);
  if (isag->isag_np != ISAKMP_NEXT_HASH)
    {
      log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  id = (struct identity *) (buffer + i);
  
  /* Check for valid ID types*/
  /* we accept non filled protoid and/or non filled port number. */
  i += sizeof(struct identity);
  st->st_peeridentity_type = id->id_type;
  if (st->st_peeridentity_protoid != id->id_protoid)
    {
      log(0, "ID Protocol Mismatch: received %d; expected %d", 
	  id->id_protoid, st->st_peeridentity_protoid, 0 );
      if (id->id_protoid==0)
	{
	  log(0, "ID Protocol 0 accepted", 0, 0 ,0);
	  st->st_peeridentity_protoid = 0;
	}
      else
	{ 
	  log(1, "ID Protocol %d rejected", id->id_protoid, 0, 0);
	  free(buffer);
	  return;
	}
    }
  if (st->st_peeridentity_port != id->id_port)
    {
      log(0, "ID Port Mismatch: received %d; expected %d", 
	  ntohs(id->id_port), ntohs(st->st_peeridentity_port), 0);
      if (id->id_port==0)
	{ 
	  log(0, "ID Port 0 accepted", 0, 0 ,0);
	  st->st_peeridentity_port = 0;
	}
      else
	{ 
	  log(1, "ID Port %d rejected", ntohs(id->id_port), 0, 0);
	  free(buffer);
	  return;
	}      
    }

  st->st_peeridentity_len = ntohs(isag->isag_length) -
    (sizeof(struct isakmp_generic) + 
     sizeof(struct identity));
  st->st_peeridentity = (u_char *) calloc(st->st_peeridentity_len,
					  sizeof(u_char));
  if (st->st_peeridentity == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_i3()", 0, 0, 0);
  
  bcopy(buffer + i, st->st_peeridentity, st->st_peeridentity_len);
  i += st->st_peeridentity_len;

#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE){
    if (id->id_type == ID_IPV4_ADDR)
      {
	bcopy(st->st_peeridentity, &(sin.sin_addr), sizeof(sin));
	fprintf(stdout, "IDir is %s\n", inet_ntoa(sin.sin_addr));
      }
  }
#endif
  
  /* HASH_R */
  isag = (struct isakmp_generic *) (buffer + i);
  
  if ((isag->isag_np != ISAKMP_NEXT_NONE) ||
      (ntohs(isag->isag_length) + i != length))
    {
      log(0, "bad packet from %s, port %d", PRINTADDRESS(sa), 
	  get_port(sa), 0);
      free(buffer);
      free(st->st_peeridentity);	/* Rollback */
      st->st_peeridentity = (u_char *) NULL;
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_generic);
  
  /* Generate HASH_R (arg3=0) for initiator (arg4=1) */
  hash = get_hash(st, &k, 0, 1);
  
  if (length - i != k)
    {
      log(0, "computed and received HASH_R lengths don't match (%d/%d)",
	  length - i, k, 0);
#ifdef DEBUG
      fprintf(stdout, "computed HASH_R(%d): ", length-i);
      format_dump(hash, length-i);
      fprintf(stdout, "received HASH_R(%d): ", length-i);
      format_dump(buffer + i, length-i);  
#endif
       free(buffer);
      free(hash);
      free(st->st_peeridentity);
      st->st_peeridentity = (u_char *) NULL;
      /* XXX Could send notification back */
      return;
    }
  
  if (bcmp(hash, buffer + i, length - i))
    {
      log(0, "received and computed HASH_R values don't match", 0, 0, 0);
#ifdef DEBUG
      fprintf(stdout, "computed HASH_R(%d): ", length-i);
      format_dump(hash, length-i);
      fprintf(stdout, "received HASH_R(%d): ", length-i);
      format_dump(buffer + i, length-i);  
#endif
      free(buffer); 
      free(hash); 
      free(st->st_peeridentity); 
      st->st_peeridentity = (u_char *) NULL;
      /* XXX Could send notification back */
      return; 
    }
#ifdef DEBUG
  else
    {
      if (arg_verbose & DEBUG_VALUES)
      {
         fprintf(stdout, "-----\nResponder Hash(%d): ", length-i);
         format_dump(hash, length-i);
  	     fprintf(stdout, "-----\n");
      }
     if (arg_verbose & DEBUG_VERBOSE)
       fprintf(stdout, "HASH_R verified\n");
    }
#endif

  /* Free old packet */
  free(st->st_packet);
  st->st_packet = (u_char *) NULL;

  /* Free computed HASH_R */
  free(hash);
    
  /* Free received packet */
  free(buffer);
    
  /* Delete pending event */
  delete_event(st);
    
  /* Advance state */
  st->st_state = OAKLEY_MAIN_I_4;

#ifdef DEBUG
 packet_dump("************** PHASE 1 COMPLETED ************* PHASE 1 COMPLETED **************\n",
	       DUMP_MSG, 0);
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "************** PHASE 1 COMPLETED ************* PHASE 1 COMPLETED **************\n");
  fclose(f_demo);
#endif DEMO

  /* Is there a quick mode exchange following ? */
  if (st->st_goal != 0)
    {
#ifdef DEBUG
      fprintf(stdout, "Initiate Quick Mode Exchange with %s, port %d, goal %d\n",
	  PRINTADDRESS(sa), get_port(sa), st->st_goal);
#endif
      /* Added kernelfd as arg to initiate_quick - 
	 needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     
      initiate_quick(sock, st, kernelfd);
    }
    
  /* Schedule for SA expiration */
  if (st->st_expire != 0)
    event_schedule(EVENT_SA_EXPIRE, st->st_expire, st, 0);
  else
    event_schedule(EVENT_SA_EXPIRE, DEFAULT_SA_EXPIRE, st, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_i3  | in ipsecdoi_mainxchg.c");
#endif
}

/************************************************************************/
/*
 * Handle HDR;KE;Ni from Initiator. Send a HDR;KE;Nr back.
 * called by comm_handle (demux.c)
 */
void ipsecdoi_handle_r1(int               sock, 
			u_char           *buffer, 
			int               length, 
			struct sockaddr   sa,
			struct state     *st)
{
  struct isakmp_hdr        *isa;
  struct isakmp_generic    *isag;
  u_char                   *packet;
  int                       i;
  int                       k;
  int                       k1;
  u_char                    tmp[LOCALSECRETSIZE];
  MP_INT                    temp;
  MP_INT                    temp2;
  u_char                   *gr;
  int                       grlen;
  int                       grbegin;
  int                       groffset;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_r1  | in ipsecdoi_mainxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"=======================================================================\n");
  fprintf(stdout,"Handle Message 3 (HDR + KE + Ni) from Initiator\nSend back HDR + KE + Nr\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 3 RECEIVED FROM INITIATOR\n");
  fclose(f_demo);
#endif DEMO
  isa = (struct isakmp_hdr *) buffer;
  
  if (length <= sizeof(struct isakmp_hdr) + 
      2 * sizeof(struct isakmp_generic))
    {
      free(buffer);
      log(0, "too short HDR;KE;Ni packet (%d bytes) from %s, port %d", 
	  length, PRINTADDRESS(sa), get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
  i = htons(isag->isag_length);
  if (i + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic) > 
      length)
    {
      free(buffer);
      log(0, "malformed packet m4 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }
  
  if (i - sizeof(struct isakmp_generic) < MINIMUM_PUBLIC_VALUE_SIZE)
    {
      free(buffer);
      log(0, "too short public size (%d) sent from %s, port %d",
	  i - sizeof(struct isakmp_generic), PRINTADDRESS(sa), 
	  get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  if (isag->isag_np != ISAKMP_NEXT_NONCE)
    {
      free(buffer);
      log(0, "unexpected header %d in HDR;KE;Ni packet from %s, port %d",
	  isag->isag_np, PRINTADDRESS(sa), get_port(sa));
      /* XXX Could send notification back */
      return;
    }
  
  mpz_init(&temp);
  mpz_init(&temp2);
  
/* Get gi from buffer & convert to MP_INT (stored in temp) */
  i += sizeof(struct isakmp_hdr);
  for (k = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
       k < i; k++)
    {
      mpz_mul_ui(&temp2, &temp, 256);
      mpz_add_ui(&temp, &temp2, buffer[k]);
    }
  
  /* Check the Ni header */
  isag = (struct isakmp_generic *) (buffer + i);
  if (ntohs(isag->isag_length) + i != length)
    {
      free(buffer);
      log(0, "malformed packet m5 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }
  
  mpz_init(&(st->st_gi));
  mpz_init(&(st->st_gr));
  mpz_init(&(st->st_sec));
  mpz_init(&(st->st_shared));
  
  st->st_gi_in_use = 1;
  st->st_gr_in_use = 1;
  st->st_sec_in_use = 1;
  st->st_shared_in_use = 1;
  
/* Move temp to st_gi */
  mpz_set(&(st->st_gi), &temp);
  
  get_rnd_bytes(tmp, LOCALSECRETSIZE);
#ifdef DEBUG
   if (arg_verbose & DEBUG_VALUES)
   {
  	fprintf(stdout, "\nResponder's Secret - y(%d) : \n", LOCALSECRETSIZE);
  	format_dump(tmp, LOCALSECRETSIZE);
   }
#endif
  
  mpz_set_ui(&temp, 0);
  
  for (k = 0; k < LOCALSECRETSIZE; k++)
    {
      mpz_mul_ui(&temp2, &temp, 256);
      mpz_add_ui(&temp, &temp2, tmp[k]);
    }
  
  mpz_set(&(st->st_sec), &temp);
  
  mpz_powm(&(st->st_gr), &groupgenerator, &(st->st_sec), &groupmodulo);
  
  mpz_powm(&(st->st_shared), &(st->st_gi), &(st->st_sec), &groupmodulo);
    
  grlen = mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1;
    if (grlen < 96)
	grlen = 96;
  gr = (u_char *) calloc(grlen, sizeof(u_char));
  if (gr == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);
  
  mpz_set(&temp, &(st->st_gr));
  
  for (k = grlen - 1 ; k >= 0; k--)
    {
      gr[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
      mpz_set(&temp, &temp2);
    }
  
  /* Strip leading byte of zeroes, if necessary */
  grbegin = 0;
  if ((gr[0] == 0) && (grlen > 96)) {
    grbegin++;
    grlen--;
  }

  mpz_clear(&temp);
  mpz_clear(&temp2);

#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "Group modulo (groupmodulo) in ipsecdoi_handle_r1: ");
      mpz_out_str(stdout, 16, &(groupmodulo));
      fprintf(stdout, "\n");
      fprintf(stdout, "Group generator (groupgenerator) in ipsecdoi_handle_r1: ");
      mpz_out_str(stdout, 16, &(groupgenerator));
      fprintf(stdout, "\n");
      fprintf(stdout, "Local secret (st_sec) in ipsecdoi_handle_r1: ");
      mpz_out_str(stdout, 16, &(st->st_sec));
      fprintf(stdout, "\n");
      fprintf(stdout, "Public value sent (st_gr) in ipsecdoi_handle_r1: ");
      mpz_out_str(stdout, 16, &(st->st_gr));
      fprintf(stdout, "\n");
      fprintf(stdout, "Public value is %d bytes long.\n", grlen);
      fprintf(stdout, "Public value received (st_gi) in ipsecdoi_handle_r1: ");
      mpz_out_str(stdout, 16, &(st->st_gi));
      fprintf(stdout, "\n");
      fprintf(stdout, "Shared secret (st_shared) in ipsecdoi_handle_r1: ");
      mpz_out_str(stdout, 16, &(st->st_shared));
      fprintf(stdout, "\n");
    }
#endif 

  /* Save Ni */
  st->st_ni = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (st->st_ni == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);
  
  bcopy(isag, st->st_ni, ntohs(isag->isag_length));
  
  /* Create Nr */
  st->st_nr = (u_char *) calloc(DEFAULTNONCESIZE + 
				sizeof(struct isakmp_generic), 
				sizeof(u_char));
  if (st->st_nr == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);
  
  isag = (struct isakmp_generic *) st->st_nr;
  isag->isag_length = ntohs(DEFAULTNONCESIZE + 
			    sizeof(struct isakmp_generic));
  
  get_rnd_bytes(st->st_nr + sizeof(struct isakmp_generic), DEFAULTNONCESIZE);
  
   i = sizeof(struct isakmp_hdr) + 2 * sizeof(struct isakmp_generic) +
	grlen +DEFAULTNONCESIZE;
  
  packet = (u_char *) calloc(i, sizeof(u_char));
  if (packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);
    
  bcopy(isa, packet, sizeof(struct isakmp_hdr));
    
  isa = (struct isakmp_hdr *) packet;
  isag = (struct isakmp_generic *) (packet + sizeof(struct isakmp_hdr));
  isag->isag_np = ISAKMP_NEXT_NONCE;
  isa->isa_length = htonl(i);

  isag->isag_length = htons(grlen + sizeof(struct isakmp_generic));
    
  i -= (sizeof(struct isakmp_generic) + DEFAULTNONCESIZE);

  groffset = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic) - grbegin;
  for (k = grbegin; k < grbegin + grlen; k++) 
    packet[k + groffset] = gr[k];
  
  isag = (struct isakmp_generic *) st->st_nr;
  bcopy(st->st_nr, packet + i, ntohs(isag->isag_length));
    
  /* Advance state */
  /* change 04/98 : do it later, when we are sure we sent the message */
  /*  st->st_state = OAKLEY_MAIN_R_2;*/

  /* Delete previous CLEANUP event */
  delete_event(st);

  /* Free received packet */
  free(buffer);

  /* No need to free old packet, since we didn't keep it */
  
  st->st_packet = packet;
  st->st_packet_len = i + DEFAULTNONCESIZE + sizeof(struct isakmp_generic);
  if (sendto(sock, packet, st->st_packet_len, 0, &sa, sizeof(sa)) !=
      st->st_packet_len)
    log(0, "sendto() failed in ipsecdoi_handle_r1()", 0, 0, 0);
#ifdef DEBUG
  packet_dump("\tRESPONDER->INITIATOR ***** MAIN MODE MESSAGE 4 : HDR + KE + Nr", DUMP_MSG);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(st->st_packet, DUMP_SENT_NO_ENC, 0);
  show_payloads(st->st_packet, 0);
#endif DEBUG
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 4 SENT BY RESPONDER\n");
  fclose(f_demo);
#endif DEMO

    /* Generate SKEYID, SKEYID_A, SKEYID_D, SKEYID_E */
  generate_skeyids(st);
  
  /* advance state */
  st->st_state = OAKLEY_MAIN_R_2;  

  event_schedule(EVENT_CLEANUP, EVENT_CLEANUP_DELAY, st, 0);    
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_r1  | in ipsecdoi_mainxchg.c");
#endif
}

/************************************************************************/
/*
 * Handle HDR;IDii;HASH_I from initiator. Send a HDR;IDir;HASH_R back.
 * Called by comm_handle (demux.c)
 */
void ipsecdoi_handle_r2(int               sock, 
			u_char           *buffer, 
			int               length, 
			struct sockaddr   sa,
			struct state     *st)
{
  struct isakmp_hdr       *isa;
  struct isakmp_generic   *isag;
  struct identity         *id;
  u_char                  *hash;
  int                      i;
  int                      k;
  int                      k1;
  int                      blocksize;
  int                      padding;
  struct sockaddr_in       sin;
  key_schedule             des_cbc_keys;

  char           *three_des_cbc_key;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_r2  | in ipsecdoi_mainxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"=======================================================================\n");
  fprintf(stdout,"Handle Message 5 (HDR* + IDii + HASH_I) from initiator\nSend back HDR* + IDir + HASH_R\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 5 (ENCRYPTED) RECEIVED FROM INITIATOR\n");
  fclose(f_demo);
#endif DEMO
  isa = (struct isakmp_hdr *) buffer;
  if (length <= sizeof(struct isakmp_hdr) +
      2 * sizeof(struct isakmp_generic) + sizeof(struct identity))
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  if (isa->isa_np != ISAKMP_NEXT_ID)
    {
      log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i = sizeof(struct isakmp_hdr);
  
  /* IDii */
  isag = (struct isakmp_generic *) (buffer + i);
  if (ntohs(isag->isag_length) + i >= length)
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa), 
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_generic);
  if (isag->isag_np != ISAKMP_NEXT_HASH)
    {
      log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  id = (struct identity *) (buffer + i);
  
  /* Check for valid ID types */
  if (st->st_peeridentity_protoid != id->id_protoid)
    {
      log(0, "ID Protocol Mismatch: received %d; expected %d", 
	  id->id_protoid, st->st_peeridentity_protoid, 0 );
      if (id->id_protoid==0)
	{
	  log(0, "ID Protocol 0 accepted", 0, 0 ,0);
	  st->st_peeridentity_protoid = 0;
	  st->st_myidentity_protoid = 0; 
	}
      else
	{ 
	  log(1, "ID Protocol %d rejected", id->id_protoid, 0, 0);
	  free(buffer);
	  return;
	}
    }
  if (st->st_peeridentity_port != id->id_port)
    {
      log(0, "ID Port Mismatch: received %d; expected %d", 
	  ntohs(id->id_port), ntohs(st->st_peeridentity_port), 0);
      if (id->id_port==0)
	{ 
	  log(0, "ID Port 0 accepted", 0, 0 ,0);
	  st->st_peeridentity_port = 0;
	  st->st_myidentity_port = 0;
	}
      else
	{ 
	  log(1, "ID Port %d rejected", ntohs(id->id_port), 0, 0);
	  free(buffer);
	  return;
	}      
    }
  
  i += sizeof(struct identity);
  st->st_peeridentity_type = id->id_type;
  st->st_peeridentity_len = ntohs(isag->isag_length) - (sizeof(struct isakmp_generic) + sizeof(struct identity));
  st->st_peeridentity = (u_char *) calloc(st->st_peeridentity_len, sizeof(u_char));
  if (st->st_peeridentity == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);
  
  bcopy(buffer + i, st->st_peeridentity, st->st_peeridentity_len);
  i += st->st_peeridentity_len;
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE){
    fprintf(stdout, "IDii type is %d, length %d\n", id->id_type, 
	    st->st_peeridentity_len);
    if (id->id_type == ID_IPV4_ADDR)
      {
	bcopy(st->st_peeridentity, &(sin.sin_addr), sizeof(sin));
	fprintf(stdout, "IDii is %s\n", inet_ntoa(sin.sin_addr));
      }
  }
#endif
  
  /* HASH_I */
  isag = (struct isakmp_generic *) (buffer + i);
  
  if ((isag->isag_np != ISAKMP_NEXT_NONE) ||
      (ntohs(isag->isag_length) + i != length))
    {
      log(0, "bad packet from %s, port %d", PRINTADDRESS(sa), 
	  get_port(sa), 0);
      free(buffer);
      free(st->st_peeridentity);	/* Rollback */
      st->st_peeridentity = (u_char *) NULL;
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_generic);
  
  /* Compute HASH_I (arg3=1) for responder (arg4=0) */
  hash = get_hash(st, &k, 1, 0);
  
  if (length - i != k)
    {
      log(0, "computed and received HASH_I lengths don't match (%d/%d)",
	  length - i, k, 0);
      free(buffer);
      free(hash);
      free(st->st_peeridentity);
      st->st_peeridentity = (u_char *) NULL;
      /* XXX Could send notification back */
      return;
    }

  if (bcmp(hash, buffer + i, length - i))
    {
      log(0, "received and computed HASH_I values don't match", 0, 0, 0);
#ifdef DEBUG
      fprintf(stdout, "computed HASH_I: ");
      format_dump(hash, length-i);
      fprintf(stdout, "received HASH_I: ");
      format_dump(buffer+i, length-i);
#endif
      free(buffer);
      free(hash);
      free(st->st_peeridentity);
      st->st_peeridentity = (u_char *) NULL;
      /* XXX Could send notification back */
      return;
    }
#ifdef DEBUG
  else
    {
      if (arg_verbose & DEBUG_VALUES)
      {
         fprintf(stdout, "-----\nInitiator Hash(%d): ", length-i);
         format_dump(hash, length-i);
  	     fprintf(stdout, "-----\n");
      }
     if (arg_verbose & DEBUG_VERBOSE)
       fprintf(stdout, "HASH_I verified\n");
    }
#endif
  
  /* Free old packet */
  free(st->st_packet);
  
  /* Free computed HASH_I */
  free(hash);
  
  st->st_packet_len = sizeof(struct isakmp_hdr) + 
    2 * sizeof(struct isakmp_generic) + k +
    st->st_myidentity_len + sizeof(struct identity);
  
  /* Padding */
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
    case OAKLEY_3DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
      
    default:
      exit_log("unknown/unsupported ISAKMP encryption algorithm %d",
	       st->st_enc, 0, 0);
    }
  
  k = (blocksize - 
       (st->st_packet_len - sizeof(struct isakmp_hdr)) % blocksize) % 
    blocksize;
  
  st->st_packet_len += k;
  
#ifdef DEBUG
  if(arg_verbose & DEBUG_VALUES)
    fprintf(stdout, "padding is %d bytes\n", k);
#endif
  
  st->st_packet = (u_char *) calloc(st->st_packet_len, sizeof(u_char));
  if (st->st_packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);
  
  /* st->st_packet[st->st_packet_len - 1] = k; */
  padding = k;
  
  bcopy(buffer, st->st_packet, sizeof(struct isakmp_hdr));
  isa = (struct isakmp_hdr *) st->st_packet;
  isa->isa_length = htonl(st->st_packet_len);
  
  /* IDir */
  i = sizeof(struct isakmp_hdr);
  isag = (struct isakmp_generic *) (st->st_packet + i);
  i += sizeof(struct isakmp_generic);
  id = (struct identity *) (st->st_packet + i);
  i += sizeof(struct identity);
  
  isag->isag_np = ISAKMP_NEXT_HASH;
  isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    sizeof(struct identity) +
			    st->st_myidentity_len);
  
  id->id_type = st->st_myidentity_type;
  id->id_protoid = st->st_myidentity_protoid;
  id->id_port = st->st_myidentity_port;
  bcopy(st->st_myidentity, st->st_packet + i, st->st_myidentity_len);
  i += st->st_myidentity_len;
  
  /* Get HASH_R (arg3=0) for responder (arg4=0) */
  hash = get_hash(st, &k, 0, 0);
  
  /* HASH_R */
  isag = (struct isakmp_generic *) (st->st_packet + i);
  i += sizeof(struct isakmp_generic);
  isag->isag_length = htons(sizeof(struct isakmp_generic) + k);
  bcopy(hash, st->st_packet + i, k);


#ifdef DEBUG
  if (arg_verbose & DEBUG_VALUES)
  {
  	fprintf(stdout, "-----\nResponder Hash(%d): \t", k);
  	format_dump(hash, k);
  	fprintf(stdout, "-----\n");
  }
#endif
  
  /* Free computed HASH_R */
  free(hash);
  
#ifdef DEBUG
  packet_dump("\tRESPONDER->INITIATOR ***** MAIN MODE MESSAGE 6 : HDR* + IDir + HASH_R", DUMP_MSG);
  if (arg_verbose & DEBUG_PACKETS)
  {
      	packet_dump(st->st_packet, DUMP_SENT_BEFORE_ENC, 0);
        fprintf(stdout, "Encrypting %d bytes; algorithm: %s(%d) \n-----IV: ",
	      st->st_packet_len - sizeof(struct isakmp_hdr),
	      ((st->st_enc == OAKLEY_DES_CBC) ? "DES_CBC" : "3DES_CBC"),
	      st->st_enc);
	    format_dump(st->st_iv, st->st_iv_len);
  }
  show_payloads(st->st_packet, 0);
  if(arg_verbose & DEBUG_VERBOSE)
    {
     fprintf(stdout, "lastblock: \t");
     format_dump(st->st_lastblock, st->st_lastblock_len);
     }
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "MAIN MODE MESSAGE 6 (ENCRYPTED) SENT BY RESPONDER\n");
  fclose(f_demo);
#endif DEMO
  
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      /* Weak key detection already done */
      deskey(des_cbc_keys, st->st_skeyid_e, DES_KEY_ENCRYPT);
      des_cbc_encrypt(st->st_packet + sizeof(struct isakmp_hdr),
		      st->st_packet_len - sizeof(struct isakmp_hdr),
		      des_cbc_keys, 
		      st->st_iv, 
		      DES_ENCRYPT);
      
      /* Last block of Phase 1, kept for Phase 2 IV generation */
      st->st_lastblock = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					   sizeof(u_char));
      if (st->st_lastblock == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);
      
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE,
	    st->st_lastblock, DES_CBC_BLOCK_SIZE);
      st->st_lastblock_len = DES_CBC_BLOCK_SIZE;
      
      /* Keep last block as IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_r2()",
		     0, 0, 0);
	}
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    st->st_iv, DES_CBC_BLOCK_SIZE);
      st->st_iv_len = DES_CBC_BLOCK_SIZE;
      break;

    case OAKLEY_3DES_CBC:
#ifdef DEBUG
      fprintf(stdout,"\nEncode with 3DES in Main exchange\n");
#endif
      /* Weak key detection already done */
      /* Encrypt */
      /* Choice to recompute the key and keymaterial everytime we receive a message encrypted with 3DES 
	 or to store the key or the key material in the state 
	 The second solution being much faster but is it crypto-correct? */
      three_des_cbc_key = (char *) malloc(sizeof(key_schedule)*3);
      if (three_des_cbc_key == (unsigned char) NULL)
	exit_log("malloc() failed for three_des_cbc_key\n", 0,0,0);
      
      deskey(* (key_schedule *) (three_des_cbc_key), 
	     st->st_skeyid_e, 
	     0);
      deskey(* (key_schedule *) (three_des_cbc_key+sizeof(key_schedule)),
	     st->st_skeyid_e + 8, 
	     1);
      deskey(* (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
	     st->st_skeyid_e+16, 
	     0);
      
      des3_cbc_encrypt(st->st_packet + sizeof( struct isakmp_hdr),
		       st->st_packet_len - sizeof( struct isakmp_hdr),
		       * (key_schedule*) (three_des_cbc_key),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
		       st->st_iv,
		       DES_ENCRYPT);

      /* free the key */
      free(three_des_cbc_key);

      /* Last block of Phase 1, kept for Phase 2 IV generation */
      st->st_lastblock = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					   sizeof(u_char));
      if (st->st_lastblock == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE,
	    st->st_lastblock, DES_CBC_BLOCK_SIZE);
      st->st_lastblock_len = DES_CBC_BLOCK_SIZE;
      
      /* Keep last block as IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_r2()",
		     0, 0, 0);
	}
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    st->st_iv, DES_CBC_BLOCK_SIZE);
      st->st_iv_len = DES_CBC_BLOCK_SIZE;

      break;
      /* XXX Support more */
    default:
      exit_log("unsupported ISAKMP encryption algorithm %d", st->st_enc,
	       0, 0);
    }     
  
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	{
      fprintf(stdout, "updated IV:\t");
      format_dump(st->st_iv, st->st_iv_len);
	  fprintf(stdout, "updated lastblock:\t");
	  format_dump(st->st_lastblock, st->st_lastblock_len);
  	  fprintf(stdout, "last encrypted block of Phase 1: ");
  	  format_dump(st->st_lastblock, st->st_lastblock_len);
  	  fprintf(stdout, "IV:\t");
  	  format_dump(st->st_iv, st->st_iv_len);
	}
#endif

  /* Get rid of received packet */
  free(buffer);
  
  /* Delete pending CLEANUP event */
  delete_event(st);

  if (sendto(sock, st->st_packet, st->st_packet_len, 0, &sa, sizeof(sa))
      != st->st_packet_len)
    log(0, "sendto() failed in ipsecdoi_handle_r2()", 0, 0, 0);

#ifdef DEBUG
  if (arg_verbose & DEBUG_PACKETS)
    	packet_dump(st->st_packet, DUMP_SENT_AFTER_ENC, st->st_packet_len);
 packet_dump("************** PHASE 1 COMPLETED ************* PHASE 1 COMPLETED **************\n",
	       DUMP_MSG, 0);
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "************** PHASE 1 COMPLETED ************* PHASE 1 COMPLETED **************\n");
  fclose(f_demo);
#endif DEMO


  /* advance state */
  st->st_state = OAKLEY_MAIN_R_3;  
  /* Expiration of SA */
  if (st->st_expire != 0)
    event_schedule(EVENT_SA_EXPIRE, st->st_expire, st, 0);
  else
    event_schedule(EVENT_SA_EXPIRE, DEFAULT_SA_EXPIRE, st, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_r2   | in ipsecdoi_mainxchg.c");
#endif
}











