/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/* ipsec_doi_quickxchg.c - constructs and handles Quick Mode packets
 * 		Handles PFS; Ordering of payloads is fixed.
 */

#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include "constants.h"
#include "state.h"
#include "packet.h"
#include "defs.h"
#include DES_H
#include SHA_H
#include MD5_H

/* No real policy yet, just take what the arguments are */
#include "argdefs.h"

#include <sadb.h>

#define PRINTADDRESS(x) (get_address(x) != (char *) NULL ? get_address(x) : "(unknown address family)")

extern char   *get_address();
extern u_char *oakley_get_sa();

/************************************************************************/
/*
 * goal :  Initiate quick mode ; prepare the first xchg of the Oakley Quick Mode :
 * HDR* + HASH1 + SA0 + SA1 + Ni [+KE] [+IDui + IDur] --> 
 * Called by  : oakley_initiate      
 */
/* Added kernelfd as arg to initiate_quick - 
   needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     

void initiate_quick(int           sock, 
		    struct state *st, 
		    int           kernelfd)
{
  int                    packet_ptr;
  int                    j;
  int                    k;
  int                    k1;
  int                    blocksize;
  u_char                *p;
  unsigned long          des_cbc_keys[16][2];

  char                  *three_des_cbc_key;
  struct state          *nst;
  struct isakmp_hdr     *isa;
  struct isakmp_generic *isag;
  struct isakmp_sa      *isasa;
  struct sockaddr_in     sin;
  struct MD5Context      md5ctx;
  SHA_CTX                shactx;
  MP_INT                 temp;
  MP_INT                 temp2;
  int                    gilen;
  int                    gibegin;
  u_char                 tmp[LOCALSECRETSIZE];
  u_char                *gi;
  int                    point_pfs_ke;
  int                    ke_length; 
  int                    gioffset;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO

#ifdef TEST_SEND_CLIENT_ID
/* Send Client ID */
  struct identity          *ide;
#endif TEST_SEND_CLIENT_ID

#ifdef DEBUG_IN_OUT
  in_out_functions(" E initiate_quick  | in ipsecdoi_quickxchg.c");
#endif
#ifdef DEBUG
  if (st->st_pfs_handle)
  	fprintf(stdout,"Initiate Quick Mode\nSend HDR* + HASH(1) + SA + Ni + KE\n");
  else
  	fprintf(stdout,"Initiate Quick Mode\nSend HDR* + HASH(1) + SA + Ni\n");
#endif
  nst = (struct state *) duplicate_state(st);

  nst->st_packet = (u_char *) calloc(8192, sizeof(u_char));
  if (nst->st_packet == (u_char *) NULL)
    exit_log("calloc() failed in initiate_quick()", 0, 0, 0);
  
  nst->st_msgid = get_msgid(st->st_peer);
  nst->st_state = OAKLEY_QUICK_I_1;
  
  /* Don't move this before the get_msgid() call */
  insert_state(nst);
  
  /* ISAKMP Header */
  isa = (struct isakmp_hdr *) nst->st_packet;
  bcopy(nst->st_icookie, isa->isa_icookie, COOKIE_SIZE);
  bcopy(nst->st_rcookie, isa->isa_rcookie, COOKIE_SIZE);
  isa->isa_np = ISAKMP_NEXT_HASH;
  isa->isa_maj = ISAKMP_MAJOR_VERSION;
  isa->isa_min = ISAKMP_MINOR_VERSION;
  isa->isa_xchg = ISAKMP_XCHG_QUICK;
  isa->isa_flags |= ISAKMP_FLAG_ENCRYPTION;
  isa->isa_msgid = nst->st_msgid;
  
  /* point after HDR */
  packet_ptr = sizeof(struct isakmp_hdr);
  
  /* HASH(1) -- first pass */
  isag = (struct isakmp_generic *) (nst->st_packet + packet_ptr);
  isag->isag_np = ISAKMP_NEXT_SA;
  if (nst->st_prf == 0)
    {
      switch (nst->st_hash)
	{
	case OAKLEY_MD5:
	  isag->isag_length = htons(16 + sizeof(struct isakmp_generic));
	  break;
	  
	case OAKLEY_SHA:
	  isag->isag_length = htons(20 + sizeof(struct isakmp_generic));
	  break;
	  
	default:
	  exit_log("unknown ISAKMP hash algorithm (%d) in initiate_quick()", nst->st_hash, 0, 0);
	}
    }
  else
    {
      switch (nst->st_prf)
	{
	default:
	  exit_log("unknown PRF (%d) in initiate_quick()",
		   nst->st_prf, 0, 0);
	}
    }
  
  /* Point after HASH(1) */
  packet_ptr += ntohs(isag->isag_length);

 /* Initialize sin so that kernel_getspi has correct destination address to
    send to kernel  */
  bcopy(&(st->st_peer), &sin, sizeof(sin));
  
  /* Added kernelfd as arg to oakley_get_sa - 
     needed for oakley_get_ah_prop's call to kernel_getspi  */     
  /* Arg#6 = 1 ==> tells oakley_get_sa that we are the initiator  */
  p = oakley_get_sa(&j, sin, nst->st_goal, nst, kernelfd, 1);
  if (p == (u_char *) NULL)
    {
      free_state(nst);
      return;
    }
  
  bcopy(p, nst->st_packet + packet_ptr, j);
  
  /* Fix NextPayload field */
  isasa = (struct isakmp_sa *) (nst->st_packet + packet_ptr);
  isasa->isasa_np = ISAKMP_NEXT_NONCE;
  
  /* Also copy SA into state */
  nst->st_sa = (u_char *) calloc(j, sizeof(u_char));
  if (nst->st_sa == (u_char *) NULL)
    exit_log("calloc() failed in initiate_quick()", 0, 0, 0);
  /*  bcopy(p, nst->st_sa, j);*/
  bcopy(nst->st_packet + packet_ptr, nst->st_sa, j);

  /* Point right after the SA */
  packet_ptr += j;

  free(p);
  
  /* Ni */
  isag = (struct isakmp_generic *) (nst->st_packet + packet_ptr);
  isag->isag_length = htons(sizeof(struct isakmp_generic) + DEFAULTNONCESIZE);
  get_rnd_bytes(nst->st_packet + packet_ptr + 
		sizeof(struct isakmp_generic), DEFAULTNONCESIZE);
    
  /* Copy Ni into state */
  nst->st_ni = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (nst->st_ni == (u_char *) NULL)
    exit_log("calloc() failed in initiate_quick()", 0, 0, 0);  
  bcopy(isag, nst->st_ni, ntohs(isag->isag_length));

  /* point right after the Ni */
  packet_ptr += ntohs(isag->isag_length);

  /* handle PFS */
  if (arg_pfs_requested)
    {
      nst->st_pfs_handle = 1;
      isag->isag_np = ISAKMP_NEXT_KE; 
      mpz_init(&(nst->st_sec));
      mpz_init(&(nst->st_gi));
      mpz_init(&(temp));
      mpz_init(&(temp2));
      
      get_rnd_bytes(tmp, LOCALSECRETSIZE);
      
      for (k =0; k < LOCALSECRETSIZE ; k++)
	{
	  mpz_mul_ui(&(nst->st_gi), &(nst->st_sec), 256);
	  mpz_add_ui(&(nst->st_sec), &(nst->st_gi), tmp[k]);
	}
      
      mpz_powm(&(nst->st_gi), &groupgenerator, &(nst->st_sec), &groupmodulo);

      mpz_init(&(temp));
      mpz_init(&(temp2));
      
      gilen = mpz_sizeinbase(&(nst->st_gi), 16) / 2 + 1;
    	if (gilen < 96)
		gilen = 96;
      gi = (u_char *) calloc(gilen, sizeof(u_char));
      if (gi == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
      
      mpz_set(&temp, &(nst->st_gi));
      
      for (k = gilen - 1 ; k >= 0; k--)
	{
	  gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	  mpz_set(&temp, &temp2);
	}
      
      /* strip leading byte of zeroes, if necessary */
      gibegin = 0;
      if ((gi[0] == 0) && (gilen > 96)) {
	gibegin++;
	gilen--;
      }
      
      mpz_clear(&temp);
      mpz_clear(&temp2);
      
      /* put the keying material in the packet */
      isag = (struct isakmp_generic *) (nst->st_packet + packet_ptr);
      isag->isag_np = ISAKMP_NEXT_NONE;
      isag->isag_length = htons(gilen + sizeof(struct isakmp_generic));
      gioffset = packet_ptr + sizeof(struct isakmp_generic) - gibegin;
      for (k = gibegin; k < gibegin + gilen; k++) 
	nst->st_packet[gioffset + k] = gi[k];

      /* point right after the KE */
      packet_ptr +=  ntohs(isag->isag_length);
      
    } /* end "if(arg_pfs_requested)" */

#ifdef TEST_SEND_CLIENT_ID
/* Send Client ID */
/* Testing - always send Client ID */
/* Allocate space in packet for Client Identities */
        isag->isag_np = ISAKMP_NEXT_ID;

      /* add the IDci (my client) payload to the packet */
     isag = (struct isakmp_generic *) (nst->st_packet + packet_ptr);
     isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    nst->st_myidentity_len + sizeof(struct identity));
     isag->isag_np = ISAKMP_NEXT_ID;
  
     ide = (struct identity *) (nst->st_packet + packet_ptr +
			     sizeof(struct isakmp_generic));
     ide->id_type = nst->st_myidentity_type;
     ide->id_protoid = 0;
     ide->id_port = 0;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
    fprintf(stdout, "Client ID (IDci): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
    bcopy(nst->st_myidentity, &(sin.sin_addr), sizeof(struct in_addr));
    fprintf(stdout, "Sent Client ID IDci is %s\n", inet_ntoa(sin.sin_addr));
    }
#endif
     bcopy(nst->st_myidentity,
	   nst->st_packet + packet_ptr +
	   sizeof(struct isakmp_generic) + sizeof(struct identity), 
	   nst->st_myidentity_len);
     packet_ptr += ntohs(isag->isag_length);

      /* add the IDcr (peer's client) payload to the packet */
     isag = (struct isakmp_generic *) (nst->st_packet + packet_ptr);
     isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    nst->st_peeridentity_len + sizeof(struct identity));
     isag->isag_np = ISAKMP_NEXT_NONE;

     ide = (struct identity *) (nst->st_packet + packet_ptr +
			     sizeof(struct isakmp_generic));
     ide->id_type = nst->st_peeridentity_type;
     ide->id_protoid = 0;
     ide->id_port = 0;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
    fprintf(stdout, "Client ID (IDcr): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
    bcopy(nst->st_peeridentity, &(sin.sin_addr), sizeof(struct in_addr));
    fprintf(stdout, "Sent Client ID IDcr is %s\n", inet_ntoa(sin.sin_addr));
    }
#endif
     bcopy(nst->st_peeridentity,
	   nst->st_packet + packet_ptr +
	   sizeof(struct isakmp_generic) + sizeof(struct identity), 
	   nst->st_peeridentity_len);
     packet_ptr += ntohs(isag->isag_length);
#endif TEST_SEND_CLIENT_ID

  /* we finally know the size of the packet */  
  isa->isa_length = packet_ptr;
  
  /* Compute the Phase 2 IV */
  if (nst->st_prf == 0)
    {
      switch (nst->st_hash)
	{
	case OAKLEY_MD5:
	  nst->st_iv_len = 16;
	  nst->st_iv = (u_char *) calloc(nst->st_iv_len,
					 sizeof(u_char));
	  if (nst->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in initiate_quick()", 0, 0, 0);
	  
	  MD5Init(&md5ctx);
	  MD5Update(&md5ctx, st->st_lastblock, st->st_lastblock_len);
	  MD5Update(&md5ctx, (u_char *) &(nst->st_msgid),
		    sizeof(nst->st_msgid));
	  MD5Final(nst->st_iv, &md5ctx);
	  
	  break;
		
	case OAKLEY_SHA:
	  nst->st_iv_len = 20;
	  nst->st_iv = (u_char *) calloc(nst->st_iv_len,
					 sizeof(u_char));
	  if (nst->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in initiate_quick()", 0, 0, 0);
	  
	  SHAInit(&shactx);
	  SHAUpdate(&shactx, st->st_lastblock, st->st_lastblock_len);
	  SHAUpdate(&shactx, (u_char *) &(nst->st_msgid),
		    sizeof(nst->st_msgid));
	  SHAFinal(&shactx);
	  bcopy((&shactx)->buffer, nst->st_iv, 20);
	  
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d in initiate_quick()", nst->st_hash, 0, 0);
	}
    }
  else
    {
      /* XXX 3DES-CBC MAC */
    }
  
  /* HASH(1) -- compute */  
  compute_hash(nst, nst->st_packet, isa->isa_length, 0);
  
  /* Padding */
  switch (nst->st_enc)
    {
    case OAKLEY_DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
    case OAKLEY_3DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;      
    default:
      exit_log("unknown/unsupported ISAKMP encryption algorithm %d",
	       nst->st_enc, 0, 0);
    }
  
  k = (isa->isa_length - sizeof(struct isakmp_hdr)) % blocksize;
  
#ifdef DEBUG
  if ((k) && (arg_verbose & DEBUG_VALUES))
    fprintf(stdout, "padding is %d bytes\n", k);
#endif

  isa->isa_length += k;		/* Add padding */
  
  /* Update lengths */
  nst->st_packet_len = isa->isa_length;
  isa->isa_length = htonl(isa->isa_length);
    
#ifdef DEBUG
  if (st->st_pfs_handle)
     packet_dump("\n\tINITIATOR->RESPONDER ***** QUICK MODE MESSAGE 1 : HDR* + HASH1 + SA + Ni + KE", DUMP_MSG, 0);
  else
     packet_dump("\n\tINITIATOR->RESPONDER ***** QUICK MODE MESSAGE 1 : HDR* + HASH1 + SA + Ni", DUMP_MSG, 0);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(nst->st_packet, DUMP_SENT_BEFORE_ENC, 0);
  show_payloads(nst->st_packet, 0);
  if (arg_verbose & DEBUG_VERBOSE)
  {
  	fprintf(stdout, "-----\nPhase 2 Initial IV: ");
  	format_dump(nst->st_iv, nst->st_iv_len);
  	fprintf(stdout, "-----\n");
  }
      if (arg_verbose & DEBUG_PACKETS)
      {
        fprintf(stdout, "Encrypting %d bytes; algorithm: %s(%d) \n-----IV: ",
	      nst->st_packet_len - sizeof(struct isakmp_hdr),
	      ((nst->st_enc == OAKLEY_DES_CBC) ? "DES_CBC" : "3DES_CBC"),
	      nst->st_enc);
	    format_dump(nst->st_iv, nst->st_iv_len);
      }
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "QUICK MODE MESSAGE 1 (ENCRYPTED) SENT BY INITIATOR\n");
  fclose(f_demo);
#endif DEMO

  /* Encrypt */
  switch (nst->st_enc)
    {
    case OAKLEY_DES_CBC:

      /* Weak key detection already done */
      deskey(des_cbc_keys, nst->st_skeyid_e, DES_KEY_ENCRYPT);
      des_cbc_encrypt(nst->st_packet + sizeof(struct isakmp_hdr),
		      nst->st_packet_len - sizeof(struct isakmp_hdr),
		      des_cbc_keys, 
		      nst->st_iv, 
		      DES_ENCRYPT);
      
      /* Keep last block as IV */
      if ((nst->st_iv != (u_char *) NULL) &&
	  (nst->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(nst->st_iv);
	  nst->st_iv = (u_char *) NULL;
	}
      
      if (nst->st_iv == (u_char *) NULL)
	{
	  nst->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					 sizeof(u_char));
	  if (nst->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in initiate_quick()",
		     0, 0, 0);
	}
      
      bcopy(nst->st_packet + nst->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    nst->st_iv, DES_CBC_BLOCK_SIZE);
      nst->st_iv_len = DES_CBC_BLOCK_SIZE;
      
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
	  fprintf(stdout, "Updated lastblock: ");
	  format_dump(nst->st_lastblock, nst->st_lastblock_len);
	}
#endif
      
      break;
      
    case OAKLEY_3DES_CBC:
      /* Weak key detection already done */
      /* Encrypt */
      /* Choice to recompute the key and keymaterial everytime we receive a message encrypted with 3DES 
	 or to store the key or the key material in the state 
	 The second solution being much faster but is it crypto-correct? */
      three_des_cbc_key = (char *) malloc(sizeof(key_schedule)*3);
      if (three_des_cbc_key == (unsigned char) NULL)
	exit_log("malloc() failed for three_des_cbc_key\n", 0,0,0);
      
      /* construct the key from st->st_skeyid_e, encrypt, erase the computed key */
      deskey(* (key_schedule *) (three_des_cbc_key), 
	     nst->st_skeyid_e, 
	     0);
      deskey(* (key_schedule *) (three_des_cbc_key+sizeof(key_schedule)),
	     nst->st_skeyid_e + 8, 
	     1);
      deskey(* (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
	     nst->st_skeyid_e+16, 
	     0);
      des3_cbc_encrypt(nst->st_packet + sizeof( struct isakmp_hdr),
		       nst->st_packet_len - sizeof( struct isakmp_hdr),
		       * (key_schedule*) (three_des_cbc_key),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
		       nst->st_iv,
		       DES_ENCRYPT);
      free(three_des_cbc_key);
          
      /* Keep last block as IV */
      if ((nst->st_iv != (u_char *) NULL) &&
	  (nst->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(nst->st_iv);
	  nst->st_iv = (u_char *) NULL;
	}
      
      if (nst->st_iv == (u_char *) NULL)
	{
	  nst->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					 sizeof(u_char));
	  if (nst->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in initiate_quick()",
		     0, 0, 0);
	}
      
      bcopy(nst->st_packet + nst->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    nst->st_iv, DES_CBC_BLOCK_SIZE);
      nst->st_iv_len = DES_CBC_BLOCK_SIZE;
      
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
      	fprintf(stdout, "Updated IV: ");
      	format_dump(nst->st_iv, nst->st_iv_len);
	  fprintf(stdout, "Updated lastblock: ");
	  format_dump(nst->st_lastblock, nst->st_lastblock_len);
	}
#endif
      
      break;
      
      /* XXX Support more */
    default:
      exit_log("unsupported encryption algorithm %d", nst->st_enc,
	       0, 0);
    }
  

#ifdef DEBUG
  if (arg_verbose & DEBUG_PACKETS)
    	packet_dump(nst->st_packet, DUMP_SENT_AFTER_ENC, nst->st_packet_len);
#endif
  if (sendto(sock, nst->st_packet, nst->st_packet_len, 0, &(nst->st_peer),
	     sizeof(nst->st_peer)) != nst->st_packet_len)
    exit_log("sendto() failed in initiate_quick()", 0, 0, 0);
#ifdef DEBUG
  else
    fprintf(stdout, "Transmitted %d bytes\n", nst->st_packet_len);
#endif
  
  event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, nst, 0);
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q initiate_quick  | in ipsecdoi_quickxchg.c");
#endif
}


/************************************************************************/
/* 
 * goal : responder side, handle first message of quick mode. :
 * receive : HDR* + HASH1 + SA0 + SA1 + Ni [+ KE] [+ IDui + IDur]
 * send : HDR* + HASH2 + SA0 + SA1 + Nr [+ KE] [+ IDur + Idui]
 * Called by comm_handle (demux.c)
 */
 /* Added kernelfd as arg to ipsecdoi_handle_quick_r1 - 
    needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     

void ipsecdoi_handle_quick_r1(int              sock, 
			      u_char          *buffer, 
			      int              length,
			      struct sockaddr  sa, 
			      struct state    *st,
			      int	       kernelfd)
{
  int                    i;
  int                    j;
  int                    sa_length;
  int                    k, k1, padding;
  int                    blocksize;

  u_char                p_tmp[8192];
  u_char                *p2;

  u_char                *p;
  u_char                *hashval;
  unsigned long          des_cbc_keys[16][2];

  char                  *three_des_cbc_key;

  struct isakmp_hdr     *isa;
  struct isakmp_generic *isag;
  struct isakmp_sa      *isasa;
  struct isakmp_sa      *isas;

  struct isakmp_sa      *save_sa;
  struct isakmp_proposal *save_proposal;

  struct sockaddr_in     sin;
  MP_INT                 temp;
  MP_INT                 temp2;
  int                    grlen;
  int                    grbegin;
  u_char                 tmp[LOCALSECRETSIZE];
  u_char                *gr;
  int                    point_pfs_ke;
  int                    ke_length; 
  int                    groffset;

  u_int32_t                spi;
  u_int32_t                peerspi;

/* Accept/Send Client ID */
  struct identity          *ide;
  int                    client_id_recvd;
  struct sockaddr_in     sin1;

#ifdef DEBUG
  int                    gxylen;
  int                    gxybegin;
  u_char                *gxy;
#endif

#ifdef DEMO
  FILE *f_demo;
#endif DEMO

#ifdef DEBUG_IN_OUT
  in_out_functions(" E  ipsecdoi_handle_quick_r1 | in ipsecdoi_quickxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
  if (arg_pfs_requested)
    fprintf(stdout,"Handle Quick Mode Message 1 (HDR* + HASH(1) + SA + Ni + KE) from Initiator\nSend back HDR* + HASH(2) + SA + Nr + KE\n");
  else
    fprintf(stdout,"Handle Quick Mode Message 1 (HDR* + HASH(1) + SA + Ni) from Initiator\nSend back HDR* + HASH(2) + SA + Nr\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "QUICK MODE MESSAGE 1 (ENCRYPTED) RECEIVED FROM INITIATOR\n");
  fclose(f_demo);
#endif DEMO

/* Accept/Send Client ID */
  client_id_recvd = 0;

  isa = (struct isakmp_hdr *) buffer;
  
  if (isa->isa_np != ISAKMP_NEXT_HASH)
    {
      log(0, "malformed packet from %s, port %d. HASH(2) expected after HDR in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
  
  if (isag->isag_np != ISAKMP_NEXT_SA)
    {
      log(0, "malformed packet from %s, port %d. SA expected after HASH(2) in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  if (length <= sizeof(struct isakmp_hdr) +
      2 * sizeof(struct isakmp_generic) + sizeof(struct isakmp_sa))
    {
      log(0, "too short packet from %s, port %d in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i = sizeof(struct isakmp_hdr);
  
  /* Check HASH(1) */
  if (ntohs(isag->isag_length) + sizeof(struct isakmp_hdr) >=
      length)
    {
      log(0, "too short packet from %s, port %d in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_generic);
  
  /* Copy the hash */
  k = ntohs(isag->isag_length) - sizeof(struct isakmp_generic);
  hashval = (u_char *) calloc(k, sizeof(u_char));
  if (hashval == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  
  bcopy(buffer + i, hashval, k);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "received HASH(1):\t");
      format_dump(buffer + i, k);
    }
#endif
  
  compute_hash(st, buffer, length, 0);
    
  if (bcmp(hashval, buffer + i, k))
    {
      free(buffer);
      free(hashval);
      log(0, "HASH(1) from %s, port %d does not verify in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      /* XXX Could send notification back */
      return;
    }

#ifdef DEBUG
  if (arg_verbose & DEBUG_VALUES)
  {
  	fprintf(stdout, "HASH(1)(%d):\t", k);
  	format_dump(buffer + i, k);
  }
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "HASH(1) verified\n");
#endif
    
  /* Point at the SA payload */
  i += k;
  
  /* Authentication succeeded */
  free(hashval);

  isasa = (struct isakmp_sa *) (buffer + i);
  if (i + ntohs(isasa->isasa_length) >= length)
    {
      log(0, "too short packet from %s, port %d in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }

  if (isasa->isasa_np != ISAKMP_NEXT_NONCE)
    {
      log(0, "malformed packet q3 from %s, port %d in handling of Quick#1", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
    
/* Arg#5 = 0 ==> tells ipsecdoi_ipsec_sa_copy that we are the responder */
  if (ipsecdoi_ipsec_sa_copy(buffer, length, sa, st, 0) == 0)
    {
      free(buffer);
      free_state(st);
#ifdef WIT
      exit_log("ipsecdoi_handle_quick_r1: no acceptable proposal from %s, port %d in handling of Quick#1",
	  PRINTADDRESS(sa), get_port(sa), 0);
#else
      log(0, "ipsecdoi_handle_quick_r1: no acceptable proposal from %s, port %d in handling of Quick#1",
	  PRINTADDRESS(sa), get_port(sa), 0);
#endif
      /* XXX Could send notification back */
      return;
    }

  /* Ni payload */
  i +=  ntohs(isasa->isasa_length);
  isag = (struct isakmp_generic *) (buffer + i );
  if (ntohs(isag->isag_length) + i  > length)
    {
      log(0, "malformed packet from %s, port %d. The Ni payload is too long in handling of Quick#1.", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }

  if ((isag->isag_np != ISAKMP_NEXT_KE ) &&
      (isag->isag_np != ISAKMP_NEXT_ID ) &&
      (isag->isag_np != ISAKMP_NEXT_NONE))

    {
/* Save payload type (from buffer) for error message */
      k1 = isag->isag_np;
      free(buffer);
#ifndef WIT
      log(0, "unsupported payload # %d received in Quick Mode from %s, port %d in handling of Quick#1",
	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      exit_log("unsupported payload # %d received in Quick Mode from %s, port %d in handling of Quick#1",
	  k1, PRINTADDRESS(sa), get_port(sa));
#endif WIT
      /* XXX Could send notification back */
      return;
    }

  /* Copy Ni into the state */
  st->st_ni = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (st->st_ni == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  bcopy(isag, st->st_ni, ntohs(isag->isag_length));
  
  /* handle received optional KE payload for PFS */
  if (isag->isag_np == ISAKMP_NEXT_KE)
    {
#ifdef WIT
/* WIT - Error exit if receiver didn't expect optional KE payload */
	if (!arg_pfs_requested)
		exit_log("Optional KE payload received but not expected", 0, 0, 0);
#endif
/* Make sure that the proposal included the group attribute for PFS exchange */
  	if (!st->st_att_group)
    	{
      		free(buffer);
      		free_state(st);
#ifdef WIT
      		exit_log("ipsecdoi_handle_quick_r1: no acceptable proposal from %s, port %d in handling of Quick#1 - group attribute missing",
	  	PRINTADDRESS(sa), get_port(sa), 0);
#else
      		log(0, "ipsecdoi_handle_quick_r1: no acceptable proposal from %s, port %d in handling of Quick#1 - group attribute missing",
	  	PRINTADDRESS(sa), get_port(sa), 0);
#endif
      /* XXX Could send notification back */
      		return;
    	}
      /* do some basic checks on the received payload 
	 and set st->st_pfs_handle to "true" */

      /* point on the KE payload */
      i += ntohs(isag->isag_length);
      
      isag = (struct isakmp_generic *) (buffer + i);
      if (ntohs(isag->isag_length) + i > length)
	{
	  log(0, "malformed packet from %s, port %d. The KE payload is too long in handling Quick#", PRINTADDRESS(sa),
	      get_port(sa), 0);
	  free(buffer);
	  /* XXX Could send notification back */
	  return;
	}
      
      /* TODO : handle IDci, IDcr optional payloads */

      if ((isag->isag_np != ISAKMP_NEXT_ID) &&
         (isag->isag_np != ISAKMP_NEXT_NONE))
	{
/* Save payload type (from buffer) for error message */
      	  k1 = isag->isag_np;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported payload # %d after the KE payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported payload # %d after the KE payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#endif

	  /* XXX Could send notification back */
	  return;
	}
      
      if (ntohs(isag->isag_length)- sizeof(struct isakmp_generic)
	  < MINIMUM_PUBLIC_VALUE_SIZE)
	{ 
	  free (buffer);
	  log(0, "malformed packet from %s, port %d. Insufficient Length of the keying material in handling of Quick#1.", PRINTADDRESS(sa),
	      get_port(sa), 0);
	  /* XXX could send notification back */
	  return;
	}

      /* set st->st_pfs_handle so that we put the optional KE payload
	 in the construction of the response */
      st->st_pfs_handle = 1; 
      /* save the location of the "peer_pfs_KE" */
      point_pfs_ke = i;
      /* save the length of the KE payload */
      ke_length = ntohs(isag->isag_length);

    } /* end "if (isag->isag_np == ISAKMP_NEXT_KE)" */
#ifdef WIT
/* WIT - Error exit if receiver didn't expect optional KE payload */
  else if (arg_pfs_requested)
		exit_log("Optional KE payload expected but not received ", 0, 0, 0);
#endif
  else if (st->st_att_group)
		log(0, "Group attribute proposed for IPsec SA without PFS", 0, 0, 0);

/* Accept Client ID */
/* Responder - Handle Client ID Payloads from Initiator */
  if (isag->isag_np == ISAKMP_NEXT_ID)
  {
      client_id_recvd = 1;
      i += ntohs(isag->isag_length);
      isag = (struct isakmp_generic *) (buffer + i);
      if ((isag->isag_np != ISAKMP_NEXT_ID))
	{
/* Save payload type (from buffer) for error message */
      	  k1 = isag->isag_np;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported payload # %d after the Client ID (IDci) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported payload # %d after the Cllient ID (IDci) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#endif

	  /* XXX Could send notification back */
	  return;
	}

      ide = (struct identity *) (buffer + i + sizeof(struct isakmp_generic));
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Client ID (IDci): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
#endif
      if (ide->id_type == ID_IPV4_ADDR)
	{
	    st->st_peerclientid_type = ID_IPV4_ADDR;
	    st->st_peerclientid_len = sizeof(struct in_addr);
  	    st->st_peerclientid = (u_char *) calloc(st->st_peerclientid,
			  sizeof(u_char));
  	    if (st->st_peerclientid == (u_char *) NULL)
    		        exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
	    bcopy(buffer + i + sizeof(struct isakmp_generic) + sizeof(struct identity), 
		st->st_peerclientid, st->st_peerclientid_len);
#ifdef DEBUG
  	    if (arg_verbose & DEBUG_VERBOSE)
		{
  		bcopy(st->st_peerclientid, &(sin.sin_addr), sizeof(struct in_addr));
	      	fprintf(stdout, "Received Client ID IDci is %s\n", inet_ntoa(sin.sin_addr));
		}
#endif
	}
      else
	{
/* Save id type (from buffer) for error message */
      	  k1 = ide->id_type;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported Client ID (IDci) type # %d (expected ID_IPV4_ADDR) from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported Client ID (IDci) type # %d (expected ID_IPV4_ADDR) from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#endif
	  /* XXX Could send notification back */
	 return;
	}
      if (ide->id_protoid)
	  {
  	  st->st_peerclientid_protoid = ide->id_protoid;
      	  log(0, "unsupported Client ID (IDci) protocol # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ide->id_protoid, PRINTADDRESS(sa), get_port(sa));
	  }
      if (ide->id_port)
	  {
  	  st->st_peerclientid_port = ide->id_port;
      	  log(0, "unsupported Client ID (IDci) port # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ntohs(ide->id_port), PRINTADDRESS(sa), get_port(sa));
	  }

      i += ntohs(isag->isag_length);
      isag = (struct isakmp_generic *) (buffer + i);
      if ((isag->isag_np != ISAKMP_NEXT_NONE))
	{
/* Save payload type (from buffer) for error message */
      	  k1 = isag->isag_np;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported payload # %d after the Client ID (IDcr) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported payload # %d after the Cllient ID (IDcr) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#endif

	  /* XXX Could send notification back */
	  return;
	}

      ide = (struct identity *) (buffer + i + sizeof(struct isakmp_generic));
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Client ID (IDcr): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
#endif
      if (ide->id_type == ID_IPV4_ADDR)
	{
	    bcopy(buffer + i + sizeof(struct isakmp_generic) + sizeof(struct identity), 
		&(sin.sin_addr), sizeof(struct in_addr));
  	    if (bcmp(st->st_myidentity, &(sin.sin_addr), st->st_myidentity_len))
		{
      	  	log(0, "Invalid Client ID (IDcr) address from %s, port %d in handling of Quick#1",
	  	   PRINTADDRESS(sa), get_port(sa), 0);
      	  	log(0, "Invalid Client ID (IDcr) address is %s",
	  	   inet_ntoa(sin.sin_addr), 0, 0);
		}
#ifdef DEBUG
  	    if (arg_verbose & DEBUG_VERBOSE)
	      	fprintf(stdout, "Received Client ID IDcr is %s\n", inet_ntoa(sin.sin_addr));
#endif
	}
      else
      	  log(0, "unsupported Client ID (IDcr) type # %d (expected ID_IPV4_ADDR) from %s, port %d in handling of Quick#1",
	  	  ide->id_type, PRINTADDRESS(sa), get_port(sa));
      if (ide->id_protoid)
	  {
  	  st->st_myclientid_protoid = ide->id_protoid;
      	  log(0, "unsupported Client ID (IDcr) protocol # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ide->id_protoid, PRINTADDRESS(sa), get_port(sa));
	  }
      if (ide->id_port)
	  {
  	  st->st_myclientid_port = ide->id_port;
      	  log(0, "unsupported Client ID (IDcr) port # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ntohs(ide->id_port), PRINTADDRESS(sa), get_port(sa));
	  }
#ifdef WIT
/* Client ID sent by Initiator - check whether it's correct */
      if (bcmp(st->st_peerclientid, &(arg_dest_addr), st->st_peerclientid_len))
	{
      	     bcopy(st->st_peerclientid, &(sin.sin_addr), sizeof(struct in_addr));
      	     bcopy(&arg_dest_addr, &(sin1.sin_addr), sizeof(struct in_addr));
       	     fprintf(stdout, "Invalid Client ID (IDci) address %s - doesn't match WIT Host to Ping ",
	      	inet_ntoa(sin.sin_addr));
       	     fprintf(stdout, "%s\n", inet_ntoa(sin1.sin_addr));
	}
#endif WIT
  }
#ifdef WIT
/* No Client ID sent by Initiator - check whether should have been sent */
  else
      if (arg_gateway_addr != arg_dest_addr)
	     {
      	     bcopy(&arg_dest_addr, &(sin1.sin_addr), sizeof(struct in_addr));
       	     fprintf(stdout, "Client ID payload expected for WIT Host to Ping %s\n",
	  	       inet_ntoa(sin1.sin_addr) );
	     }
#endif WIT

#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
  	fprintf (stdout, "Correct first quick mode message from initiator, build the response \n");
#endif

  /* Padding */
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
    case OAKLEY_3DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
    default:
      exit_log("unknown/unsupported encryption algorithm %d specified",
	       st->st_enc, 0, 0);
    }
  
  padding = (blocksize - 
       (length - sizeof(struct isakmp_hdr)) % blocksize) % blocksize;

  st->st_packet = (u_char *) calloc(length+padding, sizeof(u_char));
  if (st->st_packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  
  /* Copy header */
  bcopy(buffer, st->st_packet, sizeof(struct isakmp_hdr));
  isa = (struct isakmp_hdr *) st->st_packet;
  isa->isa_length = 0;
  i = sizeof(struct isakmp_hdr);
    
  /* HASH(2) -- first pass */
  isag = (struct isakmp_generic *) (st->st_packet + i);
  isag->isag_np = ISAKMP_NEXT_SA;
  
  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  isag->isag_length = htons(16 + sizeof(struct isakmp_generic));
	  break;
	  
	case OAKLEY_SHA:
	  isag->isag_length = htons(20 + sizeof(struct isakmp_generic));
	  break;
	  
	default:
	  exit_log("unknown hash algorithm (%d) specified in ipsecdoi_handle_quick_r1()", st->st_hash, 0, 0);
	}
    }
  else
    {
      switch (st->st_prf)
	{
	case OAKLEY_PRF_3DES_CBC_MAC:
	  isag->isag_length = htons(8 + sizeof(struct isakmp_generic));
	  break;
	  
	default:
	  exit_log("unknown PRF (%d) specified in ipsecdoi_handle_quick_r1()",
		   st->st_prf, 0, 0);
	}
    }

  i += ntohs(isag->isag_length);
  
  /* SA handling */
  /*  isas = (struct isakmp_sa *) (st->st_packet + i);
  isas->isasa_np = ISAKMP_NEXT_NONCE;
  isas->isasa_doi = ntohl(ISAKMP_DOI_IPSEC);
  k = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
    ntohs(isag->isag_length) + IPSEC_DOI_SITUATION_LENGTH;
  st->st_packet[k - 1] = SIT_IDENTITY_ONLY;*/
  
  /* Set up a reverse path SA with similar characteristics */
  switch (st->st_peer_protoid)
    {
    case PROTO_IPSEC_AH:
      st->st_goal = GOAL_AUTHENTICATE;
      break;
      
    case PROTO_IPSEC_ESP:
      st->st_goal = GOAL_ENCRYPT;
      break;
      
	default:
	  exit_log("unknown protocol %d negotiated", st->st_peer_protoid,
		   0, 0);
    }
  
  /* Just do what the peer does */
  if (st->st_att_encapsulation_mode == ENCAPSULATION_MODE_TUNNEL)
    st->st_goal |= GOAL_TUNNEL;
  
  /* Initialize sin so that kernel_getspi has correct destination address to
     send to kernel  */
  bcopy(&(st->st_peer), &sin, sizeof(sin));
  
  /* Added kernelfd as arg to oakley_get_sa - 
     needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     
  /* Arg#6 = 0 ==> tells oakley_get_sa that we are the responder  */

  /* in order to accept any appropriate proposal sent by the peer, we do not have 
     to construct the sa payload in accordance to whatever we should have expected.
     On the contrary, we should be able to just send back the proposal we accepted. 
     To do so, do not go through oakley_get_sa() */

  bzero(p_tmp, 8192);
    
  save_sa = (struct isakmp_sa *) p_tmp;
  
  save_sa->isasa_np = ISAKMP_NEXT_NONCE;
  save_sa->isasa_doi = htonl(ISAKMP_DOI_IPSEC);

  k1 = sizeof(struct isakmp_sa);
  /* We don't do labeled integrity/secrecy, so forget the LDI */
  p_tmp[k1 + IPSEC_DOI_SITUATION_LENGTH - 1] = SIT_IDENTITY_ONLY;
  k1 += IPSEC_DOI_SITUATION_LENGTH;
  sa_length = k1;
  save_proposal = (struct isakmp_proposal *) st->st_proposal;  
  bcopy(save_proposal , p_tmp + k1 , ntohs(save_proposal->isap_length));
  save_proposal = (struct isakmp_proposal *) (p_tmp + k1);
  
  save_proposal->isap_spisize = DOI_SPI_SIZE_IPSEC;
  save_proposal->isap_notrans = 1;
  save_proposal->isap_np = 0;
  k1 += sizeof (struct isakmp_proposal);
  peerspi = ntohl(*(unsigned long *)st->st_peer_spi);
  spi = kernel_getspi(kernelfd, sin,  0, 
	peerspi, st->st_att_encapsulation_mode);
  bcopy(&spi, p_tmp + k1, sizeof(spi));

  st->st_spi = (u_char *) calloc(DOI_SPI_SIZE_IPSEC, sizeof(u_char));
  if (st->st_spi == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  bcopy(&spi, st->st_spi, sizeof(spi));
  st->st_spi_len = DOI_SPI_SIZE_IPSEC;
  k1 += DOI_SPI_SIZE_IPSEC;
  sa_length += ntohs(save_proposal->isap_length);

  p2 = (u_char *) calloc(sa_length, sizeof(u_char));
  if (p2 == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  bcopy(p_tmp, p2, sa_length);
  bcopy(p2, st->st_packet + i, sa_length);
  
  isas = (struct isakmp_sa *) (st->st_packet + i);
  isas->isasa_length = htons(sa_length);
  isas->isasa_np = ISAKMP_NEXT_NONCE;
  isas->isasa_doi = ntohl(ISAKMP_DOI_IPSEC);

#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
  {
  	format_dump(save_proposal, (ntohs(save_proposal->isap_length)));
  	fprintf(stdout, "\n sa (%d):\n", ntohs(isas->isasa_length));
  	format_dump(isas, ntohs(isas->isasa_length));
  }
#endif

  /* Also copy SA into state */
  st->st_sa = (u_char *) calloc(sa_length, sizeof(u_char));
  if (st->st_sa == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  bcopy(p2, st->st_sa, sa_length);
  free(p2);

  i += ntohs(isas->isasa_length);
  
  /* Nr */
  isag = (struct isakmp_generic *) (st->st_packet + i);
  isag->isag_length = htons(DEFAULTNONCESIZE + 
			    sizeof(struct isakmp_generic));
  if (st->st_pfs_handle)
    {
      isag->isag_np =  ISAKMP_NEXT_KE;
    }
  else
    {
      isag->isag_np = ISAKMP_NEXT_NONE;
    }
  
  get_rnd_bytes((u_char *)isag + sizeof(struct isakmp_generic),
		DEFAULTNONCESIZE);
  /* Copy Nr into state */
  st->st_nr = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (st->st_nr == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
  
  bcopy(isag, st->st_nr, ntohs(isag->isag_length));
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "Nr sent: ");
      format_dump(st->st_nr + sizeof(struct isakmp_generic), 
		  ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
    }
#endif
  
  /* point at the end of the Ni payload*/
  /* end of message if no KE payload */
  i += ntohs(isag->isag_length);
  
  /* handle an optional KE payload sent by the initiator */
  /* compute the new keying material, make the hash with the shared 
     secret from the ephemeral DH, send our public value */
     
  if (st->st_pfs_handle)
    {
#ifdef DEBUG
    if (arg_verbose & DEBUG_VERBOSE)
      fprintf(stdout, "Handle PFS (Optional KE payload) \n");
#endif
      mpz_init(&(st->st_gi));
      mpz_init(&(st->st_gr));
      mpz_init(&(st->st_sec));
      mpz_init(&(st->st_shared));
      mpz_init(&temp);
      mpz_init(&temp2);
      
      /* Get gi from buffer & convert to MP_INT (stored in temp)*/
      for (k =  point_pfs_ke + sizeof(struct isakmp_generic);
	   k < point_pfs_ke + ke_length ; k++)
	{
	  mpz_mul_ui(&temp2, &temp, 256);
	  mpz_add_ui(&temp, &temp2, buffer[k]);
	}
      /* Move temp to st_gi */
      mpz_set(&(st->st_gi), &temp);
      get_rnd_bytes(tmp, LOCALSECRETSIZE);
      
      mpz_set_ui(&temp, 0);
      
      for (k = 0; k < LOCALSECRETSIZE; k++)
	{
	  mpz_mul_ui(&temp2, &temp, 256);
	  mpz_add_ui(&temp, &temp2, tmp[k]);
	}
      mpz_set(&(st->st_sec), &temp);
      mpz_powm(&(st->st_gr), &groupgenerator, &(st->st_sec), &groupmodulo);
      mpz_powm(&(st->st_shared), &(st->st_gi), &(st->st_sec), &groupmodulo);
      grlen = mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1;
    	if (grlen < 96)
		grlen = 96;
      gr = (u_char *) calloc(grlen, sizeof(u_char));
      if (gr == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
      mpz_set(&temp, &(st->st_gr));
      for (k = grlen - 1 ; k >= 0; k--)
	{
	  gr[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	  mpz_set(&temp, &temp2);
	}
      /* Strip leading byte of zeroes, if necessary */
      grbegin = 0;
      if ((gr[0] == 0) && (grlen > 96)) {
	grbegin++;
	grlen--;
      }
      mpz_clear(&temp);
      mpz_clear(&temp2);

      /* add the KE payload to the packet */
      isag = (struct isakmp_generic *) (st->st_packet + i);
      isag->isag_np = ISAKMP_NEXT_NONE;
      isag->isag_length = htons(sizeof(struct isakmp_generic) + grlen);
      groffset = i + sizeof(struct isakmp_generic);
      for (k = 0 ; k < grlen ; k++)
	{
	  st->st_packet[k + groffset] = gr[k + grbegin]; 
	}

      i += ntohs(isag->isag_length);
#ifdef DEBUG
      gxylen = mpz_sizeinbase(&(st->st_shared), 16) / 2 + 1;
    	if (gxylen < 96)
		gxylen = 96;
      gxy = (u_char *) calloc(gxylen, sizeof(u_char));
      if (gxy == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
      
      mpz_init(&temp);
      mpz_init(&temp2);
      mpz_set(&temp, &(st->st_shared));
      
      for (k = gxylen - 1; k >= 0; k--)
	{
	  gxy[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	  mpz_set(&temp, &temp2);
	}
      
      /* Strip leading byte of zeroes, if necessary*/
      gxybegin = 0;
      if ((gxy[0] == 0) && (gxylen > 96)) {
	gxybegin++;
	gxylen--;
      }
      mpz_clear(&temp);
      mpz_clear(&temp2);
  if (arg_verbose & DEBUG_VALUES)
  {
  	fprintf(stdout, "***** Diffie-Hellman Exchange #2 (for PFS) Values:\n");
    fprintf(stdout, "Initiator's Secret #2 (for PFS) - g_x(%d) : \n", 
      	ke_length - sizeof(struct isakmp_generic));
    format_dump(buffer + point_pfs_ke + sizeof(struct isakmp_generic), 
    	ke_length - sizeof(struct isakmp_generic) );
  	fprintf(stdout, "Responder's Secret #2 (for PFS) - g_y(%d) : \n", grlen);
    format_dump(gr+grbegin, grlen);      
  	fprintf(stdout, "Shared Secret #2 (for PFS) - g_xy(%d):\n", gxylen);
    format_dump(gxy+gxybegin, gxylen);
  	fprintf(stdout, "-----\n");
  }
#endif   
      
    } /* end "if (st->st_pfs_handle)" */
/* Send Client ID */
/* Receiver only sends Client ID Payloads if Initiator sent them*/
  if (client_id_recvd)
     {
        isag->isag_np = ISAKMP_NEXT_ID;

      /* add the IDci (peer's client) payload to the packet */
     isag = (struct isakmp_generic *) (st->st_packet + i);
     isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    st->st_peerclientid_len + sizeof(struct identity));
     isag->isag_np = ISAKMP_NEXT_ID;
  
     ide = (struct identity *) (st->st_packet + i +
			     sizeof(struct isakmp_generic));
     ide->id_type = st->st_peerclientid_type;
     ide->id_protoid = st->st_peerclientid_protoid;
     ide->id_port = st->st_peerclientid_port;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
    fprintf(stdout, "Client ID (IDci): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
    bcopy(st->st_peerclientid, &(sin.sin_addr), sizeof(struct in_addr));
    fprintf(stdout, "Sent Client ID IDci is %s\n", inet_ntoa(sin.sin_addr));
    }
#endif
     bcopy(st->st_peerclientid,
	   st->st_packet + i +
	   sizeof(struct isakmp_generic) + sizeof(struct identity), 
	   st->st_peerclientid_len);
     i += ntohs(isag->isag_length);

      /* add the IDcr (my client) payload to the packet */
     isag = (struct isakmp_generic *) (st->st_packet + i);
     isag->isag_length = htons(sizeof(struct isakmp_generic) +
			    st->st_myidentity_len + sizeof(struct identity));
     isag->isag_np = ISAKMP_NEXT_NONE;

     ide = (struct identity *) (st->st_packet + i +
			     sizeof(struct isakmp_generic));
     ide->id_type = st->st_myidentity_type;
     ide->id_protoid = 0;
     ide->id_port = 0;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
    fprintf(stdout, "Client ID (IDcr): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
    bcopy(st->st_myidentity, &(sin.sin_addr), sizeof(struct in_addr));
    fprintf(stdout, "Sent Client ID IDcr is %s\n", inet_ntoa(sin.sin_addr));
    }
#endif
     bcopy(st->st_myidentity,
	   st->st_packet + i +
	   sizeof(struct isakmp_generic) + sizeof(struct identity), 
	   st->st_myidentity_len);
     i += ntohs(isag->isag_length);
     }
  st->st_packet_len = i;
  isa->isa_length = htonl(i);
  
#ifdef DEBUG
 if(arg_verbose & DEBUG_VALUES)
    fprintf(stdout, "packet length (before padding) is %d bytes\n", i);
#endif
  
  /* Compute HASH(2) */
  compute_hash(st, st->st_packet, st->st_packet_len, 1);
  
  /* Update state of exchange */
  st->st_state = OAKLEY_QUICK_R_2;
  
  /* Free received packet */
  free(buffer);

  st->st_packet_len += padding;
  isa->isa_length = htonl(st->st_packet_len);
  
#ifdef DEBUG
  if ((padding) && (arg_verbose & DEBUG_VALUES))
    fprintf(stdout, "added %d bytes of padding\n", padding);
  if (st->st_pfs_handle)
    packet_dump("\n\tRESPONDER->INITIATOR ***** QUICK MODE MESSAGE 2 : HDR* + HASH2 + SA + Nr + KE", DUMP_MSG, 0);
  else
    packet_dump("\n\tRESPONDER->INITIATOR ***** QUICK MODE MESSAGE 2 : HDR* + HASH2 + SA + Nr", DUMP_MSG, 0);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(st->st_packet, DUMP_SENT_BEFORE_ENC, st->st_packet_len);
  show_payloads(st->st_packet, 0); 
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "IV: ");
      format_dump(st->st_iv, st->st_iv_len);
      fprintf(stdout, "lastblock:\t");
      format_dump(st->st_lastblock, st->st_lastblock_len);
    }
      if (arg_verbose & DEBUG_PACKETS)
      {
        fprintf(stdout, "Encrypting %d bytes; algorithm: %s(%d) \n-----IV: ",
	      st->st_packet_len - sizeof(struct isakmp_hdr),
	      ((st->st_enc == OAKLEY_DES_CBC) ? "DES_CBC" : "3DES_CBC"),
	      st->st_enc);
	    format_dump(st->st_iv, st->st_iv_len);
      }
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "QUICK MODE MESSAGE 2 (ENCRYPTED) SENT BY RESPONDER\n");
  fclose(f_demo);
#endif DEMO

  /* Encrypt */
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      /* Weak key detection already done */
      deskey(des_cbc_keys, st->st_skeyid_e, DES_KEY_ENCRYPT);
      des_cbc_encrypt(st->st_packet + sizeof(struct isakmp_hdr),
		      st->st_packet_len - sizeof(struct isakmp_hdr),
		      des_cbc_keys, 
		      st->st_iv, 
		      DES_ENCRYPT);
  
      /* Keep last block as IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()",
		     0, 0, 0);
	}
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    st->st_iv, DES_CBC_BLOCK_SIZE);
      st->st_iv_len = DES_CBC_BLOCK_SIZE;      
      break;

    case OAKLEY_3DES_CBC:
      /* Weak key detection already done */
      /* Encrypt */
      /* Choice to recompute the key and keymaterial everytime we receive a message encrypted with 3DES 
	 or to store the key or the key material in the state 
	 The second solution being much faster but is it crypto-correct? */
      three_des_cbc_key = (char *) malloc(sizeof(key_schedule)*3);
      if (three_des_cbc_key == (unsigned char) NULL)
	exit_log("malloc failed for three_des_cbc_key\n", 0,0,0);
      deskey(* (key_schedule *) (three_des_cbc_key), 
	     st->st_skeyid_e, 
	     0);
      deskey(* (key_schedule *) (three_des_cbc_key + sizeof(key_schedule)),
	     st->st_skeyid_e + 8, 
	     1);
      deskey(* (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
	     st->st_skeyid_e + 16, 
	     0);
      des3_cbc_encrypt(st->st_packet + sizeof( struct isakmp_hdr),
		       st->st_packet_len - sizeof( struct isakmp_hdr),
		       * (key_schedule*) (three_des_cbc_key),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
		       st->st_iv,
		       DES_ENCRYPT);
      /* free the key */
      free(three_des_cbc_key);
      
      /* Keep last block as IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_quick_r1()",
		     0, 0, 0);
	}
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    st->st_iv, DES_CBC_BLOCK_SIZE);
      st->st_iv_len = DES_CBC_BLOCK_SIZE;
      break;

      /* XXX Support more */
    default:
      exit_log("unsupported encryption algorithm %d", st->st_enc,
	       0, 0);
    }
  
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	{
      fprintf(stdout, "Updated IV:\n");
      format_dump(st->st_iv, st->st_iv_len);
	  fprintf(stdout, "Updated lastblock: ");
	  format_dump(st->st_lastblock, st->st_lastblock_len);
	}
#endif

  /* Transmit */
#ifdef DEBUG
  if (arg_verbose & DEBUG_PACKETS)
    	packet_dump(st->st_packet, DUMP_SENT_AFTER_ENC, st->st_packet_len);
#endif
  if (sendto(sock, st->st_packet, st->st_packet_len, 0, &sa, sizeof(sa))
      != st->st_packet_len)
    log(0, "sendto() failed in ipsecdoi_handle_quick_r1()", 0, 0, 0);
#ifdef DEBUG
  else if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "transmitted %d bytes\n", st->st_packet_len);
#endif
  
  /* Schedule */
  event_schedule(EVENT_CLEANUP, EVENT_CLEANUP_DELAY, st, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_quick_r1  | in ipsecdoi_quickxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
}



/************************************************************************/
/*
 * goal : initiator side, handle message from Responder in Quick Mode.
 * received : HDR* + HASH2 + SA0 + SA1 + Nr [+ KE] [+ IDur + Idui]
 * send : HDR*, HASH3
 * Called by : comm_handle (demux.c)
 */
void  ipsecdoi_handle_quick_i1(int             sock, 
			       u_char         *buffer, 
			       int             length,
			       struct sockaddr sa, 
			       struct state   *st, 
			       int             kernelfd)
{
  int                     i;
  int                     k;
  int                     k1;
  int                     blocksize;
  int                     hashlen;
  unsigned long           des_cbc_keys[16][2];

  char                   *three_des_cbc_key;
  u_char                 *hashval;
  struct isakmp_hdr      *isa;
  struct isakmp_generic  *isag;
  struct isakmp_sa       *isasa;
  MP_INT                  temp;
  MP_INT                  temp2;
  struct sockaddr_in     sin;

  struct identity          *ide;
  struct sockaddr_in     sin1;

#ifdef DEBUG
  int                    gilen;
  int                    gibegin;
  u_char*                gi;
  int                    gxylen;
  int                    gxybegin;
  u_char*                  gxy;
#endif

#ifdef DEMO
  FILE *f_demo;
#endif DEMO

#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_quick_i1  | in ipsecdoi_quickxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
  if (st->st_pfs_handle)
    fprintf(stdout,"Handle Quick Mode Message 2 (HDR* + HASH(2) + SA + Nr + KE) from Responder\nSend back HDR* + HASH(3)\n");
  else
    fprintf(stdout,"Handle Quick Mode Message 2 (HDR* + HASH(2) + SA + Nr) from Responder\nSend back HDR* + HASH(3)\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "QUICK MODE MESSAGE 2 (ENCRYPTED) RECEIVED FROM RESPONDER\n");
  fclose(f_demo);
#endif DEMO
  isa = (struct isakmp_hdr *) buffer;
  
  if (length <= sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
      2 * sizeof(struct isakmp_generic))
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
  
  if ((isa->isa_np != ISAKMP_NEXT_HASH) ||
      (isag->isag_np != ISAKMP_NEXT_SA))
    {
      log(0, "malformed packet q5 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i = sizeof(struct isakmp_hdr);
  if (i + ntohs(isag->isag_length) >= length)
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  i += sizeof(struct isakmp_generic);
  
  /* Check HASH(2) */
  k = ntohs(isag->isag_length) - sizeof(struct isakmp_generic);
  hashval = (u_char *) calloc(k, sizeof(u_char));
  if (hashval == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);
  bcopy(buffer + i, hashval, k);
  
  compute_hash(st, buffer, length, 1);
  
  if (bcmp(buffer + i, hashval, 
	   ntohs(isag->isag_length) - sizeof(struct isakmp_generic)))
    {
      log(0, "HASH(2) authentication failed", 0, 0, 0);
#ifdef DEBUG
      fprintf(stdout, "HASH2 received(%d):\t",ntohs(isag->isag_length) - sizeof(struct isakmp_generic) );
      format_dump(hashval, ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
      fprintf(stdout, "HASH2 computed(%d):\t", ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
      format_dump(buffer + i, ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
#endif
      free(hashval);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  free(hashval);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VALUES)
  {
      fprintf(stdout, "-----\nHASH(2)(%d):\t",ntohs(isag->isag_length) - sizeof(struct isakmp_generic) );
      format_dump(hashval, ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
  	  fprintf(stdout, "-----\n");
  }
  if (arg_verbose & DEBUG_VERBOSE)
  	fprintf(stdout, "HASH(2) verified\n");
#endif
  
  i = sizeof(struct isakmp_hdr) + ntohs(isag->isag_length);
  
  /* SA */
  isasa = (struct isakmp_sa *) (buffer + i);  
  if (ntohs(isasa->isasa_length) + i >= length)
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
	/* XXX Could send notification back */
      return;
    }
  
  if (isasa->isasa_np != ISAKMP_NEXT_NONCE)
    {
      log(0, "malformed packet q6 from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  /* Copy the SA we got into the state, while verifying it */
/* Arg#5 = 1 ==> tells ipsecdoi_ipsec_sa_copy that we are the initiator */
  if (ipsecdoi_ipsec_sa_copy(buffer, length, sa, st, 1) == 0)
    {
      free(buffer);
      free_state(st);
#ifdef WIT
      exit_log("ipsecdoi_handle_quick_i1: no acceptable proposal from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
#else
      log(0, "ipsecdoi_handle_quick_i1: no acceptable proposal from %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
#endif
	/* XXX Could send notification back */
      return;
    }

  i += ntohs(isasa->isasa_length);
  
  /* Nr */
  isag = (struct isakmp_generic *) (buffer + i);

  if (ntohs(isag->isag_length) + i > length)
    {
      log(0, "malformed packet q7 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }

  /* Copy Nr into the state */
  st->st_nr = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
  if (st->st_nr == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);
  bcopy(isag, st->st_nr, ntohs(isag->isag_length));
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "Nr received:\t");
      format_dump(st->st_nr +  sizeof(struct isakmp_generic),  
		  ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
    }
#endif
 
  
  /* handle PFS if PFS was requested */
  if (st->st_pfs_handle)
    {
      if (isag->isag_np != ISAKMP_NEXT_KE)
	{
	  log(0, "malformed packet q6 from %s, port %d\n next payload is %d (expected KE)", PRINTADDRESS(sa),
	      get_port(sa), isag->isag_np);
	  free(buffer);
	  /* XXX Could send notification back */
	  return;
	}
     /* point on the KE payload */
      i += ntohs(isag->isag_length);
      isag = (struct isakmp_generic *) (buffer + i);

      mpz_init(&temp);
      mpz_init(&temp2);

      /* Get gr from buffer & convert to MP_INT (stored in temp) */
      for (k = i + sizeof(struct isakmp_generic);
	   k < i + ntohs(isag->isag_length) ; k++)
	{
	  mpz_mul_ui(&temp2, &temp, 256);
	  mpz_add_ui(&temp, &temp2, buffer[k]);
	}
      
      mpz_init(&(st->st_shared));
      mpz_init(&(st->st_gr));
      
      /* Move temp to st_gr */
      mpz_set(&(st->st_gr), &temp);
      
      /* Compute shared secret */
      mpz_powm(&(st->st_shared), 
	       &(st->st_gr), 
	       &(st->st_sec), 
	       &groupmodulo);
     
#ifdef DEBUG
      gilen =  mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1;
    	if (gilen < 96)
		gilen = 96;
      gi = (u_char *) calloc(gilen, sizeof(u_char));
      if (gi == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);      
      mpz_init(&temp);
      mpz_init(&temp2);
      mpz_set(&temp, &(st->st_gi));
      for (k = gilen - 1; k >= 0; k--)
	{
	  gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	  mpz_set(&temp, &temp2);
	}
      
      /* Strip leading byte of zeroes, if necessary*/
      gibegin = 0;
      if ((gi[0] == 0) && (gilen > 96)) {
	gibegin++;
	gilen--;
      }
    mpz_clear(&temp);
    mpz_clear(&temp2);

      gxylen = mpz_sizeinbase(&(st->st_shared), 16) / 2 + 1;
    	if (gxylen < 96)
		gxylen = 96;
      gxy = (u_char *) calloc(gxylen, sizeof(u_char));
      if (gxy == (u_char *) NULL)
	exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);
      
      mpz_init(&temp);
      mpz_init(&temp2);
      mpz_set(&temp, &(st->st_shared));
      
      for (k = gxylen - 1; k >= 0; k--)
	{
	  gxy[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	  mpz_set(&temp, &temp2);
	}
      
      /* Strip leading byte of zeroes, if necessary*/
      gxybegin = 0;
      if ((gxy[0] == 0) && (gxylen > 96)) {
	gxybegin++;
	gxylen--;
      }
      mpz_clear(&temp);
      mpz_clear(&temp2);
  if (arg_verbose & DEBUG_VALUES)
  {
  	fprintf(stdout, "***** Diffie-Hellman Exchange #2 (for PFS) Values:\n");
  	fprintf(stdout, "Initiator's Secret #2 (for PFS) - g_x(%d) : \n", gilen);
    format_dump(gi+gibegin, gilen);      
    fprintf(stdout, "Responders's Secret #2 (for PFS) - g_y(%d) : \n", 
    	ntohs(isag->isag_length) - sizeof(struct isakmp_generic));
    format_dump(buffer + i + sizeof(struct isakmp_generic), 
    	ntohs(isag->isag_length) - sizeof(struct isakmp_generic) );
  	fprintf(stdout, "Shared Secret #2 (for PFS) - g_xy(%d):\n", gxylen);
      format_dump(gxy+gxybegin, gxylen);
  	fprintf(stdout, "-----\n");
  }
#endif   


    } /* end of "handle_pfs" */

/* Accept Client ID */
/* Initiator - Handle Client ID Payloads from Responder */
  if (isag->isag_np == ISAKMP_NEXT_ID)
  {
      i += ntohs(isag->isag_length);
      isag = (struct isakmp_generic *) (buffer + i);
      if ((isag->isag_np != ISAKMP_NEXT_ID))
	{
/* Save payload type (from buffer) for error message */
      	  k1 = isag->isag_np;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported payload # %d after the Client ID (IDci) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported payload # %d after the Cllient ID (IDci) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#endif

	  /* XXX Could send notification back */
	  return;
	}

      ide = (struct identity *) (buffer + i + sizeof(struct isakmp_generic));
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Client ID (IDci): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
#endif
      if (ide->id_type == ID_IPV4_ADDR)
	{
	    bcopy(buffer + i + sizeof(struct isakmp_generic) + sizeof(struct identity), 
		&(sin.sin_addr), sizeof(sin));
  	    if (bcmp(st->st_myidentity, &(sin.sin_addr), st->st_myidentity_len))
		{
      	  	log(0, "Invalid Client ID (IDci) address from %s, port %d in handling of Quick#1",
	  	    PRINTADDRESS(sa), get_port(sa), 0);
      	  	log(0, "Invalid Client ID (IDci) address is %s",
	  	   inet_ntoa(sin.sin_addr), 0, 0);
		}
#ifdef DEBUG
  	    if (arg_verbose & DEBUG_VERBOSE)
	      	fprintf(stdout, "Received Client ID IDci is %s\n", inet_ntoa(sin.sin_addr));
#endif
	}
      else
      	  log(0, "unsupported Client ID (IDci) type # %d (expected ID_IPV4_ADDR) from %s, port %d in handling of Quick#1",
	  	  ide->id_type, PRINTADDRESS(sa), get_port(sa));
      if (ide->id_protoid)
	  {
  	  st->st_myclientid_protoid = ide->id_protoid;
      	  log(0, "unsupported Client ID (IDci) protocol # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ide->id_protoid, PRINTADDRESS(sa), get_port(sa));
	  }
      if (ide->id_port)
	  {
  	  st->st_myclientid_port = ide->id_port;
      	  log(0, "unsupported Client ID (IDci) port # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ntohs(ide->id_port), PRINTADDRESS(sa), get_port(sa));
	  }

      i += ntohs(isag->isag_length);
      isag = (struct isakmp_generic *) (buffer + i);
      if ((isag->isag_np != ISAKMP_NEXT_NONE))
	{
/* Save payload type (from buffer) for error message */
      	  k1 = isag->isag_np;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported payload # %d after the Client ID (IDcr) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported payload # %d after the Cllient ID (IDcr) payload received in Quick Mode from %s, port %d in handling of Quick#1",
	  	  k1, PRINTADDRESS(sa), get_port(sa));
#endif

	  /* XXX Could send notification back */
	  return;
	}

      ide = (struct identity *) (buffer + i + sizeof(struct isakmp_generic));
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "Client ID (IDcr): type = %d, protocol = %d, port = %d\n", 
		ide->id_type, ide->id_protoid, ntohs(ide->id_port));
#endif
      if (ide->id_type == ID_IPV4_ADDR)
	{
	    st->st_peerclientid_type = ID_IPV4_ADDR;
	    st->st_peerclientid_len = sizeof(struct in_addr);
  	    st->st_peerclientid = (u_char *) calloc(st->st_peerclientid,
			  sizeof(u_char));
  	    if (st->st_peerclientid == (u_char *) NULL)
    		        exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);
	    bcopy(buffer + i + sizeof(struct isakmp_generic) + sizeof(struct identity), 
		st->st_peerclientid, st->st_peerclientid_len);
#ifdef DEBUG
  	    if (arg_verbose & DEBUG_VERBOSE)
		{
  		bcopy(st->st_peerclientid, &(sin.sin_addr), sizeof(struct in_addr));
	      	fprintf(stdout, "Received Client ID IDcr is %s\n", inet_ntoa(sin.sin_addr));
		}
#endif
	}
      else
	{
/* Save id type (from buffer) for error message */
      	  k1 = ide->id_type;
      	  free(buffer);
#ifndef WIT
      	  log(0, "unsupported Client ID (IDcr) type # %d (expected ID_IPV4_ADDR) from %s, port %d in handling of Quick#1",
	  	  ide->id_type, PRINTADDRESS(sa), get_port(sa));
#else
      	  exit_log("unsupported Client ID (IDcr) type # %d (expected ID_IPV4_ADDR) from %s, port %d in handling of Quick#1",
	  	  ide->id_type, PRINTADDRESS(sa), get_port(sa));
#endif
	  /* XXX Could send notification back */
	  return;
	}

      if (ide->id_protoid)
	  {
  	  st->st_peerclientid_protoid = ide->id_protoid;
      	  log(0, "unsupported Client ID (IDcr) protocol # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ide->id_protoid, PRINTADDRESS(sa), get_port(sa));
	  }
      if (ide->id_port)
	  {
  	  st->st_peerclientid_port = ide->id_port;
      	  log(0, "unsupported Client ID (IDcr) port # %d (expected 0) from %s, port %d in handling of Quick#1",
	  	  ntohs(ide->id_port), PRINTADDRESS(sa), get_port(sa));
	  }
#ifdef WIT
/* Client ID sent by Responder - check whether it's correct */
      if (bcmp(st->st_peerclientid, &(arg_dest_addr), st->st_peerclientid_len))
	{
      	     bcopy(st->st_peerclientid, &(sin.sin_addr), sizeof(struct in_addr));
      	     bcopy(&arg_dest_addr, &(sin1.sin_addr), sizeof(struct in_addr));
       	     fprintf(stdout, "Invalid Client ID (IDcr) address %s - doesn't match WIT Host to Ping ",
	      	 inet_ntoa(sin.sin_addr));
       	     fprintf(stdout, "%s\n", inet_ntoa(sin1.sin_addr));
	}
#endif WIT
  }
#ifdef WIT
/* No Client ID sent by Responder - check whether should have been sent */
  else
      if (arg_gateway_addr != arg_dest_addr)
	     {
      	     bcopy(&arg_dest_addr, &(sin1.sin_addr), sizeof(struct in_addr));
       	     fprintf(stdout, "Client ID payload expected for WIT Host to Ping %s\n",
	  	       inet_ntoa(sin1.sin_addr) );
	     }
#endif WIT

  /* XXX Handle additional payloads...right now we error */
  if (isag->isag_np != ISAKMP_NEXT_NONE)
    {
/* Save payload type (from buffer) for error message */
      k1 = isag->isag_np;
      free(buffer);
#ifndef WIT
      log(0, "unsupported payload # %d received in Quick Mode from %s, port %d in handling of Quick#2",
	  k1, PRINTADDRESS(sa), get_port(sa));
#else
      exit_log("unsupported payload # %d received in Quick Mode from %s, port %d in handling of Quick#2",
	  k1, PRINTADDRESS(sa), get_port(sa));
#endif WIT

      /* XXX Could send notification back */
      return;
    }
  
 
  /* Copy accepted proposal into the state */
  st->st_proposal = (u_char *) calloc(ntohs(isasa->isasa_length), 
				      sizeof(u_char));
  if (st->st_proposal == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);
  bcopy(isasa, st->st_proposal, ntohs(isasa->isasa_length));
  
  /* XXX Check that the proposal is one of those we sent originally */
  
  /* Get rid of old packet */
  free(st->st_packet);
  st->st_packet = (u_char *) NULL;
  
  /* Delete old pending event */
  delete_event(st);
  
  /* New packet */
  if (st->st_prf == 0)
    {
      switch (st->st_hash)
	{
	case OAKLEY_MD5:
	  hashlen = 16;
	  break;
	  
	case OAKLEY_SHA:
	  hashlen = 20;
	  break;
	  
	default:
	  exit_log("unknown/unsupported hash algorithm %d specified",
		   st->st_hash, 0, 0);
	}
    }
  else
    {
      switch (st->st_prf)
	{
	case OAKLEY_PRF_3DES_CBC_MAC:
	  hashlen = DES_CBC_BLOCK_SIZE;
	  break;
	default:
	  exit_log("unknown/unsupported PRF algorithm %d specified",
		   st->st_prf, 0, 0);
	}
    }
  
  st->st_packet_len = sizeof(struct isakmp_hdr) + 
    sizeof(struct isakmp_generic) + hashlen;

  /* Padding */
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
    case OAKLEY_3DES_CBC:
      blocksize = DES_CBC_BLOCK_SIZE;
      break;
      
    default:
      exit_log("unknown/unsupported encryption algorithm %d specified",
	       st->st_enc, 0, 0);
    }
  
  k = (blocksize - 
       (st->st_packet_len - sizeof(struct isakmp_hdr)) % blocksize) %
    blocksize;
  st->st_packet_len += k;

#ifdef DEBUG
  if ((k) && (arg_verbose & DEBUG_VALUES))
    fprintf(stdout, "added %d bytes of padding\n", k);
#endif

  st->st_packet = (u_char *) calloc(st->st_packet_len, sizeof(u_char));
  if (st->st_packet == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);
  
  bcopy(buffer, st->st_packet, sizeof(struct isakmp_hdr));
  
  isa = (struct isakmp_hdr *) st->st_packet;
  isa->isa_np = ISAKMP_NEXT_HASH;
  
  /* HASH(3) */
  isag = (struct isakmp_generic *) (st->st_packet + 
				    sizeof(struct isakmp_hdr));
  isag->isag_length = htons(hashlen + sizeof(struct isakmp_generic));
  
  /* Compute HASH(3) */
  compute_hash_3(st, st->st_packet);
  
  /* Free old packets */
  free(buffer);
  
  /* Advance state */
  st->st_state = OAKLEY_QUICK_I_2;
  
  isa->isa_length = htonl(st->st_packet_len);
  
#ifdef DEBUG
  packet_dump("\n\tINITIATOR->RESPONDER ***** QUICK MODE MESSAGE 3 : HDR* + HASH3", DUMP_MSG, 0);
  if (arg_verbose & DEBUG_PACKETS)
  	packet_dump(st->st_packet, DUMP_SENT_BEFORE_ENC, 0);
  show_payloads(st->st_packet, 0);
if(arg_verbose & DEBUG_VERBOSE)
  {
    fprintf(stdout, "lastblock:\t");
    format_dump(st->st_lastblock, st->st_lastblock_len);
 }
      if (arg_verbose & DEBUG_PACKETS)
      {
        fprintf(stdout, "Encrypting %d bytes; algorithm: %s(%d) \n-----IV: ",
	      st->st_packet_len - sizeof(struct isakmp_hdr),
	      ((st->st_enc == OAKLEY_DES_CBC) ? "DES_CBC" : "3DES_CBC"),
	      st->st_enc);
	    format_dump(st->st_iv, st->st_iv_len);
      }
#endif  
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "QUICK MODE MESSAGE 3 (ENCRYPTED) SENT BY INITIATOR\n");
  fclose(f_demo);
#endif DEMO

  /* Encrypt */
  switch (st->st_enc)
    {
    case OAKLEY_DES_CBC:
      /* Weak key detection already done */
      deskey(des_cbc_keys, st->st_skeyid_e, DES_KEY_ENCRYPT);
      des_cbc_encrypt(st->st_packet + sizeof(struct isakmp_hdr),
		      st->st_packet_len - sizeof(struct isakmp_hdr),
		      des_cbc_keys, 
		      st->st_iv, 
		      DES_ENCRYPT);

      /* Keep last block as IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_quick_i1()",
		     0, 0, 0);
	}
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    st->st_iv, DES_CBC_BLOCK_SIZE);
      st->st_iv_len = DES_CBC_BLOCK_SIZE;
      break;


    case OAKLEY_3DES_CBC:
      /* Weak key detection already done */
      /* Encrypt */
      /* Choice to recompute the key and keymaterial everytime we receive a message encrypted with 3DES 
	 or to store the key or the key material in the state 
	 The second solution being much faster but is it crypto-correct? */
      three_des_cbc_key = (char *) malloc(sizeof(key_schedule)*3);
      if (three_des_cbc_key == (unsigned char) NULL)
	exit_log("malloc failed for three_des_cbc_key\n", 0,0,0);
      
      /* construct the key from st->st_skeyid_e, encrypt, erase the computed key */
      deskey(* (key_schedule *) (three_des_cbc_key), 
	     st->st_skeyid_e, 
	     0);
      deskey(* (key_schedule *) (three_des_cbc_key+sizeof(key_schedule)),
	     st->st_skeyid_e + 8, 
	     1);
      deskey(* (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
	     st->st_skeyid_e+16, 
	     0);
      des3_cbc_encrypt(st->st_packet + sizeof( struct isakmp_hdr),
		       st->st_packet_len - sizeof( struct isakmp_hdr),
		       * (key_schedule*) (three_des_cbc_key),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)),
		       * (key_schedule*) (three_des_cbc_key+sizeof(key_schedule)+sizeof(key_schedule)),
		       st->st_iv,
		       DES_ENCRYPT);
      free(three_des_cbc_key);
         
      /* Keep last block as IV */
      if ((st->st_iv != (u_char *) NULL) &&
	  (st->st_iv_len < DES_CBC_BLOCK_SIZE))
	{
	  free(st->st_iv);
	  st->st_iv = (u_char *) NULL;
	}
      
      if (st->st_iv == (u_char *) NULL)
	{
	  st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					sizeof(u_char));
	  if (st->st_iv == (u_char *) NULL)
	    exit_log("calloc() failed in ipsecdoi_handle_quick_i1()",
		     0, 0, 0);
	}
      
      bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
	    st->st_iv, DES_CBC_BLOCK_SIZE);
      st->st_iv_len = DES_CBC_BLOCK_SIZE;
      break;
      
      /* XXX Support more */
    default:
      exit_log("unsupported ISAKMP encryption algorithm %d", st->st_enc,
	       0, 0);
    }
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	{
         fprintf(stdout, "Updated IV:\t");
         format_dump(st->st_iv, st->st_iv_len);
	  fprintf(stdout, "Updated lastblock:\t");
	  format_dump(st->st_lastblock, st->st_lastblock_len);
	}
      if (arg_verbose & DEBUG_PACKETS)
	  packet_dump(st->st_packet, DUMP_SENT_AFTER_ENC, st->st_packet_len);
#endif  
  /* Transmit */
  if (sendto(sock, st->st_packet, st->st_packet_len, 0, &sa, sizeof(sa))
      != st->st_packet_len)
    log(0, "sendto() failed in ipsecdoi_handle_quick_i1()", 0, 0, 0);

  /* Derive new keying material and call kernel to update partial SA */
  /* Arg#4 = 1 ==> tells compute_keymat that we are the initiator */
  /* CHANGE */
  compute_keymat(st, 0, kernelfd, 1);

#ifdef DEBUG
	success_printout(st);
#endif

/* WIT - Message to let WIT know that PlutoPlus has successfully completed */
#ifdef WIT
fprintf(stdout, "SUCCESSFUL NEGOTIATION\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOI_FILE, "a");
  fprintf(f_demo, "************** PHASE 2 COMPLETED ************* PHASE 2 COMPLETED **************\n");
  fprintf(f_demo, "SUCCESSFUL NEGOTIATION\n");
  fclose(f_demo);
#endif DEMO

  /* free the base of the keying material if pfs was requested */
  if (st->st_pfs_handle)
    {
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    fprintf (stdout, "\nPFS was requested, free the base of the keying material.\n");
#endif
  /*      mpz_clear(&(st->st_gi));
	  mpz_clear(&(st->st_gr));*/
      mpz_clear(&(st->st_sec));
      mpz_clear(&(st->st_shared));
    }
  /* 
   * XXX temporary missing piece
   * XXX Unless the commit bit is set, we need to send a message
   * XXX to the kernel to establish the new SA.
     */
#ifdef WIT
 /* WIT - Exit after a single successful negotiation */
 	exit(0);
#endif WIT
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_quick_i1  | in ipsecdoi_quickxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
#endif
}

/************************************************************************/
/*
 * Goal : Responder side, handle last message of Quick Mode.
 * received : HDR* + HASH3
 * Called by : comm_handle (demux.c)
 */
void ipsecdoi_handle_quick_r2(int              sock, 
			      u_char          *buffer, 
			      int              length,
			      struct sockaddr  sa, 
			      struct state    *st,
			      int              kernelfd)
{
  struct isakmp_hdr        *isa;
  struct isakmp_generic    *isag;
  int                       i;
  int                       k;
  int                       j;
  u_char                   *hashval;

#ifdef DEMO
  FILE *f_demo;
#endif DEMO


#ifdef DEBUG_IN_OUT
  in_out_functions(" E ipsecdoi_handle_quick_r2  | in ipsecdoi_quickxchg.c");
#endif
#ifdef DEBUG
  fprintf(stdout,"\n\n=======================================================================\n");
  fprintf(stdout,"Handle Quick Mode Message 3 (HDR* + HASH(3) ) from Initiator\nSend nothing back\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "QUICK MODE MESSAGE 3 (ENCRYPTED) RECEIVED FROM INITIATOR\n");
  fclose(f_demo);
#endif DEMO
  isa = (struct isakmp_hdr *) buffer;
  
  /* Standard checks */
  if (length <= sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic))
    {
      log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  if (isa->isa_np != ISAKMP_NEXT_HASH)
    {
      log(0, "malformed packet q8 from %s, port %d", PRINTADDRESS(sa),
	  get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
  
  if ((ntohs(isag->isag_length) + sizeof(struct isakmp_hdr) != length) ||
      (isag->isag_np != ISAKMP_NEXT_NONE))
    {
      log(0, "malformed packet q9 from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
      free(buffer);
      /* XXX Could send notification back */
      return;
    }
  
  /* Check HASH(3) */
  i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
  k = ntohs(isag->isag_length) - sizeof(struct isakmp_generic);
  hashval = (u_char *) calloc(k, sizeof(u_char));
  if (hashval == (u_char *) NULL)
    exit_log("calloc() failed in ipsecdoi_handle_quick_r2()", 0, 0, 0);
  bcopy(buffer + i, hashval, k);
    
  compute_hash_3(st, buffer);
  
  if (bcmp(buffer + i, hashval, k))
    {
      log(0, "HASH(3) authentication failed with %s, port %d\n",
	  PRINTADDRESS(sa), get_port(sa), 0);
#ifdef DEBUG
  fprintf(stdout, "HASH3 received(%d):\t", k);
  format_dump(hashval, k);
  fprintf(stdout, "HASH3 Sent    (%d):\t", k);
  format_dump(buffer + i, k);
#endif
  
      free(buffer);
      free(hashval);
      /* XXX Could send notification back */
      return;
    }
  
  free(hashval);
  
#ifdef DEBUG
     if (arg_verbose & DEBUG_VERBOSE)
  		fprintf(stdout, "HASH(3) authentication succeeded\n");
#endif
  
  /* Delete old event */
  delete_event(st);
  
  /* Advance state */
  st->st_state = OAKLEY_QUICK_R_3;
  
  /* Free received packet */
  free(buffer);

  /* Derive new keying material and call kernel to update partial SA */
  /* Arg#4 = 0 ==> tells compute_keymat that we are the responder */
  /* CHANGE */
  compute_keymat(st, 0, kernelfd, 0);

  /* free the base of the keying material if pfs was requested */
  if (st->st_pfs_handle)
    {
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	fprintf (stdout, "\nPFS was requested, free the base of the keying material.(just be sure. :-)\n");
#endif
      /*      mpz_clear(&(st->st_gi));
	      mpz_clear(&(st->st_gr));*/
      mpz_clear(&(st->st_sec));
      mpz_clear(&(st->st_shared));
    }
  
#ifdef DEBUG
	success_printout(st);
#endif

/* WIT - Message to let WIT know that PlutoPlus has successfully completed */
#ifdef WIT
fprintf(stdout, "SUCCESSFUL NEGOTIATION\n");
#endif
#ifdef DEMO
  f_demo = fopen(DEMOR_FILE, "a");
  fprintf(f_demo, "************** PHASE 2 COMPLETED ************* PHASE 2 COMPLETED **************\n");
  fprintf(f_demo, "SUCCESSFUL NEGOTIATION\n");
  fclose(f_demo);
#endif DEMO

  /* 
   * XXX temporary missing piece
   * XXX Unless the commit bit is set, we need to send a message
   * XXX to the kernel to establish the new SA.
   */
#ifdef WIT
 /* WIT - Exit after a single successful negotiation */
 	exit(0);
#endif WIT

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q ipsecdoi_handle_quick_r2  | in ipsecdoi_quickxchg.c");
#endif
}
