/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/*
 * kernel_comm.c - 
 * 	This file implements all the kernel communicating routines.
 * 	Eventually it should use PF_KEY or whatever kernel API is available
 */

#include <stdio.h>
#include <asm/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "constants.h"
#include <linux/netlink.h>
#include <sadb.h>
#include <fcntl.h>
#include <errno.h>
#include "defs.h"
#include "state.h"
#include "argdefs.h"
#include <signal.h>

static int kernelfd_sav;
void kernel_unregister();

/**********************************************************************/
/*
 * Handle a kernel request. Supposedly, there's a message in
 * the kernelsock socket.
 */
void kernel_handle(int kernelfd, int sock)
{
  struct sadb_msg        *sm;
  int                     sadblen, outbound;
  struct sockaddr_in      sin;
  int                     goal;
    char buffer[sizeof(struct sadb_msg)+1];
  struct state     *st;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernel_handle | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernel_handle | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not read from kernel socket */
  return;
#endif ZZZZR
  
  sadblen = read(kernelfd, (char *)buffer, (sizeof(struct sadb_msg)));
  if (sadblen == -1)
    exit_log("read() kernelfd (ike) failed in kernel_handle() errno=%d", errno, 0, 0);
  
#ifdef TEST_KERN
    fprintf(stdout, "read %d bytes from kernelfd (ike) socket in kernel_handle\n", sadblen);
#endif TEST_KERN
  sm = (struct sadb_msg *)buffer;
  
  switch (sm->m_sadbm.sadbm_type)
    {
    case SADBM_ACQUIRE:
#ifdef TEST_KERN
	  fprintf(stdout, "received SADBM_ACQUIRE (spi=%lx)\n", 
			sm->sadb_info.dst.sa_info.peer_spi);
#endif TEST_KERN
#ifdef ZZZZI
      /* ZZZZ Code to negotiate with self - initiate with peer port = our_port + 1 */
      sin.sin_port = htons(our_port + 1);
#else
      sin.sin_port = htons(our_port);
#endif ZZZZI 
      sin.sin_family = sm->sadb_info.dst.dst.addr_family; 
      sin.sin_addr.s_addr = sm->sadb_info.dst.dst.addr_union.ip_addr.s_addr; 
      
      /* Goal should be determined by policy engine 
	 For now can be: GOAL_AUTHENTICATE and/or GOAL_ENCRYPT and/or GOAL_TUNNEL */
      goal = arg_key_mgmt_goal;  
      
#ifdef TEST_KERN
	  fprintf(stdout, "initiating exchange with [%s], port %d, goal %d\n", 
	      inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), goal);
#endif
      
      /* Added kernelfd as arg to ipsecdoi_initiate - 
	 needed for oakley_get_[ah/esp]_prop's call to kernel_getspi  */     
      ipsecdoi_initiate(sock, sin, goal, KEY_OAKLEY, kernelfd,
			htonl(sm->sadb_info.dst.sa_info.peer_spi));
      break;
    case SADBM_DELETE:
#ifdef ZZZZI
      /* ZZZZ Code to negotiate with self - set peer port = our_port + 1 */
      sin.sin_port = htons(our_port + 1);
#else
      sin.sin_port = htons(our_port);
#endif ZZZZI 
      sin.sin_family = sm->sadb_info.dst.dst.addr_family; 
      sin.sin_addr.s_addr = sm->sadb_info.dst.dst.addr_union.ip_addr.s_addr; 

#ifdef TEST_KERN
	  fprintf(stdout, "received SADBM_DELETE (addr=%s spi=%lx peerspi=%lx flags=%x)\n", 
	        inet_ntoa(sin.sin_addr),
  		sm->sadb_info.dst.sa_info.spi,
  		sm->sadb_info.dst.sa_info.peer_spi,
    		sm->sadb_info.dst.sa_info.flags); 
#endif

#ifdef TEST_PR_STATE
	pr_state_table();
#endif TEST_PR_STATE

if (sm->sadb_info.dst.sa_info.flags & IPSEC_OUTBOUND_FLAG)
      {
	st = (struct state *) find_phase2_state_to_delete(sin, 
		htonl(sm->sadb_info.dst.sa_info.peer_spi), 1);
  	if (st != (struct state *) NULL)  
	         st->st_flags |= ST_FLAG_OUTBOUND_DELETED;
      }
else
      {
	st = (struct state *) find_phase2_state_to_delete(sin, 
		htonl(sm->sadb_info.dst.sa_info.spi), 0);
  	if (st != (struct state *) NULL)  
	         st->st_flags |= ST_FLAG_INBOUND_DELETED;
      }
  if (st == (struct state *) NULL)  
      log(0, "kernel_handle: received DELETE kernel message - SA NOT found addr=%s spi=%lx peerspi=%lx",
		inet_ntoa(sin.sin_addr),
  		sm->sadb_info.dst.sa_info.spi,
  		sm->sadb_info.dst.sa_info.peer_spi);
  else 
      {
#ifdef TEST_KERN
      fprintf(stdout, "kernel_handle: received DELETE kernel message - SA found addr=%s spi=%lx peerspi=%lx\n",
		inet_ntoa(sin.sin_addr),
  		sm->sadb_info.dst.sa_info.spi,
  		sm->sadb_info.dst.sa_info.peer_spi);
#endif

        if ((st->st_flags & ST_FLAG_OUTBOUND_DELETED) &&
              ((st->st_flags & ST_FLAG_INBOUND_DELETED)))
           {
		delete_state(st);
		free_state(st);
           }
       }
      break;
    case SADBM_GETSPI:
#ifdef TEST_KERN
	  fprintf(stdout, "received SADBM_GETSPI ****ERROR\n");
#endif
    default:
      log(0, "kernel_handle: received invalid kernel message %d", sm->m_sadbm.sadbm_type, 0, 0);
      break;
    }
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernel_handle | in kernel_comm.c");
#endif
}

/**********************************************************************/
int kernelsock_open(void)
{
  int               kernelfd;
  struct sadb_msg   m_sadbmsg;
  int               sadblen;
  int               l, k;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernelsock_open | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernelsock_open | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not open kernel socket */
  return 0;
#endif ZZZZR
/* Open /dev/ike for kernel communications - if busy, sleep and try again */
    for (k = 0; k < 5; k++) {
	if ((kernelfd = open("/dev/ike", O_RDWR)) > 0)
		break;
	if ((errno != EBUSY) || (k == 4))
		exit_log("bad device /dev/ike in kernelsock_open", 0, 0, 0);
	sleep(3);
    }
  
  /* Register ISAKMP daemon with IPSEC */
  bzero((char *)&m_sadbmsg, sizeof(m_sadbmsg));
  m_sadbmsg.m_sadbm.sadbm_msglen = l = sizeof(struct sadb_msg);
  m_sadbmsg.m_sadbm.sadbm_version = SADBM_VERSION;
  m_sadbmsg.m_sadbm.sadbm_type = SADBM_REGISTER;
  m_sadbmsg.m_sadbm.sadbm_flags = SADBM_IKE;
  m_sadbmsg.sadb_info.udp_port = our_port;
#ifdef TEST_KERN
	  fprintf(stdout, "sending SADBM_REGISTER \n");
#endif
  /* XXX temporary code to cause kernel to kick key mgmt for initiator (send SADBM_ACQUIRE)  */
  if (arg_initiator){
#ifdef TEST_KERN
	  fprintf(stdout, "sending SADBM_REGISTER with kick\n");
#endif
    m_sadbmsg.request_type = KICK_KEY_MGMT;
    if (arg_key_mgmt_goal & GOAL_TUNNEL)
    /* For tunnel-mode SA ==> negotiate with the gateway */
      m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = arg_gateway_addr; 
    else
      m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = arg_dest_addr; 
  }
  else
  {
#ifdef TEST_KERN
	  fprintf(stdout, "sending SADBM_REGISTER without kick\n");
#endif
    m_sadbmsg.request_type = NO_KICK_KEY_MGMT;
  }
/* Once PlutoPlus has sent SADBM_REGISTER to kernel, 
 *      must send SADBM_UNREGISTER before exiting on signal 
 */
      signal(SIGKILL, kernel_unregister);
      signal(SIGINT, kernel_unregister);
      signal(SIGTERM, kernel_unregister);
      signal(SIGQUIT, kernel_unregister);
  if ((sadblen = write(kernelfd, (char *)&m_sadbmsg, l)) < 0) 
    exit_log("write() kernelfd (ike) failed in kernelsock_open", 0, 0, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernelsock_open | in kernel_comm.c");
#endif
  kernelfd_sav = kernelfd;
  return kernelfd;
}

/**********************************************************************/
int kernelsock_read(int kernelfd)
{
  int       sadblen;
char buffer[sizeof(struct sadb_msg)+1];
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernelsock_read | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernelsock_read | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not read from kernel socket */
  return 0;
#endif ZZZZR
  sadblen = read(kernelfd, (char *)buffer, (sizeof(struct sadb_msg)));
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernelsock_read | in kernel_comm.c");
#endif
  return sadblen;
}

/**********************************************************************/
void kernelsock_close(int kernelfd)
{
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernelsock_close | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernelsock_close | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not close kernel socket */
  return;
#endif ZZZZR
  close(kernelfd);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernelsock_close | in kernel_comm.c");
#endif
}

/**********************************************************************/
u_int32_t kernel_getspi(int                 kernelfd, 
			struct sockaddr_in  sin, 
			int                 initiator,
			u_int32_t           peerspi,
  			u_int16_t           encapsulation_mode)
{
  struct sadb_msg    m_sadbmsg;
  struct sadb_msg   *sm;
  int                sadblen;
  int                l;
  u_int32_t          spi;
    char buffer[sizeof(struct sadb_msg)+1];
  int			kernelfd1;
  struct timeval	tv;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernel_getspi | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernel_getspi | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not get SPI from kernel */
  return htonl(257);
#endif ZZZZR
  bzero((char *)&m_sadbmsg, sizeof(m_sadbmsg));
  /* if receive key_acquire message, want to send key_getspi msg to kernel */
  m_sadbmsg.m_sadbm.sadbm_msglen = l =  sizeof(struct sadb_msg);
  m_sadbmsg.m_sadbm.sadbm_version = SADBM_VERSION;
  m_sadbmsg.m_sadbm.sadbm_flags = SADBM_IKE;
  m_sadbmsg.m_sadbm.sadbm_flags |= SADBM_IKE_GETSPI;
  m_sadbmsg.m_sadbm.sadbm_type = SADBM_GETSPI;
#ifdef TEST_KERN
	  fprintf(stdout, "sending SADBM_GETSPI\n");
#endif
  
  /* Tell kernel to add partial SA to database in addition to computing SPI */
  m_sadbmsg.request_type = SADB_ADD_PARTIAL_SA;
  
  m_sadbmsg.sadb_info.dst.prefix_len = 32; 
  m_sadbmsg.sadb_info.dst.dst.addr_family = sin.sin_family; 
#ifndef WIT
  m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = sin.sin_addr.s_addr;  
#else
/* Fix 08/30 - tunnel SA for WIT */
  m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = arg_dest_addr;  
#endif WIT
#ifdef TEST_KERN
      fprintf(stdout, "sending SADBM_GETSPI with ADD_PARTIAL_SA: destination = %s\n", 
	  inet_ntoa(m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr));
#endif
  m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_family = sin.sin_family; 
#ifndef WIT
    m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr.s_addr = sin.sin_addr.s_addr;  
#else
  /* Set peer_addr to gateway_addr to enable tunneling */
  if (encapsulation_mode == ENCAPSULATION_MODE_TUNNEL)
    m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr.s_addr = arg_gateway_addr; 
  else
/* Fix 08/30 - tunnel SA for WIT */
    m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr.s_addr = arg_dest_addr; 
#endif WIT
#ifdef TEST_KERN
      fprintf(stdout, "SADBM_GETSPI: peer address = %s\n", 
	  inet_ntoa(m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr));
#endif
  /* SA added through key negotiation  */ 
  m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_NEG_KM_FLAG;

    /* Create one-way inbound partial SA with own spi */
    m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_INBOUND_FLAG; 
#ifdef ZZZZI
      /* ZZZZ Code to negotiate with self - Make SA 2-way */
  m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;
#endif ZZZZI 

    /* Set peer spi to 0 - will be set to actual spi when SA is updated */
  m_sadbmsg.sadb_info.dst.sa_info.peer_spi = 0; 

/* Set lifetime of Partial SA - 
 *         necessary so kernel doesn't expire Partial SA before it's updated */
  gettimeofday(&tv, NULL);
  m_sadbmsg.sadb_info.dst.sa_info.lifetime.time_expired = 
		PARTIAL_SA_EXPIRE + tv.tv_sec;
/* Tell kernel not to expire Partial SA on kbytes */
  m_sadbmsg.sadb_info.dst.sa_info.lifetime.bytes_remaining = -1;

/* Use /dev/ike2 to give SADBM_GETSPI its own channel */
  if ((kernelfd1 = open("/dev/ike2", O_RDWR)) < 0)
	exit_log("bad device /dev/ike2 (kernelfd1) in kernel_getspi", 0, 0, 0);

  if ((sadblen = write(kernelfd1, (char *)&m_sadbmsg, l)) < 0) 
    exit_log("write() kernelfd1 (ike2) failed in kernel_getspi", 0, 0, 0);
  
#ifdef TEST_KERN
    fprintf(stdout, "wrote %d bytes to kernelfd1 (ike2) socket\n", l);
#endif
  
  sadblen = read(kernelfd1, (char *)buffer, (sizeof(struct sadb_msg)));
  if (sadblen == -1)
    exit_log("read() kernelfd1 (ike2) failed in kernel_getspi()", 0, 0, 0);
  
  sm = (struct sadb_msg *)buffer;
  spi = sm->sadb_info.dst.sa_info.spi; 
#ifdef TEST_KERN
    fprintf(stdout, "read %d bytes (spi=%x msgtype=%d) from kernelfd1 (ike2) socket in kernel_getspi\n", sadblen, spi, sm->m_sadbm.sadbm_type);
#endif
  close(kernelfd1);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernel_getspi | in kernel_comm.c");
#endif
  return htonl(spi);
}

/**********************************************************************/
/* For successful negotiation, send SADBM_UPDATE (inbound) and
 *	SADBM_ADD (outbound) to kernel 
 */
void kernel_update(int            kernelfd, 
		   struct state  *st, 
		   int            initiator)
{
  struct sadb_msg      m_sadbmsg;
  unsigned long        spi;
  struct sockaddr_in   sin;
  int                  k;
  int                  l;
  int                  sadblen;
  u_int8_t             ipsec_enc_alg;
  u_int8_t             ipsec_auth_alg;
  struct timeval	tv;
#ifdef TEST_KERN
  time_t tttv, tttv1;
#endif TEST_KERN
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernel_update | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernel_update | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not call kernel to update SA */
  return;
#endif ZZZZR
  bzero((char *)&m_sadbmsg, sizeof(m_sadbmsg));
  m_sadbmsg.m_sadbm.sadbm_msglen = l = sizeof(struct sadb_msg);
  m_sadbmsg.m_sadbm.sadbm_version = SADBM_VERSION;
  m_sadbmsg.m_sadbm.sadbm_flags = SADBM_IKE;
  m_sadbmsg.m_sadbm.sadbm_type = SADBM_UPDATE;
  
  m_sadbmsg.request_type = SADB_DST;
  
  /* Destination */
  bcopy(&st->st_peer,&sin,sizeof(sin));
  m_sadbmsg.sadb_info.dst.dst.addr_family = sin.sin_family; 
#ifndef WIT
/* If valid peerclientid was found - use it as the destination */
    if (st->st_peerclientid)
	bcopy(st->st_peerclientid, &(sin.sin_addr), sizeof(struct in_addr));

    	m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = sin.sin_addr.s_addr;
#else
/* Fix 08/30 - tunnel SA for WIT */
    m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = arg_dest_addr; 
#endif WIT
#ifdef TEST_KERN
    fprintf(stdout, "SADBM_UPDATE: destination = %s\n", 
	inet_ntoa(m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr));
#endif
  
  /* prefix length */
  m_sadbmsg.sadb_info.dst.prefix_len = 32;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "SADBM_UPDATE: prefix_len = %d\n", 
	m_sadbmsg.sadb_info.dst.prefix_len);
#endif
  
  /* SPI */
  bcopy(st->st_spi,&spi, sizeof(spi));
  m_sadbmsg.sadb_info.dst.sa_info.spi = ntohl(spi); 
#ifdef TEST_KERN
    fprintf(stdout,"sending SADBM_UPDATE: spi (USED-I) = %lx\n",m_sadbmsg.sadb_info.dst.sa_info.spi); 
#endif
  
  /* UID - not used */
  
  /* Peer address */
  /* Peer address set to destination address  */
  bcopy(&st->st_peer, &sin,sizeof(sin));
  m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_family = sin.sin_family; 
#ifndef WIT
    m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr.s_addr = sin.sin_addr.s_addr;  
#else
  /* Set peer_addr to gateway_addr to enable tunneling */
  if (st->st_att_encapsulation_mode == ENCAPSULATION_MODE_TUNNEL)
    m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr.s_addr = arg_gateway_addr; 
  else
/* Fix 08/30 - tunnel SA for WIT */
    m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr.s_addr = arg_dest_addr; 
#endif WIT
#ifdef TEST_KERN
    fprintf(stdout, "SADBM_UPDATE: peer address = %s\n", 
	inet_ntoa(m_sadbmsg.sadb_info.dst.sa_info.peer_addr.addr_union.ip_addr));
#endif
  
  /* Peer_SPI - for 1-way inbound SA, set Peer SPI to SPI */
  m_sadbmsg.sadb_info.dst.sa_info.peer_spi = m_sadbmsg.sadb_info.dst.sa_info.spi; 
#ifdef TEST_KERN
    fprintf(stdout,"SADBM_UPDATE: peer_spi = %lx\n",m_sadbmsg.sadb_info.dst.sa_info.peer_spi); 
#endif
  
  /* Transport_Mechanism */
  if (st->st_att_encapsulation_mode == ENCAPSULATION_MODE_TUNNEL)
    m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_TUNNEL_FLAG;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      if (st->st_att_encapsulation_mode == ENCAPSULATION_MODE_TUNNEL)
	fprintf(stdout,"SADBM_UPDATE: tunnel flag turned on\n");
      else
	fprintf(stdout,"SADBM_UPDATE: tunnel flag turned off\n"); 
      
      fprintf(stdout,"SADBM_UPDATE: encapsulation mode = %x flags= %x \n", 
	  st->st_att_encapsulation_mode, m_sadbmsg.sadb_info.dst.sa_info.flags); 
    }
#endif
  
  /* Replay flag */
  if (arg_replay_protection)
    m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_CHECK_REPLAY_FLAG;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    {
      if (arg_replay_protection)
	fprintf(stdout,"SADBM_UPDATE: replay protect flag turned on\n");
      else
	fprintf(stdout,"SADBM_UPDATE: replay protect flag turned off\n"); 
      
      fprintf(stdout,"SADBM_UPDATE: replay_protection mode = %x flags= %x \n", 
	  arg_replay_protection, m_sadbmsg.sadb_info.dst.sa_info.flags);     
      fprintf(stdout,"SADBM_UPDATE: initiator = %x \n", initiator); 
    }
#endif
  /* SA added through key negotiation  */ 
  m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_NEG_KM_FLAG;
    /* Make SA 1-way inbound */
    m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_INBOUND_FLAG;
#ifdef ZZZZI
      /* ZZZZ Code to negotiate with self - Make SA 2-way */
  m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;
#endif ZZZZI 
  
#ifdef TEST_KERN
    fprintf(stdout,"SADBM_UPDATE: flags = %x\n", 
	m_sadbmsg.sadb_info.dst.sa_info.flags);
#endif

/* Lifetime - send kernel -1 if lifetime not measured in kbytes */
  m_sadbmsg.sadb_info.dst.sa_info.lifetime.bytes_remaining = 
		(st->st_att_sa_duration_kbytes)?
		st->st_att_sa_duration_kbytes * 1000 : -1 ;
#ifdef TEST_KERN
    fprintf(stdout,"SADBM_UPDATE: lifetime (bytes) = %d\n", 
	m_sadbmsg.sadb_info.dst.sa_info.lifetime.bytes_remaining);
#endif TEST_KERN

/* Lifetime - send kernel -1 if lifetime not measured in seconds */
  gettimeofday(&tv, NULL);
  m_sadbmsg.sadb_info.dst.sa_info.lifetime.time_expired = 
		(st->st_att_sa_duration_seconds)?
		st->st_att_sa_duration_seconds + tv.tv_sec : -1 ;
#ifdef TEST_KERN
    fprintf(stdout,"SADBM_UPDATE: lifetime (seconds) = %d\n", 
		(st->st_att_sa_duration_seconds)?
		st->st_att_sa_duration_seconds : -1);
     tttv = time((time_t *) NULL);
     tttv1 = tttv + st->st_att_sa_duration_seconds;
     fprintf(stdout,"SADBM_UPDATE: timenow=%s", ctime(&tttv));
     fprintf(stdout,"SADBM_UPDATE: expiration time = %s", ctime(&tttv1));
#endif TEST_KERN
  
  /* Protocol */
  if (st->st_protoid == PROTO_IPSEC_ESP)
    m_sadbmsg.sadb_info.dst.sa_info.protocol = IPSEC_IPPROTO_ESP;
  else
    m_sadbmsg.sadb_info.dst.sa_info.protocol = IPSEC_IPPROTO_AH;
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout,"SADBM_UPDATE: protocol = %d\n",
	m_sadbmsg.sadb_info.dst.sa_info.protocol);
#endif
  
  switch (st->st_protoid)
    {
    case PROTO_IPSEC_ESP:
      
      /* ESP: Crypto Alg_ID */
      switch (st->st_transid)
	{
	case ESP_NULL:
	  ipsec_enc_alg = IPSEC_ESP_NULL;
	  break;
	case ESP_DES_IV64:
	  ipsec_enc_alg = IPSEC_RFC_1829;
	  break;
	case ESP_DES:
	  ipsec_enc_alg = IPSEC_ESP_DES_CBC;
	  break;
	case ESP_3DES:
	  ipsec_enc_alg = IPSEC_DES3_ALG;
	  break;
	case ESP_RC5:
	  ipsec_enc_alg = IPSEC_ESP_RC5_CBC;
	  break;
	case ESP_IDEA:
	  ipsec_enc_alg = IPSEC_ESP_IDEA_CBC;
	  break;
	case ESP_BLOWFISH:
	  ipsec_enc_alg = IPSEC_ESP_BLOWFISH_CBC;
	  break;
	default:
	  ipsec_enc_alg = IPSEC_ESP_DES_CBC;
	  break;
	}
      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_alg_id = ipsec_enc_alg;
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: ESP crypto alg = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_alg_id);
#endif
      
	/* Initial Replay Sequence Number - depends on esp alg */
      if ((ipsec_enc_alg == IPSEC_ESP_DES_CBC) ||
	  (ipsec_enc_alg == IPSEC_DES3_ALG) ||
	  (ipsec_enc_alg == IPSEC_ESP_RC5_CBC) ||
	  (ipsec_enc_alg == IPSEC_ESP_BLOWFISH_CBC) ||
	  (ipsec_enc_alg == IPSEC_ESP_IDEA_CBC) ||
	  (ipsec_enc_alg == IPSEC_ESP_NULL))
	m_sadbmsg.sadb_info.dst.sa_info.sn = 1;
      
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: ESP sn = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.sn);
#endif
      
      /* ESP: Crypto Ivec_Length */
      /* XXX Temporary code: should be handled by IPSEC policy mechanism */
      /* XXX set enc iv len as specified by -v option (saved in arg_enc_iv_len)
	 (enc iv len is ALWAYS 0 for ESP_NULL; ALWAYS 8 for ESP_DES_IV64
           default is 8 for other algorithms) */
	if (st->st_transid ==  ESP_DES_IV64)
      		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_ivec_length = 8; 
	else if (st->st_transid == ESP_NULL)
      		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_ivec_length = 0; 
	else
      		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_ivec_length = arg_enc_iv_len; 
#ifdef DEBUG
      if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: ESP iv len = %d\n", 
	    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_ivec_length);
#endif
      
      /* ESP Key Processing
	 Need to break up key material between crypto key and auth key
	 (Use 1st st->st_esp_keymat_len bytes for enc key, 
         next st->st_ah_keymat_len bytes for auth key          */
      /* ESP: Outbound Crypto Key Length & Key */
      /* XXX Temporary code: should be handled by IPSEC policy mechanism */
      /* XXX set enc key len as specified by -k option (saved in enc_key_len) 
	 (default: 8 for IPSEC_RFC_1829, IPSEC_DES_CBC, DES3; 16 for others) */
      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len = st->st_esp_keymat_len;
      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_len = st->st_esp_keymat_len;

      /* Copy enc_key_len # of bytes from keymat */
	/* Create inbound SA with own key */
	bcopy(st->st_keymat, 
	      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_data,
	      st->st_esp_keymat_len);
       /* Make both keys the same */
	bcopy(st->st_keymat, 
	      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_data,
	      st->st_esp_keymat_len);
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
	fprintf(stdout,"SADBM_UPDATE: ESP outbound crypto key len = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len);
	  fprintf(stdout, "SADBM_UPDATE: ESP outbound crypto key:\n ");
 	  format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_data, 
 		      st->st_esp_keymat_len );
	  fprintf(stdout,"\nSADBM_UPDATE: ESP inbound crypto key len = %d\n",
	      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_len);
	  fprintf(stdout, "SADBM_UPDATE: ESP inbound crypto key:\n ");
	  format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_data, 
		      st->st_esp_keymat_len);
	  fprintf(stdout, "\n");
	}
#endif
      
      /* ESP: Auth Alg_ID */
      if ((st->st_att_authentication_alg)) 
	{
	  if ((st->st_att_authentication_alg == AUTH_ALGO_HMAC_SHA_1))
	    ipsec_auth_alg = IPSEC_HMACSHA96;
	  else
	    ipsec_auth_alg = IPSEC_HMACMD596;
	  m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_alg_id = ipsec_auth_alg;
#ifdef DEBUG
	  if(arg_verbose & DEBUG_VERBOSE)
	    fprintf(stdout,"SADBM_UPDATE: ESP auth alg = %d\n",
		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_alg_id);
#endif
	  
	  /* Initial Replay Sequence Number - depends on auth alg 
	     Only set it for IPSEC_CIPHER_TEST_ALG - 
	     other ESP alg's have their own requirements for sn */
	  if ((ipsec_enc_alg == IPSEC_CIPHER_TEST_ALG) &&
	      ((ipsec_auth_alg == IPSEC_HMACMD596) ||
	       (ipsec_auth_alg == IPSEC_HMACSHA96)))
	    m_sadbmsg.sadb_info.dst.sa_info.sn = 1;
	  
#ifdef DEBUG
	  if(arg_verbose & DEBUG_VERBOSE)
	    fprintf(stdout,"SADBM_UPDATE: ESP auth sn = %d\n",
		m_sadbmsg.sadb_info.dst.sa_info.sn);
#endif
	  
	  /* ESP: Auth Ivec_Length */
	  /* XXXX state structure (state.h) does not have an entry for ESP's auth ivec len */
	  /* XXXX ESP: Auth Ivec_Length - must be 0 */
	  
	  /* ESP: Outbound Auth Key Length & Key */
	  /* XXXX Temporary code: should be handled by IPSEC policy mechanism */
	  /* XXXX set key len as specified by auth_key_len
	     (20 for IPSEC_HMACSHA96; 16 for others) */
	  m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len = st->st_ah_keymat_len;
	  m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_len = st->st_ah_keymat_len;

	  /* XXXX Copy auth_key_len # of bytes following enc_key from keymat */
	/* Create inbound SA with own key */
	      bcopy(st->st_keymat + st->st_esp_keymat_len, 
		    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_data,
		    st->st_ah_keymat_len);
        /* Make both keys the same */
	      bcopy(st->st_keymat + st->st_esp_keymat_len, 
		    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_data,
		    st->st_ah_keymat_len);
#ifdef DEBUG
	  if (arg_verbose & DEBUG_VERBOSE)
	    {
	    fprintf(stdout,"SADBM_UPDATE: ESP outbound auth key len = %d\n",
		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len);
	      fprintf(stdout, "SADBM_UPDATE: ESP outbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	      fprintf(stdout,"\nSADBM_UPDATE: ESP inbound auth key len = %d\n",
		  m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_len);
	      fprintf(stdout, "SADBM_UPDATE: ESP inbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	      fprintf(stdout, "\n");
	    }
#endif
	  
	  /* ESP: Auth Data_Length */
	  /* XXXX Temporary code: should be handled by IPSEC policy mechanism */
	  /* XXXX set auth data len as specified by auth_data_len
	     (5 for IPSEC_RFC_1852, 6 for AH_HMAC_SHA192, 3 for IPSEC_HMACMD596 and IPSEC_HMACSHA96; 
	     4 for others) */
	  m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_data_length = 3;
#ifdef DEBUG
	  if (arg_verbose & DEBUG_VERBOSE)
	    fprintf(stdout,"SADBM_UPDATE: ESP auth data len = %d\n", 
		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_data_length);
#endif
	}
      break;
      
    case PROTO_IPSEC_AH:
      
      /* AH: Auth Alg_ID */
      switch (st->st_transid)
	{
	case AH_MD5:
	  ipsec_auth_alg = IPSEC_HMACMD596;
	  break;
	case AH_SHA:
	  ipsec_auth_alg = IPSEC_HMACSHA96;
	  break;
	default:
	  ipsec_auth_alg = IPSEC_HMACMD596;
	  break;
	}
      m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_alg_id = ipsec_auth_alg;
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: AH auth alg = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_alg_id);
#endif
      
      /* Initial Replay Sequence Number - depends on auth alg */
      if ((ipsec_auth_alg == IPSEC_HMACMD596) ||
	  (ipsec_auth_alg == IPSEC_HMACSHA96))
	m_sadbmsg.sadb_info.dst.sa_info.sn = 1;
      
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: AH sn = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.sn);
#endif
      
      /* AH: Auth Ivec_Length */
      /* XXXX AH: Auth Ivec_Length - must be 0 */
      m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_ivec_length = 0;
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: AH iv len = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_ivec_length);
#endif
      
      /* AH: Outbound Auth Key Length & Key */
      /* XXXX Temporary code: should be handled by IPSEC policy mechanism */
      /* XXXX set key len as specified by auth_key_len
	 (20 for IPSEC_HMACSHA96; 16 for others) */
      m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len = st->st_ah_keymat_len;
      m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_len = st->st_ah_keymat_len;

	/* Create inbound SA with own key */
	  bcopy(st->st_keymat, 
		m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_data,
       		st->st_ah_keymat_len);
       /* Make both keys the same */
	  bcopy(st->st_keymat, 
		m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_data,
       		st->st_ah_keymat_len);
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
	fprintf(stdout,"SADBM_UPDATE: AH outbound key len = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len);
	  fprintf(stdout, "SADBM_UPDATE: AH outbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	  fprintf(stdout, "\nSADBM_UPDATE: AH inbound key len = %d\n",
		  m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_len, 0, 0);
	  fprintf(stdout, "SADBM_UPDATE: AH inbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	  fprintf(stdout, "\n");
	}
#endif
      
      /* AH: Auth Data_Length */
      /* XXXX Temporary code: should be handled by IPSEC policy mechanism */
      /* XXXX set auth data len as specified by auth_data_len
	 (5 for IPSEC_RFC_1852, 6 for AH_HMAC_SHA192, 3 for IPSEC_HMACMD596 and IPSEC_HMACSHA96; 
	 4 for others) */
      m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_data_length = 3;
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout,"SADBM_UPDATE: AH data len = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_data_length);
#endif
      break;
      
    default: exit_log("Unknown ipsec protocol %d", st->st_protoid, 0, 0);
      break;
    }

/* Update the 1-way inbound partial SA using an SADBM_UPDATE command */
  if ((sadblen = write(kernelfd, (char *)&m_sadbmsg, l)) < 0) 
    exit_log("write() kernelfd (ike) failed in kernel_update", 0, 0, 0);
#ifdef TEST_KERN
    fprintf(stdout, "wrote %d bytes to kernelfd (ike) socket\n", l);
#endif

/* Prepare to add a 1-way outbound SA using an SADBM_ADD command */
  m_sadbmsg.m_sadbm.sadbm_type = SADBM_SET;
  
  /* SPI - for 1-way outbound SA, use Peer SPI for both SPIs */
  bcopy(st->st_peer_spi,&spi, sizeof(spi));
  m_sadbmsg.sadb_info.dst.sa_info.spi = ntohl(spi); 
  m_sadbmsg.sadb_info.dst.sa_info.peer_spi = m_sadbmsg.sadb_info.dst.sa_info.spi; 
#ifdef TEST_KERN
    fprintf(stdout,"SADBM_ADD: spi = %lx\n",m_sadbmsg.sadb_info.dst.sa_info.spi); 
    fprintf(stdout,"sending SADBM_ADD: peer_spi (USED-O) = %lx\n",m_sadbmsg.sadb_info.dst.sa_info.peer_spi); 
#endif TEST_KERN

    /* Make SA 1-way outbound */
    m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;
    /* Turn OFF inbound flag */
    m_sadbmsg.sadb_info.dst.sa_info.flags ^= IPSEC_INBOUND_FLAG;

  /* Keys - for 1-way outbound SA, use Peer Keys */
  switch (st->st_protoid)
    {
    case PROTO_IPSEC_ESP:

      /* XXXX Copy enc_key_len # of bytes from peer_keymat */
	/* Create outbound SA with peer's key */
	bcopy(st->st_peer_keymat, 
	      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_data,
	      st->st_esp_keymat_len);
       /* Make both keys the same */
	bcopy(st->st_peer_keymat, 
	      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_data,
	      st->st_esp_keymat_len);
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
	fprintf(stdout,"SADBM_ADD: ESP outbound crypto key len = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len);
	  fprintf(stdout, "SADBM_ADD: ESP outbound crypto key:\n ");
 	  format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_data, 
 		      st->st_esp_keymat_len );
	  fprintf(stdout,"\nSADBM_ADD: ESP inbound crypto key len = %d\n",
	      m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_len);
	  fprintf(stdout, "SADBM_ADD: ESP inbound crypto key:\n ");
	  format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_data, 
		      st->st_esp_keymat_len);
	  fprintf(stdout, "\n");
	}
#endif

      if ((st->st_att_authentication_alg)) 
	{
	  /* XXXX Copy auth_key_len # of bytes following enc_key from peer_keymat */
	/* Create outbound SA with peer's key */
	      bcopy(st->st_peer_keymat + st->st_esp_keymat_len, 
		    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_data,
		    st->st_ah_keymat_len);
        /* Make both keys the same */
	      bcopy(st->st_peer_keymat + st->st_esp_keymat_len, 
		    m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_data,
		    st->st_ah_keymat_len);
#ifdef DEBUG
	  if (arg_verbose & DEBUG_VERBOSE)
	    {
	    fprintf(stdout,"SADBM_ADD: ESP outbound auth key len = %d\n",
		m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len);
	      fprintf(stdout, "SADBM_ADD: ESP outbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	      fprintf(stdout,"\nSADBM_ADD: ESP inbound auth key len = %d\n",
		  m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_len);
	      fprintf(stdout, "SADBM_ADD: ESP inbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	      fprintf(stdout, "\n");
	    }
#endif
	}
      break;
      
    case PROTO_IPSEC_AH:

	/* AH: Create outbound SA with peer's key */
	  bcopy(st->st_peer_keymat, 
		m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_data,
       		st->st_ah_keymat_len);
       /* Make both keys the same */
	  bcopy(st->st_peer_keymat, 
		m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_data,
       		st->st_ah_keymat_len);
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	{
	fprintf(stdout,"SADBM_ADD: AH outbound key len = %d\n",
	    m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len);
	  fprintf(stdout, "SADBM_ADD: AH outbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	  fprintf(stdout, "\nSADBM_ADD: AH inbound key len = %d\n",
		  m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_len, 0, 0);
	  fprintf(stdout, "SADBM_ADD: AH inbound auth key:\n ");
	      format_dump(m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_data,
			  st->st_ah_keymat_len);
	  fprintf(stdout, "\n");
	}
#endif
      break;
      
    default: exit_log("Unknown ipsec protocol %d", st->st_protoid, 0, 0);
      break;
    }

/* NOW add a 1-way outbound SA using an SADBM_ADD command */
  if ((sadblen = write(kernelfd, (char *)&m_sadbmsg, l)) < 0) 
    exit_log("write() kernelfd (ike) failed in kernel_update", 0, 0, 0);
#ifdef TEST_KERN
    fprintf(stdout, "wrote %d bytes to kernelfd (ike) socket\n", l);
#endif

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernel_update | in kernel_comm.c");
#endif
  
}

/**********************************************************************/
/* For failed negotiation, send fake SADBM_UPDATE to kernel 
 *	so kernel can delete destination address from address array
 */
void kernel_update_ng(int            kernelfd, 
		   struct state  *st)
{
  struct sadb_msg      m_sadbmsg;
  struct sockaddr_in   sin;
  int                  l;
  int                  sadblen;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernel_update_ng | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernel_update_ng | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not call kernel to update SA */
  return;
#endif ZZZZR
  bzero((char *)&m_sadbmsg, sizeof(m_sadbmsg));
  m_sadbmsg.m_sadbm.sadbm_msglen = l = sizeof(struct sadb_msg);
  m_sadbmsg.m_sadbm.sadbm_version = SADBM_VERSION;
  m_sadbmsg.m_sadbm.sadbm_flags = SADBM_IKE;
  m_sadbmsg.m_sadbm.sadbm_type = SADBM_UPDATE;
  
  m_sadbmsg.request_type = SADB_DST;
  
  /* Destination */
  bcopy(&st->st_peer,&sin,sizeof(sin));
  m_sadbmsg.sadb_info.dst.dst.addr_family = sin.sin_family; 
  m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr.s_addr = sin.sin_addr.s_addr;
#ifdef TEST_KERN
    fprintf(stdout, "SADBM_UPDATE: failed destination = %s\n", 
	inet_ntoa(m_sadbmsg.sadb_info.dst.dst.addr_union.ip_addr));
#endif
  
  /* SA added through key negotiation  */ 
  m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_NEG_KM_FLAG;

  /* Protocol - fake protocol to let kernel know that negotiation failed */
    m_sadbmsg.sadb_info.dst.sa_info.protocol = IPSEC_IPPROTO_NG;
#ifdef TEST_KERN
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout,"SADBM_UPDATE: failed protocol = %d\n",
	m_sadbmsg.sadb_info.dst.sa_info.protocol);
#endif

/* Negotiation failed - tell kernel to delete destination address from 
 * 	address array */
  if ((sadblen = write(kernelfd, (char *)&m_sadbmsg, l)) < 0) 
    exit_log("write() kernelfd (ike) failed in kernel_update_ng", 0, 0, 0);
#ifdef TEST_KERN
    fprintf(stdout, "wrote %d bytes to kernelfd (ike) socket\n", l);
#endif

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernel_update_ng | in kernel_comm.c");
#endif
}

/**********************************************************************/
/* kernel_unregister:
 * Send SADBM_UNREGISTER message to IPSEC, then exit
 *
 * signo > 0 ===> kernel_unregister called as a result of trapped signal -
 *		  unregister, then call exit_log_nr to log PlutoPlus error 
 *		  message (and exit) without call to kernel_unregister
 * signo = 0 ===> kernel_unregister called by exit_log - 
 *		  unregister, then return (don't recursively call exit_log)
 */ 
void kernel_unregister(int signo)
{
  struct sadb_msg      m_sadbmsg;
  int                  l;
  int                  sadblen;
#ifdef DEBUG_IN_OUT
  in_out_functions(" E kernel_unregister | in kernel_comm.c");
#ifdef ZZZZR
  in_out_functions(" Q-ZZZZR kernel_unregister | in kernel_comm.c");
#endif ZZZZR
#endif DEBUG_IN_OUT
#ifdef ZZZZR
  /* ZZZZ Code to negotiate with self - responder does not unregister */
    if (signo)
    	exit_log_nr("PlutoPlus exiting - received signal # %d", signo, 0, 0);
    else
	return;
#endif ZZZZR
  /* Unregister ISAKMP daemon with IPSEC */
  bzero((char *)&m_sadbmsg, sizeof(m_sadbmsg));
  m_sadbmsg.m_sadbm.sadbm_msglen = l = sizeof(struct sadb_msg);
  m_sadbmsg.m_sadbm.sadbm_version = SADBM_VERSION;
  m_sadbmsg.m_sadbm.sadbm_type = SADBM_UNREGISTER;
  m_sadbmsg.m_sadbm.sadbm_flags = SADBM_IKE;
  m_sadbmsg.sadb_info.udp_port = our_port;
#ifdef TEST_KERN
	  fprintf(stdout, "sending SADBM_UNREGISTER \n");
#endif
  if ((sadblen = write(kernelfd_sav, (char *)&m_sadbmsg, l)) < 0) 
    exit_log_nr("write() kernelfd_sav (ike) failed in kernel_unregister", 0, 0, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q kernel_unregister | in kernel_comm.c");
#endif
    if (signo)
    	exit_log_nr("PlutoPlus exiting - received signal # %d", signo, 0, 0);
}

