/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/* rnd.c */

#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include <fcntl.h>
#include "constants.h"
#include "defs.h"
#include SHA_H

/* for the verbose mode */
#include "argdefs.h"

#define RANDOM_POOL_SIZE   20
static u_char random_pool[RANDOM_POOL_SIZE];
static volatile u_char i, j, k;


/********************************************************************/
/*
 * Signal handler.
 */
void
rnd_handler()
{

   k <<= 1;        /* Shift left by 1 */
    j++;           
    k |= (i & 0x1); /* Get lsbit of counter */

    if (j != 8)
      signal(SIGVTALRM, rnd_handler);
}

/********************************************************************/
/*
 * Do the clock skew trick.
 */
static u_char
generate_rnd_byte(void)
{
  struct itimerval tmval;
  struct itimerval ntmval;
  u_char           c;

    if (read(linux_random, &c, sizeof(c)) == -1)
      exit_log("read() failed in generate_rnd_byte()", 0, 0, 0);
    return c;

    i = 0;
    j = 0;

    ntmval.it_interval.tv_sec = 0;
    ntmval.it_interval.tv_usec = 1;
    ntmval.it_value.tv_sec = 0;
    ntmval.it_value.tv_usec = 1;
    signal(SIGVTALRM, rnd_handler);
    setitimer(ITIMER_VIRTUAL, &ntmval, &tmval);
    
    while (j != 8)
      i++;
    
    setitimer(ITIMER_VIRTUAL, &tmval, &ntmval);
    signal(SIGVTALRM, SIG_IGN);

    return k;
}



/********************************************************************/
/*
 * Get a single random byte.
 */
u_char get_rnd_byte(void)
{
  SHA_CTX shactx;
  
  SHAInit(&shactx);
  
  random_pool[RANDOM_POOL_SIZE - 1] = generate_rnd_byte();
  random_pool[0] = generate_rnd_byte();
  
  SHAUpdate(&shactx, random_pool, RANDOM_POOL_SIZE);
  SHAFinal(&shactx);
  bcopy((&shactx)->buffer, random_pool);
  
  return random_pool[0];
}



/********************************************************************/
/*
 * A random number generator based on clock skew. I first heard of
 * the idea by John Ioannidis, who heard it from Matt Blaze and/or
 * Jack Lacy. Seems to work.
 * If we're on OpenBSD, use the arc4random() generator instead; if
 * on Linux, use /dev/random.
 */
void get_rnd_bytes(u_char *buffer, 
		   int     length)
{
  int i;
#ifdef __OpenBSD__
  for (i = 0; i < length; i++)
    buffer[i] = arc4random() % 256;
#else    
  for (i = 0; i < length; i++)
    buffer[i] = get_rnd_byte();
#endif
}


/********************************************************************/
/*
 * Initialize the random pool.
 */
void init_rnd_pool(void)
{
  int i;
  SHA_CTX shactx;           /* unsigned long state[5];   *
			     * unsigned long count[2];   * 
			     * unsigned char buffer[64]; */  
  
  SHAInit(&shactx);
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "opening the random file : %s\n", LINUX_RANDOM_FILE);
#endif
  linux_random = open(LINUX_RANDOM_FILE, O_RDONLY);
  if (linux_random == -1)
    exit_log("open() failed in init_rnd_pool()", 0, 0, 0);
  
  get_rnd_bytes(random_pool, RANDOM_POOL_SIZE);
  
  SHAUpdate(&shactx, random_pool, RANDOM_POOL_SIZE);
  SHAFinal(&shactx);
  bcopy((&shactx)->buffer, random_pool);
  
}
