/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/*
 * state.c - 
 * 	This file has the functions that handle the
 * 	state hash table and the messageid generation.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include "constants.h"
#include "state.h"
#include "defs.h"
#include "argdefs.h"

#define PRINTADDRESS(x) (get_address(x) != (char *) NULL ? get_address(x) : "(un
known address family)")

extern char *get_address();

#define STATE_TABLE_SIZE 32

static struct state       *statetable[STATE_TABLE_SIZE];


/****************************************************************************/
/*
 * Initialize the modulo and generator.
 */
void init_nums(void)
{
#ifdef DEBUG_IN_OUT
  in_out_functions(" E init_nums | in state.c");
#endif
  if (mpz_init_set_str(&groupmodulo, GROUPDEFAULT, 16) != 0)
    exit_log("mpz_init_set_str() failed in init_nums()", 0, 0, 0);
  
  if (mpz_init_set_str(&groupgenerator, DEFAULTGENERATOR, 10) != 0)
      exit_log("mpz_init_set_str() failed in init_nums()", 0, 0, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q init_nums | in state.c");
#endif
}



/****************************************************************************/
/*
 * Had to go somewhere, might as well be this file. Initialize
 * global variables.
 */
void init_vars(void)
{
#ifdef DEBUG_IN_OUT
  in_out_functions(" E init_vars | in state.c");
#endif
  
  
  if (our_port == 0)
    our_port = PORT;	/* ie 500, can be manually defined in DEBUG mode */
  
  /* 
   * Generate the secret value for responder cookies, and
   * schedule an event for refresh.
   */
  get_rnd_bytes(secret_of_the_day, SECRET_VALUE_LENGTH);
  event_schedule(EVENT_REINIT_SECRET, EVENT_REINIT_SECRET_DELAY, NULL, 0);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q init_vars | in state.c");
#endif
}

/************************************************************************/
/*
 * Get a msgid for Phase 2 messages.
 * Called by : initiate_quick in ipsec_doi_quickxchg.c 
 * 
 */
u_int32_t get_msgid(struct sockaddr sa)
{
  u_int32_t          msgid;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E get_msgid  | in state.c");
#endif
/* Make Message ID Random */
  get_rnd_bytes(&(msgid), sizeof(u_int32_t));
#ifdef DEBUG
	  if (arg_verbose & DEBUG_VERBOSE)
	    fprintf(stdout, "Setting Message ID for %s to %x\n",
		PRINTADDRESS(sa), msgid);
#endif
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q get_msgid   | in state.c");
#endif
  
/* Return msgid in Network Byte Order */
  return htonl(msgid);
}

/****************************************************************************/
/* Called by : insert_state (state.c)
 * Called by : delete_state (state.c)
 * Called by : find_full_state (state.c)
 * Called by : find_half_state (state.c) */
static u_int state_hash(u_char         *icookie, 
			u_char         *rcookie, 
			struct sockaddr sa)
{
  u_int   i;
  u_int   j;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E state_hash | in state.c");
#endif
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
    {
      fprintf(stdout, "ICOOKIE: ");
      for (i = 0; i < COOKIE_SIZE; i++)
	fprintf(stdout, "%02x ", icookie[i]);
      fprintf(stdout, "\n");
      fprintf(stdout, "RCOOKIE: ");
      for (i = 0; i < COOKIE_SIZE; i++)
	fprintf(stdout, "%02x ", rcookie[i]);
      fprintf(stdout, "\n");
      fprintf(stdout, "sockaddr: ");
      for (i = 0; i < 4; i++)
	fprintf(stdout, "%02x ", sa.sa_data[i]);
      fprintf(stdout, "\n");
    }
#endif
  
  for (i = 0, j = 0; j < COOKIE_SIZE; j++)
    i += icookie[j] + rcookie[j];
  
  i += sa.sa_family;
  
  for (j = 0; j < FULL_INET_ADDRESS_SIZE; j++)
    i += sa.sa_data[j];
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)  
  	fprintf(stdout, "state hash entry %d\n", i % STATE_TABLE_SIZE);
#endif
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q state_hash | in state.c");
#endif
  

  return i % STATE_TABLE_SIZE;
}




/****************************************************************************/
/* 
 * Get a state object.
 * Called by oakley_initiate (ipsec_doi.c)
 * Called by ipsecdoi_handle_rfirst (ipsec_doi_mainxchg.c)
 * Called by duplicate_state (state.c)
 */
struct state * get_state(void)
{
  struct state *st;
    
#ifdef DEBUG_IN_OUT
  in_out_functions(" E get_state  | in state.c");
#endif
  
  st = (struct state *) calloc(1, sizeof(struct state));
  if (st == (struct state *) NULL)
    exit_log("calloc() failed in get_state()", 0, 0, 0);
    
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q get_state | in state.c");
#endif
  
  return st;
}




/****************************************************************************/
/*
 * Initialize the state table.
 * called by main (main.c)
 */
void init_state(void)
{
  int l;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E init_state | in state.c");
#endif
  
  for (l = 0; l < STATE_TABLE_SIZE; l++)
    statetable[l] = (struct state *) NULL;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q init_state | in state.c");
#endif
}




/****************************************************************************/
/*
 * Insert a state object in the hash table. The object is inserted
 * at the begining of list.
 * Called by : oakley_initiate in ipsec_doi.c
 * Called by ipsecdoi_handle_rfirst in ipsec_doi_mainxchg.c
 * Called by : initiate_quick in ipsec_doi_quickxchg.c
 * Called by : comm_handle in demux.c
 */
void insert_state(struct state *st)
{
  int i;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E insert_state | in state.c");
#endif
  
  i = state_hash(st->st_icookie, st->st_rcookie, st->st_peer);
  
  st->st_prev = (struct state *) NULL;
  st->st_next = statetable[i];
  statetable[i] = st;
  if (st->st_next != (struct state *) NULL)
    st->st_next->st_prev = st;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q insert_state | in state.c");
#endif
}




/****************************************************************************/
/*
 * Delete a state object from the hash table, but don't free it
 * Called by : ipsecdoi_handle_i1 (ipsec_doi_mainxchg.c)
 * Called by : event_handle (timer.c)
 */
void delete_state(struct state *st)
{
  int i;
  
  /* If it's not the first in the list */
#ifdef DEBUG_IN_OUT
  in_out_functions(" E delete_state | in state.c");
#endif  
    fprintf(stdout, "delete_state #1 (st=%x)\n", st);
  
  if (st->st_prev != (struct state *) NULL)
    {
      st->st_prev->st_next = st->st_next;
    fprintf(stdout, "delete_state #2 (st=%x)\n", st);
      
      if (st->st_next != (struct state *) NULL)
	st->st_next->st_prev = st->st_prev;
    fprintf(stdout, "delete_state #3 (st=%x)\n", st);
      
#ifdef DEBUG_IN_OUT
      in_out_functions(" Q delete_state #1| in state.c");
#endif
      return;
    }
  
  i = state_hash(st->st_icookie, st->st_rcookie, st->st_peer);
  statetable[i] = st->st_next;
  if (st->st_next != (struct state *) NULL)
    st->st_next->st_prev = (struct state *) NULL;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q delete_state #2| in state.c");
#endif
}




/****************************************************************************/
/*
 * Free a state object
 * Called by : oakley_initiate (ipsec_doi.c)
 * Called by ipsecdoi_handle_rfirst (ipsec_doi_mainxchg.c)
 * Called by : initiate_quick (ipsec_doi_quickxchg.c)
 * Called by event_handle (timer.c)
 */
void free_state(struct state *st)
{
  int k;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E free_state | in state.c");
#endif
  if ((st->st_packet != (u_char *) NULL) &&
      (st->st_packet_len != 0))
    {
    free(st->st_packet);
    }
  
  if (st->st_gi_in_use)
    mpz_clear(&(st->st_gi));
  
  if (st->st_gr_in_use)
    mpz_clear(&(st->st_gr));
  
  if (st->st_sec_in_use)
    mpz_clear(&(st->st_sec));
  
  if (st->st_shared_in_use)
    mpz_clear(&(st->st_shared));
  
  if (st->st_proposal != (u_char *) NULL)
    free(st->st_proposal);

  if (st->st_sa != (u_char *) NULL)
    free(st->st_sa);
  
  if (st->st_ni != (u_char *) NULL)
    free(st->st_ni);
  
  if (st->st_nr != (u_char *) NULL)
    free(st->st_nr);
  
  if (st->st_skeyid != (u_char *) NULL)
    free(st->st_skeyid);
  
  if (st->st_skeyid_d != (u_char *) NULL)
    free(st->st_skeyid_d);
  
  if (st->st_skeyid_a != (u_char *) NULL)
    free(st->st_skeyid_a);
  
  if (st->st_skeyid_e != (u_char *) NULL)
    free(st->st_skeyid_e);
  
  if (st->st_myidentity != (u_char *) NULL)
    free(st->st_myidentity);
  
  if (st->st_peeridentity != (u_char *) NULL)
    free(st->st_peeridentity);
  
  if (st->st_peerclientid != (u_char *) NULL)
    free(st->st_peerclientid);
  
  if (st->st_iv != (u_char *) NULL)
    free(st->st_iv);
  
  if (st->st_lastblock != (u_char *) NULL)
    free(st->st_lastblock);

  if (st->st_spi != (u_char *) NULL)
    free(st->st_spi);
  
  if (st->st_keymat != (u_char *) NULL)
    free(st->st_keymat);
  
  if (st->st_peer_spi != (u_char *) NULL)
    free(st->st_peer_spi);
  
  if (st->st_peer_keymat != (u_char *) NULL)
    free(st->st_peer_keymat);
  
  free(st);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q free_state | in state.c");
#endif
}




/****************************************************************************/
/*
 * Duplicate a state object, for Phase 2.
 * Called by : comm_handle in demux.c
 * Called by initiate_quick (ipsec_doi_quickxchg.c)
 */
struct state * duplicate_state(struct state *st)
{
  struct state *nst;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E duplicate_state | in state.c");
#endif
  nst = get_state();
  
  nst->st_phase1 = st;
  
  bcopy(st->st_icookie, nst->st_icookie, COOKIE_SIZE);
  bcopy(st->st_rcookie, nst->st_rcookie, COOKIE_SIZE);
  nst->st_peer = st->st_peer;
  
  nst->st_doi = st->st_doi;
  nst->st_situation = st->st_situation;
  
  /* Copy SKEYID_D */
  nst->st_skeyid_d = (u_char *) calloc(st->st_skeyid_d_len, sizeof(u_char));
  if (nst->st_skeyid_d == (u_char *) NULL)
    exit_log("calloc() failed in duplicate_state()", 0, 0, 0);
  
  bcopy(st->st_skeyid_d, nst->st_skeyid_d, st->st_skeyid_d_len);
  nst->st_skeyid_d_len = st->st_skeyid_d_len;
  
  /* Copy SKEYID_E */
  nst->st_skeyid_e = (u_char *) calloc(st->st_skeyid_e_len, sizeof(u_char));
  if (nst->st_skeyid_e == (u_char *) NULL)
    exit_log("calloc() failed in duplicate_state()", 0, 0, 0);
  
  bcopy(st->st_skeyid_e, nst->st_skeyid_e, st->st_skeyid_e_len);
  nst->st_skeyid_e_len = st->st_skeyid_e_len;
  
  /* Copy SKEYID_A */
  nst->st_skeyid_a = (u_char *) calloc(st->st_skeyid_a_len, sizeof(u_char));
  if (nst->st_skeyid_a == (u_char *) NULL)
    exit_log("calloc() failed in duplicate_state()", 0, 0, 0);
  
  bcopy(st->st_skeyid_a, nst->st_skeyid_a, st->st_skeyid_a_len);
  nst->st_skeyid_a_len = st->st_skeyid_a_len;
  
  /* Copy myidentity */
  nst->st_myidentity = (u_char *) calloc(st->st_myidentity_len, 
					 sizeof(u_char));
  if (nst->st_myidentity == (u_char *) NULL)
    exit_log("calloc() failed in duplicate_state()", 0, 0, 0);
  
  bcopy(st->st_myidentity, nst->st_myidentity, st->st_myidentity_len);
  nst->st_myidentity_len = st->st_myidentity_len;
  nst->st_myidentity_type = st->st_myidentity_type;
  
  /* Copy peeridentity */
  nst->st_peeridentity = (u_char *) calloc(st->st_peeridentity_len, 
					   sizeof(u_char));
  if (nst->st_peeridentity == (u_char *) NULL)
    exit_log("calloc() failed in duplicate_state()", 0, 0, 0);
  
  bcopy(st->st_peeridentity, nst->st_peeridentity, st->st_peeridentity_len);
  nst->st_peeridentity_len = st->st_peeridentity_len;
  nst->st_peeridentity_type = st->st_peeridentity_type;
  
  /* Copy goal */
  nst->st_goal = st->st_goal;
  
  nst->st_hash = st->st_hash;
  nst->st_enc = st->st_enc;
  nst->st_prf = st->st_prf;
  nst->st_auth = st->st_auth;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q duplicate_state | in state.c");
#endif
    
  return nst;
}




/****************************************************************************/
/*
 * Find a state object.
 * Called by : comm_handle in demux.c
 */
struct state * find_full_state(u_char          *icookie, 
			       u_char          *rcookie, 
			       struct sockaddr  sa, 
			       u_int32_t        msgid)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) && (!bcmp(&(x.sa_data), &(y.sa_data), FULL_INET_ADDRESS_SIZE)))
  
  struct state *st;
#ifdef DEBUG_IN_OUT
  in_out_functions(" E find_full_state | in state.c");
#endif
  
  st = statetable[state_hash(icookie, rcookie, sa)];
  
#ifdef DEBUG
  if (arg_verbose & DEBUG_VERBOSE)
    fprintf(stdout, "find_full_state() hash %d pointer %p\n", 
	state_hash(icookie, rcookie, sa), st);
#endif
  
  while (st != (struct state *) NULL)
    if (SA_EQUAL(sa, st->st_peer) &&
	(!bcmp(icookie, st->st_icookie, COOKIE_SIZE)) &&
	(!bcmp(rcookie, st->st_rcookie, COOKIE_SIZE)) &&
	(msgid == st->st_msgid))
      break;
    else
      st = st->st_next;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q find_full_state | in state.c");
#endif
  
  return st;
#undef SA_EQUAL
}

/****************************************************************************/
/*
 * Find an ISAKMP SA state object.
 * Called by oakley_initiate (ipsec_doi.c)
 */
struct state * find_phase1_state(struct sockaddr sa)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) && (!bcmp(&(x.sa_data), &(y.sa_data), FULL_INET_ADDRESS_SIZE)))
  struct state *st;
  int           i;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E find_phase1_state | in state.c");
#endif
  
  for (i = 0; i < STATE_TABLE_SIZE; i++)
    for (st = statetable[i]; st != (struct state *) NULL; st = st->st_next)
      if ((SA_EQUAL(sa, st->st_peer)) &&      /* Host we want */
	  (st->st_msgid == 0) &&	        /* ISAKMP SA */
	  (st->st_protoid == PROTO_ISAKMP) && 
	  ((st->st_state == OAKLEY_MAIN_I_4) ||
	   (st->st_state == OAKLEY_MAIN_R_3)))
	return st;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q find_phase1_state | in state.c");
#endif
  
  return (struct state *) NULL;
#undef SA_EQUAL
}

/****************************************************************************/
/*
 * Find an IPSEC SA state object.
 *    Before initiating a Phase 2 negotiation,
 *     		find the existing SA which is expiring (st1)
 *     		and check to see if a replacement has already been negotiated
 *			(st_repl)
 * Called by oakley_initiate (ipsec_doi.c)
 *     spi is in network byte order
 */
struct state * find_phase2_state_to_expire(struct sockaddr sa, u_int32_t spi)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) && (!bcmp(&(x.sa_data), &(y.sa_data), FULL_INET_ADDRESS_SIZE)))
  struct state *st, *st1;
  int           i, k, repl_found;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E find_phase2_state_to_expire | in state.c");
#endif

  st1 = (struct state *) NULL;
  repl_found = 0;
  
  for (i = 0; i < STATE_TABLE_SIZE; i++)
    for (st = statetable[i]; st != (struct state *) NULL; st = st->st_next)
        if ((SA_EQUAL(sa, st->st_peer)) &&      /* Host we want */
         (st->st_msgid != 0) &&	        /* Phase 2 IPSEC SA */
	 ((st->st_state == OAKLEY_QUICK_I_2) || /* Fully-established SA */
	 (st->st_state == OAKLEY_QUICK_R_3))) /* Fully-established SA */
        {
		/* Make sure this is the SA that is about to expire */
	   if (!bcmp(&spi, st->st_peer_spi, st->st_peer_spi_len))
#ifdef TEST_KERN
            {
    fprintf(stdout, "Phase 2 State (expiring) found for %s msgid=%lx flags=%x state=%d spi=%lx\n", 
	      PRINTADDRESS(sa), ntohl(st->st_msgid), st->st_flags, st->st_state, ntohl(spi));
#endif TEST_KERN
       		st1 = st;
		if (repl_found)
		       {
		/* Both expiring SA and replacement SA found */
	                st1->st_flags |= ST_FLAG_REPL_FOUND;
fprintf(stdout, 
"expiring & repl found: %s spi = %lx flags = %x st1 = %x repl_found = %x\n",
	      PRINTADDRESS(sa), ntohl(spi), st1->st_flags, st1, repl_found);
			return st1;
		       }
#ifdef TEST_KERN
            }
#endif TEST_KERN
        else if (!(st->st_flags & ST_FLAG_OUTBOUND_EXPIRED) &&
              (!(st->st_flags & ST_FLAG_OUTBOUND_DELETED)))
            {
#ifdef TEST_KERN
    fprintf(stdout, "Phase 2 State (replacement) found for %s msgid=%lx flags=%x state=%d old spi=%lx\n", 
	      PRINTADDRESS(sa), ntohl(st->st_msgid), st->st_flags, st->st_state, ntohl(spi));
#endif TEST_KERN
	        repl_found = 1;
		if (st1)
		       {
		/* Both expiring SA and replacement SA found */
	               st1->st_flags |= ST_FLAG_REPL_FOUND;
fprintf(stdout, 
"expiring & repl found: %s spi = %lx flags = %x st1 = %x repl_found = %x\n",
	      PRINTADDRESS(sa), ntohl(spi), st1->st_flags, st1, repl_found);
			return st1;
		       }
            }
        }

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q find_phase2_state_to_expire | in state.c");
#endif

   if (repl_found)
{
fprintf(stdout, 
"repl found but not expiring: %s spi = %lx st1 = %x repl_found = %x\n",
	      PRINTADDRESS(sa), ntohl(spi), st1, repl_found);
/* Replacement SA found, but expiring SA not found */
      return (struct state *) -1;
}
   else
{
fprintf(stdout, 
"repl NOT found: %s spi = %lx st1 = %x repl_found = %x\n",
	      PRINTADDRESS(sa), ntohl(spi), st1, repl_found);
/* Replacement SA not found
 * Return pointer to expiring SA (if found) or NULL */
      return st1;
}
#undef SA_EQUAL
}

/****************************************************************************/
/*
 * Find an IPSEC SA state object that the kernel is deleting.
 * 
 * Called by oakley_initiate (ipsec_doi.c)
 *     spi is in network byte order
 */
struct state * find_phase2_state_to_delete(struct sockaddr sa, 
			u_int32_t spi, int outbound)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) && (!bcmp(&(x.sa_data), &(y.sa_data), FULL_INET_ADDRESS_SIZE)))
  struct state *st;
  struct state *st1;
  int           i;
  int           k;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E find_phase2_state_to_delete | in state.c");
#endif
  
  for (i = 0; i < STATE_TABLE_SIZE; i++)
    for (st = statetable[i]; st != (struct state *) NULL; st = st->st_next)
      if ((SA_EQUAL(sa, st->st_peer)) &&      /* Host we want */
         (st->st_msgid != 0) &&	        /* Phase 2 IPSEC SA */
		/* Make sure this is the SA to be deleted */
	 (outbound && !bcmp(&spi, st->st_peer_spi, st->st_peer_spi_len) || 
	 (!outbound && !bcmp(&spi, st->st_spi, st->st_spi_len))) && 
	 ((st->st_state == OAKLEY_QUICK_I_2) || /* Fully-established SA */
	 (st->st_state == OAKLEY_QUICK_R_3))) /* Fully-established SA */
#ifdef TEST_KERN
       {
    fprintf(stdout, "Phase 2 State (delete) found for %s msgid=%lx flags=%x state=%d spi=%lx\n", 
	      PRINTADDRESS(sa), ntohl(st->st_msgid), st->st_flags, st->st_state, ntohl(spi));
#endif TEST_KERN
	return st;  
#ifdef TEST_KERN
       }
#endif TEST_KERN

#ifdef DEBUG_IN_OUT
  in_out_functions(" Q find_phase2_state_to_delete | in state.c");
#endif
  
  return (struct state *) NULL;
#undef SA_EQUAL
}

/****************************************************************************/
/*
 * Find a state object in the hash table, without using
 * the rcookie (to find state objects where we haven't yet
 * received a message from the responder).
 * Called by : comm_handle (demux.c)
 */
struct state *
find_half_state(u_char *icookie, struct sockaddr sa)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) && (!bcmp(&(x.sa_data), &(y.sa_data), FULL_INET_ADDRESS_SIZE)))

  struct state  *st;
  u_char         rcookie[COOKIE_SIZE];

#ifdef DEBUG_IN_OUT
  in_out_functions(" E find_half_state | in state.c");
#endif
  
  bzero(rcookie, COOKIE_SIZE);
  st = statetable[state_hash(icookie, rcookie, sa)];
  
  while (st != (struct state *) NULL)
    if (SA_EQUAL(sa, st->st_peer) &&
	(st->st_msgid == 0) &&	
	(!bcmp(icookie, st->st_icookie, COOKIE_SIZE)))
      break;
    else
      st = st->st_next;
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q find_half_state | in state.c");
#endif
  
  return st;
#undef SA_EQUAL
}

#ifdef TEST_PR_STATE
void pr_state_table()
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) && (!bcmp(&(x.sa_data), &(y.sa_data), FULL_INET_ADDRESS_SIZE)))
  struct state *st;
  int           i;
  int           k;

#ifdef DEBUG_IN_OUT
  in_out_functions(" E pr_state_table | in state.c");
#endif

  for (i = 0; i < STATE_TABLE_SIZE; i++)
    for (st = statetable[i]; st != (struct state *) NULL; st = st->st_next)
   {
    fprintf(stdout, "State found (i=%d st_next=%x) for %s msgid=%x flags=%x state=%d st=%x\n", i, st->st_next,
      PRINTADDRESS(st->st_peer), ntohl(st->st_msgid), st->st_flags, 
      st->st_state, st);
    fprintf(stdout, "\tSPI    (%d) :\t", st->st_spi_len);
      format_dump(st->st_spi, st->st_spi_len);
    fprintf(stdout, "\tPEER SPI    (%d) :\t", st->st_peer_spi_len);
      format_dump(st->st_peer_spi, st->st_peer_spi_len);
    fprintf(stdout, "\tInitiator Cookie(%d): ", COOKIE_SIZE);
      format_dump(st->st_icookie, COOKIE_SIZE); 
    fprintf(stdout, "\tResponder Cookie(%d): ", COOKIE_SIZE);
      format_dump(st->st_rcookie, COOKIE_SIZE); 
   }
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q pr_state_table | in state.c");
#endif
#undef SA_EQUAL
}
#endif TEST_PR_STATE
