/* This program is an implementation of the IKE Internet Standard.
 * PlutoPlus Copyright (C) 1999 Sheila Frankel - for details see COPYING.
 * Pluto Copyright (C) 1997 Angelos D. Keromytis - for details see COPYING.others.
 */

/*
 * timer.c - 
 * 	This file has the event handling routines. Events are
 * 	kept as a linked list of event structures. These structures
 * 	have information like event type, expiration time and a pointer
 * 	to event specific data (for example, to a state structure).
 */

#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "constants.h"
#include "state.h"
#include "defs.h"
#include "argdefs.h"

#define PRINTADDRESS(x) (get_address(x) != (char *) NULL ? get_address(x) : "(unknown address family)")

static struct event *evlist = (struct event *) NULL;


/*******************************************************************************/
/*
 * This routine returns a user readable form of an address contained in a
 * sockaddr structure. The return value CAN be a statically allocated
 * object (as is the case with inet_ntoa()).
 * Used in combination with 'PRINTADDRESS which is used in 
 * ipsec_doi*.c, state.c and demux.c.
 */
char * get_address(struct sockaddr sa)
{
  struct sockaddr_in sin;

  switch (sa.sa_family)
    {
    case AF_INET:
      bcopy(&sa, &sin, sizeof(sa));
      return inet_ntoa(sin.sin_addr);
      
    default:
      return (char *) NULL;
    }
}



/*******************************************************************************/
/*
 * Return a port (if applicable), from a struct sockaddr. If not applicable,
 * return -1.
 * Called almost everywhere...
 */
int get_port(struct sockaddr sa)
{
  struct sockaddr_in sin;
    
  switch (sa.sa_family)
    {
    case AF_INET:
      bcopy(&sa, &sin, sizeof(sa));
      return ntohs(sin.sin_port);
      
    default:
      return -1;
    }
}



/*******************************************************************************/
/*
 * This routine places an event in the event list.
 */
void event_schedule(int      type, 
		    time_t   tm, 
		    void    *ptr, 
		    int      ptrlen)
{
  struct event  *ev;
  struct event  *evt;
  struct state  *st;
  time_t tttv, tttv1;
  
  ev = (struct event *) calloc(1, sizeof(struct event));
  if (ev == (struct event *) NULL)
    exit_log("calloc() failed in event_schedule()", 0, 0, 0);
  
  tttv = time((time_t *) NULL);
  ev->ev_type = type;
  ev->ev_time = tm + time((time_t *) NULL);
  tttv1 = ev->ev_time;
  ev->ev_ptr = ptr;
  ev->ev_ptr_len = ptrlen;
  
  /* 
   * If the event is a retransmission, put a backpointer to the event
   * in the state object, so we can delete the event if we receive
   * a reply.
   * Same if event is a cleanup.
   */
  if ((type == EVENT_RETRANSMIT) || (type == EVENT_CLEANUP) ||
      (type == EVENT_SA_EXPIRE))
    {
      st = (struct state *) ptr;
      st->st_event = ev;
    }
  
  switch (ev->ev_type)
    {
    case EVENT_REINIT_SECRET:
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
       fprintf(stdout, "event_schedule: EVENT_REINIT_SECRET timenow=%s\n",
 		ctime(&tttv));
#endif
	break;
    case EVENT_RETRANSMIT:
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "event_schedule: EVENT_RETRANSMIT timenow=%s\n",
 		ctime(&tttv));
#endif
	break;
    case EVENT_CLEANUP:
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "event_schedule: EVENT_CLEANUP timenow=%s\n",
 		ctime(&tttv));
#endif
	break;
    case EVENT_SA_EXPIRE:
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "event_schedule: EVENT_SA_EXPIRE timenow=%s\n",
 		ctime(&tttv));
#endif
	break;
   default:
#ifdef DEBUG
  if(arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "event_schedule: ***ERROR - unrecognized event\n");
#endif
	break;
    }
  
  if (evlist == (struct event *) NULL) /* No other events */
    {
      evlist = ev;
      return;
    }
  
  /* We're installing the most close-to-happen event */
  if (evlist->ev_time >= ev->ev_time)
    {
      ev->ev_next = evlist;
      evlist = ev;
      return;
    }
  
  for (evt = evlist; evt->ev_next != (struct event *) NULL;
       evt = evt->ev_next)
    if (evt->ev_next->ev_time >= ev->ev_time)
      break;
  
  ev->ev_next = evt->ev_next;
  evt->ev_next = ev;

}



/*******************************************************************************/
/*
 * Handle the first event on the list.
 */
void event_handle(int   kernelfd, 
		  int   sock)
{
  time_t              tm;
  struct event       *ev;
  struct state       *st;
  struct sockaddr     sa;
  struct sockaddr_in  sin;
  time_t tttv;
  
#ifdef DEBUG_IN_OUT
  in_out_functions(" E event_handle | in timer.c");
#endif
  ev = evlist;
  
  if (ev == (struct event *) NULL)    /* Just paranoid */
    {
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "empty event list, yet we're called\n");
#endif
      return;
    }
  
  tm = time((time_t *) NULL);
  
  if (tm < ev->ev_time)
    {
#ifdef DEBUG
      if (arg_verbose & DEBUG_VERBOSE)
	fprintf(stdout, "called while no event expired (%u/%u, %d)\n", tm, ev->ev_time, ev->ev_type);
#endif
	/*
	 * This will happen if the most close-to-expire event was
	 * a retransmission or cleanup, and we received a packet
	 * at the same time as the event expired. Due to the processing
	 * order in call_server(), the packet processing will happen first, 
	 * and the event will be removed.
	 */
      return;
    }

  evlist = evlist->ev_next;		/* Ok, we'll handle this event */
    
  switch (ev->ev_type)
    {
    case EVENT_REINIT_SECRET:
      get_rnd_bytes(secret_of_the_day, SECRET_VALUE_LENGTH);
      event_schedule(EVENT_REINIT_SECRET, EVENT_REINIT_SECRET_DELAY,
		     NULL, 0);
      break;
      
    case EVENT_RETRANSMIT:
      st = (struct state *) ev->ev_ptr;
      sa = st->st_peer;
      if (st->st_retransmit >= MAXIMUM_RETRANSMISSIONS)
	{
#ifdef DEBUG
	  fprintf(stdout, 
	      "max number of retransmissions(%d) reached for %s, port %d\n",
	      st->st_retransmit, PRINTADDRESS(sa), get_port(sa));
#endif
	  
/* For failed negotiation, send fake SADBM_UPDATE to kernel 
 *	so kernel can delete destination address from address array
 */
	  kernel_update_ng(kernelfd, st);
	  delete_state(st);
	  free_state(st);

#ifdef WIT
/* WIT - Exit if max number of retransmissions reached */
        exit_log("Max number of retransmissions exceeded (%d)", st->st_retransmit, 0, 0);
#endif WIT
	  
	  /* XXX Send some message to the kernel */
	  break;
	}
      else
	st->st_retransmit++;
      
      if (sendto(sock, st->st_packet, st->st_packet_len, 0,
		 (struct sockaddr *) &(st->st_peer), 
		 sizeof(st->st_peer)) != st->st_packet_len)
	log(1, "sendto() failed in event_handle, for %s, port %d",
	    PRINTADDRESS(sa), get_port(sa), 0);
      
      event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
      break;
      
    case EVENT_SA_EXPIRE:
      /* XXX Do something with kernelfd */
      st = (struct state *) ev->ev_ptr;
      sa = st->st_peer;
  tttv = time((time_t *) NULL);
      log(0, "SA expired for %s, port %d msgid=%x", PRINTADDRESS(sa), 
	  get_port(sa), ntohl(st->st_msgid));
      log(0, "SA expired for %s, port %d time=%s", PRINTADDRESS(sa), 
	  get_port(sa), ctime(&tttv));
#ifdef TEST_KERN
    fprintf(stdout, "\tSPI    (%d) :\t", st->st_spi_len);
      format_dump(st->st_spi, st->st_spi_len);
#endif TEST_KERN
      delete_state(st);
      free_state(st);
      break;
      
    case EVENT_CLEANUP:
      st = (struct state *) ev->ev_ptr;
      sa = st->st_peer;
  tttv = time((time_t *) NULL);
      log(0, "responder state expired for %s, port %d msgid=%x",
	  PRINTADDRESS(sa), get_port(sa), ntohl(st->st_msgid));
      log(0, "responder state expired for %s, port %d time=%s",
	  PRINTADDRESS(sa), get_port(sa), ctime(&tttv));
#ifdef TEST_KERN
    fprintf(stdout, "\tSPI    (%d) :\t", st->st_spi_len);
      format_dump(st->st_spi, st->st_spi_len);
#endif TEST_KERN
      delete_state(st);		/* Remove from state table */
      free_state(st);		/* Free the memory */
      break;
      
    default:
      log(0, "unknown event %d expired, ignoring", ev->ev_type, 0, 0);
    }
  
  free(ev);
#ifdef DEBUG_IN_OUT
  in_out_functions(" Q event_handle | in timer.c");
#endif
}



/*******************************************************************************/
/*
 * Return the time until the next event in the queue
 * expires, -1 if no jobs in queue.
 */
int32_t next_event(void)
{
  time_t tm;
  
  if (evlist == (struct event *) NULL)
    return -1;
  
  tm = time((time_t*) NULL);
  
  if (evlist->ev_time - tm <= 0)
    return 0;
  else
    return (evlist->ev_time - tm);
}



/*******************************************************************************/
/*
 * Delete an event.
 */
void delete_event(struct state *st)
{
  struct event *ev;
  
  if (st->st_event == (struct event *) NULL)
    return;
  
  if (evlist == (struct event *) NULL)
    return;
    
  if (st->st_event == evlist)
    {
      if (evlist->ev_type == EVENT_RETRANSMIT)
	st->st_retransmit = 0;
      evlist = evlist->ev_next;
      free(st->st_event);
      st->st_event = (struct event *) NULL;
      return;
    }
  
  for (ev = evlist; ev->ev_next != (struct event *) NULL;
       ev = ev->ev_next)
    if (ev->ev_next == st->st_event)
      {
	if (ev->ev_type == EVENT_RETRANSMIT)
	  st->st_retransmit = 0;
	ev->ev_next = ev->ev_next->ev_next;
	free(st->st_event);
	st->st_event = (struct event *) NULL;
	return;
      }
  
#ifdef DEBUG
  log(0, "event %d to be deleted not found", st->st_event->ev_type, 0, 0);
#endif
}
