/*-
 * Copyright (c) 2006 Michael Richardson <mcr@xelerance.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Effort sponsored in part by Hifn inc.
 */

/*
 * this program maps the hifn vulcan Public Key engine and runs various
 * test case on it.
 *
 */

#include <stdio.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dev/hifn/hifn7751reg.h>

unsigned char * mapvulcanpk(void)
{
	unsigned char *mapping;
	int fd=open("/dev/vulcanpk", O_RDWR);
	
	if(fd == -1) {
	  perror("open");
	  exit(6);
	}

	/* HIFN_1_PUB_MEMEND */
	mapping = mmap(NULL, 4096, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);
	
	if(mapping == NULL) {
		perror("mmap");
		exit(4);
	}
	
	return mapping;
}

#include "hexdump.c"

void print_status(u_int32_t stat)
{
	printf("status: %08x ", stat);
	if(stat & HIFN_PUBSTS_DONE) {
		printf("done ");
	}
	if(stat & HIFN_PUBSTS_CARRY) {
		printf("carry ");
	}
	if(stat & HIFN_PUBSTS_FIFO_EMPTY) {
		printf("empty ");
	}
	if(stat & HIFN_PUBSTS_FIFO_FULL) {
		printf("full ");
	}
	if(stat & HIFN_PUBSTS_FIFO_OVFL) {
		printf("overflow ");
	}
	if(stat & HIFN_PUBSTS_FIFO_WRITE) {
		printf("write=%d ", (stat & HIFN_PUBSTS_FIFO_WRITE)>>16);
	}
	if(stat & HIFN_PUBSTS_FIFO_READ) {
		printf("read=%d ", (stat & HIFN_PUBSTS_FIFO_READ)>>24);
	}
	printf("\n");
}


#define PUB_WORD(offset) *(volatile u_int32_t *)(&mapping[offset])
#define PUB_WORD_WRITE(offset, value) printf("write-1 %04x = %08x\n", offset, value), PUB_WORD(offset)=value

static __inline void
bus_space_write_barrier(void)
{
  __asm __volatile("" : : : "memory");
}

#ifdef ENHANCED_MODE
inline void write_pkop(unsigned char *mapping,
		       u_int32_t oplen, u_int32_t op)
{
	volatile u_int32_t *opfifo;
	static int entry=0;

	opfifo = (volatile u_int32_t *)(mapping+HIFN_1_PUB_FIFO_OPLEN);

	if(entry >= 32) entry=0;

	printf("oplen=%08x op=%08x (entry=%d)\n", oplen, op, entry/2);
	bus_space_write_barrier();
	opfifo[entry++]=oplen;
	bus_space_write_barrier();
	opfifo[entry++]=op;
	bus_space_write_barrier();
}
#else
inline void write_pkop(unsigned char *mapping,
		       u_int32_t oplen, u_int32_t op)
{
	volatile u_int32_t *opfifo;

	opfifo = (volatile u_int32_t *)(mapping+HIFN_1_PUB_OPLEN);

	{
	  unsigned int opcode,m,b,a;
	  opcode = (op>>24)&0xff;
	  m      = (op>>16)&0xff;
	  b      = (op>>8)&0xff;
	  a      = op & 0xff;
	  
	  op = (opcode << 18)|(m<<12)|(b<<6)|(a<<0);
	}
	{
	  unsigned int red,exp,mod;
	  red = (oplen >> 24)&0xff;
	  exp = (oplen >> 8)&0xfff;
	  mod = (oplen >> 0)&0xff;
	  
	  oplen = ((red&0xf) << 18) | ((exp&0x7ff) << 7) | (mod & 0x7f);
	}
	printf("oplen=%08x op=%08x\n", oplen, op);
	opfifo[0]=oplen;
	opfifo[1]=op;
}
#endif

#define PKVALUE_BITS 3072
#define PKVALUE_LEN  (PKVALUE_BITS/8)

struct pkprogram {
	unsigned char *aValues[16];
	unsigned char *bValues[16];
	unsigned char *modValue;
	unsigned char *oValue;
	u_int64_t pk_program[16];
};

int pk_verbose_execute=0;

void sanitycheckpk(unsigned char *mapping)
{
	u_int32_t stat;

#ifdef ENHANCED_MODE
	printf("running in enhanced mode\n");
#else
	printf("running in compat mode\n");
#endif

#if 0
	printf("sanitycheck: %p size=%d\n", mapping, HIFN_1_PUB_MEMEND);
	hexdump(mapping, 0, 256);
#endif

#ifdef ENHANCED_MODE
	PUB_WORD_WRITE(HIFN_1_PUB_MODE, PUB_WORD(HIFN_1_PUB_MODE)|HIFN_PKMODE_ENHANCED);
#else
	PUB_WORD_WRITE(HIFN_1_PUB_MODE, PUB_WORD(HIFN_1_PUB_MODE)&~HIFN_PKMODE_ENHANCED);
#endif

	/* enable RNG again in case we reset things */
	PUB_WORD_WRITE(HIFN_1_RNG_CONFIG, PUB_WORD(HIFN_1_RNG_CONFIG) | HIFN_RNGCFG_ENA);

	printf("setting to PK mode=%08x\n", PUB_WORD(HIFN_1_PUB_MODE));

#if 1
	/* clear out public key memory */
	memset(mapping+HIFN_1_PUB_MEM, 0, HIFN_1_PUB_MEMEND-HIFN_1_PUB_MEM);

	/* verify that memory cleared */
	{
		int i;
		volatile unsigned int *n = (volatile unsigned int *)&mapping[HIFN_1_PUB_MEM];
		for(i=HIFN_1_PUB_MEM; i< HIFN_1_PUB_MEMEND; i+=4) {
			if(*n!=0) {
				printf("failed to clear pubkey memory at offset=%04x (%04x,%04x) (value=%02x)\n", i, HIFN_1_PUB_MEM, HIFN_1_PUB_MEMEND, mapping[i]);
				exit(4);
			}
			n++;
		}
	}
	printf("public key memory cleared properly\n");
#endif

	/* clear out PUBLIC DONE */
	PUB_WORD_WRITE(HIFN_1_PUB_IEN, 0); /* disable interrupts */
	bus_space_write_barrier();

	print_status(PUB_WORD(HIFN_1_PUB_STATUS));
	PUB_WORD_WRITE(HIFN_1_PUB_STATUS, HIFN_PUBSTS_DONE | HIFN_PUBSTS_FIFO_EMPTY);
	print_status(PUB_WORD(HIFN_1_PUB_STATUS));

	/* now setup memory for 3072 bit add */
	//PUB_WORD_WRITE(HIFN_1_PUB_IEN, HIFN_PUBIEN_DONE);

	bus_space_write_barrier();
	/* A = 1, offset = 0 */
	PUB_WORD_WRITE(HIFN_1_PUB_MEM, 0x000000001);

	/* B = 2, offset = (3072/8)/64 = 6 */
	PUB_WORD_WRITE(HIFN_1_PUB_MEM + (3072/8), 0x00000002);

	/* initalize result area to nonsense */
	PUB_WORD_WRITE(HIFN_1_PUB_MEM + (2*3072/8), 0x00000010);
	
	bus_space_write_barrier();

	/* perform addition */
	write_pkop(mapping, 
		   /* sizes are ModLen=48(*32=1536), EXP_len=0, RED_len=0 */
		   (0<<24)|(0<<8)|(48),
		   /* opcode 0001(ADD), with A=0, B=1 (6), M=4 (24) */
		   (1<<24)|(24<<16)|(3<<8)|(0<<0));

	bus_space_write_barrier();

	print_status(PUB_WORD(HIFN_1_PUB_STATUS));

	{
		int count=5;
		volatile int i;

		while(--count>0 &&
		      ((i = PUB_WORD(HIFN_1_PUB_STATUS)) & HIFN_PUBSTS_DONE) != 1) {
			print_status(i);
			hexdump(mapping,HIFN_1_PUB_MEM+(0*384), 16);
			hexdump(mapping,HIFN_1_PUB_MEM+(1*384), 16);
			hexdump(mapping,HIFN_1_PUB_MEM+(2*384), 16);
			sleep(1);
		}
		if(count==0) {
			printf("post bar-1 register dump\n");
			hexdump(mapping, 0, HIFN_1_PUB_MEM);

			printf("failed\n");
			exit(2);
		}
	}

	print_status(PUB_WORD(HIFN_1_PUB_STATUS));
	printf("pubkey result\n");
	hexdump(mapping,HIFN_1_PUB_MEM+(0*384), 16);
	hexdump(mapping,HIFN_1_PUB_MEM+(1*384), 16);
	hexdump(mapping,HIFN_1_PUB_MEM+(2*384), 16);
}

void do_diffiehellman(unsigned char *mapping)
{
}

void do_rsasignature(unsigned char *mapping)
{
}

void do_rsaverify(unsigned char *mapping)
{
}



int main(int argc, char *argv[])
{
	unsigned char *vulcanpkreg;

	vulcanpkreg = mapvulcanpk();
	if(vulcanpkreg == NULL) exit(5);

	sanitycheckpk(vulcanpkreg);

	do_diffiehellman(vulcanpkreg);
	do_rsasignature(vulcanpkreg);
	do_rsaverify(vulcanpkreg);

	exit(0);
}
	
