/*

  silcauth.c

  Author: Pekka Riikonen <priikone@silcnet.org>

  Copyright (C) 2001 - 2007 Pekka Riikonen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

*/
/* $Id: silcauth.c,v 1.42 2007/07/21 17:18:20 priikone Exp $ */

#include "silc.h"
#include "silcauth.h"

/******************************************************************************

                           Authentication Payload

******************************************************************************/

/* Authentication Payload structure */
struct SilcAuthPayloadStruct {
  SilcStack stack;
  unsigned char *random_data;
  unsigned char *auth_data;
  SilcUInt16 auth_len;
  SilcUInt16 len;
  SilcUInt16 auth_method;
  SilcUInt16 random_len;
};

/* Parses and returns Authentication Payload */

SilcAuthPayload silc_auth_payload_parse(SilcStack stack,
					const unsigned char *data,
					SilcUInt32 data_len)
{
  SilcBufferStruct buffer;
  SilcAuthPayload newp;
  int ret;

  SILC_LOG_DEBUG(("Parsing Authentication Payload"));

  silc_buffer_set(&buffer, (unsigned char *)data, data_len);

  if (stack)
    stack = silc_stack_alloc(0, stack);

  newp = silc_scalloc(stack, 1, sizeof(*newp));
  if (!newp) {
    silc_stack_free(stack);
    return NULL;
  }
  newp->stack = stack;

  /* Parse the payload */
  ret = silc_buffer_sunformat(stack, &buffer,
			      SILC_STR_UI_SHORT(&newp->len),
			      SILC_STR_UI_SHORT(&newp->auth_method),
			      SILC_STR_UI16_NSTRING_ALLOC(&newp->random_data,
							  &newp->random_len),
			      SILC_STR_UI16_NSTRING_ALLOC(&newp->auth_data,
							  &newp->auth_len),
			      SILC_STR_END);
  if (ret == -1) {
    silc_sfree(stack, newp);
    silc_stack_free(stack);
    return NULL;
  }

  if (newp->len != silc_buffer_len(&buffer) ||
      newp->random_len + newp->auth_len > silc_buffer_len(&buffer) - 8) {
    silc_auth_payload_free(newp);
    return NULL;
  }

  /* Authentication data must be provided */
  if (newp->auth_len < 1)  {
    silc_auth_payload_free(newp);
    return NULL;
  }

  /* If password authentication, random data must not be set */
  if (newp->auth_method == SILC_AUTH_PASSWORD && newp->random_len) {
    silc_auth_payload_free(newp);
    return NULL;
  }

  /* If public key authentication, random data must be at least 128 bytes */
  if (newp->auth_method == SILC_AUTH_PUBLIC_KEY && newp->random_len < 128) {
    silc_auth_payload_free(newp);
    return NULL;
  }

  return newp;
}

/* Encodes authentication payload into buffer and returns it */

SilcBuffer silc_auth_payload_encode(SilcStack stack,
				    SilcAuthMethod method,
				    const unsigned char *random_data,
				    SilcUInt16 random_len,
				    const unsigned char *auth_data,
				    SilcUInt16 auth_len)
{
  SilcBuffer buffer;
  SilcUInt32 len;
  unsigned char *autf8 = NULL;
  SilcUInt32 autf8_len;

  SILC_LOG_DEBUG(("Encoding Authentication Payload"));

  /* Passphrase MUST be UTF-8 encoded, encode if it is not */
  if (method == SILC_AUTH_PASSWORD && !silc_utf8_valid(auth_data, auth_len)) {
    autf8_len = silc_utf8_encoded_len(auth_data, auth_len, 0);
    if (!autf8_len)
      return NULL;
    autf8 = silc_scalloc(stack, autf8_len, sizeof(*autf8));
    auth_len = silc_utf8_encode(auth_data, auth_len, 0, autf8, autf8_len);
    auth_data = (const unsigned char *)autf8;
  }

  len = 2 + 2 + 2 + random_len + 2 + auth_len;
  buffer = silc_buffer_salloc_size(stack, len);
  if (!buffer) {
    silc_sfree(stack, autf8);
    return NULL;
  }

  silc_buffer_sformat(stack, buffer,
		      SILC_STR_UI_SHORT(len),
		      SILC_STR_UI_SHORT(method),
		      SILC_STR_UI_SHORT(random_len),
		      SILC_STR_UI_XNSTRING(random_data, random_len),
		      SILC_STR_UI_SHORT(auth_len),
		      SILC_STR_UI_XNSTRING(auth_data, auth_len),
		      SILC_STR_END);

  silc_sfree(stack, autf8);
  return buffer;
}

/* Frees authentication payload. */

void silc_auth_payload_free(SilcAuthPayload payload)
{
  if (payload) {
    SilcStack stack = payload->stack;

    if (payload->random_data) {
      memset(payload->random_data, 0, payload->random_len);
      silc_sfree(stack, payload->random_data);
    }
    if (payload->auth_data) {
      memset(payload->auth_data, 0, payload->auth_len);
      silc_sfree(stack, payload->auth_data);
    }

    silc_sfree(stack, payload);
    silc_stack_free(stack);
  }
}

/* Get authentication method */

SilcAuthMethod silc_auth_get_method(SilcAuthPayload payload)
{
  return payload->auth_method;
}

/* Get the public data from the auth payload. */

unsigned char *silc_auth_get_public_data(SilcAuthPayload payload,
					 SilcUInt32 *pubdata_len)
{
  if (pubdata_len)
    *pubdata_len = (SilcUInt32)payload->random_len;

  return payload->random_data;
}

/* Get the authentication data. If this is passphrase it is UTF-8 encoded. */

unsigned char *silc_auth_get_data(SilcAuthPayload payload,
				  SilcUInt32 *auth_len)
{
  if (auth_len)
    *auth_len = (SilcUInt32)payload->auth_len;

  return payload->auth_data;
}

/******************************************************************************

                           Authentication Routines

******************************************************************************/

/* Encodes the authentication data for hashing and signing as the protocol
   dictates. */

static unsigned char *
silc_auth_public_key_encode_data(SilcStack stack,
				 SilcPublicKey public_key,
				 const unsigned char *randomdata,
				 SilcUInt32 random_len, const void *id,
				 SilcIdType type, SilcUInt32 *ret_len)
{
  SilcBuffer buf;
  unsigned char *pk, id_data[32], *ret;
  SilcUInt32 pk_len, id_len;

  pk = silc_pkcs_public_key_encode(stack, public_key, &pk_len);
  if (!pk)
    return NULL;

  if (!silc_id_id2str(id, type, id_data, sizeof(id_data), &id_len)) {
    silc_free(pk);
    return NULL;
  }

  buf = silc_buffer_salloc_size(stack, random_len + id_len + pk_len);
  if (!buf) {
    silc_free(pk);
    return NULL;
  }
  silc_buffer_sformat(stack, buf,
		      SILC_STR_UI_XNSTRING(randomdata, random_len),
		      SILC_STR_UI_XNSTRING(id_data, id_len),
		      SILC_STR_UI_XNSTRING(pk, pk_len),
		      SILC_STR_END);

  ret = silc_buffer_steal(buf, ret_len);

  silc_buffer_sfree(stack, buf);
  silc_sfree(stack, pk);

  return ret;
}

typedef struct {
  SilcStack stack;
  unsigned char *pubdata;
  SilcUInt32 pubdata_len;
  SilcAuthGenerated generated;
  void *context;
} *SilcAuthGenerateContext;

/* Signature callback */

static void
silc_auth_public_key_auth_generate_cb(SilcBool success,
				      const unsigned char *signature,
				      SilcUInt32 signature_len,
				      void *context)
{
  SilcAuthGenerateContext a = context;
  SilcStack stack = a->stack;
  SilcBuffer buf;

  if (!success) {
    a->generated(NULL, context);
    silc_sfree(stack, a->pubdata);
    silc_sfree(stack, a);
    silc_stack_free(stack);
    return;
  }

  /* Encode Authentication Payload */
  buf = silc_auth_payload_encode(stack, SILC_AUTH_PUBLIC_KEY, a->pubdata,
				 a->pubdata_len, signature, signature_len);

  a->generated(buf, context);

  silc_buffer_sfree(stack, buf);
  silc_sfree(stack, a->pubdata);
  silc_sfree(stack, a);
  silc_stack_free(stack);
}

/* Generates Authentication Payload with authentication data. This is used
   to do public key based authentication. This generates the random data
   and the actual authentication data. Returns NULL on error. */

SilcAsyncOperation
silc_auth_public_key_auth_generate(SilcPublicKey public_key,
				   SilcPrivateKey private_key,
				   SilcRng rng, SilcHash hash,
				   const void *id, SilcIdType type,
				   SilcAuthGenerated generated,
				   void *context)
{
  unsigned char randomdata[256];

  /* Get random data */
  if (rng)
    silc_rng_get_rn_data(rng, sizeof(randomdata), randomdata,
			 sizeof(randomdata));
  else
    silc_rng_global_get_rn_data(rng, sizeof(randomdata), randomdata,
				sizeof(randomdata));

  return silc_auth_public_key_auth_generate_wpub(public_key, private_key,
						 randomdata, sizeof(randomdata),
						 hash, rng, id, type, generated,
						 context);
}

/* Generates Authentication Payload with authentication data. This is used
   to do public key based authentication. This generates the random data
   and the actual authentication data. Returns NULL on error. */

SilcAsyncOperation
silc_auth_public_key_auth_generate_wpub(SilcPublicKey public_key,
					SilcPrivateKey private_key,
					const unsigned char *pubdata,
					SilcUInt32 pubdata_len,
					SilcHash hash,
					SilcRng rng,
					const void *id, SilcIdType type,
					SilcAuthGenerated generated,
					void *context)
{
  SilcAuthGenerateContext a;
  SilcAsyncOperation op;
  unsigned char *tmp;
  SilcUInt32 tmp_len;
  SilcStack stack;

  SILC_LOG_DEBUG(("Generating Authentication Payload with data"));

  /* We use the Crypto Toolkit's stack since we're doing crypto */
  stack = silc_stack_alloc(2048, silc_crypto_stack());

  a = silc_scalloc(stack, 1, sizeof(*a));
  if (!a) {
    generated(NULL, context);
    return NULL;
  }
  a->stack = stack;

  /* Encode the auth data */
  tmp = silc_auth_public_key_encode_data(stack, public_key, pubdata,
					 pubdata_len, id, type, &tmp_len);
  if (!tmp) {
    silc_sfree(stack, a);
    silc_stack_free(stack);
    generated(NULL, context);
    return NULL;
  }

  a->pubdata = silc_smemdup(stack, pubdata, pubdata_len);
  if (!a->pubdata) {
    memset(tmp, 0, tmp_len);
    silc_sfree(stack, tmp);
    silc_sfree(stack, a);
    silc_stack_free(stack);
    generated(NULL, context);
    return NULL;
  }

  /* Compute the hash and the signature. */
  op = silc_pkcs_sign(private_key, tmp, tmp_len, TRUE, hash, rng,
		      silc_auth_public_key_auth_generate_cb, a);

  memset(tmp, 0, tmp_len);
  silc_sfree(stack, tmp);

  return op;
}

/* Verifies the authentication data. */

SilcAsyncOperation
silc_auth_public_key_auth_verify(SilcAuthPayload payload,
				 SilcPublicKey public_key,
				 SilcHash hash,
				 const void *id,
				 SilcIdType type,
				 SilcAuthResultCb result,
				 void *context)
{
  SilcAsyncOperation op;
  unsigned char *tmp;
  SilcUInt32 tmp_len;

  SILC_LOG_DEBUG(("Verifying authentication data"));

  /* Encode auth data */
  tmp = silc_auth_public_key_encode_data(payload->stack,
					 public_key, payload->random_data,
					 payload->random_len,
					 id, type, &tmp_len);
  if (!tmp) {
    SILC_LOG_DEBUG(("Authentication failed"));
    result(FALSE, context);
    return NULL;
  }

  /* Verify the authentication data */
  op = silc_pkcs_verify(public_key, payload->auth_data,
			payload->auth_len, tmp, tmp_len, hash,
			result, context);

  memset(tmp, 0, tmp_len);
  silc_sfree(payload->stack, tmp);

  return op;
}

/* Same as above but the payload is not parsed yet. This will parse it. */

SilcAsyncOperation
silc_auth_public_key_auth_verify_data(const unsigned char *payload,
				      SilcUInt32 payload_len,
				      SilcPublicKey public_key,
				      SilcHash hash,
				      const void *id,
				      SilcIdType type,
				      SilcAuthResultCb result,
				      void *context)
{
  SilcAsyncOperation op;
  SilcAuthPayload auth_payload;

  auth_payload = silc_auth_payload_parse(silc_crypto_stack(), payload,
					 payload_len);
  if (!auth_payload) {
    SILC_LOG_DEBUG(("Authentication failed"));
    result(FALSE, context);
    return NULL;
  }

  op = silc_auth_public_key_auth_verify(auth_payload, public_key, hash,
					id, type, result, context);

  silc_auth_payload_free(auth_payload);

  return op;
}

/* Verifies the authentication data directly from the Authentication
   Payload. Supports all authentication methods. If the authentication
   method is passphrase based then the `auth_data' and `auth_data_len'
   are the passphrase and its length. If the method is public key
   authentication then the `auth_data' is the SilcPublicKey and the
   `auth_data_len' is ignored. */

SilcAsyncOperation
silc_auth_verify(SilcAuthPayload payload, SilcAuthMethod auth_method,
		 const void *auth_data, SilcUInt32 auth_data_len,
		 SilcHash hash, const void *id, SilcIdType type,
		 SilcAuthResultCb result, void *context)
{
  SILC_LOG_DEBUG(("Verifying authentication"));

  if (!payload || auth_method != payload->auth_method) {
    result(FALSE, context);
    return NULL;
  }

  switch (payload->auth_method) {
  case SILC_AUTH_NONE:
    /* No authentication */
    SILC_LOG_DEBUG(("No authentication required"));
    result(TRUE, context);
    return NULL;

  case SILC_AUTH_PASSWORD:
    /* Passphrase based authentication. The `pkcs', `hash', `id' and `type'
       arguments are not needed. */

    /* Sanity checks */
    if ((payload->auth_len == 0) || !auth_data ||
	payload->auth_len != auth_data_len)
      break;

    if (!memcmp(payload->auth_data, auth_data, auth_data_len)) {
      SILC_LOG_DEBUG(("Passphrase Authentication successful"));
      result(TRUE, context);
      return NULL;
    }
    break;

  case SILC_AUTH_PUBLIC_KEY:
    /* Public key based authentication */
    return silc_auth_public_key_auth_verify(payload, (SilcPublicKey)auth_data,
					    hash, id, type, result, context);
    break;

  default:
    break;
  }

  SILC_LOG_DEBUG(("Authentication failed"));
  result(FALSE, context);

  return NULL;
}

/* Same as above but parses the authentication payload before verify. */

SilcAsyncOperation
silc_auth_verify_data(const unsigned char *payload,
		      SilcUInt32 payload_len,
		      SilcAuthMethod auth_method,
		      const void *auth_data,
		      SilcUInt32 auth_data_len, SilcHash hash,
		      const void *id, SilcIdType type,
		      SilcAuthResultCb result, void *context)
{
  SilcAsyncOperation op;
  SilcAuthPayload auth_payload;

  auth_payload = silc_auth_payload_parse(silc_crypto_stack(), payload,
					 payload_len);
  if (!auth_payload || (auth_payload->auth_len == 0)) {
    result(FALSE, context);
    return NULL;
  }

  op = silc_auth_verify(auth_payload, auth_method, auth_data, auth_data_len,
			 hash, id, type, result, context);

  silc_auth_payload_free(auth_payload);

  return op;
}

/******************************************************************************

                            Key Agreement Payload

******************************************************************************/

/* The Key Agreement protocol structure */
struct SilcKeyAgreementPayloadStruct {
  SilcUInt16 hostname_len;
  unsigned char *hostname;
  SilcUInt16 protocol;
  SilcUInt16 port;
};

/* Parses and returns an allocated Key Agreement payload. */

SilcKeyAgreementPayload
silc_key_agreement_payload_parse(const unsigned char *payload,
				 SilcUInt32 payload_len)
{
  SilcBufferStruct buffer;
  SilcKeyAgreementPayload newp;
  int ret;

  SILC_LOG_DEBUG(("Parsing Key Agreement Payload"));

  newp = silc_calloc(1, sizeof(*newp));
  if (!newp)
    return NULL;

  /* Parse the payload */
  silc_buffer_set(&buffer, (unsigned char *)payload, payload_len);
  ret = silc_buffer_unformat(&buffer,
			     SILC_STR_UI16_NSTRING_ALLOC(&newp->hostname,
							 &newp->hostname_len),
			     SILC_STR_UI_SHORT(&newp->protocol),
			     SILC_STR_UI_SHORT(&newp->port),
			     SILC_STR_END);
  if (ret == -1 || newp->hostname_len > silc_buffer_len(&buffer) - 6) {
    silc_free(newp);
    return NULL;
  }

  return newp;
}

/* Encodes the Key Agreement protocol and returns the encoded buffer */

SilcBuffer silc_key_agreement_payload_encode(const char *hostname,
					     SilcUInt16 protocol,
					     SilcUInt16 port)
{
  SilcBuffer buffer;
  SilcUInt32 len = hostname ? strlen(hostname) : 0;

  SILC_LOG_DEBUG(("Encoding Key Agreement Payload"));

  buffer = silc_buffer_alloc_size(2 + len + 4);
  if (!buffer)
    return NULL;
  silc_buffer_format(buffer,
		     SILC_STR_UI_SHORT(len),
		     SILC_STR_UI_XNSTRING(hostname, len),
		     SILC_STR_UI_SHORT(protocol),
		     SILC_STR_UI_SHORT(port),
		     SILC_STR_END);

  return buffer;
}

/* Frees the Key Agreement protocol */

void silc_key_agreement_payload_free(SilcKeyAgreementPayload payload)
{
  if (payload) {
    silc_free(payload->hostname);
    silc_free(payload);
  }
}

/* Returns the hostname in the payload */

char *silc_key_agreement_get_hostname(SilcKeyAgreementPayload payload)
{
  return payload->hostname;
}

/* Returns the protocol in the payload */

SilcUInt16 silc_key_agreement_get_protocol(SilcKeyAgreementPayload payload)
{
  return payload->protocol;
}

/* Returns the port in the payload */

SilcUInt16 silc_key_agreement_get_port(SilcKeyAgreementPayload payload)
{
  return payload->port;
}
